﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_Abort.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os.h>
#include <nnt/nntest.h>
#include <algorithm>

#include <hos/nvhdcp_up.h>

#include "../Shared/testHdcp_Config.h"
#include "../Shared/testHdcp_Helper.h"
#include "../Shared/testHdcp_Util-hardware.nx.h"
#include "../Shared/HdcpRawTestInitializer.h"

/**
 * @brief   Check if the processing time is expected.
 */
TEST(RawDisconnected, ProcessingTime)
{
    nnt::hdcp::HdcpRawTestInitializer initializer;
    initializer.ShowWarningBasedOnHdmiConnection(false);
    bool isHdmiConnected = initializer.IsHdmiConnected();
    ASSERT_EQ(false, isHdmiConnected);

    HDCP_CLIENT_HANDLE handle;
    HDCP_RET_ERROR error;
    int64_t start = 0LL;
    int64_t end = 0LL;

    start = nnt::hdcp::GetCurrentMilliSeconds();
    error = hdcp_open(&handle, nnt::hdcp::HdcpDeviceId);
    end = nnt::hdcp::GetCurrentMilliSeconds();
    // Within 100 msec.
    EXPECT_GE(100, end - start);
    EXPECT_EQ(HDCP_RET_SUCCESS, error);

    // Disable HDCP and check if HDCP is disabled.
    start = nnt::hdcp::GetCurrentMilliSeconds();
    error = hdcp_enable(handle, false);
    end = nnt::hdcp::GetCurrentMilliSeconds();
    EXPECT_GE(100, end - start);
    EXPECT_EQ(HDCP_RET_SUCCESS, error);
    start = nnt::hdcp::GetCurrentMilliSeconds();
    error = hdcp_status(handle);
    end = nnt::hdcp::GetCurrentMilliSeconds();
    EXPECT_GE(100, end - start);
    EXPECT_EQ(HDCP_RET_NOT_PLUGGED, error);

    // Enable HDCP.
    start = nnt::hdcp::GetCurrentMilliSeconds();
    error = hdcp_enable(handle, true);
    end = nnt::hdcp::GetCurrentMilliSeconds();
    EXPECT_GE(100, end - start);
    EXPECT_EQ(HDCP_RET_SUCCESS, error);
    start = nnt::hdcp::GetCurrentMilliSeconds();
    error = hdcp_status(handle);
    end = nnt::hdcp::GetCurrentMilliSeconds();
    EXPECT_GE(100, end - start);
    EXPECT_EQ(HDCP_RET_NOT_PLUGGED, error);

    // The status of HDCP goes to "ENABLED" within 10 sec.
    for (int sec = 1; sec <= nnt::hdcp::AuthenticationTimeLimitSec; ++sec) {
        NN_LOG("Authenticating HDCP... %d sec.\n", sec);
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000));
        error = hdcp_status(handle);
        EXPECT_EQ(HDCP_RET_NOT_PLUGGED, error);
    }

    // Disable HDCP.
    start = nnt::hdcp::GetCurrentMilliSeconds();
    error = hdcp_enable(handle, false);
    end = nnt::hdcp::GetCurrentMilliSeconds();
    // My understanding is that HDCP can be disabled quickly against "enabled".
    // So the short time limit is set here.
    EXPECT_GE(100, end - start);
    EXPECT_EQ(HDCP_RET_SUCCESS, error);
    start = nnt::hdcp::GetCurrentMilliSeconds();
    error = hdcp_status(handle);
    end = nnt::hdcp::GetCurrentMilliSeconds();
    EXPECT_GE(100, end - start);
    EXPECT_EQ(HDCP_RET_NOT_PLUGGED, error);

    start = nnt::hdcp::GetCurrentMilliSeconds();
    error = hdcp_close(handle);
    end = nnt::hdcp::GetCurrentMilliSeconds();
    EXPECT_GE(100, end - start);
    EXPECT_EQ(HDCP_RET_SUCCESS, error);
}

/**
 * @brief   Check if state transition is working well.
 */
TEST(RawDisconnected, StateTransition)
{
    nnt::hdcp::HdcpRawTestInitializer initializer;
    initializer.ShowWarningBasedOnHdmiConnection(false);
    bool isHdmiConnected = initializer.IsHdmiConnected();
    ASSERT_EQ(false, isHdmiConnected);

    HDCP_CLIENT_HANDLE handle;
    HDCP_RET_ERROR error;
    int64_t start = 0LL;
    int64_t end = 0LL;
    bool isSent;

    error = hdcp_open(&handle, nnt::hdcp::HdcpDeviceId);
    ASSERT_EQ(HDCP_RET_SUCCESS, error);

    std::uintptr_t communicationBuffer;
    std::uintptr_t blockBuffer;
    typedef struct
    {
        nn::os::MessageQueueType communicationQueue;
        nn::os::MessageQueueType blockQueue;
    } HdcpTestStateTransitionType;
    HdcpTestStateTransitionType testType;
    nn::os::InitializeMessageQueue(&testType.communicationQueue, &communicationBuffer, 1);
    nn::os::InitializeMessageQueue(&testType.blockQueue, &blockBuffer, 1);
    std::uintptr_t message = reinterpret_cast<std::uintptr_t>(nullptr);
    nn::os::SendMessageQueue(&testType.blockQueue, message);

    auto handler = [](HDCP_CLIENT_HANDLE handle, NvU32 state, void* context)
    {
        NN_SDK_LOG("[%d] %s() handle:[%p] state:[%d] context:[%p] a\n", __LINE__, __FUNCTION__, handle, state, context);
        NN_UNUSED(handle);
        auto testType = reinterpret_cast<HdcpTestStateTransitionType*>(context);
        std::uintptr_t message;
        nn::os::ReceiveMessageQueue(&message, &testType->blockQueue);
        // Timeout should not occur here.
        bool isSent = nn::os::TimedSendMessageQueue(&testType->communicationQueue, state, nn::TimeSpan::FromMilliSeconds(1000));
        EXPECT_EQ(true, isSent);
        nn::os::SendMessageQueue(&testType->blockQueue, message);
    };

    error = hdcp_event_state_transit(handle, handler, &testType);
    EXPECT_EQ(HDCP_RET_SUCCESS, error);

    error = hdcp_init_asyncevents(handle);
    EXPECT_EQ(HDCP_RET_SUCCESS, error);

    // Clear all events.
    error = hdcp_enable(handle, true);
    EXPECT_EQ(HDCP_RET_SUCCESS, error);
    error = hdcp_enable(handle, false);
    EXPECT_EQ(HDCP_RET_SUCCESS, error);
    while (nn::os::TimedReceiveMessageQueue(&message, &testType.communicationQueue, nn::TimeSpan::FromMilliSeconds(1000)))
        ;

    // No state transition occurs.
    error = hdcp_enable(handle, true);
    EXPECT_EQ(HDCP_RET_SUCCESS, error);
    isSent = nn::os::TimedReceiveMessageQueue(&message, &testType.communicationQueue, nn::TimeSpan::FromMilliSeconds(1000));
    EXPECT_EQ(false, isSent);
    error = hdcp_enable(handle, false);
    EXPECT_EQ(HDCP_RET_SUCCESS, error);
    isSent = nn::os::TimedReceiveMessageQueue(&message, &testType.communicationQueue, nn::TimeSpan::FromMilliSeconds(1000));
    EXPECT_EQ(false, isSent);

    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000));

    error = hdcp_close(handle);
    ASSERT_EQ(HDCP_RET_SUCCESS, error);
}
