﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/grc/grc_Api.h>
#include <nn/grc/grc_Application.h>
#include <nn/capsrv/capsrv_AlbumAccess.h>

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/applet/applet_Apis.h>
#include <nn/grc/grc_ResultPrivate.h>
#include "testGrc_Macro.h"
#include "testGrc_MemoryManagement.h"

static const int Width = 1280;
static const int Height = 720;

#define NNT_GRC_INITIALIZE_FOR_APPLICATION(aruidValue)  \
    {                                                           \
        nn::applet::AppletResourceUserId nntGrcInitializeAruid = {static_cast<uint64_t>(aruidValue)};   \
        nn::sf::SharedPointer<nn::grcsrv::IMovieMaker> pMovieMakerProxy; \
        NNT_GRC_EXPECT_SUCCESS(nn::grc::CreateMovieMakerForApplication(&pMovieMakerProxy, nntGrcInitializeAruid, {0x010000000000B1C1ull})); \
        NNT_GRC_EXPECT_SUCCESS(nn::grc::InitializeForApplication(std::move(pMovieMakerProxy))); \
    }

#define NNT_GRC_SCOPED_INITIALIZE_FOR_APPLICATION(aruidValue)   \
    NNT_GRC_INITIALIZE_FOR_APPLICATION(aruidValue)              \
    NN_UTIL_SCOPE_EXIT{ nn::grc::FinalizeForApplication(); };

namespace {

    void CleanAlbum() NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::InitializeAlbumAccess());
        NN_UTIL_SCOPE_EXIT{ nn::capsrv::FinalizeAlbumAccess(); };

        for(nn::capsrv::AlbumStorageType s = 0; s < nn::capsrv::AlbumStorageCount; s++)
        {
            if(!nn::capsrv::IsAlbumMounted(s))
            {
                continue;
            }

            for(;;)
            {
                nn::capsrv::AlbumEntry entries[100];
                int count = 0;
                (void)nn::capsrv::GetAlbumFileList(&count, entries, 100, s);
                if(count == 0)
                {
                    break;
                }

                for(int i = 0; i < count; i++)
                {
                    (void)nn::capsrv::DeleteAlbumFile(&entries[i].fileId);
                }
            }
        }

    }
}

#define NNT_GRC_SCOPED_LAYER(varName, aruidValue)   \
    uint64_t varName = 0;               \
    nn::applet::AppletResourceUserId nntGrcLayerAruid = {static_cast<uint64_t>(aruidValue)};   \
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::grc::InitializeByHipc());    \
    NN_UTIL_SCOPE_EXIT{ nn::grc::FinalizeByHipc(); };               \
    nn::sf::SharedPointer<nn::grcsrv::IOffscreenRecorder> pService; \
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::grc::CreateOffscreenRecorder(&pService, MakeTransferMemory(GetRecorderFirmwareMemory(), GetRecorderFirmwareMemorySize()), GetRecorderFirmwareMemorySize())); \
    NN_ABORT_UNLESS_RESULT_SUCCESS(pService->CreateOffscreenLayer(&varName, nntGrcLayerAruid)); \
    NN_UTIL_SCOPE_EXIT{ pService->DestroyOffscreenLayer(varName); };

NNT_GRC_TEST_SERVERSTATE(Recording_StartAbort)
{
    const int RepeatCount = 25;
    uint64_t aruid = __LINE__;

    InitializeGraphics();
    CleanAlbum();
    NNT_GRC_SCOPED_LAYER(hLayer, aruid);

    NNT_GRC_SCOPED_INITIALIZE_FOR_APPLICATION(aruid);
    nn::grc::MovieLayer* pLayer;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::grc::OpenMovieLayer(&pLayer, hLayer, Width, Height));
    NN_UTIL_SCOPE_EXIT{ nn::grc::CloseMovieLayer(pLayer); };

    for(int i = 0; i < RepeatCount; i++)
    {
        NN_LOG("Loop %d/%d\n", i + 1, RepeatCount);
        NN_LOG("[%d]start movie file\n", i);
        NNT_EXPECT_RESULT_SUCCESS(nn::grc::StartMovieFile(pLayer, nn::grc::OffscreenRecordingParameter::GetDefaultValue()));
        NN_LOG("[%d]abort movie file\n", i);
        NNT_EXPECT_RESULT_SUCCESS(nn::grc::AbortMovieFile(pLayer));
    }
}

NNT_GRC_TEST_SERVERSTATE(Recording_StartClose)
{
    const int RepeatCount = 25;
    uint64_t aruid = __LINE__;

    InitializeGraphics();
    CleanAlbum();
    NNT_GRC_SCOPED_LAYER(hLayer, aruid);

    NNT_GRC_SCOPED_INITIALIZE_FOR_APPLICATION(aruid);

    for(int i = 0; i < RepeatCount; i++)
    {
        NN_LOG("Loop %d/%d\n", i + 1, RepeatCount);
        nn::grc::MovieLayer* pLayer;
        NN_LOG("[%d]open layer\n", i);
        NNT_EXPECT_RESULT_SUCCESS(nn::grc::OpenMovieLayer(&pLayer, hLayer, Width, Height));
        NN_LOG("[%d]start movie file\n", i);
        NNT_EXPECT_RESULT_SUCCESS(nn::grc::StartMovieFile(pLayer, nn::grc::OffscreenRecordingParameter::GetDefaultValue()));
        NN_LOG("[%d]close layer\n", i);
        nn::grc::CloseMovieLayer(pLayer);
    }
}

NNT_GRC_TEST_SERVERSTATE(Recording_StartFinalize)
{
    const int RepeatCount = 25;
    uint64_t aruid = __LINE__;

    InitializeGraphics();
    CleanAlbum();
    NNT_GRC_SCOPED_LAYER(hLayer, aruid);

    for(int i = 0; i < RepeatCount; i++)
    {
        NN_LOG("Loop %d/%d\n", i + 1, RepeatCount);
        NN_LOG("[%d]initialize shim\n", i);
        NNT_GRC_INITIALIZE_FOR_APPLICATION(aruid);
        nn::grc::MovieLayer* pLayer;
        NN_LOG("[%d]open layer\n", i);
        NNT_EXPECT_RESULT_SUCCESS(nn::grc::OpenMovieLayer(&pLayer, hLayer, Width, Height));
        NN_LOG("[%d]start movie file\n", i);
        NNT_EXPECT_RESULT_SUCCESS(nn::grc::StartMovieFile(pLayer, nn::grc::OffscreenRecordingParameter::GetDefaultValue()));
        NN_LOG("[%d]finalize shim\n", i);
        nn::grc::FinalizeForApplication();
    }
}

NNT_GRC_TEST_SERVERSTATE(Recording_StartDestroyFinalize)
{
    const int RepeatCount = 25;

    InitializeGraphics();
    CleanAlbum();

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::grc::InitializeByHipc());
    NN_UTIL_SCOPE_EXIT{ nn::grc::FinalizeByHipc(); };
    nn::sf::SharedPointer<nn::grcsrv::IOffscreenRecorder> pService;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::grc::CreateOffscreenRecorder(&pService, MakeTransferMemory(GetRecorderFirmwareMemory(), GetRecorderFirmwareMemorySize()), GetRecorderFirmwareMemorySize()));

    for(int i = 0; i < RepeatCount; i++)
    {
        NN_LOG("Loop %d/%d\n", i + 1, RepeatCount);
        uint64_t aruid = 0x1111 + i;
        NN_LOG("[%d]create layer\n", i);
        uint64_t hLayer = 0;
        NN_ABORT_UNLESS_RESULT_SUCCESS(pService->CreateOffscreenLayer(&hLayer, {aruid}));
        NN_LOG("[%d]initialize shim\n", i);
        NNT_GRC_INITIALIZE_FOR_APPLICATION(aruid);
        nn::grc::MovieLayer* pLayer;
        NN_LOG("[%d]open layer\n", i);
        NNT_EXPECT_RESULT_SUCCESS(nn::grc::OpenMovieLayer(&pLayer, hLayer, Width, Height));
        NN_LOG("[%d]start movie file\n", i);
        NNT_EXPECT_RESULT_SUCCESS(nn::grc::StartMovieFile(pLayer, nn::grc::OffscreenRecordingParameter::GetDefaultValue()));
        NN_LOG("[%d]destroy layer\n", i);
        pService->DestroyOffscreenLayer(hLayer);
        NN_LOG("[%d]finalize shim\n", i);
        nn::grc::FinalizeForApplication();
    }
}

NNT_GRC_TEST_SERVERSTATE(Recording_StartShutdownFinalize)
{
    const int RepeatCount = 25;


    InitializeGraphics();
    CleanAlbum();

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::grc::InitializeByHipc());
    NN_UTIL_SCOPE_EXIT{ nn::grc::FinalizeByHipc(); };

    for(int i = 0; i < RepeatCount; i++)
    {
        NN_LOG("Loop %d/%d\n", i + 1, RepeatCount);
        uint64_t aruid = 0x1234 + i;
        NN_LOG("[%d]make recorder\n", i);
        nn::sf::SharedPointer<nn::grcsrv::IOffscreenRecorder> pService;
        NNT_EXPECT_RESULT_SUCCESS(nn::grc::CreateOffscreenRecorder(&pService, MakeTransferMemory(GetRecorderFirmwareMemory(), GetRecorderFirmwareMemorySize()), GetRecorderFirmwareMemorySize()));
        NN_LOG("[%d]create layer\n", i);
        uint64_t hLayer = 0;
        NNT_EXPECT_RESULT_SUCCESS(pService->CreateOffscreenLayer(&hLayer, {aruid}));
        NN_LOG("[%d]initialize shim\n", i);
        NNT_GRC_INITIALIZE_FOR_APPLICATION(aruid);
        nn::grc::MovieLayer* pLayer;
        NN_LOG("[%d]open layer\n", i);
        NNT_EXPECT_RESULT_SUCCESS(nn::grc::OpenMovieLayer(&pLayer, hLayer, Width, Height));
        NN_LOG("[%d]start movie file\n", i);
        NNT_EXPECT_RESULT_SUCCESS(nn::grc::StartMovieFile(pLayer, nn::grc::OffscreenRecordingParameter::GetDefaultValue()));
        NN_LOG("[%d]destroy recorder\n", i);
        pService.Reset();
        NN_LOG("[%d]finalize shim\n", i);
        nn::grc::FinalizeForApplication();
    }
}

NNT_GRC_TEST_SERVERSTATE(Recording_StartParameter)
{
    uint64_t aruid = __LINE__;

    InitializeGraphics();
    CleanAlbum();
    NNT_GRC_SCOPED_LAYER(hLayer, aruid);

    {
        NN_LOG("initialize shim\n");
        NNT_GRC_INITIALIZE_FOR_APPLICATION(aruid);
        nn::grc::MovieLayer* pLayer;
        NN_LOG("open layer\n");
        NNT_EXPECT_RESULT_SUCCESS(nn::grc::OpenMovieLayer(&pLayer, hLayer, Width, Height));

        NN_LOG("checking invalid videoBitRate\n");
        {
            auto param = nn::grc::OffscreenRecordingParameter::GetDefaultValue();
            param.videoBitRate = 0;
            NNT_EXPECT_RESULT_FAILURE(nn::grc::ResultAlbumInvalidParameter,
                nn::grc::StartMovieFile(pLayer, param)
            );
        }

        NN_LOG("checking invalid videoWidth\n");
        {
            auto param = nn::grc::OffscreenRecordingParameter::GetDefaultValue();
            param.videoWidth = 9999;
            NNT_EXPECT_RESULT_FAILURE(nn::grc::ResultAlbumInvalidParameter,
                nn::grc::StartMovieFile(pLayer, param)
            );
        }

        NN_LOG("checking invalid videoHeight\n");
        {
            auto param = nn::grc::OffscreenRecordingParameter::GetDefaultValue();
            param.videoHeight = 999;
            NNT_EXPECT_RESULT_FAILURE(nn::grc::ResultAlbumInvalidParameter,
                nn::grc::StartMovieFile(pLayer, param)
            );
        }

        NN_LOG("checking invalid videoFrameRate\n");
        {
            auto param = nn::grc::OffscreenRecordingParameter::GetDefaultValue();
            param.videoFrameRate = 31;
            NNT_EXPECT_RESULT_FAILURE(nn::grc::ResultAlbumInvalidParameter,
                nn::grc::StartMovieFile(pLayer, param)
            );
        }

        NN_LOG("checking invalid videoFrameCountBetweenIdr\n");
        {
            auto param = nn::grc::OffscreenRecordingParameter::GetDefaultValue();
            param.videoFrameCountBetweenIdr = 0;
            NNT_EXPECT_RESULT_FAILURE(nn::grc::ResultAlbumInvalidParameter,
                nn::grc::StartMovieFile(pLayer, param)
            );
        }

        NN_LOG("checking invalid audioSamepleRate\n");
        {
            auto param = nn::grc::OffscreenRecordingParameter::GetDefaultValue();
            param.audioSampleRate = 0;
            NNT_EXPECT_RESULT_FAILURE(nn::grc::ResultAlbumInvalidParameter,
                nn::grc::StartMovieFile(pLayer, param)
            );
        }

        NN_LOG("checking invalid audioChannelCount\n");
        {
            auto param = nn::grc::OffscreenRecordingParameter::GetDefaultValue();
            param.audioChannelCount = 1;
            NNT_EXPECT_RESULT_FAILURE(nn::grc::ResultAlbumInvalidParameter,
                nn::grc::StartMovieFile(pLayer, param)
            );
        }

        NN_LOG("checking invalid audioFormat\n");
        {
            auto param = nn::grc::OffscreenRecordingParameter::GetDefaultValue();
            param.audioFormat = nn::audio::SampleFormat_PcmInt32;
            NNT_EXPECT_RESULT_FAILURE(nn::grc::ResultAlbumInvalidParameter,
                nn::grc::StartMovieFile(pLayer, param)
            );
        }

        NN_LOG("finalize shim\n");
        nn::grc::FinalizeForApplication();
    }
}
