﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "testPerf_Main.h"
#include <nnt.h>

// 繰り返し初期化と終了を行う
TEST(GpuMeasure, RepeatInitialize)
{
    nn::os::SetThreadCoreMask(nn::os::GetCurrentThread(), 0, 1);
    GpuMeterManager gpuMeterManager;

    for(int i = 1; i < 2000; i++)
    {
        gpuMeterManager.Initialize(3, 2, i);
        nn::perf::GpuMeter* gpuMeter = gpuMeterManager.GetMeter();

        GetGfw()->BeginFrame(0);
        {
            nn::gfx::CommandBuffer* rootCommandBuffer = GetGfw()->GetRootCommandBuffer(0);
            gpuMeter->BeginMeasure(rootCommandBuffer);
            SleepGpu();
            gpuMeter->EndMeasure(rootCommandBuffer);
        }
        GetGfw()->EndFrame(0);

        gpuMeterManager.Finalize();
    }

    SUCCEED();
}

// 設定
TEST(GpuMeasure, Set)
{
    nn::os::SetThreadCoreMask(nn::os::GetCurrentThread(), 0, 1);
    GpuMeterManager gpuMeterManager;
    nn::perf::GpuMeter* gpuMeter;

    gpuMeterManager.Initialize(1, 2, 1);
    gpuMeter = gpuMeterManager.GetMeter();

    EXPECT_EQ(nn::util::Color4u8::Green(), gpuMeter->GetColor());
    EXPECT_EQ(60.f, gpuMeter->GetFrameRate());
    EXPECT_EQ(0, nn::util::Strncmp(gpuMeter->GetName(), GetNullName(), 128));

    gpuMeter->SetColor(nn::util::Color4u8::Red());
    gpuMeter->SetFrameRate(30.f);
    gpuMeter->SetName("test");

    EXPECT_EQ(nn::util::Color4u8::Red(), gpuMeter->GetColor());
    EXPECT_EQ(30.f, gpuMeter->GetFrameRate());
    EXPECT_EQ(0, nn::util::Strncmp(gpuMeter->GetName(), "test", 128));

    gpuMeterManager.Finalize();
}

// アタッチデタッチをテスト
TEST(GpuMeasure, TestAttach)
{
    const int meterCount = 3;
    GpuMeterManager gpuMeterManager[meterCount];
    nn::perf::GpuMeter* gpuMeter[meterCount];
    int sectionCount;

    sectionCount = 1;
    for(int i = 0; i < meterCount; ++i)
    {
        gpuMeterManager[i].Initialize(1, 2, sectionCount);
        gpuMeter[i] = gpuMeterManager[i].GetMeter();
    }

    gpuMeter[0]->AttachLoadMeter(gpuMeter[1]);
    gpuMeter[0]->AttachLoadMeter(gpuMeter[2]);

    ASSERT_FALSE(gpuMeter[0]->IsLinked());
    ASSERT_TRUE(gpuMeter[1]->IsLinked());
    ASSERT_TRUE(gpuMeter[2]->IsLinked());

    gpuMeter[0]->DetachLoadMeter(gpuMeter[1]);
    gpuMeter[0]->DetachLoadMeter(gpuMeter[2]);

    ASSERT_FALSE(gpuMeter[0]->IsLinked());
    ASSERT_FALSE(gpuMeter[1]->IsLinked());
    ASSERT_FALSE(gpuMeter[2]->IsLinked());

    gpuMeter[0]->AttachLoadMeter(gpuMeter[1]);
    gpuMeter[1]->AttachLoadMeter(gpuMeter[2]);

    ASSERT_FALSE(gpuMeter[0]->IsLinked());
    ASSERT_TRUE(gpuMeter[1]->IsLinked());
    ASSERT_TRUE(gpuMeter[2]->IsLinked());

    for(int i = 0; i < meterCount; ++i)
    {
        gpuMeterManager[i].Finalize();
    }

    SUCCEED();
}

// 設定した最大区間を計測する
TEST(GpuMeasure, MeasureMaxSection)
{
    nn::os::SetThreadCoreMask(nn::os::GetCurrentThread(), 0, 1);
    GpuMeterManager gpuMeterManager;
    nn::perf::GpuMeter* gpuMeter;
    int sectionCount;

    sectionCount = 1;
    gpuMeterManager.Initialize(1, 2, sectionCount);
    gpuMeter = gpuMeterManager.GetMeter();

    GetGfw()->BeginFrame(0);
    {
        nn::gfx::CommandBuffer* rootCommandBuffer = GetGfw()->GetRootCommandBuffer(0);
        for(int i = 0; i < sectionCount; i++)
        {
            gpuMeter->BeginMeasure(rootCommandBuffer, "test1");
            SleepGpu();
            gpuMeter->EndMeasure(rootCommandBuffer);
        }
    }
    GetGfw()->EndFrame(0);
    GetGfw()->ExecuteCommand(0);
    GetGfw()->QueuePresentTexture(1);
    GetGfw()->QueueFinish();

    gpuMeter->Next();
    EXPECT_EQ(sectionCount, gpuMeter->GetLastSectionCount());
    EXPECT_LT(0, (gpuMeter->GetLastTotalEnd() - gpuMeter->GetLastTotalBegin()).ToTimeSpan().GetNanoSeconds());
    EXPECT_EQ(gpuMeter->GetLastTotalBegin(), gpuMeter->GetLastResult(0).begin);
    EXPECT_EQ(gpuMeter->GetLastTotalEnd(), gpuMeter->GetLastResult(sectionCount - 1).end);

    for(int i = 0; i < sectionCount; i++)
    {
        const nn::perf::LoadMeterBase::Section& section = gpuMeter->GetLastResult(i);
        EXPECT_LT(0, (section.end - section.begin).ToTimeSpan().GetNanoSeconds());
        EXPECT_EQ(0, section.depth);
        EXPECT_EQ(0, section.tag);
        EXPECT_EQ("test1", section.name);
    }
    gpuMeterManager.Finalize();

    sectionCount = 10240;
    gpuMeterManager.Initialize(1, 2, sectionCount);
    gpuMeter = gpuMeterManager.GetMeter();
    GetGfw()->BeginFrame(0);
    {
        nn::gfx::CommandBuffer* rootCommandBuffer = GetGfw()->GetRootCommandBuffer(0);
        for(int i = 0; i < sectionCount; i++)
        {
            gpuMeter->BeginMeasure(rootCommandBuffer, "test2", i);
            SleepGpu();
            gpuMeter->EndMeasure(rootCommandBuffer);
        }
    }
    GetGfw()->EndFrame(0);
    GetGfw()->ExecuteCommand(0);
    GetGfw()->QueuePresentTexture(1);
    GetGfw()->QueueFinish();

    gpuMeter->Next();
    EXPECT_EQ(sectionCount, gpuMeter->GetLastSectionCount());
    EXPECT_LT(0, (gpuMeter->GetLastTotalEnd() - gpuMeter->GetLastTotalBegin()).ToTimeSpan().GetNanoSeconds());
    EXPECT_EQ(gpuMeter->GetLastTotalBegin(), gpuMeter->GetLastResult(0).begin);
    EXPECT_EQ(gpuMeter->GetLastTotalEnd(), gpuMeter->GetLastResult(sectionCount - 1).end);

    for(int i = 0; i < sectionCount; i++)
    {
        const nn::perf::LoadMeterBase::Section& section = gpuMeter->GetLastResult(i);
        EXPECT_LT(0, (section.end - section.begin).ToTimeSpan().GetNanoSeconds());
        EXPECT_EQ(0, section.depth);
        EXPECT_EQ(i, section.tag);
        EXPECT_EQ("test2", section.name);
    }
    gpuMeterManager.Finalize();

    SUCCEED();
}

// バッファ数に対応した結果が取得できるか
TEST(GpuMeasure, TestBufferCount)
{
    nn::os::SetThreadCoreMask(nn::os::GetCurrentThread(), 0, 1);
    GpuMeterManager gpuMeterManager;
    nn::perf::GpuMeter* gpuMeter;
    int sectionCount;

    sectionCount = 1;
    // バッファ数を 4 に設定
    // 3(Buffer-1)回目に呼ばれた Next() で計測した結果が集計される。
    // (デフォルトのバッファ数 2 では、計測後 1(buffer-1)回目に呼ばれた Next() で集計される)
    gpuMeterManager.Initialize(1, 4, sectionCount);
    gpuMeter = gpuMeterManager.GetMeter();

    // 1 フレーム目
    GetGfw()->BeginFrame(0);
    {
        nn::gfx::CommandBuffer* rootCommandBuffer = GetGfw()->GetRootCommandBuffer(0);
        gpuMeter->BeginMeasure(rootCommandBuffer,"Test1", 1, nn::util::Color4u8::Red());
        SleepGpu();
        gpuMeter->EndMeasure(rootCommandBuffer);
    }
    GetGfw()->EndFrame(0);
    GetGfw()->ExecuteCommand(0);
    GetGfw()->QueuePresentTexture(1);
    GetGfw()->QueueFinish();

    // 1 回目の Next()。空の結果を取得。
    gpuMeter->Next();
    {
        const nn::perf::LoadMeterBase::Section& section = gpuMeter->GetLastResult(0);
        EXPECT_EQ(0, section.begin.ToTimeSpan().GetNanoSeconds());
        EXPECT_EQ(0, section.end.ToTimeSpan().GetNanoSeconds());
        EXPECT_EQ(NULL, section.name);
        EXPECT_EQ(0, section.tag);
        EXPECT_EQ(nn::util::Color4u8::Green(), section.color);
    }

    // 2 フレーム目
    GetGfw()->BeginFrame(0);
    {
        nn::gfx::CommandBuffer* rootCommandBuffer = GetGfw()->GetRootCommandBuffer(0);
        gpuMeter->BeginMeasure(rootCommandBuffer,"Test2", 2, nn::util::Color4u8::Blue());
        SleepGpu();
        gpuMeter->EndMeasure(rootCommandBuffer);
    }
    GetGfw()->EndFrame(0);
    GetGfw()->ExecuteCommand(0);
    GetGfw()->QueuePresentTexture(1);
    GetGfw()->QueueFinish();

    // 2 回目の Next()。空の結果を取得。
    gpuMeter->Next();
    {
        const nn::perf::LoadMeterBase::Section& section = gpuMeter->GetLastResult(0);
        EXPECT_EQ(0, section.begin.ToTimeSpan().GetNanoSeconds());
        EXPECT_EQ(0, section.end.ToTimeSpan().GetNanoSeconds());
        EXPECT_EQ(NULL, section.name);
        EXPECT_EQ(0, section.tag);
        EXPECT_EQ(nn::util::Color4u8::Green(), section.color);
    }

    // 3 フレーム目
    GetGfw()->BeginFrame(0);
    {
        nn::gfx::CommandBuffer* rootCommandBuffer = GetGfw()->GetRootCommandBuffer(0);
        gpuMeter->BeginMeasure(rootCommandBuffer,"Test3", 3, nn::util::Color4u8::Yellow());
        SleepGpu();
        gpuMeter->EndMeasure(rootCommandBuffer);
    }
    GetGfw()->EndFrame(0);
    GetGfw()->ExecuteCommand(0);
    GetGfw()->QueuePresentTexture(1);
    GetGfw()->QueueFinish();

    // 3 回目の Next()。1 フレーム目の結果を取得。
    gpuMeter->Next();
    {
        const nn::perf::LoadMeterBase::Section& section = gpuMeter->GetLastResult(0);
        EXPECT_LT(0, (section.end - section.begin).ToTimeSpan().GetNanoSeconds());
        EXPECT_LT(0, section.begin.ToTimeSpan().GetNanoSeconds());
        EXPECT_LT(0, section.end.ToTimeSpan().GetNanoSeconds());
        EXPECT_EQ("Test1", section.name);
        EXPECT_EQ(1, section.tag);
        EXPECT_EQ(nn::util::Color4u8::Red(), section.color);
    }

    // 4 フレーム目
    GetGfw()->BeginFrame(0);
    GetGfw()->EndFrame(0);
    GetGfw()->ExecuteCommand(0);
    GetGfw()->QueuePresentTexture(1);
    GetGfw()->QueueFinish();

    // 4 回目の Next()。2 フレーム目の結果を取得。
    gpuMeter->Next();
    {
        const nn::perf::LoadMeterBase::Section& section = gpuMeter->GetLastResult(0);
        EXPECT_LT(0, (section.end - section.begin).ToTimeSpan().GetNanoSeconds());
        EXPECT_LT(0, section.begin.ToTimeSpan().GetNanoSeconds());
        EXPECT_LT(0, section.end.ToTimeSpan().GetNanoSeconds());
        EXPECT_EQ("Test2", section.name);
        EXPECT_EQ(2, section.tag);
        EXPECT_EQ(nn::util::Color4u8::Blue(), section.color);
    }

    // 5 フレーム目
    GetGfw()->BeginFrame(0);
    GetGfw()->EndFrame(0);
    GetGfw()->ExecuteCommand(0);
    GetGfw()->QueuePresentTexture(1);
    GetGfw()->QueueFinish();

    // 5 回目の Next()。3 フレーム目の結果を取得。
    gpuMeter->Next();
    {
        const nn::perf::LoadMeterBase::Section& section = gpuMeter->GetLastResult(0);
        EXPECT_LT(0, (section.end - section.begin).ToTimeSpan().GetNanoSeconds());
        EXPECT_LT(0, section.begin.ToTimeSpan().GetNanoSeconds());
        EXPECT_LT(0, section.end.ToTimeSpan().GetNanoSeconds());
        EXPECT_EQ("Test3", section.name);
        EXPECT_EQ(3, section.tag);
        EXPECT_EQ(nn::util::Color4u8::Yellow(), section.color);
    }

    gpuMeterManager.Finalize();

    SUCCEED();
} //NOLINT(impl/function_size)

// Generic 版では連続でタイムスタンプを呼ぶと止まる不具合があるので以下のテストは行わない
#if defined(NN_BUILD_CONFIG_SPEC_NX)
// 入れ子で計測できるか判断する
TEST(GpuMeasure, MeasureNest)
{
    nn::os::SetThreadCoreMask(nn::os::GetCurrentThread(), 0, 1);
    GpuMeterManager gpuMeterManager;
    nn::perf::GpuMeter* gpuMeter;
    int sectionCount;

    sectionCount = 16;
    gpuMeterManager.Initialize(1, 2, sectionCount);
    gpuMeter = gpuMeterManager.GetMeter();

    GetGfw()->BeginFrame(0);
    {
        nn::gfx::CommandBuffer* rootCommandBuffer = GetGfw()->GetRootCommandBuffer(0);
        for(int i = 0; i < sectionCount; i++)
        {
            gpuMeter->BeginMeasure(rootCommandBuffer);
        }
        SleepGpu();
        for(int i = 0; i < sectionCount; i++)
        {
            gpuMeter->EndMeasure(rootCommandBuffer);
        }
    }
    GetGfw()->EndFrame(0);
    GetGfw()->ExecuteCommand(0);
    GetGfw()->QueuePresentTexture(1);
    GetGfw()->QueueFinish();

    gpuMeter->Next();

    EXPECT_EQ(sectionCount, gpuMeter->GetLastSectionCount());
    EXPECT_LT(0, (gpuMeter->GetLastTotalEnd() - gpuMeter->GetLastTotalBegin()).ToTimeSpan().GetNanoSeconds());
    EXPECT_EQ(gpuMeter->GetLastTotalBegin(), gpuMeter->GetLastResult(0).begin);
    EXPECT_EQ(gpuMeter->GetLastTotalEnd(), gpuMeter->GetLastResult(0).end);

    for(int i = 0; i < sectionCount; i++)
    {
        const nn::perf::LoadMeterBase::Section& section = gpuMeter->GetLastResult(i);
        EXPECT_LT(0, section.end.ToTimeSpan().GetNanoSeconds() - section.begin.ToTimeSpan().GetNanoSeconds());
        EXPECT_EQ(i, section.depth);
    }
    gpuMeterManager.Finalize();

    sectionCount = 5000;
    gpuMeterManager.Initialize(1, 2, sectionCount);
    gpuMeter = gpuMeterManager.GetMeter();
    GetGfw()->BeginFrame(0);
    {
        nn::gfx::CommandBuffer* rootCommandBuffer = GetGfw()->GetRootCommandBuffer(0);
        for(int i = 0; i < sectionCount; i++)
        {
            gpuMeter->BeginMeasure(rootCommandBuffer);
        }
        SleepGpu();
        for(int i = 0; i < sectionCount; i++)
        {
            gpuMeter->EndMeasure(rootCommandBuffer);
        }
    }
    GetGfw()->EndFrame(0);
    GetGfw()->ExecuteCommand(0);
    GetGfw()->QueuePresentTexture(1);
    GetGfw()->QueueFinish();

    gpuMeter->Next();

    EXPECT_EQ(sectionCount, gpuMeter->GetLastSectionCount());
    EXPECT_LT(0, (gpuMeter->GetLastTotalEnd() - gpuMeter->GetLastTotalBegin()).ToTimeSpan().GetNanoSeconds());
    EXPECT_EQ(gpuMeter->GetLastTotalBegin(), gpuMeter->GetLastResult(0).begin);
    EXPECT_EQ(gpuMeter->GetLastTotalEnd(), gpuMeter->GetLastResult(0).end);
    for(int i = 0; i < sectionCount; i++)
    {
        const nn::perf::LoadMeterBase::Section& section = gpuMeter->GetLastResult(i);
        EXPECT_LT(0, (section.end - section.begin).ToTimeSpan().GetNanoSeconds());
        EXPECT_EQ(i, section.depth);
    }
    gpuMeterManager.Finalize();

    SUCCEED();
}
#endif

// Next() が呼ばれない時に結果を取得した場合
TEST(GpuMeasure, MeasureNull)
{
    nn::os::SetThreadCoreMask(nn::os::GetCurrentThread(), 0, 1);
    GpuMeterManager gpuMeterManager;
    nn::perf::GpuMeter* gpuMeter;
    int sectionCount;

    sectionCount = 1024;
    gpuMeterManager.Initialize(1, 2, sectionCount);
    gpuMeter = gpuMeterManager.GetMeter();
    GetGfw()->BeginFrame(0);
    {
        nn::gfx::CommandBuffer* rootCommandBuffer = GetGfw()->GetRootCommandBuffer(0);
        for(int i = 0; i < sectionCount; i++)
        {
            gpuMeter->BeginMeasure(rootCommandBuffer);
            gpuMeter->EndMeasure(rootCommandBuffer);
        }
    }
    GetGfw()->EndFrame(0);

    EXPECT_EQ(0, gpuMeter->GetLastSectionCount());
    EXPECT_EQ(0, gpuMeter->GetLastTotalBegin().ToTimeSpan().GetNanoSeconds());
    EXPECT_EQ(0, gpuMeter->GetLastTotalEnd().ToTimeSpan().GetNanoSeconds());

    for(int i = 0; i < 1024; i++)
    {
        const nn::perf::LoadMeterBase::Section& section = gpuMeter->GetLastResult(i);
        EXPECT_EQ(0, section.begin.ToTimeSpan().GetNanoSeconds());
        EXPECT_EQ(0, section.end.ToTimeSpan().GetNanoSeconds());
        EXPECT_EQ(0, section.depth);
        EXPECT_EQ(0, section.tag);
        EXPECT_EQ(NULL, section.name);
    }

    gpuMeterManager.Finalize();

    SUCCEED();
}

// 計測結果が妥当か判断する
TEST(GpuMeasure, CheckResult)
{
    nn::os::SetThreadCoreMask(nn::os::GetCurrentThread(), 0, 1);
    GpuMeterManager gpuMeterManager;
    nn::perf::GpuMeter* gpuMeter;
    int sectionCount = 64;

    gpuMeterManager.Initialize(1, 2, sectionCount);
    gpuMeter = gpuMeterManager.GetMeter();

    GetGfw()->BeginFrame(0);
    {
        nn::gfx::CommandBuffer* rootCommandBuffer = GetGfw()->GetRootCommandBuffer(0);
        for(int i = 0; i < sectionCount; i++)
        {
            gpuMeter->BeginMeasure(rootCommandBuffer);
            SleepGpu();
            gpuMeter->EndMeasure(rootCommandBuffer);
        }
    }
    GetGfw()->EndFrame(0);
    GetGfw()->ExecuteCommand(0);
    GetGfw()->QueuePresentTexture(1);
    GetGfw()->QueueFinish();

    gpuMeter->Next();

    EXPECT_EQ(sectionCount, gpuMeter->GetLastSectionCount());
    EXPECT_LT(0, (gpuMeter->GetLastTotalEnd() - gpuMeter->GetLastTotalBegin()).ToTimeSpan().GetNanoSeconds());
    EXPECT_EQ(gpuMeter->GetLastTotalBegin(), gpuMeter->GetLastResult(0).begin);
    EXPECT_EQ(gpuMeter->GetLastTotalEnd(), gpuMeter->GetLastResult(sectionCount - 1).end);

    for(int i = 0; i < sectionCount; i++)
    {
        const nn::perf::LoadMeterBase::Section& section = gpuMeter->GetLastResult(i);
        EXPECT_LT(0, (section.end - section.begin).ToTimeSpan().GetNanoSeconds());
        EXPECT_EQ(0, section.depth);
        EXPECT_EQ(0, section.tag);
        EXPECT_EQ(NULL, section.name);
    }

    GetGfw()->BeginFrame(0);
    {
        nn::gfx::CommandBuffer* rootCommandBuffer = GetGfw()->GetRootCommandBuffer(0);

        gpuMeter->SetColor(nn::util::Color4u8::White());

        gpuMeter->BeginMeasure(rootCommandBuffer);
        SleepGpu();
        gpuMeter->EndMeasure(rootCommandBuffer);

        gpuMeter->BeginMeasure(rootCommandBuffer,"Test1");
        SleepGpu();
        gpuMeter->EndMeasure(rootCommandBuffer);

        gpuMeter->BeginMeasure(rootCommandBuffer,"Test2", 32, nn::util::Color4u8::Red());
        SleepGpu();
        gpuMeter->EndMeasure(rootCommandBuffer);

        gpuMeter->BeginMeasure(rootCommandBuffer,"Test3", 16);
        SleepGpu();
        gpuMeter->EndMeasure(rootCommandBuffer);

        gpuMeter->SetColor(nn::util::Color4u8::Blue());

        gpuMeter->BeginMeasure(rootCommandBuffer,16);
        SleepGpu();
        gpuMeter->EndMeasure(rootCommandBuffer);

        gpuMeter->BeginMeasure(rootCommandBuffer,4, nn::util::Color4u8::Yellow());
        SleepGpu();
        gpuMeter->EndMeasure(rootCommandBuffer);

        gpuMeter->BeginMeasure(rootCommandBuffer,"Test4", nn::util::Color4u8::Black());
        SleepGpu();
        gpuMeter->EndMeasure(rootCommandBuffer);
    }
    GetGfw()->EndFrame(0);
    GetGfw()->ExecuteCommand(0);
    GetGfw()->QueuePresentTexture(1);
    GetGfw()->QueueFinish();

    gpuMeter->Next();

    EXPECT_EQ(7, gpuMeter->GetLastSectionCount());
    EXPECT_LT(0, (gpuMeter->GetLastTotalEnd() - gpuMeter->GetLastTotalBegin()).ToTimeSpan().GetNanoSeconds());
    EXPECT_EQ(gpuMeter->GetLastTotalBegin(), gpuMeter->GetLastResult(0).begin);
    EXPECT_EQ(gpuMeter->GetLastTotalEnd(), gpuMeter->GetLastResult(6).end);

    const nn::perf::LoadMeterBase::Section& section = gpuMeter->GetLastResult(0);
    EXPECT_LT(0, (section.end - section.begin).ToTimeSpan().GetNanoSeconds());
    EXPECT_EQ(NULL, section.name);
    EXPECT_EQ(0, section.tag);
    EXPECT_EQ(nn::util::Color4u8::White(), section.color);

    const nn::perf::LoadMeterBase::Section& section1 = gpuMeter->GetLastResult(1);
    EXPECT_LT(0, (section1.end - section1.begin).ToTimeSpan().GetNanoSeconds());
    EXPECT_EQ("Test1", section1.name);
    EXPECT_EQ(0, section1.tag);
    EXPECT_EQ(nn::util::Color4u8::White(), section1.color);

    const nn::perf::LoadMeterBase::Section& section2 = gpuMeter->GetLastResult(2);
    EXPECT_LT(0, (section2.end - section2.begin).ToTimeSpan().GetNanoSeconds());
    EXPECT_EQ("Test2", section2.name);
    EXPECT_EQ(32, section2.tag);
    EXPECT_EQ(nn::util::Color4u8::Red(), section2.color);

    const nn::perf::LoadMeterBase::Section& section3 = gpuMeter->GetLastResult(3);
    EXPECT_LT(0, (section3.end - section3.begin).ToTimeSpan().GetNanoSeconds());
    EXPECT_EQ("Test3", section3.name);
    EXPECT_EQ(16, section3.tag);
    EXPECT_EQ(nn::util::Color4u8::White(), section3.color);

    const nn::perf::LoadMeterBase::Section& section4 = gpuMeter->GetLastResult(4);
    EXPECT_LT(0, (section4.end - section4.begin).ToTimeSpan().GetNanoSeconds());
    EXPECT_EQ(NULL, section4.name);
    EXPECT_EQ(16, section4.tag);
    EXPECT_EQ(nn::util::Color4u8::Blue(), section4.color);

    const nn::perf::LoadMeterBase::Section& section5 = gpuMeter->GetLastResult(5);
    EXPECT_LT(0, (section5.end - section5.begin).ToTimeSpan().GetNanoSeconds());
    EXPECT_EQ(NULL, section5.name);
    EXPECT_EQ(4, section5.tag);
    EXPECT_EQ(nn::util::Color4u8::Yellow(), section5.color);

    const nn::perf::LoadMeterBase::Section& section6 = gpuMeter->GetLastResult(6);
    EXPECT_LT(0, (section6.end - section6.begin).ToTimeSpan().GetNanoSeconds());
    EXPECT_EQ("Test4", section6.name);
    EXPECT_EQ(0, section6.tag);
    EXPECT_EQ(nn::util::Color4u8::Black(), section6.color);

    EXPECT_EQ(nn::util::Color4u8::Blue(), gpuMeter->GetColor());

    gpuMeterManager.Finalize();

    SUCCEED();
} //NOLINT(impl/function_size)

namespace{

const size_t                    threadStackSize = 0x2000;
NN_OS_ALIGNAS_THREAD_STACK char threadStack[3][threadStackSize];
nn::os::TlsSlot tlsSlot;
nn::os::ThreadType              thread[3];
TlsSlotValue                    tlsSlotValue[3];

int GetCoreNumberFunction()
{
    int coreNumber = reinterpret_cast<TlsSlotValue*>(nn::os::GetTlsValue(tlsSlot))->coreNumber;
    return *reinterpret_cast<int*>(&coreNumber);
}

void ThreadMeasureGpu(void *tlsValue)
{
    nn::os::SetTlsValue(tlsSlot, reinterpret_cast<uintptr_t>(tlsValue));
    int coreNumber = reinterpret_cast<TlsSlotValue*>(nn::os::GetTlsValue(tlsSlot))->coreNumber;
    const char* name = reinterpret_cast<TlsSlotValue*>(nn::os::GetTlsValue(tlsSlot))->name;
    nn::util::Color4u8 color = reinterpret_cast<TlsSlotValue*>(nn::os::GetTlsValue(tlsSlot))->color;
    nn::gfx::CommandBuffer* commandBuffer = &GetSubCommandBuffer(coreNumber)->object;
    nn::perf::GpuMeter* gpuMeter = reinterpret_cast<TlsSlotValue*>(nn::os::GetTlsValue(tlsSlot))->meter;

    EXPECT_EQ(nn::util::Color4u8::Green(), gpuMeter->GetColor());
    commandBuffer->Begin();
    gpuMeter->SetColor(color);
    gpuMeter->BeginMeasure(commandBuffer, name, coreNumber);
    SleepGpu(coreNumber);
    gpuMeter->EndMeasure(commandBuffer);
    commandBuffer->End();
    EXPECT_EQ(color, gpuMeter->GetColor());
}
}

// 複数スレッド 3コアで動作する場合
TEST(GpuMeasure, CheckMultiThread)
{
    GpuMeterManager gpuMeterManager;
    nn::perf::GpuMeter* gpuMeter;
    int sectionCount = 3;

    gpuMeterManager.Initialize(3, 2, sectionCount);
    gpuMeter = gpuMeterManager.GetMeter();
    InitializeSubCommandBuffer();

    // アフィニティマスクの設定
    nn::Bit64 core0 = 1;
    nn::Bit64 core1 = 2;
    nn::Bit64 core2 = 4;

    {
        nn::Result result = nn::os::AllocateTlsSlot(&tlsSlot, NULL);
        NN_ASSERT(result.IsSuccess(), "Cannot allocate TLS slot.");

        tlsSlotValue[0].coreNumber = 0;
        tlsSlotValue[0].meter = gpuMeter;
        tlsSlotValue[0].name = "Test0";
        tlsSlotValue[0].color = nn::util::Color4u8::Red();
        result = nn::os::CreateThread(&thread[0], ThreadMeasureGpu, &tlsSlotValue[0], threadStack[0], threadStackSize, nn::os::DefaultThreadPriority);
        nn::os::SetThreadName(&thread[0], "Thread0");
        nn::os::SetThreadCoreMask(&thread[0], 0, core0);

        tlsSlotValue[1].coreNumber = 1;
        tlsSlotValue[1].meter = gpuMeter;
        tlsSlotValue[1].name = "Test1";
        tlsSlotValue[1].color = nn::util::Color4u8::Green();
        result = nn::os::CreateThread(&thread[1], ThreadMeasureGpu, &tlsSlotValue[1], threadStack[1], threadStackSize, nn::os::DefaultThreadPriority);
        nn::os::SetThreadName(&thread[1], "Thread1");
        nn::os::SetThreadCoreMask(&thread[1], 1, core1);

        tlsSlotValue[2].coreNumber = 2;
        tlsSlotValue[2].meter = gpuMeter;
        tlsSlotValue[2].name = "Test2";
        tlsSlotValue[2].color = nn::util::Color4u8::Blue();
        result = nn::os::CreateThread(&thread[2], ThreadMeasureGpu, &tlsSlotValue[2], threadStack[2], threadStackSize, nn::os::DefaultThreadPriority);
        nn::os::SetThreadName(&thread[2], "Thread2");
        nn::os::SetThreadCoreMask(&thread[2], 2, core2);

        nn::os::SetThreadCoreMask(nn::os::GetCurrentThread(), 0, core0);
        nn::os::SetTlsValue(tlsSlot, reinterpret_cast<uintptr_t>(&tlsSlotValue[0]));
    }

    gpuMeter->SetGetCoreNumberFunction(GetCoreNumberFunction);

    GetGfw()->BeginFrame(0);
    {
        GetGfw()->ResetCommandBuffer(GetSubCommandBuffer(0));
        GetGfw()->ResetCommandBuffer(GetSubCommandBuffer(1));
        GetGfw()->ResetCommandBuffer(GetSubCommandBuffer(2));
        nn::os::StartThread(&thread[0]);
        nn::os::StartThread(&thread[1]);
        nn::os::StartThread(&thread[2]);
        nn::os::WaitThread(&thread[0]);
        nn::os::WaitThread(&thread[1]);
        nn::os::WaitThread(&thread[2]);
        GetGfw()->GetRootCommandBuffer(0)->CallCommandBuffer(&GetSubCommandBuffer(0)->object);
        GetGfw()->GetRootCommandBuffer(0)->CallCommandBuffer(&GetSubCommandBuffer(1)->object);
        GetGfw()->GetRootCommandBuffer(0)->CallCommandBuffer(&GetSubCommandBuffer(2)->object);
    }
    GetGfw()->EndFrame(0);
    GetGfw()->ExecuteCommand(0);
    GetGfw()->QueuePresentTexture(1);
    GetGfw()->QueueFinish();

    gpuMeter->Next();

    EXPECT_EQ(sectionCount, gpuMeter->GetLastSectionCount());
    EXPECT_LT(0, (gpuMeter->GetLastTotalEnd() - gpuMeter->GetLastTotalBegin()).ToTimeSpan().GetNanoSeconds());
    EXPECT_EQ(gpuMeter->GetLastTotalBegin(), gpuMeter->GetLastResult(0).begin);

    for(int i = 0; i < sectionCount; i++)
    {
        const nn::perf::LoadMeterBase::Section& section = gpuMeter->GetLastResult(i);
        EXPECT_LT(0, (section.end - section.begin).ToTimeSpan().GetNanoSeconds());
        switch (section.tag)
        {
        case 0:
            EXPECT_EQ("Test0", section.name);
            EXPECT_EQ(nn::util::Color4u8::Red(), section.color);
            break;
        case 1:
            EXPECT_EQ("Test1", section.name);
            EXPECT_EQ(nn::util::Color4u8::Green(), section.color);
            break;
        case 2:
            EXPECT_EQ("Test2", section.name);
            EXPECT_EQ(nn::util::Color4u8::Blue(), section.color);
            break;
        default:
            FAIL();
            break;
        }
    }

    nn::os::DestroyThread(&thread[0]);
    nn::os::DestroyThread(&thread[1]);
    nn::os::DestroyThread(&thread[2]);
    nn::os::FreeTlsSlot(tlsSlot);

    FinalizeSubCommandBuffer();
    gpuMeterManager.Finalize();

    SUCCEED();
}
