﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "testPerf_Main.h"
#include <nnt.h>

// 繰り返し初期化と終了を行う
TEST(CpuMeasure, RepeatInitialize)
{
    CpuMeterManager cpuMeterManager;

    for(int i = 1; i < 1000; i++)
    {
        cpuMeterManager.Initialize(3, 2, i);
        nn::perf::CpuMeter* cpuMeter = cpuMeterManager.GetMeter();

        cpuMeter->BeginMeasure();
        Sleep();
        cpuMeter->EndMeasure();

        cpuMeterManager.Finalize();
    }

    SUCCEED();
}

// info 初期値
TEST(CpuMeasure, InfoDefault)
{
    nn::perf::LoadMeterInfo info;
    EXPECT_EQ(2, info.GetBufferCount());
    EXPECT_EQ(0, info.GetCoreCount());
    EXPECT_EQ(0, info.GetSectionCountMax());
}

// 設定
TEST(CpuMeasure, Set)
{
    CpuMeterManager cpuMeterManager;
    nn::perf::CpuMeter* cpuMeter;

    cpuMeterManager.Initialize(1, 2, 1);
    cpuMeter = cpuMeterManager.GetMeter();

    EXPECT_EQ(nn::util::Color4u8::Green(), cpuMeter->GetColor());
    EXPECT_EQ(60.f, cpuMeter->GetFrameRate());
    EXPECT_EQ(0, nn::util::Strncmp(cpuMeter->GetName(), GetNullName(), 128));

    cpuMeter->SetColor(nn::util::Color4u8::Red());
    cpuMeter->SetFrameRate(30.f);
    cpuMeter->SetName("test");

    EXPECT_EQ(nn::util::Color4u8::Red(), cpuMeter->GetColor());
    EXPECT_EQ(30.f, cpuMeter->GetFrameRate());
    EXPECT_EQ(0, nn::util::Strncmp(cpuMeter->GetName(), "test", 128));

    cpuMeterManager.Finalize();
}

// アタッチデタッチをテスト
TEST(CpuMeasure, TestAttach)
{
    const int meterCount = 3;
    CpuMeterManager cpuMeterManager[meterCount];
    nn::perf::CpuMeter* cpuMeter[meterCount];
    int sectionCount;

    sectionCount = 1;
    for(int i = 0; i < meterCount; ++i)
    {
        cpuMeterManager[i].Initialize(1, 2, sectionCount);
        cpuMeter[i] = cpuMeterManager[i].GetMeter();
    }

    cpuMeter[0]->AttachLoadMeter(cpuMeter[1]);
    cpuMeter[0]->AttachLoadMeter(cpuMeter[2]);

    ASSERT_FALSE(cpuMeter[0]->IsLinked());
    ASSERT_TRUE(cpuMeter[1]->IsLinked());
    ASSERT_TRUE(cpuMeter[2]->IsLinked());

    cpuMeter[0]->DetachLoadMeter(cpuMeter[1]);
    cpuMeter[0]->DetachLoadMeter(cpuMeter[2]);

    ASSERT_FALSE(cpuMeter[0]->IsLinked());
    ASSERT_FALSE(cpuMeter[1]->IsLinked());
    ASSERT_FALSE(cpuMeter[2]->IsLinked());

    cpuMeter[0]->AttachLoadMeter(cpuMeter[1]);
    cpuMeter[1]->AttachLoadMeter(cpuMeter[2]);

    ASSERT_FALSE(cpuMeter[0]->IsLinked());
    ASSERT_TRUE(cpuMeter[1]->IsLinked());
    ASSERT_TRUE(cpuMeter[2]->IsLinked());

    for(int i = 0; i < meterCount; ++i)
    {
        cpuMeterManager[i].Finalize();
    }
}

// 設定した最大区間を計測する
TEST(CpuMeasure, MeasureMaxSection)
{
    CpuMeterManager cpuMeterManager;
    nn::perf::CpuMeter* cpuMeter;
    int sectionCount;

    sectionCount = 1;
    cpuMeterManager.Initialize(1, 2, sectionCount);
    cpuMeter = cpuMeterManager.GetMeter();
    for(int i = 0; i < sectionCount; i++)
    {
        cpuMeter->BeginMeasure("test1");
        Sleep();
        cpuMeter->EndMeasure();
    }
    cpuMeter->Next();
    EXPECT_EQ(sectionCount, cpuMeter->GetLastSectionCount());
    EXPECT_LT(0, (cpuMeter->GetLastTotalEnd() - cpuMeter->GetLastTotalBegin()).ToTimeSpan().GetNanoSeconds());
    EXPECT_EQ(cpuMeter->GetLastTotalBegin(), cpuMeter->GetLastResult(0).begin);
    EXPECT_EQ(cpuMeter->GetLastTotalEnd(), cpuMeter->GetLastResult(sectionCount - 1).end);

    for(int i = 0; i < sectionCount; i++)
    {
        const nn::perf::LoadMeterBase::Section& section = cpuMeter->GetLastResult(i);
        EXPECT_LT(0, (section.end - section.begin).ToTimeSpan().GetNanoSeconds());
        EXPECT_EQ(0, section.depth);
        EXPECT_EQ(0, section.tag);
        EXPECT_EQ("test1", section.name);
    }
    cpuMeterManager.Finalize();

    sectionCount = 5000;
    cpuMeterManager.Initialize(1, 2, sectionCount);
    cpuMeter = cpuMeterManager.GetMeter();
    for(int i = 0; i < sectionCount; i++)
    {
        cpuMeter->BeginMeasure("test2", i);
        Sleep();
        cpuMeter->EndMeasure();
    }
    cpuMeter->Next();
    EXPECT_EQ(sectionCount, cpuMeter->GetLastSectionCount());
    EXPECT_LT(0, (cpuMeter->GetLastTotalEnd() - cpuMeter->GetLastTotalBegin()).ToTimeSpan().GetNanoSeconds());
    EXPECT_EQ(cpuMeter->GetLastTotalBegin(), cpuMeter->GetLastResult(0).begin);
    EXPECT_EQ(cpuMeter->GetLastTotalEnd(), cpuMeter->GetLastResult(sectionCount - 1).end);

    for(int i = 0; i < sectionCount; i++)
    {
        const nn::perf::LoadMeterBase::Section& section = cpuMeter->GetLastResult(i);
        EXPECT_LT(0, (section.end - section.begin).ToTimeSpan().GetNanoSeconds());
        EXPECT_EQ(0, section.depth);
        EXPECT_EQ(i, section.tag);
        EXPECT_EQ("test2", section.name);
    }
    cpuMeterManager.Finalize();

    SUCCEED();
}

// バッファ数に対応した結果が取得できるか
TEST(CpuMeasure, TestBufferCount)
{
    nn::os::SetThreadCoreMask(nn::os::GetCurrentThread(), 0, 1);
    CpuMeterManager cpuMeterManager;
    nn::perf::CpuMeter* cpuMeter;
    int sectionCount;

    sectionCount = 1;
    // バッファ数を 4 に設定
    // 3(Buffer-1)回目に呼ばれた Next() で計測した結果が集計される。
    // (デフォルトのバッファ数 2 では、計測後 1(buffer-1)回目に呼ばれた Next() で集計される)
    cpuMeterManager.Initialize(1, 4, sectionCount);
    cpuMeter = cpuMeterManager.GetMeter();

    // 1 フレーム目
    cpuMeter->BeginMeasure("Test1", 1, nn::util::Color4u8::Red());
    Sleep();
    cpuMeter->EndMeasure();

    // 1 回目の Next()。空の結果を取得。
    cpuMeter->Next();
    {
        const nn::perf::LoadMeterBase::Section& section = cpuMeter->GetLastResult(0);
        EXPECT_EQ(0, section.begin.ToTimeSpan().GetNanoSeconds());
        EXPECT_EQ(0, section.end.ToTimeSpan().GetNanoSeconds());
        EXPECT_EQ(NULL, section.name);
        EXPECT_EQ(0, section.tag);
        EXPECT_EQ(nn::util::Color4u8::Green(), section.color);
    }

    // 2 フレーム目
    cpuMeter->BeginMeasure("Test2", 2, nn::util::Color4u8::Blue());
    Sleep();
    cpuMeter->EndMeasure();

    // 2 回目の Next()。空の結果を取得。
    cpuMeter->Next();
    {
        const nn::perf::LoadMeterBase::Section& section = cpuMeter->GetLastResult(0);
        EXPECT_EQ(0, section.begin.ToTimeSpan().GetNanoSeconds());
        EXPECT_EQ(0, section.end.ToTimeSpan().GetNanoSeconds());
        EXPECT_EQ(NULL, section.name);
        EXPECT_EQ(0, section.tag);
        EXPECT_EQ(nn::util::Color4u8::Green(), section.color);
    }

    // 3 フレーム目
    cpuMeter->BeginMeasure("Test3", 3, nn::util::Color4u8::Yellow());
    Sleep();
    cpuMeter->EndMeasure();

    // 3 回目の Next()。1 フレーム目の結果を取得。
    cpuMeter->Next();
    {
        const nn::perf::LoadMeterBase::Section& section = cpuMeter->GetLastResult(0);
        EXPECT_LT(0, (section.end - section.begin).ToTimeSpan().GetNanoSeconds());
        EXPECT_LT(0, section.begin.ToTimeSpan().GetNanoSeconds());
        EXPECT_LT(0, section.end.ToTimeSpan().GetNanoSeconds());
        EXPECT_EQ("Test1", section.name);
        EXPECT_EQ(1, section.tag);
        EXPECT_EQ(nn::util::Color4u8::Red(), section.color);
    }

    // 4 フレーム目
    // 4 回目の Next()。2 フレーム目の結果を取得。
    cpuMeter->Next();
    {
        const nn::perf::LoadMeterBase::Section& section = cpuMeter->GetLastResult(0);
        EXPECT_LT(0, (section.end - section.begin).ToTimeSpan().GetNanoSeconds());
        EXPECT_LT(0, section.begin.ToTimeSpan().GetNanoSeconds());
        EXPECT_LT(0, section.end.ToTimeSpan().GetNanoSeconds());
        EXPECT_EQ("Test2", section.name);
        EXPECT_EQ(2, section.tag);
        EXPECT_EQ(nn::util::Color4u8::Blue(), section.color);
    }

    // 5 フレーム目
    // 5 回目の Next()。3 フレーム目の結果を取得。
    cpuMeter->Next();
    {
        const nn::perf::LoadMeterBase::Section& section = cpuMeter->GetLastResult(0);
        EXPECT_LT(0, (section.end - section.begin).ToTimeSpan().GetNanoSeconds());
        EXPECT_LT(0, section.begin.ToTimeSpan().GetNanoSeconds());
        EXPECT_LT(0, section.end.ToTimeSpan().GetNanoSeconds());
        EXPECT_EQ("Test3", section.name);
        EXPECT_EQ(3, section.tag);
        EXPECT_EQ(nn::util::Color4u8::Yellow(), section.color);
    }

    cpuMeterManager.Finalize();

    SUCCEED();
}

// 入れ子で計測できるか判断する
TEST(CpuMeasure, MeasureNest)
{
    CpuMeterManager cpuMeterManager;
    nn::perf::CpuMeter* cpuMeter;
    int sectionCount;

    sectionCount = 16;
    cpuMeterManager.Initialize(1, 2, sectionCount);
    cpuMeter = cpuMeterManager.GetMeter();
    for(int i = 0; i < sectionCount; i++)
    {
        cpuMeter->BeginMeasure();
    }
    Sleep();
    for(int i = 0; i < sectionCount; i++)
    {
        cpuMeter->EndMeasure();
    }
    cpuMeter->Next();

    EXPECT_EQ(sectionCount, cpuMeter->GetLastSectionCount());
    EXPECT_LT(0, (cpuMeter->GetLastTotalEnd() - cpuMeter->GetLastTotalBegin()).ToTimeSpan().GetNanoSeconds());
    EXPECT_EQ(cpuMeter->GetLastTotalBegin(), cpuMeter->GetLastResult(0).begin);
    EXPECT_EQ(cpuMeter->GetLastTotalEnd(), cpuMeter->GetLastResult(0).end);

    for(int i = 0; i < sectionCount; i++)
    {
        const nn::perf::LoadMeterBase::Section& section = cpuMeter->GetLastResult(i);
        EXPECT_LT(0, (section.end - section.begin).ToTimeSpan().GetNanoSeconds());
        EXPECT_EQ(i, section.depth);
    }
    cpuMeterManager.Finalize();

    sectionCount = 10240;
    cpuMeterManager.Initialize(1, 2, sectionCount);
    cpuMeter = cpuMeterManager.GetMeter();
    for(int i = 0; i < sectionCount; i++)
    {
        cpuMeter->BeginMeasure();
    }
    Sleep();
    for(int i = 0; i < sectionCount; i++)
    {
        cpuMeter->EndMeasure();
    }
    cpuMeter->Next();

    EXPECT_EQ(sectionCount, cpuMeter->GetLastSectionCount());
    EXPECT_LT(0, (cpuMeter->GetLastTotalEnd() - cpuMeter->GetLastTotalBegin()).ToTimeSpan().GetNanoSeconds());
    EXPECT_EQ(cpuMeter->GetLastTotalBegin(), cpuMeter->GetLastResult(0).begin);
    EXPECT_EQ(cpuMeter->GetLastTotalEnd(), cpuMeter->GetLastResult(0).end);

    for(int i = 0; i < sectionCount; i++)
    {
        const nn::perf::LoadMeterBase::Section& section = cpuMeter->GetLastResult(i);
        EXPECT_LT(0, (section.end - section.begin).ToTimeSpan().GetNanoSeconds());
        EXPECT_EQ(i, section.depth);
    }
    cpuMeterManager.Finalize();
}

// Next() が呼ばれない時に結果を取得した場合
TEST(CpuMeasure, MeasureNull)
{
    CpuMeterManager cpuMeterManager;
    nn::perf::CpuMeter* cpuMeter;
    int sectionCount;

    sectionCount = 1024;
    cpuMeterManager.Initialize(1, 2, sectionCount);
    cpuMeter = cpuMeterManager.GetMeter();
    for(int i = 0; i < sectionCount; i++)
    {
        cpuMeter->BeginMeasure();
        cpuMeter->EndMeasure();
    }

    EXPECT_EQ(0, cpuMeter->GetLastSectionCount());
    EXPECT_EQ(0, cpuMeter->GetLastTotalBegin().ToTimeSpan().GetNanoSeconds());
    EXPECT_EQ(0, cpuMeter->GetLastTotalEnd().ToTimeSpan().GetNanoSeconds());

    for(int i = 0; i < 1024; i++)
    {
        const nn::perf::LoadMeterBase::Section& section = cpuMeter->GetLastResult(i);
        EXPECT_EQ(0, section.begin.ToTimeSpan().GetNanoSeconds());
        EXPECT_EQ(0, section.end.ToTimeSpan().GetNanoSeconds());
        EXPECT_EQ(0, section.depth);
        EXPECT_EQ(0, section.tag);
        EXPECT_EQ(NULL, section.name);
    }

    cpuMeterManager.Finalize();
}

// 計測結果が妥当か判断する
TEST(CpuMeasure, CheckResult)
{
    CpuMeterManager cpuMeterManager;
    nn::perf::CpuMeter* cpuMeter;
    int sectionCount = 64;

    cpuMeterManager.Initialize(1, 2, sectionCount);
    cpuMeter = cpuMeterManager.GetMeter();

    for(int i = 0; i < sectionCount; i++)
    {
        cpuMeter->BeginMeasure();
        Sleep();
        cpuMeter->EndMeasure();
    }
    cpuMeter->Next();

    EXPECT_EQ(sectionCount, cpuMeter->GetLastSectionCount());
    EXPECT_LT(0, (cpuMeter->GetLastTotalEnd() - cpuMeter->GetLastTotalBegin()).ToTimeSpan().GetNanoSeconds());
    EXPECT_EQ(cpuMeter->GetLastTotalBegin(), cpuMeter->GetLastResult(0).begin);
    EXPECT_EQ(cpuMeter->GetLastTotalEnd(), cpuMeter->GetLastResult(sectionCount - 1).end);

    for(int i = 0; i < sectionCount; i++)
    {
        const nn::perf::LoadMeterBase::Section& section = cpuMeter->GetLastResult(i);
        EXPECT_LT(0, (section.end - section.begin).ToTimeSpan().GetNanoSeconds());
        EXPECT_EQ(0, section.depth);
        EXPECT_EQ(0, section.tag);
        EXPECT_EQ(NULL, section.name);
    }

    cpuMeter->SetColor(nn::util::Color4u8::White());

    cpuMeter->BeginMeasure();
    Sleep();
    cpuMeter->EndMeasure();

    cpuMeter->BeginMeasure("Test1");
    Sleep();
    cpuMeter->EndMeasure();

    cpuMeter->BeginMeasure("Test2", 32, nn::util::Color4u8::Red());
    Sleep();
    cpuMeter->EndMeasure();

    cpuMeter->BeginMeasure("Test3", 16);
    Sleep();
    cpuMeter->EndMeasure();

    cpuMeter->SetColor(nn::util::Color4u8::Blue());

    cpuMeter->BeginMeasure(16);
    Sleep();
    cpuMeter->EndMeasure();

    cpuMeter->BeginMeasure(4, nn::util::Color4u8::Yellow());
    Sleep();
    cpuMeter->EndMeasure();

    cpuMeter->BeginMeasure("Test4", nn::util::Color4u8::Black());
    Sleep();
    cpuMeter->EndMeasure();

    cpuMeter->Next();

    EXPECT_EQ(7, cpuMeter->GetLastSectionCount());
    EXPECT_LT(0, (cpuMeter->GetLastTotalEnd() - cpuMeter->GetLastTotalBegin()).ToTimeSpan().GetNanoSeconds());
    EXPECT_EQ(cpuMeter->GetLastTotalBegin(), cpuMeter->GetLastResult(0).begin);
    EXPECT_EQ(cpuMeter->GetLastTotalEnd(), cpuMeter->GetLastResult(6).end);

    const nn::perf::LoadMeterBase::Section& section = cpuMeter->GetLastResult(0);
    EXPECT_LT(0, (section.end - section.begin).ToTimeSpan().GetNanoSeconds());
    EXPECT_EQ(NULL, section.name);
    EXPECT_EQ(0, section.tag);
    EXPECT_EQ(nn::util::Color4u8::White(), section.color);

    const nn::perf::LoadMeterBase::Section& section1 = cpuMeter->GetLastResult(1);
    EXPECT_LT(0, (section1.end - section1.begin).ToTimeSpan().GetNanoSeconds());
    EXPECT_EQ("Test1", section1.name);
    EXPECT_EQ(0, section1.tag);
    EXPECT_EQ(nn::util::Color4u8::White(), section1.color);

    const nn::perf::LoadMeterBase::Section& section2 = cpuMeter->GetLastResult(2);
    EXPECT_LT(0, (section2.end - section2.begin).ToTimeSpan().GetNanoSeconds());
    EXPECT_EQ("Test2", section2.name);
    EXPECT_EQ(32, section2.tag);
    EXPECT_EQ(nn::util::Color4u8::Red(), section2.color);

    const nn::perf::LoadMeterBase::Section& section3 = cpuMeter->GetLastResult(3);
    EXPECT_LT(0, (section3.end - section3.begin).ToTimeSpan().GetNanoSeconds());
    EXPECT_EQ("Test3", section3.name);
    EXPECT_EQ(16, section3.tag);
    EXPECT_EQ(nn::util::Color4u8::White(), section3.color);

    const nn::perf::LoadMeterBase::Section& section4 = cpuMeter->GetLastResult(4);
    EXPECT_LT(0, (section4.end - section4.begin).ToTimeSpan().GetNanoSeconds());
    EXPECT_EQ(NULL, section4.name);
    EXPECT_EQ(16, section4.tag);
    EXPECT_EQ(nn::util::Color4u8::Blue(), section4.color);

    const nn::perf::LoadMeterBase::Section& section5 = cpuMeter->GetLastResult(5);
    EXPECT_LT(0, (section5.end - section5.begin).ToTimeSpan().GetNanoSeconds());
    EXPECT_EQ(NULL, section5.name);
    EXPECT_EQ(4, section5.tag);
    EXPECT_EQ(nn::util::Color4u8::Yellow(), section5.color);

    const nn::perf::LoadMeterBase::Section& section6 = cpuMeter->GetLastResult(6);
    EXPECT_LT(0, (section6.end - section6.begin).ToTimeSpan().GetNanoSeconds());
    EXPECT_EQ("Test4", section6.name);
    EXPECT_EQ(0, section6.tag);
    EXPECT_EQ(nn::util::Color4u8::Black(), section6.color);

    EXPECT_EQ(nn::util::Color4u8::Blue(), cpuMeter->GetColor());

    cpuMeterManager.Finalize();
}
