﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "testPerf_Main.h"
#include <nnt.h>

// 繰り返し初期化と終了を行う
TEST(CpuMeasureLoadMeterCenter, RepeatInitialize)
{
    nn::os::SetThreadCoreMask(nn::os::GetCurrentThread(), 0, 1);
    for(int i = 1; i < 1000; i++)
    {
        LoadMeterCenterManager::Initialize(false, 3, 0, 2, 0, i);

        NN_PERF_BEGIN_FRAME();
        NN_PERF_BEGIN_MEASURE();
        Sleep();
        NN_PERF_END_MEASURE();
        NN_PERF_END_FRAME();

        LoadMeterCenterManager::Finalize();
    }

    SUCCEED();
}

// info 初期値
TEST(CpuMeasureLoadMeterCenter, InfoDefault)
{
    nn::perf::LoadMeterCenterInfo info;
    EXPECT_EQ(2, info.GetCpuBufferCount());
    EXPECT_EQ(2, info.GetGpuBufferCount());
    EXPECT_EQ(0, info.GetCoreCount());
    EXPECT_EQ(0, info.GetCpuSectionCountMax());
    EXPECT_EQ(0, info.GetGpuSectionCountMax());
    EXPECT_EQ(0, info.GetUserMeterCount());
}

// 設定
TEST(CpuMeasureLoadMeterCenter, Set)
{
    nn::os::SetThreadCoreMask(nn::os::GetCurrentThread(), 0, 1);
    LoadMeterCenterManager::Initialize(false, 1, 1, 2, 0, 1);

    nn::perf::CpuMeter* frameMeter = NN_PERF_GET_FRAME_METER();
    nn::perf::CpuMeter* coreMeter = NN_PERF_GET_CORE_METER(0);
    nn::perf::CpuMeter* userMeter = NN_PERF_GET_USER_METER(0);

    EXPECT_EQ(nn::util::Color4u8(16,16,16,150), frameMeter->GetColor());
    EXPECT_EQ(nn::util::Color4u8::Green(), coreMeter->GetColor());
    EXPECT_EQ(nn::util::Color4u8::Green(), userMeter->GetColor());

    EXPECT_EQ(60.f, nn::perf::LoadMeterCenter::GetFrameRate());
    EXPECT_EQ(0, nn::util::Strncmp(frameMeter->GetName(), "Frame", 128));
    EXPECT_EQ(0, nn::util::Strncmp(coreMeter->GetName(), "CPU0", 128));
    EXPECT_EQ(0, nn::util::Strncmp(userMeter->GetName(), "USER0", 128));

    NN_PERF_SET_FRAME_RATE(30.f);
    NN_PERF_SET_COLOR(nn::util::Color4u8::Red());
    NN_PERF_SET_COLOR_INDEX(0, nn::util::Color4u8::Yellow());
    NN_PERF_SET_METER_NAME(0, "Test1");

    EXPECT_EQ(nn::util::Color4u8(16,16,16,150), frameMeter->GetColor());
    EXPECT_EQ(nn::util::Color4u8::Red(), coreMeter->GetColor());
    EXPECT_EQ(nn::util::Color4u8::Yellow(), userMeter->GetColor());

    EXPECT_EQ(30.f, nn::perf::LoadMeterCenter::GetFrameRate());
    EXPECT_EQ(0, nn::util::Strncmp(frameMeter->GetName(), "Frame", 128));
    EXPECT_EQ(0, nn::util::Strncmp(coreMeter->GetName(), "CPU0", 128));
    EXPECT_EQ(0, nn::util::Strncmp(userMeter->GetName(), "Test1", 128));

    LoadMeterCenterManager::Finalize();
}

// アタッチされているかをテスト
TEST(CpuMeasureLoadMeterCenter, TestAttach)
{
    nn::os::SetThreadCoreMask(nn::os::GetCurrentThread(), 0, 1);
    LoadMeterCenterManager::Initialize(false, 3, 3, 2, 0, 1);

    nn::perf::CpuMeter* frameMeter = NN_PERF_GET_FRAME_METER();
    nn::perf::CpuMeter* coreMeter[3];
    coreMeter[0] = NN_PERF_GET_CORE_METER(0);
    coreMeter[1] = NN_PERF_GET_CORE_METER(1);
    coreMeter[2] = NN_PERF_GET_CORE_METER(2);
    nn::perf::CpuMeter* userMeter[3];
    userMeter[0] = NN_PERF_GET_USER_METER(0);
    userMeter[1] = NN_PERF_GET_USER_METER(1);
    userMeter[2] = NN_PERF_GET_USER_METER(2);

    ASSERT_FALSE(frameMeter->IsLinked());

    ASSERT_TRUE(coreMeter[0]->IsLinked());
    ASSERT_TRUE(coreMeter[1]->IsLinked());
    ASSERT_TRUE(coreMeter[2]->IsLinked());

    ASSERT_TRUE(userMeter[0]->IsLinked());
    ASSERT_TRUE(userMeter[1]->IsLinked());
    ASSERT_TRUE(userMeter[2]->IsLinked());

    LoadMeterCenterManager::Finalize();
}


// 設定した最大区間を計測する
TEST(CpuMeasureLoadMeterCenter, MeasureMaxSection)
{
    int sectionCount = 1;
    nn::os::SetThreadCoreMask(nn::os::GetCurrentThread(), 0, 1);
    LoadMeterCenterManager::Initialize(false, 1, 0, 2, 0, sectionCount);

    nn::perf::CpuMeter* frameMeter = NN_PERF_GET_FRAME_METER();
    nn::perf::CpuMeter* coreMeter = NN_PERF_GET_CORE_METER(0);

    NN_PERF_BEGIN_FRAME();
    for(int i = 0; i < sectionCount; i++)
    {
        NN_PERF_BEGIN_MEASURE_NAME("test1");
        Sleep();
        NN_PERF_END_MEASURE();
    }
    NN_PERF_END_FRAME();

    EXPECT_EQ(1, frameMeter->GetLastSectionCount());
    EXPECT_LT(0, (frameMeter->GetLastTotalEnd() - frameMeter->GetLastTotalBegin()).ToTimeSpan().GetNanoSeconds());
    EXPECT_EQ(sectionCount, coreMeter->GetLastSectionCount());
    EXPECT_LT(0, (coreMeter->GetLastTotalEnd() - coreMeter->GetLastTotalBegin()).ToTimeSpan().GetNanoSeconds());
    EXPECT_EQ(coreMeter->GetLastTotalBegin(), coreMeter->GetLastResult(0).begin);
    EXPECT_EQ(coreMeter->GetLastTotalEnd(), coreMeter->GetLastResult(sectionCount - 1).end);

    for(int i = 0; i < sectionCount; i++)
    {
        const nn::perf::LoadMeterBase::Section& section = coreMeter->GetLastResult(i);
        EXPECT_LT(0, (section.end - section.begin).ToTimeSpan().GetNanoSeconds());
        EXPECT_EQ(0, section.depth);
        EXPECT_EQ(0, section.tag);
        EXPECT_EQ("test1", section.name);
    }
    LoadMeterCenterManager::Finalize();

    sectionCount = 5000;
    nn::os::SetThreadCoreMask(nn::os::GetCurrentThread(), 0, 1);
    LoadMeterCenterManager::Initialize(false, 1, 0, 2, 0, sectionCount);
    frameMeter = NN_PERF_GET_FRAME_METER();
    coreMeter = NN_PERF_GET_CORE_METER(0);

    NN_PERF_BEGIN_FRAME();
    for(int i = 0; i < sectionCount; i++)
    {
        NN_PERF_BEGIN_MEASURE_NAME_AND_TAG("test2", i);
        Sleep();
        NN_PERF_END_MEASURE();
    }
    NN_PERF_END_FRAME();

    EXPECT_EQ(1, frameMeter->GetLastSectionCount());
    EXPECT_LT(0, (frameMeter->GetLastTotalEnd() - frameMeter->GetLastTotalBegin()).ToTimeSpan().GetNanoSeconds());
    EXPECT_EQ(sectionCount, coreMeter->GetLastSectionCount());
    EXPECT_LT(0, (coreMeter->GetLastTotalEnd() - coreMeter->GetLastTotalBegin()).ToTimeSpan().GetNanoSeconds());
    EXPECT_EQ(coreMeter->GetLastTotalBegin(), coreMeter->GetLastResult(0).begin);
    EXPECT_EQ(coreMeter->GetLastTotalEnd(), coreMeter->GetLastResult(sectionCount - 1).end);

    for(int i = 0; i < sectionCount; i++)
    {
        const nn::perf::LoadMeterBase::Section& section = coreMeter->GetLastResult(i);
        EXPECT_LT(0, (section.end - section.begin).ToTimeSpan().GetNanoSeconds());
        EXPECT_EQ(0, section.depth);
        EXPECT_EQ(i, section.tag);
        EXPECT_EQ("test2", section.name);
    }
    LoadMeterCenterManager::Finalize();
}

// バッファ数に対応した結果が取得できるか
TEST(CpuMeasureLoadMeterCenter, TestBufferCount)
{
    nn::os::SetThreadCoreMask(nn::os::GetCurrentThread(), 0, 1);

    // バッファ数を 3 に設定
    // 2(Buffer-1)回目に呼ばれた NN_PERF_END_FRAME で計測した結果が集計される。
    // (デフォルトのバッファ数 2 では、計測後 1(buffer-1)回目に呼ばれた NN_PERF_END_FRAME で集計される)
    LoadMeterCenterManager::Initialize(false, 1, 0, 3, 0, 1);
    nn::perf::CpuMeter* frameMeter = NN_PERF_GET_FRAME_METER();
    nn::perf::CpuMeter* coreMeter = NN_PERF_GET_CORE_METER(0);

    // 1 フレーム目
    NN_PERF_BEGIN_FRAME();
    NN_PERF_BEGIN_MEASURE_NAME_AND_TAG("Test1", 1);
    Sleep();
    NN_PERF_END_MEASURE();
    NN_PERF_END_FRAME();

    // 1 フレーム後。空の結果を取得(GetLastSectionCount() == 0 の場合はそもそもアクセスしない方がベター)。
    {
        EXPECT_EQ(0, frameMeter->GetLastSectionCount());
        EXPECT_EQ(0, frameMeter->GetLastTotalBegin().ToTimeSpan().GetNanoSeconds());
        EXPECT_EQ(0, frameMeter->GetLastTotalEnd().ToTimeSpan().GetNanoSeconds());
        EXPECT_EQ(0, coreMeter->GetLastSectionCount());
        EXPECT_EQ(0, coreMeter->GetLastTotalBegin().ToTimeSpan().GetNanoSeconds());
        EXPECT_EQ(0, coreMeter->GetLastTotalEnd().ToTimeSpan().GetNanoSeconds());
        const nn::perf::LoadMeterBase::Section& section = coreMeter->GetLastResult(0);
        EXPECT_EQ(0, section.begin.ToTimeSpan().GetNanoSeconds());
        EXPECT_EQ(0, section.end.ToTimeSpan().GetNanoSeconds());
        EXPECT_EQ(NULL, section.name);
        EXPECT_EQ(0, section.tag);
        EXPECT_EQ(nn::util::Color4u8::Green(), section.color);
    }

    // 2 フレーム目
    NN_PERF_BEGIN_FRAME();
    NN_PERF_BEGIN_MEASURE_NAME_AND_TAG("Test2", 2);
    Sleep();
    NN_PERF_END_MEASURE();
    NN_PERF_END_FRAME();

    // 2  フレーム後。1 フレーム目の結果を取得。
    {
        EXPECT_EQ(1, frameMeter->GetLastSectionCount());
        EXPECT_LT(0, (frameMeter->GetLastTotalEnd() - frameMeter->GetLastTotalBegin()).ToTimeSpan().GetNanoSeconds());
        EXPECT_EQ(1, coreMeter->GetLastSectionCount());
        EXPECT_LT(0, (coreMeter->GetLastTotalEnd() - coreMeter->GetLastTotalBegin()).ToTimeSpan().GetNanoSeconds());
        const nn::perf::LoadMeterBase::Section& section = coreMeter->GetLastResult(0);
        EXPECT_LT(0, (section.end - section.begin).ToTimeSpan().GetNanoSeconds());
        EXPECT_LT(0, section.begin.ToTimeSpan().GetNanoSeconds());
        EXPECT_LT(0, section.end.ToTimeSpan().GetNanoSeconds());
        EXPECT_EQ("Test1", section.name);
        EXPECT_EQ(1, section.tag);
    }

    // 3 フレーム目
    NN_PERF_BEGIN_FRAME();
    NN_PERF_BEGIN_MEASURE_NAME_AND_TAG("Test3", 3);
    Sleep();
    NN_PERF_END_MEASURE();
    NN_PERF_END_FRAME();

    // 3  フレーム後。2 フレーム目の結果を取得。
    {
        EXPECT_EQ(1, frameMeter->GetLastSectionCount());
        EXPECT_LT(0, (frameMeter->GetLastTotalEnd() - frameMeter->GetLastTotalBegin()).ToTimeSpan().GetNanoSeconds());
        EXPECT_EQ(1, coreMeter->GetLastSectionCount());
        EXPECT_LT(0, (coreMeter->GetLastTotalEnd() - coreMeter->GetLastTotalBegin()).ToTimeSpan().GetNanoSeconds());
        const nn::perf::LoadMeterBase::Section& section = coreMeter->GetLastResult(0);
        EXPECT_LT(0, (section.end - section.begin).ToTimeSpan().GetNanoSeconds());
        EXPECT_LT(0, section.begin.ToTimeSpan().GetNanoSeconds());
        EXPECT_LT(0, section.end.ToTimeSpan().GetNanoSeconds());
        EXPECT_EQ("Test2", section.name);
        EXPECT_EQ(2, section.tag);
    }

    // 4 フレーム目
    NN_PERF_BEGIN_FRAME();
    NN_PERF_END_FRAME();

    // 4 フレーム後。3 フレーム目の結果を取得。
    {
        EXPECT_EQ(1, frameMeter->GetLastSectionCount());
        EXPECT_LT(0, (frameMeter->GetLastTotalEnd() - frameMeter->GetLastTotalBegin()).ToTimeSpan().GetNanoSeconds());
        EXPECT_EQ(1, coreMeter->GetLastSectionCount());
        EXPECT_LT(0, (coreMeter->GetLastTotalEnd() - coreMeter->GetLastTotalBegin()).ToTimeSpan().GetNanoSeconds());
        const nn::perf::LoadMeterBase::Section& section = coreMeter->GetLastResult(0);
        EXPECT_LT(0, (section.end - section.begin).ToTimeSpan().GetNanoSeconds());
        EXPECT_LT(0, section.begin.ToTimeSpan().GetNanoSeconds());
        EXPECT_LT(0, section.end.ToTimeSpan().GetNanoSeconds());
        EXPECT_EQ("Test3", section.name);
        EXPECT_EQ(3, section.tag);
    }
    LoadMeterCenterManager::Finalize();
}

// 入れ子で計測できるか判断する
TEST(CpuMeasureLoadMeterCenter, MeasureNest)
{
    int sectionCount = 1;
    nn::os::SetThreadCoreMask(nn::os::GetCurrentThread(), 0, 1);
    LoadMeterCenterManager::Initialize(false, 1, 0, 2, 0, sectionCount);
    nn::perf::CpuMeter* coreMeter = NN_PERF_GET_CORE_METER(0);

    NN_PERF_BEGIN_FRAME();
    for(int i = 0; i < sectionCount; i++)
    {
        NN_PERF_BEGIN_MEASURE_NAME_AND_TAG("Test3", 3);
    }
    Sleep();
    for(int i = 0; i < sectionCount; i++)
    {
        NN_PERF_END_MEASURE();
    }
    NN_PERF_END_FRAME();

    EXPECT_EQ(sectionCount, coreMeter->GetLastSectionCount());
    EXPECT_LT(0, (coreMeter->GetLastTotalEnd() - coreMeter->GetLastTotalBegin()).ToTimeSpan().GetNanoSeconds());
    EXPECT_EQ(coreMeter->GetLastTotalBegin(), coreMeter->GetLastResult(0).begin);
    EXPECT_EQ(coreMeter->GetLastTotalEnd(), coreMeter->GetLastResult(0).end);

    for(int i = 0; i < sectionCount; i++)
    {
        const nn::perf::LoadMeterBase::Section& section = coreMeter->GetLastResult(i);
        EXPECT_LT(0, (section.end - section.begin).ToTimeSpan().GetNanoSeconds());
        EXPECT_EQ(i, section.depth);
    }
    LoadMeterCenterManager::Finalize();

    sectionCount = 10240;
    nn::os::SetThreadCoreMask(nn::os::GetCurrentThread(), 0, 1);
    LoadMeterCenterManager::Initialize(false, 1, 0, 2, 0, sectionCount);
    coreMeter = NN_PERF_GET_CORE_METER(0);

    NN_PERF_BEGIN_FRAME();
    for(int i = 0; i < sectionCount; i++)
    {
        NN_PERF_BEGIN_MEASURE_NAME_AND_TAG("Test3", 3);
    }
    Sleep();
    for(int i = 0; i < sectionCount; i++)
    {
        NN_PERF_END_MEASURE();
    }
    NN_PERF_END_FRAME();

    EXPECT_EQ(sectionCount, coreMeter->GetLastSectionCount());
    EXPECT_LT(0, (coreMeter->GetLastTotalEnd() - coreMeter->GetLastTotalBegin()).ToTimeSpan().GetNanoSeconds());
    EXPECT_EQ(coreMeter->GetLastTotalBegin(), coreMeter->GetLastResult(0).begin);
    EXPECT_EQ(coreMeter->GetLastTotalEnd(), coreMeter->GetLastResult(0).end);

    for(int i = 0; i < sectionCount; i++)
    {
        const nn::perf::LoadMeterBase::Section& section = coreMeter->GetLastResult(i);
        EXPECT_LT(0, (section.end - section.begin).ToTimeSpan().GetNanoSeconds());
        EXPECT_EQ(i, section.depth);
    }
    LoadMeterCenterManager::Finalize();
}

// Next() が呼ばれない時に結果を取得した場合
TEST(CpuMeasureLoadMeterCenter, MeasureNull)
{
    LoadMeterCenterManager::Initialize(false, 1, 0, 3, 0, 1);
    nn::perf::CpuMeter* frameMeter = NN_PERF_GET_FRAME_METER();
    nn::perf::CpuMeter* coreMeter = NN_PERF_GET_CORE_METER(0);

    NN_PERF_BEGIN_FRAME();
    NN_PERF_BEGIN_MEASURE_NAME_AND_TAG("Test1", 1);
    Sleep();
    NN_PERF_END_MEASURE();

    // 空の結果を取得。
    {
        EXPECT_EQ(0, frameMeter->GetLastSectionCount());
        EXPECT_EQ(0, frameMeter->GetLastTotalBegin().ToTimeSpan().GetNanoSeconds());
        EXPECT_EQ(0, frameMeter->GetLastTotalEnd().ToTimeSpan().GetNanoSeconds());
        EXPECT_EQ(0, coreMeter->GetLastSectionCount());
        EXPECT_EQ(0, coreMeter->GetLastTotalBegin().ToTimeSpan().GetNanoSeconds());
        EXPECT_EQ(0, coreMeter->GetLastTotalEnd().ToTimeSpan().GetNanoSeconds());
        const nn::perf::LoadMeterBase::Section& section = coreMeter->GetLastResult(0);
        EXPECT_EQ(0, section.begin.ToTimeSpan().GetNanoSeconds());
        EXPECT_EQ(0, section.end.ToTimeSpan().GetNanoSeconds());
        EXPECT_EQ(NULL, section.name);
        EXPECT_EQ(0, section.tag);
        EXPECT_EQ(nn::util::Color4u8::Green(), section.color);
    }
    NN_PERF_END_FRAME();
    LoadMeterCenterManager::Finalize();
}

// 計測結果が妥当か判断する
TEST(CpuMeasureLoadMeterCenter, CheckResult)
{
    int sectionCount = 64;
    nn::os::SetThreadCoreMask(nn::os::GetCurrentThread(), 0, 1);
    LoadMeterCenterManager::Initialize(false, 1, 0, 2, 0, sectionCount);
    nn::perf::CpuMeter* coreMeter = NN_PERF_GET_CORE_METER(0);

    NN_PERF_BEGIN_FRAME();

    NN_PERF_SET_COLOR(nn::util::Color4u8::White());

    NN_PERF_BEGIN_MEASURE();
    Sleep();
    NN_PERF_END_MEASURE();

    NN_PERF_BEGIN_MEASURE_NAME("Test1");
    Sleep();
    NN_PERF_END_MEASURE();

    NN_PERF_SET_COLOR(nn::util::Color4u8::Blue());

    NN_PERF_BEGIN_MEASURE_NAME_AND_TAG("Test2", 32);
    Sleep();
    NN_PERF_END_MEASURE();

    NN_PERF_BEGIN_MEASURE_TAG(16);
    Sleep();
    NN_PERF_END_MEASURE();

    NN_PERF_END_FRAME();

    EXPECT_EQ(4, coreMeter->GetLastSectionCount());
    EXPECT_LT(0, (coreMeter->GetLastTotalEnd() - coreMeter->GetLastTotalBegin()).ToTimeSpan().GetNanoSeconds());
    EXPECT_EQ(coreMeter->GetLastTotalBegin(), coreMeter->GetLastResult(0).begin);
    EXPECT_EQ(coreMeter->GetLastTotalEnd(), coreMeter->GetLastResult(4 - 1).end);

    const nn::perf::LoadMeterBase::Section& section = coreMeter->GetLastResult(0);
    EXPECT_LT(0, (section.end - section.begin).ToTimeSpan().GetNanoSeconds());
    EXPECT_EQ(NULL, section.name);
    EXPECT_EQ(0, section.tag);
    EXPECT_EQ(nn::util::Color4u8::White(), section.color);

    const nn::perf::LoadMeterBase::Section& section1 = coreMeter->GetLastResult(1);
    EXPECT_LT(0, (section1.end - section1.begin).ToTimeSpan().GetNanoSeconds());
    EXPECT_EQ("Test1", section1.name);
    EXPECT_EQ(0, section1.tag);
    EXPECT_EQ(nn::util::Color4u8::White(), section1.color);

    const nn::perf::LoadMeterBase::Section& section2 = coreMeter->GetLastResult(2);
    EXPECT_LT(0, (section2.end - section2.begin).ToTimeSpan().GetNanoSeconds());
    EXPECT_EQ("Test2", section2.name);
    EXPECT_EQ(32, section2.tag);
    EXPECT_EQ(nn::util::Color4u8::Blue(), section2.color);

    const nn::perf::LoadMeterBase::Section& section3 = coreMeter->GetLastResult(3);
    EXPECT_LT(0, (section3.end - section3.begin).ToTimeSpan().GetNanoSeconds());
    EXPECT_EQ(NULL, section3.name);
    EXPECT_EQ(16, section3.tag);
    EXPECT_EQ(nn::util::Color4u8::Blue(), section3.color);

    EXPECT_EQ(nn::util::Color4u8::Blue(), coreMeter->GetColor());

    LoadMeterCenterManager::Finalize();
}

// 有効無効
TEST(CpuMeasureLoadMeterCenter, CheckEnabled)
{
    int sectionCount = 64;
    nn::os::SetThreadCoreMask(nn::os::GetCurrentThread(), 0, 1);
    LoadMeterCenterManager::Initialize(false, 1, 0, 2, 0, sectionCount);
    nn::perf::CpuMeter* frameMeter = NN_PERF_GET_FRAME_METER();
    nn::perf::CpuMeter* coreMeter = NN_PERF_GET_CORE_METER(0);

    NN_PERF_SET_ENABLED(false);
    NN_PERF_BEGIN_FRAME();
    NN_PERF_BEGIN_MEASURE_NAME_AND_TAG("Test2", 32);
    Sleep();
    NN_PERF_END_MEASURE();
    NN_PERF_END_FRAME();

    EXPECT_EQ(0, frameMeter->GetLastSectionCount());
    EXPECT_EQ(0, frameMeter->GetLastTotalBegin().ToTimeSpan().GetNanoSeconds());
    EXPECT_EQ(0, frameMeter->GetLastTotalEnd().ToTimeSpan().GetNanoSeconds());
    EXPECT_EQ(0, coreMeter->GetLastSectionCount());
    EXPECT_EQ(0, coreMeter->GetLastTotalBegin().ToTimeSpan().GetNanoSeconds());
    EXPECT_EQ(0, coreMeter->GetLastTotalEnd().ToTimeSpan().GetNanoSeconds());

    NN_PERF_SET_ENABLED(true);
    NN_PERF_BEGIN_FRAME();
    NN_PERF_BEGIN_MEASURE_NAME_AND_TAG("Test2", 32);
    Sleep();
    NN_PERF_END_MEASURE();
    NN_PERF_END_FRAME();

    EXPECT_EQ(1, frameMeter->GetLastSectionCount());
    EXPECT_LT(0, (frameMeter->GetLastTotalEnd() - frameMeter->GetLastTotalBegin()).ToTimeSpan().GetNanoSeconds());
    EXPECT_EQ(1, coreMeter->GetLastSectionCount());
    EXPECT_LT(0, (coreMeter->GetLastTotalEnd() - coreMeter->GetLastTotalBegin()).ToTimeSpan().GetNanoSeconds());

    LoadMeterCenterManager::Finalize();
}

namespace{
const size_t                    threadStackSize = 0x4000;
NN_OS_ALIGNAS_THREAD_STACK char threadStack[3][threadStackSize];

nn::os::TlsSlot tlsSlot;
int GetCoreNumberFunction()
{
    int coreNumber = reinterpret_cast<TlsSlotValue*>(nn::os::GetTlsValue(tlsSlot))->coreNumber;
    return *reinterpret_cast<int*>(&coreNumber);
}

void ThreadMeasureCpu(void *tlsValue)
{
    nn::os::SetTlsValue(tlsSlot, reinterpret_cast<uintptr_t>(tlsValue));
    const char* name = reinterpret_cast<TlsSlotValue*>(nn::os::GetTlsValue(tlsSlot))->name;

    NN_PERF_BEGIN_MEASURE_NAME(name);
    Sleep();
    NN_PERF_END_MEASURE();
}
}

// 計測結果の取得
TEST(CpuMeasureLoadMeterCenter, CheckProfileResult)
{
    // アフィニティマスクの設定
    nn::Bit64 core0 = 1;
    nn::Bit64 core1 = 2;
    nn::Bit64 core2 = 4;

    nn::os::ThreadType              thread[3];
    TlsSlotValue                    tlsSlotValue[3];
    {
        nn::Result result = nn::os::AllocateTlsSlot(&tlsSlot, NULL);
        NN_ASSERT(result.IsSuccess(), "Cannot allocate TLS slot.");

        tlsSlotValue[0].coreNumber = 0;
        tlsSlotValue[0].name = "Test0";
        result = nn::os::CreateThread(&thread[0], ThreadMeasureCpu, &tlsSlotValue[0], threadStack[0], threadStackSize, nn::os::DefaultThreadPriority);
        nn::os::SetThreadName(&thread[0], "Thread0");
        nn::os::SetThreadCoreMask(&thread[0], 0, core0);

        tlsSlotValue[1].coreNumber = 1;
        tlsSlotValue[1].name = "Test1";
        result = nn::os::CreateThread(&thread[1], ThreadMeasureCpu, &tlsSlotValue[1], threadStack[1], threadStackSize, nn::os::DefaultThreadPriority);
        nn::os::SetThreadName(&thread[1], "Thread1");
        nn::os::SetThreadCoreMask(&thread[1], 1, core1);

        tlsSlotValue[2].coreNumber = 2;
        tlsSlotValue[2].name = "Test2";
        result = nn::os::CreateThread(&thread[2], ThreadMeasureCpu, &tlsSlotValue[2], threadStack[2], threadStackSize, nn::os::DefaultThreadPriority);
        nn::os::SetThreadName(&thread[2], "Thread2");
        nn::os::SetThreadCoreMask(&thread[2], 2, core2);

        nn::os::SetThreadCoreMask(nn::os::GetCurrentThread(), 0, core0);
        nn::os::SetTlsValue(tlsSlot, reinterpret_cast<uintptr_t>(&tlsSlotValue[0]));
    }

    int sectionCount = 64;
    nn::os::SetThreadCoreMask(nn::os::GetCurrentThread(), 0, 1);
    LoadMeterCenterManager::Initialize(false, 3, 2, 2, 0, sectionCount);
    NN_PERF_SET_GET_CORE_NUMBER_FUNCTION(GetCoreNumberFunction);
    NN_PERF_SET_METER_NAME(0, "Meter1");
    NN_PERF_SET_METER_NAME(1, "Meter2");

    nn::perf::CpuMeter* coreMeter[3];
    coreMeter[0] = NN_PERF_GET_CORE_METER(0);
    coreMeter[1] = NN_PERF_GET_CORE_METER(1);
    coreMeter[2] = NN_PERF_GET_CORE_METER(2);

    nn::perf::CpuMeter* userMeter[2];
    userMeter[0] = NN_PERF_GET_USER_METER(0);
    userMeter[1] = NN_PERF_GET_USER_METER(1);

    // 計測前の空結果
    EXPECT_EQ(0, NN_PERF_GET_CALL_COUNT_CPU(NULL, NULL, 0));
    EXPECT_EQ(0, NN_PERF_GET_CALL_COUNT_CPU(NULL, NULL, 8));
    EXPECT_EQ(0, NN_PERF_GET_CALL_COUNT_CPU(NULL, NULL, 16));
    EXPECT_EQ(0, NN_PERF_GET_CALL_COUNT_CPU(NULL, NULL, 32));
    EXPECT_EQ(0, NN_PERF_GET_CALL_COUNT_CPU(NULL, "Test1", 0));
    EXPECT_EQ(0, NN_PERF_GET_CALL_COUNT_CPU(NULL, "Test2", 0));
    EXPECT_EQ(0, NN_PERF_GET_CALL_COUNT_CPU(NULL, "Test1", 8));
    EXPECT_EQ(0, NN_PERF_GET_CALL_COUNT_CPU("CPU0", NULL, 0));
    EXPECT_EQ(0, NN_PERF_GET_CALL_COUNT_CPU("CPU1", NULL, 0));
    EXPECT_EQ(0, NN_PERF_GET_CALL_COUNT_CPU("CPU2", NULL, 0));
    EXPECT_EQ(0, NN_PERF_GET_CALL_COUNT_CPU("Meter1", NULL, 0));
    EXPECT_EQ(0, NN_PERF_GET_CALL_COUNT_CPU("Meter2", NULL, 0));
    EXPECT_EQ(0, NN_PERF_GET_CALL_COUNT_CPU("CPU0", "Test2", 32));
    EXPECT_EQ(0, NN_PERF_GET_CALL_COUNT_CPU("Meter1", "Test1", 32));
    EXPECT_EQ(0 ,NN_PERF_GET_CALL_COUNT_CPU("Meter2", "Test1", 32));

    EXPECT_EQ(0, NN_PERF_GET_AVERAGE_ELAPSED_TIME_CPU("CPU0", NULL, 0).GetNanoSeconds());
    EXPECT_EQ(0, NN_PERF_GET_AVERAGE_ELAPSED_TIME_CPU(NULL, NULL, 0).GetNanoSeconds());
    EXPECT_EQ(0, NN_PERF_GET_AVERAGE_ELAPSED_TIME_CPU(NULL, "Test1", 0).GetNanoSeconds());

    EXPECT_EQ(0, NN_PERF_GET_ELAPSED_TIME_CPU("CPU0", "Test2", 32).GetNanoSeconds());
    EXPECT_EQ(0, NN_PERF_GET_ELAPSED_TIME_CPU("CPU2", NULL, 0).GetNanoSeconds());
    EXPECT_EQ(0, NN_PERF_GET_ELAPSED_TIME_CPU(NULL, NULL, 16).GetNanoSeconds());

    int64_t min = nn::TimeSpan( std::numeric_limits<nn::TimeSpan>::min() ).GetNanoSeconds();
    int64_t max = nn::TimeSpan::FromSeconds( 999 ).GetNanoSeconds();

    EXPECT_EQ(min, NN_PERF_GET_MAX_ELAPSED_TIME_CPU("CPU0", NULL, 0).GetNanoSeconds());
    EXPECT_EQ(min, NN_PERF_GET_MAX_ELAPSED_TIME_CPU(NULL, NULL, 0).GetNanoSeconds());
    EXPECT_EQ(min, NN_PERF_GET_MAX_ELAPSED_TIME_CPU(NULL, "Test1", 0).GetNanoSeconds());

    EXPECT_EQ(max, NN_PERF_GET_MIN_ELAPSED_TIME_CPU("CPU0", NULL, 0).GetNanoSeconds());
    EXPECT_EQ(max, NN_PERF_GET_MIN_ELAPSED_TIME_CPU("Meter2", NULL, 0).GetNanoSeconds());
    EXPECT_EQ(max, NN_PERF_GET_MIN_ELAPSED_TIME_CPU(NULL, NULL, 32).GetNanoSeconds());

    NN_PERF_BEGIN_FRAME();
    {
        nn::os::StartThread(&thread[0]);
        nn::os::StartThread(&thread[1]);
        nn::os::StartThread(&thread[2]);
        nn::os::WaitThread(&thread[0]);
        nn::os::WaitThread(&thread[1]);
        nn::os::WaitThread(&thread[2]);

        NN_PERF_BEGIN_MEASURE();
        Sleep();
        NN_PERF_END_MEASURE();

        NN_PERF_BEGIN_MEASURE_TAG(16);
        Sleep();
        NN_PERF_END_MEASURE();

        {
            NN_PERF_AUTO_MEASURE_NAME("Test1");
            Sleep();
        }

        NN_PERF_BEGIN_MEASURE_NAME_AND_TAG("Test2", 32);
        Sleep();
        NN_PERF_END_MEASURE();

        NN_PERF_BEGIN_MEASURE_NAME_AND_TAG("Test1", 32);
        Sleep();
        NN_PERF_END_MEASURE();

        {
            NN_PERF_AUTO_MEASURE_NAME("Test2");
            Sleep();
        }

        NN_PERF_BEGIN_MEASURE_TAG(16);
        Sleep();
        NN_PERF_END_MEASURE();

        NN_PERF_BEGIN_MEASURE_INDEX_TAG(1, 16);
        Sleep();
        NN_PERF_END_MEASURE_INDEX(1);

        {
            NN_PERF_AUTO_MEASURE_INDEX_NAME_AND_TAG(1, "Test1", 32);
            Sleep();
        }

        NN_PERF_BEGIN_MEASURE_INDEX_NAME_AND_TAG(0, "Test1", 8);
        Sleep();
        NN_PERF_END_MEASURE_INDEX(0);

        {
            NN_PERF_AUTO_MEASURE_INDEX_TAG(0, 16);
            Sleep();
        }
    }
    NN_PERF_END_FRAME();

    EXPECT_EQ(8, coreMeter[0]->GetLastSectionCount());
    EXPECT_EQ(1, coreMeter[1]->GetLastSectionCount());
    EXPECT_EQ(1, coreMeter[2]->GetLastSectionCount());
    EXPECT_EQ(2, userMeter[0]->GetLastSectionCount());
    EXPECT_EQ(2, userMeter[1]->GetLastSectionCount());

    EXPECT_EQ(14, NN_PERF_GET_CALL_COUNT_CPU(NULL, NULL, 0));
    EXPECT_EQ(1, NN_PERF_GET_CALL_COUNT_CPU(NULL, NULL, 8));
    EXPECT_EQ(4, NN_PERF_GET_CALL_COUNT_CPU(NULL, NULL, 16));
    EXPECT_EQ(3, NN_PERF_GET_CALL_COUNT_CPU(NULL, NULL, 32));
    EXPECT_EQ(5, NN_PERF_GET_CALL_COUNT_CPU(NULL, "Test1", 0));
    EXPECT_EQ(3, NN_PERF_GET_CALL_COUNT_CPU(NULL, "Test2", 0));
    EXPECT_EQ(1, NN_PERF_GET_CALL_COUNT_CPU(NULL, "Test1", 8));
    EXPECT_EQ(8, NN_PERF_GET_CALL_COUNT_CPU("CPU0", NULL, 0));
    EXPECT_EQ(1, NN_PERF_GET_CALL_COUNT_CPU("CPU1", NULL, 0));
    EXPECT_EQ(1, NN_PERF_GET_CALL_COUNT_CPU("CPU2", NULL, 0));
    EXPECT_EQ(2, NN_PERF_GET_CALL_COUNT_CPU("Meter1", NULL, 0));
    EXPECT_EQ(2, NN_PERF_GET_CALL_COUNT_CPU("Meter2", NULL, 0));
    EXPECT_EQ(1, NN_PERF_GET_CALL_COUNT_CPU("CPU0", "Test2", 32));
    EXPECT_EQ(0, NN_PERF_GET_CALL_COUNT_CPU("Meter1", "Test1", 32));
    EXPECT_EQ(1 ,NN_PERF_GET_CALL_COUNT_CPU("Meter2", "Test1", 32));

    EXPECT_LT(0, NN_PERF_GET_ELAPSED_TIME_CPU("CPU0", "Test2", 32).GetNanoSeconds());
    EXPECT_LT(0, NN_PERF_GET_ELAPSED_TIME_CPU("CPU2", NULL, 0).GetNanoSeconds());
    EXPECT_LT(0, NN_PERF_GET_ELAPSED_TIME_CPU(NULL, NULL, 16).GetNanoSeconds());

    EXPECT_LT(min, NN_PERF_GET_AVERAGE_ELAPSED_TIME_CPU("CPU0", NULL, 0).GetNanoSeconds());
    EXPECT_LT(min, NN_PERF_GET_AVERAGE_ELAPSED_TIME_CPU(NULL, NULL, 0).GetNanoSeconds());
    EXPECT_LT(min, NN_PERF_GET_AVERAGE_ELAPSED_TIME_CPU(NULL, "Test1", 0).GetNanoSeconds());

    EXPECT_GT(max, NN_PERF_GET_AVERAGE_ELAPSED_TIME_CPU("CPU0", NULL, 0).GetNanoSeconds());
    EXPECT_GT(max, NN_PERF_GET_AVERAGE_ELAPSED_TIME_CPU(NULL, NULL, 0).GetNanoSeconds());
    EXPECT_GT(max, NN_PERF_GET_AVERAGE_ELAPSED_TIME_CPU(NULL, "Test1", 0).GetNanoSeconds());

    EXPECT_LE(NN_PERF_GET_AVERAGE_ELAPSED_TIME_CPU("CPU0", NULL, 0).GetNanoSeconds(), NN_PERF_GET_MAX_ELAPSED_TIME_CPU("CPU0", NULL, 0).GetNanoSeconds());
    EXPECT_LE(NN_PERF_GET_AVERAGE_ELAPSED_TIME_CPU(NULL, NULL, 0).GetNanoSeconds(), NN_PERF_GET_MAX_ELAPSED_TIME_CPU(NULL, NULL, 0).GetNanoSeconds());
    EXPECT_LE(NN_PERF_GET_AVERAGE_ELAPSED_TIME_CPU(NULL, "Test1", 0).GetNanoSeconds(), NN_PERF_GET_MAX_ELAPSED_TIME_CPU(NULL, "Test1", 0).GetNanoSeconds());

    EXPECT_GE(NN_PERF_GET_AVERAGE_ELAPSED_TIME_CPU("CPU0", NULL, 0).GetNanoSeconds(), NN_PERF_GET_MIN_ELAPSED_TIME_CPU("CPU0", NULL, 0).GetNanoSeconds());
    EXPECT_GE(NN_PERF_GET_AVERAGE_ELAPSED_TIME_CPU("Meter2", NULL, 0).GetNanoSeconds(), NN_PERF_GET_MIN_ELAPSED_TIME_CPU("Meter2", NULL, 0).GetNanoSeconds());
    EXPECT_GE(NN_PERF_GET_AVERAGE_ELAPSED_TIME_CPU(NULL, NULL, 32).GetNanoSeconds(), NN_PERF_GET_MIN_ELAPSED_TIME_CPU(NULL, NULL, 32).GetNanoSeconds());

    nn::os::DestroyThread(&thread[0]);
    nn::os::DestroyThread(&thread[1]);
    nn::os::DestroyThread(&thread[2]);
    nn::os::FreeTlsSlot(tlsSlot);

    LoadMeterCenterManager::Finalize();
} //NOLINT(impl/function_size)
