﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{UniformBuffer.cpp,PageSampleNvnTutorialLibrary}
 *
 * @brief
 *  This file defines a helper class for setting up uniform
 *  blocks. This class sets up the NVNbuffer object for the
 *  block and provides methods for setting the buffers data
 *  and getting information about it.
 */


#include <cstring>
#include <nn/nn_Assert.h>
#include <nvn/nvn_FuncPtrInline.h>
#include <nvngdSupport/UniformBuffer.h>
#include <nvngdSupport/TutorialUtil.h>

/*
 * UniformBuffer Default Constructor
 * -------------------
 * Sets members to default unitialized values.
 */
UniformBuffer::UniformBuffer() : m_pDevice(NULL),
                                 m_BufferSize(0),
                                 m_pMappedBuffer(NULL),
                                 m_BufferAddress(0)
{
}

/*
 * UniformBuffer::Init
 * -------------------
 * Initialization for the uniform buffer helper class. Creates
 * an NVNbuffer of the given size at the correct alignment.
 */
void UniformBuffer::Init(NVNdevice* pDevice, size_t bufferSize, MemoryPool* pMemoryPool)
{
    m_pDevice = pDevice;
    m_BufferSize = bufferSize;

    int bufferAlignment = 0;
    nvnDeviceGetInteger(m_pDevice, NVN_DEVICE_INFO_UNIFORM_BUFFER_ALIGNMENT, &bufferAlignment);
    ptrdiff_t offset = pMemoryPool->GetNewMemoryChunkOffset(bufferSize, bufferAlignment);

        /* Setup a buffer builder with the uniform access bit */
    NVNbufferBuilder bufferBuilder;
    nvnBufferBuilderSetDefaults(&bufferBuilder);
    nvnBufferBuilderSetDevice(&bufferBuilder, m_pDevice);
    nvnBufferBuilderSetStorage(&bufferBuilder, pMemoryPool->GetMemoryPool(), offset, m_BufferSize);

        /* Initialize a buffer from the pool. */
    if (!nvnBufferInitialize(&m_Buffer, &bufferBuilder))
    {
        NN_ASSERT(0, "Failed to initialize uniform block's buffer");
    }

        /*
         * Grab a pointer for cpu access to the buffer as
         * as well as the buffer's gpu address.
         */
    m_pMappedBuffer = nvnBufferMap(&m_Buffer);
    m_BufferAddress = nvnBufferGetAddress(&m_Buffer);
}

/*
 * UniformBuffer::SetData
 * ----------------------
 * Copy the passed in data into the uniform buffer's
 * memory pool. Data written to the pool will automatically
 * be available to all GPU commands submitted after the write.
 */
void UniformBuffer::SetData(const void* pData, size_t dataSize)
{
        /* Check for Init */
    NN_ASSERT(m_BufferSize, "Init needs to be called first.");

        /* Check that there is enough room for the given data. */
    NN_ASSERT(dataSize <= m_BufferSize);

        /*
         * Memcpy the passed in data into the memory pool through
         * the mapped pointer.
         */
    memcpy(m_pMappedBuffer, pData, dataSize);
}

void UniformBuffer::SetDataOffset(const void * pData, size_t dataSize, uint32_t offset)
{
    /* Check for Init */
    NN_ASSERT(m_BufferSize, "Init needs to be called first.");

    /* Check that there is enough room for the given data. */
    NN_ASSERT(dataSize + offset <= m_BufferSize);

    /*
    * Memcpy the passed in data into the memory pool through
    * the mapped pointer.
    */
    uint8_t* temp = reinterpret_cast<uint8_t*>(m_pMappedBuffer);
    memcpy(temp + offset, pData, dataSize);
}

/*
 * UniformBuffer::GetSizeOfBuffer
 * ------------------------------
 * Returns the size of the buffer.
 */
size_t UniformBuffer::GetSizeOfBuffer() const
{
    NN_ASSERT(m_BufferSize, "Init needs to be called first.");

    return m_BufferSize;
}

/*
 * UniformBuffer::GetBufferAddress
 * -------------------------------
 * Returns the buffer's gpu address for binding.
 */
NVNbufferAddress UniformBuffer::GetBufferAddress() const
{
    NN_ASSERT(m_BufferSize, "Init needs to be called first.");

    return m_BufferAddress;
}

/*
 * UniformBuffer::GetSizeOfBuffer
 * ------------------------------
 * Cleans up the uniform buffer class.
 */
void UniformBuffer::Finalize()
{
    if (m_BufferSize)
    {
        nvnBufferFinalize(&m_Buffer);
    }

    m_BufferSize = 0;
    m_pDevice = NULL;
    m_pMappedBuffer = NULL;
    m_BufferAddress = 0;
}

/*
 * UniformBuffer Destructor
 * ------------------------
 * Cleans up the uniform buffer class.
 */
UniformBuffer::~UniformBuffer()
{
    Finalize();
}
