﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{ShaderTypeEnum.cpp,PageSampleNvnTutorialLibrary}
 *
 * @brief
 *  This file defines some helper functions that give a more generic
 *  interface to the helper shader header files create by the offline asset
 *  file generating tool.
 */

#include <nn/nn_Assert.h>
#include <nvngdSupport/ShaderTypeEnum.h>

namespace ShaderTypes
{

/*
 * ShaderTypes::GetShaderTypeEnum
 * ------------------------------
 * Get the enum type from the name of the shader.
 */
ShaderType GetShaderTypeEnum(const std::string& name)
{
    if(name == "SimpleTexturedModel")
        return ShaderType_SimpleTexturedModel;
    else if(name == "CubeMap")
        return ShaderType_CubeMap;
    else if(name == "FontShader")
        return ShaderType_FontShader;
    else
        return ShaderType_Num;
}

/*
 * ShaderTypes::Attributes_GetAttributeLocation
 * --------------------------------------------
 * Get the location of an attribute with a given name from
 * the shader of the given type.
 */
int32_t Attributes_GetAttributeLocation(const std::string& name, ShaderType shaderType)
{
    switch (shaderType)
    {
    case ShaderType_SimpleTexturedModel:
        return SimpleTexturedModel::Attributes::GetAttributeLocation(name);
        break;
    case ShaderType_CubeMap:
        return CubeMap::Attributes::GetAttributeLocation(name);
        break;
    case ShaderType_FontShader:
        return FontShader::Attributes::GetAttributeLocation(name);
        break;
    default:
        NN_ASSERT(0, "Bad shaderType");
        return -1;
    }
}

/*
 * ShaderTypes::Attributes_GetAttributeLocationFucntion
 * --------------------------------------------
 * Gets the pointer to the function that gets the location of
 * an attribute for a given shader type.
 */
ShaderFunction Attributes_GetAttributeLocationFunction(ShaderType shaderType)
{
    switch (shaderType)
    {
    case ShaderType_SimpleTexturedModel:
        return SimpleTexturedModel::Attributes::GetAttributeLocation;
        break;
    case ShaderType_CubeMap:
        return CubeMap::Attributes::GetAttributeLocation;
        break;
    case ShaderType_FontShader:
        return FontShader::Attributes::GetAttributeLocation;
        break;
    default:
        NN_ASSERT(0, "Bad shaderType");
        return NULL;
    }
}

/*
 * ShaderTypes::Attributes_GetNumAttributes
 * ----------------------------------------
 * Get the number of attributes for a given shader type.
 */
uint32_t Attributes_GetNumAttributes(ShaderType shaderType)
{
    switch (shaderType)
    {
    case ShaderType_SimpleTexturedModel:
        return SimpleTexturedModel::Attributes::GetNumAttributes();
        break;
    case ShaderType_CubeMap:
        return CubeMap::Attributes::GetNumAttributes();
        break;
    case ShaderType_FontShader:
        return FontShader::Attributes::GetNumAttributes();
        break;
    default:
        NN_ASSERT(0, "Bad shaderType");
        return static_cast<uint32_t>(-1);
    }
}

/*
 * ShaderTypes::Attributes_GetStagesReferencedIn
 * ---------------------------------------------
 * Get the stages referenced in value for a given attribute
 * from a given sahder type.
 */
uint8_t Attributes_GetStagesReferencedIn(const std::string& name, ShaderType shaderType)
{
    switch (shaderType)
    {
    case ShaderType_SimpleTexturedModel:
        return SimpleTexturedModel::Attributes::GetStagesReferencedIn(name);
        break;
    case ShaderType_CubeMap:
        return CubeMap::Attributes::GetStagesReferencedIn(name);
        break;
    case ShaderType_FontShader:
        return FontShader::Attributes::GetStagesReferencedIn(name);
        break;
    default:
        NN_ASSERT(0, "Bad shaderType");
        return static_cast<uint8_t>(-1);
    }
}


/*
 * ShaderTypes::BlockFS_GetBinding
 * -------------------------------
 * Get the uniform block binding for "BlockFS" for a given
 * shader stage of a given shader type.
 */
uint32_t BlockFS_GetBinding(uint32_t shaderStage, ShaderType shaderType)
{
    switch (shaderType)
    {
    case ShaderType_SimpleTexturedModel:
        return SimpleTexturedModel::BlockFSUniformBlockData::GetBinding(shaderStage);
        break;
    case ShaderType_CubeMap:
        return CubeMap::BlockFSUniformBlockData::GetBinding(shaderStage);
        break;
    case ShaderType_FontShader:
        return FontShader::BlockSamplerFSUniformBlockData::GetBinding(shaderStage);
        break;
    default:
        NN_ASSERT(0, "Bad shaderType");
        return static_cast<uint32_t>(-1);
    }
}

/*
 * ShaderTypes::BlockFS_GetStagesReferencedIn
 * ------------------------------------------
 * Get the stages referenced in for the uniform block "BlockFS"
 * for a given shader type.
 */
uint8_t BlockFS_GetStagesReferencedIn(ShaderType shaderType)
{
    switch (shaderType)
    {
    case ShaderType_SimpleTexturedModel:
        return SimpleTexturedModel::BlockFSUniformBlockData::GetStagesReferencedIn();
        break;
    case ShaderType_CubeMap:
        return CubeMap::BlockFSUniformBlockData::GetStagesReferencedIn();
        break;
    case ShaderType_FontShader:
        return FontShader::BlockSamplerFSUniformBlockData::GetStagesReferencedIn();
        break;
    default:
        NN_ASSERT(0, "Bad shaderType");
        return static_cast<uint8_t>(-1);
    }
}

/*
 * ShaderTypes::BlockFS_GetUniformBlockSize
 * ----------------------------------------
 * Get the size of "BlockFS" for a given shader type.
 */
uint32_t BlockFS_GetUniformBlockSize(ShaderType shaderType)
{
    switch (shaderType)
    {
    case ShaderType_SimpleTexturedModel:
        return sizeof(SimpleTexturedModel::BlockFSUniformBlockData);
        break;
    case ShaderType_CubeMap:
        return sizeof(CubeMap::BlockFSUniformBlockData);
        break;
    case ShaderType_FontShader:
        return sizeof(FontShader::BlockSamplerFSUniformBlockData);
        break;
    default:
        NN_ASSERT(0, "Bad shaderType");
        return static_cast<uint32_t>(-1);
    }
}

/*
 * ShaderTypes::BlockFS_SetUniform_u_bindlessTex
 * ---------------------------------------------
 * Set the "u_bindlessTex" element of "BlockFS" for a given shader type.
 */
void BlockFS_SetUniform_u_bindlessTex(void* block, const uint64_t& uniformData, ShaderType shaderType)
{
    switch (shaderType)
    {
    case ShaderType_SimpleTexturedModel:
        SimpleTexturedModel::BlockFSUniformBlockData* simpleFSBlock;
        simpleFSBlock = reinterpret_cast<SimpleTexturedModel::BlockFSUniformBlockData*>(block);
        simpleFSBlock->SetUniform_u_bindlessTex(uniformData);
        break;
    case ShaderType_CubeMap:
        CubeMap::BlockFSUniformBlockData* cubeFSBlock;
        cubeFSBlock = reinterpret_cast<CubeMap::BlockFSUniformBlockData*>(block);
        cubeFSBlock->SetUniform_u_bindlessTex(uniformData);
        break;
    case ShaderType_FontShader:
        FontShader::BlockSamplerFSUniformBlockData* fontFSBlock;
        fontFSBlock = reinterpret_cast<FontShader::BlockSamplerFSUniformBlockData*>(block);
        fontFSBlock->SetUniform_u_bindlessTex(uniformData);
        break;
    default:
        NN_ASSERT(0, "Bad shaderType");
    }
}

/*
 * ShaderTypes::BlockVS_SetUniform_u_modelMtx
 * ------------------------------------------
 * Set the "u_modelMtx" element of "BlockVS" for a given shader type.
 */
void BlockVS_SetUniform_u_modelMtx(void* block, const float (&uniformData)[16], ShaderType shaderType)
{
    switch (shaderType)
    {
    case ShaderType_SimpleTexturedModel:
        SimpleTexturedModel::BlockVSUniformBlockData* simpleVSBlock;
        simpleVSBlock = reinterpret_cast<SimpleTexturedModel::BlockVSUniformBlockData*>(block);
        simpleVSBlock->SetUniform_u_modelMtx(uniformData);
        break;
    case ShaderType_CubeMap:
        CubeMap::BlockVSUniformBlockData* cubeVSBlock;
        cubeVSBlock = reinterpret_cast<CubeMap::BlockVSUniformBlockData*>(block);
        cubeVSBlock->SetUniform_u_modelMtx(uniformData);
        break;
    case ShaderType_FontShader:
    default:
        NN_ASSERT(0, "Bad shaderType");
    }
}

/*
 * ShaderTypes::BlockVS_SetUniform_u_projMtx
 * -----------------------------------------
 * Set the "u_projMtx" element of "BlockVS" for a given shader type.
 */
void BlockVS_SetUniform_u_projMtx(void* block, const float (&uniformData)[16], ShaderType shaderType)
{
    switch (shaderType)
    {
    case ShaderType_SimpleTexturedModel:
        SimpleTexturedModel::BlockVSUniformBlockData* simpleVSBlock;
        simpleVSBlock = reinterpret_cast<SimpleTexturedModel::BlockVSUniformBlockData*>(block);
        simpleVSBlock->SetUniform_u_projMtx(uniformData);
        break;
    case ShaderType_CubeMap:
        CubeMap::BlockVSUniformBlockData* cubeVSBlock;
        cubeVSBlock = reinterpret_cast<CubeMap::BlockVSUniformBlockData*>(block);
        cubeVSBlock->SetUniform_u_projMtx(uniformData);
        break;
    case ShaderType_FontShader:
    default:
        NN_ASSERT(0, "Bad shaderType");
    }
}

/*
 * ShaderTypes::BlockVS_SetUniform_u_viewMtx
 * -----------------------------------------
 * Set the "u_viewMtx" element of "BlockVS" for a given shader type.
 */
void BlockVS_SetUniform_u_viewMtx(void* block, const float (&uniformData)[16], ShaderType shaderType)
{
    switch (shaderType)
    {
    case ShaderType_SimpleTexturedModel:
        SimpleTexturedModel::BlockVSUniformBlockData* simpleVSBlock;
        simpleVSBlock = reinterpret_cast<SimpleTexturedModel::BlockVSUniformBlockData*>(block);
        simpleVSBlock->SetUniform_u_viewMtx(uniformData);
        break;
    case ShaderType_CubeMap:
        CubeMap::BlockVSUniformBlockData* cubeVSBlock;
        cubeVSBlock = reinterpret_cast<CubeMap::BlockVSUniformBlockData*>(block);
        cubeVSBlock->SetUniform_u_viewMtx(uniformData);
        break;
    case ShaderType_FontShader:
    default:
        NN_ASSERT(0, "Bad shaderType");
    }
}

/*
 * ShaderTypes::BlockVS_GetBinding
 * -------------------------------
 * Get the binding location of the "BlockVS" uniform block for a
 * given shader type.
 */
uint32_t BlockVS_GetBinding(uint32_t shaderStage, ShaderType shaderType)
{
    switch (shaderType)
    {
    case ShaderType_SimpleTexturedModel:
        return SimpleTexturedModel::BlockVSUniformBlockData::GetBinding(shaderStage);
        break;
    case ShaderType_CubeMap:
        return CubeMap::BlockVSUniformBlockData::GetBinding(shaderStage);
        break;
    case ShaderType_FontShader:
        return FontShader::BlockVSUniformBlockData::GetBinding(shaderStage);
        break;
    default:
        NN_ASSERT(0, "Bad shaderType");
        return static_cast<uint32_t>(-1);
    }
}

/*
 * ShaderTypes::BlockVS_GetStagesReferencedIn
 * ------------------------------------------
 * Get the stages referenced in of the "BlockVS" uniform block for a
 * given shader type.
 */
uint8_t BlockVS_GetStagesReferencedIn(ShaderType shaderType)
{
    switch (shaderType)
    {
    case ShaderType_SimpleTexturedModel:
        return SimpleTexturedModel::BlockVSUniformBlockData::GetStagesReferencedIn();
        break;
    case ShaderType_CubeMap:
        return CubeMap::BlockVSUniformBlockData::GetStagesReferencedIn();
        break;
    case ShaderType_FontShader:
        return FontShader::BlockVSUniformBlockData::GetStagesReferencedIn();
        break;
    default:
        NN_ASSERT(0, "Bad shaderType");
        return static_cast<uint8_t>(-1);
    }
}

/*
 * ShaderTypes::BlockVS_GetUniformBlockSize
 * ----------------------------------------
 * Get the size of of the "BlockVS" uniform block for a
 * given shader type.
 */
uint32_t BlockVS_GetUniformBlockSize(ShaderType shaderType)
{
    switch (shaderType)
    {
    case ShaderType_SimpleTexturedModel:
        return sizeof(SimpleTexturedModel::BlockVSUniformBlockData);
        break;
    case ShaderType_CubeMap:
        return sizeof(CubeMap::BlockVSUniformBlockData);
        break;
    default:
        NN_ASSERT(0, "Bad shaderType");
        return static_cast<uint32_t>(-1);
    }
}

/*
 * ShaderTypes::BlockVS_SetUniform_u_scale
 * ---------------------------------------
 * Set the "u_scale" element of "BlockVS" for a given shader type.
 */
void BlockVS_SetUniform_u_scale(void* block, const float (&uniformData)[2], ShaderType shaderType)
{
    switch (shaderType)
    {
    case ShaderType_FontShader:
        FontShader::BlockVSUniformBlockData* fontVSBlock;
        fontVSBlock = reinterpret_cast<FontShader::BlockVSUniformBlockData*>(block);
        fontVSBlock->SetUniform_u_scale(uniformData);
        break;
    default:
        NN_ASSERT(0, "Bad shaderType");
    }
}

} // namespace end
