﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{GlSimple.cpp,PageSampleGlSimple}
 *
 * @brief
 *  Basic example of initializing and using OpenGL
 */

/**
 * @page PageSampleGlSimple GlSimple
 * @tableofcontents
 *
 * @brief
 *  This sample demonstrates basic initialization of OpenGL and
 *  renders a single triangle.
 *
 * @section PageSampleGlSimple_SectionBrief Overview
 *  This sample demonstrates basic initialization of OpenGL and
 *  renders a single triangle.  This is acccomplished through the
 *  use of OpenGL ES.
 *
 * @subsection PageGlSimple_SectionExpectedOutput Expected Output
 * @image html GlSimple.png
 *
 * @section GlSimple_SectionFileStructure File Structure
 *  The main sample file and Visual Studio solutions can be found at
 *  @link ../../../Samples/Sources/Applications/GlSimple Samples/Sources/Applications/GlSimple @endlink
 *
 * @section PageSampleGlSimple_SectionNecessaryEnvironment System Requirements
 *  No extra system requirements.
 *
 * @section PageSampleGlSimple_SectionHowToOperate Operation Procedure
 *  This sample runs on its own without any additional input. Touching
 *  the touch screen on NX will exit the program.
 *
 * @section PageSampleGlSimple_SectionPrecaution Precautions
 *  None.
 *
 * @section PageSampleGlSimple_SectionHowToExecute Execution Procedure
 *  Build the Visual Solution in the desired configuration and run it.
 *
 * @section PageSampleGlSimple_SectionDetail Description
 *
 * @subsection PageSampleGlSimple_SectionSampleProgram Sample Program
 *  Below is the source code for the main tutorial file for this sample.
 *
 *  GlSimple.cpp
 *  @includelineno GlSimple.cpp
 *
 * @subsection PageSampleGlSimple_SectionSampleDetail Sample Program Description
 *  This application renders a spinning green triangle
 */
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/fs.h>

#if defined( NN_BUILD_TARGET_PLATFORM_OS_NN ) && defined( NN_BUILD_APISET_NX )
         /*
          * In this sample, we use OpenGL ES 3.2 context and APIs.
          * If you use other GL version, you should include appropriate headers instead of this.
          */
    #include <GLES3/gl32.h>
#else
    #include <GL/glew.h>
    #define GL_APIENTRY GLAPIENTRY
#endif

#include <nnt.h>
#include <nnt/graphics/testGraphics_Path.h>
#include <nnt/graphics/testGraphics_GetHostExecutableFilepath.h>
#include <nnt/graphics/testGraphics_Png.h>
#include <nnt/graphics/testGraphics_CreateDirectories.h>
#include <nnt/graphics/testGraphics_InitializeAllocatorFunctionForStandardAllocator.h>

#include "GraphicsHelper.h"

namespace
{
    const char* VertexShaderSource =
        "#version 320 es\n"
        "precision highp float;\n"
        "layout( location = 0 ) in vec2 a_position;\n"
        "uniform float u_time;\n"
        "void main() {\n"
        "    mat2 xform = mat2(cos(u_time), sin(u_time),\n"
        "                      -sin(u_time), cos(u_time));\n"
        "    gl_Position = vec4(xform * a_position, 0.0, 1.0);\n"
        "}\n";

    const char* FragmentShaderSource =
        "#version 320 es\n"
        "precision highp float;\n"
        "layout( location = 0 ) out vec4 o_Color;\n"
        "void main() {\n"
        "    o_Color = vec4(0.0, 1.0, 0.0, 1.0);\n"
        "}\n";

    GraphicsHelper s_GraphicsHelper;

    GLuint s_PosLocation;
    GLuint s_TimeLocation;

    GLuint s_VertexShaderId;
    GLuint s_FragmentShaderId;
    GLuint s_ShaderProgramId;
    GLuint s_VertexBufferId;

    void GL_APIENTRY DebugCallbackFunc(GLenum source, GLenum type, GLuint id, GLenum severity, GLsizei length, const GLchar* message, const void* userParam)
    {
        NN_UNUSED(length);
        NN_UNUSED(userParam);

        NN_LOG("GL Debug Callback:\n");
        NN_LOG("  source:       0x%08x\n", source);
        NN_LOG("  type:         0x%08x\n", type);
        NN_LOG("  id:           0x%08x\n", id);
        NN_LOG("  severity:     0x%08x\n", severity);
        NN_LOG("  message:      %s\n", message);

        NN_ASSERT(type != GL_DEBUG_TYPE_ERROR, "GL Error occurs.");
    }
} // namespace

void InitializeGraphics()
{
    s_GraphicsHelper.Initialize();
}

void InitializeDebugCallback()
{
    glDebugMessageCallback(DebugCallbackFunc, NULL);
}

void InitializeShaders()
{
    GLint result;
    GLchar shaderLog[1024];
    GLsizei shaderLogSize;

    s_VertexShaderId = glCreateShader(GL_VERTEX_SHADER);
    NN_ASSERT(s_VertexShaderId != 0, "Failed to create vertex shader\n");
    glShaderSource(s_VertexShaderId, 1, &VertexShaderSource, 0);
    glCompileShader(s_VertexShaderId);
    glGetShaderiv(s_VertexShaderId, GL_COMPILE_STATUS, &result);
    if (!result)
    {
        glGetShaderInfoLog(s_VertexShaderId, sizeof(shaderLog), &shaderLogSize, shaderLog);
        NN_ASSERT(false, "Failed to compile vertex shader: %s\n", shaderLog);
    }

    s_FragmentShaderId = glCreateShader(GL_FRAGMENT_SHADER);
    NN_ASSERT(s_FragmentShaderId != 0, "Failed to create fragment shader\n");
    glShaderSource(s_FragmentShaderId, 1, &FragmentShaderSource, 0);
    glCompileShader(s_FragmentShaderId);
    glGetShaderiv(s_FragmentShaderId, GL_COMPILE_STATUS, &result);
    if (!result)
    {
        glGetShaderInfoLog(s_FragmentShaderId, sizeof(shaderLog), &shaderLogSize, shaderLog);
        NN_ASSERT(false, "Failed to compile fragment shader: %s\n", shaderLog);
    }

    s_ShaderProgramId = glCreateProgram();
    NN_ASSERT(s_ShaderProgramId != 0, "Failed to create shader program\n");

    glAttachShader(s_ShaderProgramId, s_VertexShaderId);
    glAttachShader(s_ShaderProgramId, s_FragmentShaderId);
    glLinkProgram(s_ShaderProgramId);
    glUseProgram(s_ShaderProgramId);

    s_PosLocation = glGetAttribLocation(s_ShaderProgramId, "a_position");
    s_TimeLocation = glGetUniformLocation(s_ShaderProgramId, "u_time");
}

void InitializeVertexData()
{
    static const GLfloat VERTEX_DATA[] = {
        -0.5, -0.5,
        0.5, -0.5,
        0.0, 1.0,
    };

    glGenBuffers(1, &s_VertexBufferId);
    glBindBuffer(GL_ARRAY_BUFFER, s_VertexBufferId);
    glBufferData(GL_ARRAY_BUFFER, sizeof(VERTEX_DATA), VERTEX_DATA, GL_STATIC_DRAW);
    glEnableVertexAttribArray(s_PosLocation);
    glVertexAttribPointer(s_PosLocation, 2, GL_FLOAT, GL_FALSE, 0, 0);
}

void SetupScene()
{
    glClearColor(0.4f, 0.55f, 0.6f, 1.0f);
}

void RenderScene()
{
    glClear(GL_COLOR_BUFFER_BIT);

    glUseProgram(s_ShaderProgramId);

    glUniform1f(s_TimeLocation, 0.0f);
    glDrawArrays(GL_TRIANGLES, 0, 3);

    glUseProgram(0);
}

void SwapBuffers()
{
    s_GraphicsHelper.SwapBuffers();
}

void Destroy()
{
    glDetachShader(s_ShaderProgramId, s_VertexShaderId);
    glDetachShader(s_ShaderProgramId, s_FragmentShaderId);

    glDeleteShader(s_VertexShaderId);
    glDeleteShader(s_FragmentShaderId);

    glDeleteProgram(s_ShaderProgramId);

    glDeleteBuffers(1, &s_VertexBufferId);
}

void FinalizeGraphics()
{
    s_GraphicsHelper.Finalize();
}

TEST(GlSimple, Run)
{
    InitializeGraphics();

    NN_LOG("GL_VERSION: %s\n", glGetString(GL_VERSION));
    NN_LOG("GL_SHADING_LANGUAGE_VERSION: %s\n", glGetString(GL_SHADING_LANGUAGE_VERSION));

    InitializeDebugCallback();
    InitializeShaders();
    InitializeVertexData();
    SetupScene();

    size_t pixelDataSize = 1280 * 720 * sizeof(GLubyte) * 4;
    GLubyte* pPixelData = (GLubyte *)malloc(pixelDataSize);
    int i = 0;
    while (++i < 180)
    {
        RenderScene();
        SwapBuffers();
    };

    glReadPixels(0, 0, 1280, 720, GL_RGBA, GL_UNSIGNED_BYTE, pPixelData);

    nn::Result result = nn::fs::MountHostRoot();
    NN_ASSERT(result.IsSuccess());

    nnt::graphics::Path outputsPath(nnt::graphics::GetHostExecutableFilepath());
    outputsPath.CombineAssign("../Outputs");
    outputsPath.NormalizeAssign();

    result = nnt::graphics::CreateDirectories(outputsPath);
    NN_ASSERT(result.IsSuccess());
    result = nn::fs::MountHost("Outputs", outputsPath.GetString());
    NN_ASSERT(result.IsSuccess());

    nnt::graphics::PngIhdr ihdr;
    ihdr.SetDefault();
    ihdr.bitDepth = 8;
    ihdr.channels = 4;
    ihdr.colorType = nnt::graphics::PngColorType_RgbAlpha;
    ihdr.height = 720;
    ihdr.width = 1280;

    nn::mem::StandardAllocator testGraphicsStandardAllocator;
    nnt::graphics::AllocatorFunction testGraphicsAllocatorFunction;

    static const size_t HeapSize = 16 * 1024 * 1024;
    testGraphicsStandardAllocator.Initialize(malloc(HeapSize), HeapSize);
    nnt::graphics::InitializeAllocatorFunctionForStandardAllocator::Initialize(
        &testGraphicsAllocatorFunction,
        &testGraphicsStandardAllocator);

    nnt::graphics::PngIO::Write(
        nnt::graphics::Path("Outputs:/GlSimple_Raw_Output.png").GetString(),
        pPixelData,
        pixelDataSize,
        &ihdr,
        &testGraphicsAllocatorFunction);

    nn::fs::Unmount("Outputs");
    nn::fs::UnmountHostRoot();

    free(pPixelData);

    Destroy();
    FinalizeGraphics();

    SUCCEED();
}
