﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <random>

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/gfx/util/gfx_TransientDescriptorAllocatorHolder.h>

#include <nnt.h>

TEST(TransientDescriptorAllocator, SingleThread)
{
    nn::gfx::DescriptorPool descPool;

    // パラメータ
    static const int HistoryLengthMax = 5;
    const int baseIndex = 1020;
    const int rangeMin = 2;
    const int rangeMax = 10;
    const int allocateCountPerSection = 50;
    const int slotCount = HistoryLengthMax * allocateCountPerSection * rangeMax + 1;
    const int sectionCount = 1000;

    // 乱数
    unsigned long randSeed =  0x2391A8BC; // 適当。
    std::mt19937 randEng;
    randEng.seed(randSeed);
    std::uniform_int_distribution<int> rand(rangeMin, rangeMax);

    // ここからテスト
    nn::gfx::util::TransientDescriptorAllocatorHolder<HistoryLengthMax> traHolder;

    EXPECT_EQ(HistoryLengthMax, static_cast<int>(traHolder.HistoryLengthMax));

    EXPECT_FALSE(traHolder.Get()->IsInitialized());
    traHolder.Initialize(&descPool, baseIndex, slotCount);
    nn::gfx::util::TransientDescriptorAllocator* pTra = traHolder.Get();
    EXPECT_TRUE(pTra != NULL);
    EXPECT_TRUE(pTra->IsInitialized());
    EXPECT_FALSE(pTra->IsRecording());
    EXPECT_EQ(0, pTra->GetHistoryLength());

    pTra->FillHistory();
    EXPECT_EQ(HistoryLengthMax, pTra->GetHistoryLength());

    // 履歴がいっぱいの状態で Fill しても OK
    pTra->FillHistory();

    for(int section = 0; section < sectionCount; section++)
    {
        pTra->Free();
        EXPECT_EQ(HistoryLengthMax - 1, pTra->GetHistoryLength());

        pTra->Begin();
        EXPECT_TRUE(pTra->IsRecording());

        for(int c = 0; c < allocateCountPerSection; c++)
        {
            int size = rand(randEng);
            int slot = pTra->Allocate(size);
            EXPECT_GE(slot, baseIndex);
            EXPECT_LT(slot, static_cast<int>(baseIndex + slotCount));
        }

        pTra->End();
        EXPECT_EQ(HistoryLengthMax, pTra->GetHistoryLength());
        EXPECT_FALSE(pTra->IsRecording());
    }

    traHolder.Finalize();
    EXPECT_FALSE(traHolder.Get()->IsInitialized());

}

TEST(TransientDescriptorAllocatorDeathTest, DeathTest)
{
    nn::gfx::DescriptorPool descPool;

    {
        nn::gfx::util::TransientDescriptorAllocatorHolder<3> holder;

        // 初期化前に呼ぶと死ぬことの確認
        EXPECT_DEATH_IF_SUPPORTED(holder.Finalize(), "");

        holder.Initialize(&descPool, 0, 100);

        // 初期化後に呼ぶと死ぬことの確認
        EXPECT_DEATH_IF_SUPPORTED(holder.Initialize(&descPool, 0, 10), "");

        holder.Finalize();
    }

    {
        static const int HistoryLength = 3;
        nn::gfx::util::TransientDescriptorAllocator tra;
        nn::gfx::util::detail::IndexRange ranges[HistoryLength + 2];

        // 初期化前に呼ぶと死ぬことの確認
        EXPECT_DEATH_IF_SUPPORTED(tra.Finalize(), "");
        EXPECT_DEATH_IF_SUPPORTED(tra.IsRecording(), "");
        EXPECT_DEATH_IF_SUPPORTED(tra.GetDescriptorPool(), "");
        EXPECT_DEATH_IF_SUPPORTED(tra.GetBaseSlotIndex(), "");
        EXPECT_DEATH_IF_SUPPORTED(tra.GetSlotCount(), "");
        EXPECT_DEATH_IF_SUPPORTED(tra.Begin(), "");
        EXPECT_DEATH_IF_SUPPORTED(tra.End(), "");
        EXPECT_DEATH_IF_SUPPORTED(tra.Free(), "");
        EXPECT_DEATH_IF_SUPPORTED(tra.Allocate(), "");
        EXPECT_DEATH_IF_SUPPORTED(tra.Allocate(10), "");
        EXPECT_DEATH_IF_SUPPORTED(tra.GetHistoryLengthMax(), "");
        EXPECT_DEATH_IF_SUPPORTED(tra.GetHistoryLength(), "");
        EXPECT_DEATH_IF_SUPPORTED(tra.FillHistory(), "");

        // 不正なパラメータで呼ぶと死ぬことの確認
        EXPECT_DEATH_IF_SUPPORTED(tra.GetRequiredMemorySize(0), "");

        tra.SetMemory(ranges, sizeof(ranges));
        EXPECT_DEATH_IF_SUPPORTED(tra.Initialize(NULL, 0, 10, HistoryLength), "");
        EXPECT_DEATH_IF_SUPPORTED(tra.Initialize(&descPool, -1, 10, HistoryLength), "");
        EXPECT_DEATH_IF_SUPPORTED(tra.Initialize(&descPool, 0, 0, HistoryLength), "");
        EXPECT_DEATH_IF_SUPPORTED(tra.Initialize(&descPool, 0, tra.SlotCountMax + 1, HistoryLength), "");
        EXPECT_DEATH_IF_SUPPORTED(tra.Initialize(&descPool, 0, 10, 0), "");
        tra.SetMemory(NULL, sizeof(ranges));
        EXPECT_DEATH_IF_SUPPORTED(tra.Initialize(&descPool, 0, 10, HistoryLength), "");
        if(tra.GetRequiredMemoryAlignment() > 1)
        {
            tra.SetMemory(reinterpret_cast<uint8_t*>(ranges) + 1, sizeof(nn::gfx::util::detail::IndexRange) * (HistoryLength + 1) );
            EXPECT_DEATH_IF_SUPPORTED(tra.Initialize(&descPool, 0, 10, HistoryLength), "");
        }
        tra.SetMemory(ranges, tra.GetRequiredMemorySize(HistoryLength) - 1);
        EXPECT_DEATH_IF_SUPPORTED(tra.Initialize(&descPool, 0, 10, HistoryLength), "");

        // --- 初期化 ---
        tra.SetMemory(ranges, sizeof(ranges));
        tra.Initialize(&descPool, 0, 10, HistoryLength);
        // --------------

        // 初期化後に呼ぶと死ぬことの確認
        EXPECT_DEATH_IF_SUPPORTED(tra.Initialize(&descPool, 0, 10, HistoryLength), "");
        EXPECT_DEATH_IF_SUPPORTED(tra.SetMemory(ranges, sizeof(ranges)), "");

        // 1 つも履歴がない状態で呼ぶと死ぬことの確認
        EXPECT_DEATH_IF_SUPPORTED(tra.Free(), "");

        // Begin 前に呼ぶと死ぬことの確認
        EXPECT_DEATH_IF_SUPPORTED(tra.End(), "");
        EXPECT_DEATH_IF_SUPPORTED(tra.Allocate(), "");
        EXPECT_DEATH_IF_SUPPORTED(tra.Allocate(2), "");

        // --- Begin ---
        tra.Begin();
        // -------------

        // Begin 後に呼ぶと死ぬことの確認
        EXPECT_DEATH_IF_SUPPORTED(tra.Begin(), "");
        EXPECT_DEATH_IF_SUPPORTED(tra.FillHistory(), "");

        // 1 つも履歴がない状態で呼ぶと死ぬことの確認(Begin しただけでは履歴は増えない)
        EXPECT_DEATH_IF_SUPPORTED(tra.Free(), "");

        // --- End ---
        tra.End();
        // -------------

        tra.FillHistory();
        EXPECT_EQ(HistoryLength, tra.GetHistoryLength());

        // 履歴がいっぱいの状態で呼ぶと死ぬことの確認
        EXPECT_DEATH_IF_SUPPORTED(tra.Begin(), "");

        // Finalize しなくても OK
    }

}

