﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include <cstdlib>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/os.h>
#include <nn/init.h>
#include <nn/fs.h>
#include <nn/mem.h>
#include <nn/vi.h>
#include <nn/gfx.h>
#include <nnt.h>
#include <nn/gfx/util/gfx_TransientDescriptorAllocatorHolder.h>

#include <nnt/gfx/testGfx_GraphicsFramework.h>
#include <nn/gfx/util/gfx_TextureCompressor.h>

#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
    #include <nv/nv_MemoryManagement.h>
#endif

#if NN_GFX_IS_TARGET_NVN && defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON )
    #include <nvnTool/nvnTool_GlslcInterface.h>
#endif


namespace {

    const size_t GraphicsSystemMemorySize =  8 * 1024 * 1024;

    nnt::gfx::GraphicsFramework g_GraphicsFramework;


    nn::gfx::util::TransientDescriptorAllocatorHolder<2> g_TransientTextureDescriptorAllocatorHolder;
    nn::gfx::util::TransientDescriptorAllocator* g_pTransientTextureDescriptorAllocator = NULL;

    nn::gfx::util::TransientDescriptorAllocatorHolder<2> g_TransientSamplerDescriptorAllocatorHolder;
    nn::gfx::util::TransientDescriptorAllocator* g_pTransientSamplerDescriptorAllocator = NULL;


    const int SourceTextureWidth = 320;
    const int SourceTextureHeight = SourceTextureWidth;

    const int DestTextureWidth = SourceTextureWidth;
    const int DestTextureHeight = SourceTextureHeight;

    const int MipLevels = 9;
    const int LayerCount = 7;


    const char* RenderTriangleShaderBinaryFilePath = "Contents:/renderTriangleShader.bnsh";
    const char* RenderTextureShaderBinaryFilePath = "Contents:/renderTextureShader.bnsh";
    const char* TextureCompressionShaderBinaryFilePath = "Contents:/textureCompressionShader.bnsh";


    const uint32_t ScreenWidhth = 1280;
    const uint32_t ScreenHeight = 720;
    const uint32_t MarginX = 4;
    const uint32_t MarginY = 4;

    nn::gfx::util::TextureCompressor            g_TextureCompressor;
    nn::gfx::Sampler                            g_SamplerMinMagMipPointEdgeClamp;
    nn::gfx::DescriptorSlot                     g_SamplerDescriptorSlot;

    // 三角形の頂点情報
    namespace Triangle
    {
        const float VertexData[] = {
            -1, -1, 0, 1, 0, 0,
            +1, -1, 0, 0, 1, 0,
            0, +1, 0, 0, 0, 1,
        };

        enum VertexLayout
        {
            Position,
            Color,
            End
        };
        typedef VertexLayout VertexShaderSlot;

        const size_t VertexStride = sizeof(float) * 6;

        const ptrdiff_t Offset[End] = {
            0,
            sizeof(float) * 3
        };

        const uint32_t IndexData[] = {
            0,1,2
        };

        nn::gfx::Buffer vertexBuffer;
        ptrdiff_t vertexOffset;

        nn::gfx::Buffer indexBuffer;
        ptrdiff_t indexOffset;

        void InitializeVertexBuffer()
        {
            nn::gfx::Buffer::InfoType info;
            {
                info.SetDefault();
                info.SetSize(sizeof(VertexData));
                info.SetGpuAccessFlags(nn::gfx::GpuAccess_VertexBuffer);
            }

            ptrdiff_t offset = 0;
            if (NN_STATIC_CONDITION(nn::gfx::Buffer::IsMemoryPoolRequired))
            {
                offset = -1;
                size_t size = sizeof(VertexData);
                size_t alignment = nn::gfx::Buffer::GetBufferAlignment(g_GraphicsFramework.GetDevice(), info);
                offset = g_GraphicsFramework.AllocatePoolMemory(
                    nnt::gfx::GraphicsFramework::MemoryPoolType_Data,
                    size, alignment);

                NN_ASSERT_NOT_EQUAL(offset, -1);
                vertexBuffer.Initialize(g_GraphicsFramework.GetDevice(), info,
                    g_GraphicsFramework.GetMemoryPool(nnt::gfx::GraphicsFramework::MemoryPoolType_Data),
                    offset, size);
            }
            else
            {
                vertexBuffer.Initialize(g_GraphicsFramework.GetDevice(), info, NULL, 0, 0);
            }

            vertexOffset = offset;

            //頂点バッファに値を書き込む
            void* pDest = vertexBuffer.Map();
            {
                std::memcpy(pDest,VertexData,sizeof(VertexData));
                vertexBuffer.FlushMappedRange(0,sizeof(VertexData));
            }
            vertexBuffer.Unmap();

        }
        void FinalizeVertexBuffer()
        {
            if (NN_STATIC_CONDITION(nn::gfx::Buffer::IsMemoryPoolRequired))
            {
                g_GraphicsFramework.FreePoolMemory(nnt::gfx::GraphicsFramework::MemoryPoolType_Data,vertexOffset);
            }

            vertexBuffer.Finalize(g_GraphicsFramework.GetDevice());
        }


        void InitializeIndexBuffer()
        {
            nn::gfx::Buffer::InfoType info;
            {
                info.SetDefault();
                info.SetSize(sizeof(IndexData));
                info.SetGpuAccessFlags(nn::gfx::GpuAccess_IndexBuffer);
            }

            ptrdiff_t offset = 0;
            if (NN_STATIC_CONDITION(nn::gfx::Buffer::IsMemoryPoolRequired))
            {
                offset = -1;
                size_t size = sizeof(IndexData);
                size_t alignment = nn::gfx::Buffer::GetBufferAlignment(g_GraphicsFramework.GetDevice(), info);
                offset = g_GraphicsFramework.AllocatePoolMemory(
                    nnt::gfx::GraphicsFramework::MemoryPoolType_Data,
                    size, alignment);

                NN_ASSERT_NOT_EQUAL(offset, -1);
                indexBuffer.Initialize(g_GraphicsFramework.GetDevice(), info,
                    g_GraphicsFramework.GetMemoryPool(nnt::gfx::GraphicsFramework::MemoryPoolType_Data),
                    offset, size);
            }
            else
            {
                indexBuffer.Initialize(g_GraphicsFramework.GetDevice(), info, NULL, 0, 0);
            }

            indexOffset = offset;

            void* pDest = indexBuffer.Map();
            {
                std::memcpy(pDest,IndexData,sizeof(IndexData));
                indexBuffer.FlushMappedRange(0,sizeof(IndexData));
            }
            indexBuffer.Unmap();

        }
        void FinalizeIndexBuffer()
        {
            if (NN_STATIC_CONDITION(nn::gfx::Buffer::IsMemoryPoolRequired))
            {
                g_GraphicsFramework.FreePoolMemory(nnt::gfx::GraphicsFramework::MemoryPoolType_Data,indexOffset);
            }
            indexBuffer.Finalize(g_GraphicsFramework.GetDevice());
        }
    }

    struct ResShaderType
    {
        ptrdiff_t   binaryOffset;
        size_t      binarySize;
        nn::gfx::ResShaderFile* pResShaderFile;
        nn::gfx::ResShaderContainer* pResShaderContainer;
        nn::gfx::ShaderCodeType codeType;
    };

    ResShaderType g_TriangleResShader;
    ResShaderType g_TextureResShader;
    ResShaderType g_TextureCompressionShader;
    void InitializeResShaderFile(ResShaderType* pResShaderType,const char* shaderFilePath)
    {
        NN_LOG("\nLoad ShaderFile: %s\n",shaderFilePath);
        nn::Result result;

        void* pFileBuffer = NULL;
        int64_t fileSize = 0;
        {
            nn::fs::FileHandle hFile;
            result = nn::fs::OpenFile(&hFile, shaderFilePath, nn::fs::OpenMode_Read);
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);

            result = nn::fs::GetFileSize(&fileSize, hFile);
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);

            pFileBuffer = g_GraphicsFramework.AllocateMemory(static_cast<size_t>(fileSize), NN_ALIGNOF(nn::util::BinaryFileHeader));
            NN_ASSERT_NOT_NULL(pFileBuffer);

            result = nn::fs::ReadFile(hFile, 0, pFileBuffer, static_cast<size_t>(fileSize));
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
            nn::fs::CloseFile(hFile);
        }

        nn::gfx::Buffer buffer;
        void* pShaderBinary = NULL;
        {
            size_t alignment = reinterpret_cast<nn::gfx::ResShaderFile*>(pFileBuffer)->ToData().fileHeader.GetAlignment();
            pResShaderType->binarySize = static_cast<size_t>(fileSize);


            nn::gfx::Buffer::InfoType info;
            {
                info.SetDefault();
                info.SetGpuAccessFlags(nn::gfx::GpuAccess_Read);
                info.SetSize(pResShaderType->binarySize);
            }

            if(NN_STATIC_CONDITION(nn::gfx::Buffer::IsMemoryPoolRequired))
            {
                pResShaderType->binaryOffset = -1;
                pResShaderType->binaryOffset = g_GraphicsFramework.AllocatePoolMemory(
                    nnt::gfx::GraphicsFramework::MemoryPoolType::MemoryPoolType_Shader,
                    pResShaderType->binarySize, alignment);
                NN_ASSERT_NOT_EQUAL(pResShaderType->binaryOffset, -1);


                buffer.Initialize(g_GraphicsFramework.GetDevice(), info,
                    g_GraphicsFramework.GetMemoryPool(nnt::gfx::GraphicsFramework::MemoryPoolType::MemoryPoolType_Shader),
                    pResShaderType->binaryOffset, pResShaderType->binarySize);
            }
            else
            {
                buffer.Initialize(g_GraphicsFramework.GetDevice(),info,NULL,0,0);
            }

            pShaderBinary = buffer.Map();
            std::memcpy(pShaderBinary, pFileBuffer, static_cast<size_t>(fileSize));
        }

        {
            g_GraphicsFramework.FreeMemory(pFileBuffer);
            pFileBuffer = NULL;
        }

        pResShaderType->pResShaderFile = nn::gfx::ResShaderFile::ResCast(pShaderBinary);
        pResShaderType->pResShaderContainer = pResShaderType->pResShaderFile->GetShaderContainer();
        NN_ASSERT_NOT_NULL(pResShaderType->pResShaderContainer);
        size_t containerOffset = pResShaderType->binaryOffset + nn::util::BytePtr(pResShaderType->pResShaderFile).Distance(pResShaderType->pResShaderContainer);
        pResShaderType->pResShaderContainer->Initialize(g_GraphicsFramework.GetDevice(), g_GraphicsFramework.GetMemoryPool(nnt::gfx::GraphicsFramework::MemoryPoolType_Shader), containerOffset, pResShaderType->binarySize);

        nn::gfx::ResShaderVariation* pVariation = pResShaderType->pResShaderContainer->GetResShaderVariation(0);
        NN_ASSERT_NOT_NULL(pVariation);

        nn::gfx::ShaderInitializeResult shaderResult = nn::gfx::ShaderInitializeResult_Success;
        pResShaderType->codeType = nn::gfx::ShaderCodeType_Binary;
        nn::gfx::ResShaderProgram* pResShaderProgram = pVariation->GetResShaderProgram(pResShaderType->codeType);
        if (pResShaderProgram)
        {
            shaderResult = pResShaderProgram->Initialize(g_GraphicsFramework.GetDevice());
        }

        if (!pResShaderProgram || shaderResult != nn::gfx::ShaderInitializeResult_Success)
        {
            pResShaderType->codeType = nn::gfx::ShaderCodeType_Source;
            pResShaderProgram = pVariation->GetResShaderProgram(pResShaderType->codeType);
            if (pResShaderProgram)
            {
                shaderResult = pResShaderProgram->Initialize(g_GraphicsFramework.GetDevice());
            }
        }
        NN_ASSERT(pResShaderProgram && shaderResult == nn::gfx::ShaderInitializeResult_Success);

        int variationCount = pResShaderType->pResShaderContainer->GetShaderVariationCount();
        for (int idxVariation = 1; idxVariation < variationCount; idxVariation++)
        {
            pVariation = pResShaderType->pResShaderContainer->GetResShaderVariation(idxVariation);
            NN_ASSERT_NOT_NULL(pVariation);
            shaderResult = pVariation->GetResShaderProgram(pResShaderType->codeType)->Initialize(g_GraphicsFramework.GetDevice());
            NN_ASSERT_EQUAL(shaderResult, nn::gfx::ShaderInitializeResult_Success);
        }

        buffer.Unmap();
        buffer.Finalize(g_GraphicsFramework.GetDevice());


        NN_LOG("Load Shader: SUCCESS\n\n");
    }
    void FinalizeResShaderFile(ResShaderType* pResShaderType)
    {
        pResShaderType->pResShaderContainer->Finalize(g_GraphicsFramework.GetDevice());
    }

    nn::gfx::Shader* GetTextureCompressionShader(nn::gfx::ImageFormat imageFormat)
    {
        NN_ASSERT_NOT_NULL(g_TextureCompressionShader.pResShaderContainer);

        int variation = -1;
        switch (imageFormat)
        {
        case nn::gfx::ImageFormat_Bc1_Unorm:
            variation = 10;
            break;

        case nn::gfx::ImageFormat_Bc1_UnormSrgb:
            variation = 11;
            break;

        case nn::gfx::ImageFormat_Bc2_Unorm:
            variation = 12;
            break;

        case nn::gfx::ImageFormat_Bc2_UnormSrgb:
            variation = 13;
            break;

        case nn::gfx::ImageFormat_Bc3_Unorm:
            variation = 14;
            break;

        case nn::gfx::ImageFormat_Bc3_UnormSrgb:
            variation = 15;
            break;

        case nn::gfx::ImageFormat_Bc4_Unorm:
            variation = 16;
            break;

        case nn::gfx::ImageFormat_Bc4_Snorm:
            variation = 17;
            break;

        case nn::gfx::ImageFormat_Bc5_Unorm:
            variation = 18;
            break;

        case nn::gfx::ImageFormat_Bc5_Snorm:
            variation = 19;
            break;

        default:
            NN_UNEXPECTED_DEFAULT;
        }

        return g_TextureCompressionShader.pResShaderContainer->GetResShaderVariation(variation)->GetResShaderProgram(g_TextureCompressionShader.codeType)->GetShader();
    }

    nn::gfx::Texture g_SourceTexture;
    void InitializeSourceTextures(nn::gfx::ImageFormat imageFormat)
    {
        NN_LOG("Initialize SourceTextures\n");
        nn::gfx::Texture::InfoType info;
        {
            info.SetDefault();
            info.SetGpuAccessFlags(nn::gfx::GpuAccess_Texture);
            info.SetWidth(SourceTextureWidth);
            info.SetHeight(SourceTextureHeight);
            info.SetImageFormat(imageFormat);
            info.SetMipCount(MipLevels);
            info.SetArrayLength(LayerCount);
        }

        if (NN_STATIC_CONDITION(nn::gfx::Texture::IsMemoryPoolRequired))
        {
            size_t size = nn::gfx::Texture::CalculateMipDataSize(g_GraphicsFramework.GetDevice(), info);
            size_t alignment = nn::gfx::Texture::CalculateMipDataAlignment(g_GraphicsFramework.GetDevice(), info);

            ptrdiff_t offset = -1;
            offset = g_GraphicsFramework.AllocatePoolMemory(
            nnt::gfx::GraphicsFramework::MemoryPoolType::MemoryPoolType_RenderTarget,
            size, alignment);
            NN_ASSERT_NOT_EQUAL(offset, -1);

            g_SourceTexture.Initialize(g_GraphicsFramework.GetDevice(),info,
                g_GraphicsFramework.GetMemoryPool(nnt::gfx::GraphicsFramework::MemoryPoolType::MemoryPoolType_RenderTarget),
                offset, size);
        }
        else
        {
            g_SourceTexture.Initialize(g_GraphicsFramework.GetDevice(),info,NULL,0,0);
        }
    }
    void FinalizeSourceTextures()
    {
        NN_LOG("Finalize SourceTextures\n");
        g_SourceTexture.Finalize(g_GraphicsFramework.GetDevice());
    }

    nn::gfx::ColorTargetView g_SourceColorTargetView[LayerCount][MipLevels];
    void InitializeSourceColorTargetViews(nn::gfx::ImageFormat imageFormat)
    {
        NN_LOG("Initialize SourceColorTargetView\n");
        nn::gfx::ColorTargetView::InfoType info;
        for(int layer = 0; layer < LayerCount; layer++)
        {
            for (int mipLevel = 0; mipLevel < MipLevels; mipLevel++)
            {
                info.SetDefault();
                info.SetTexturePtr(&g_SourceTexture);
                info.SetImageFormat(imageFormat);
                info.SetMipLevel(mipLevel);
                info.EditArrayRange().SetBaseArrayIndex(layer);
                info.EditArrayRange().SetArrayLength(1);
                g_SourceColorTargetView[layer][mipLevel].Initialize(g_GraphicsFramework.GetDevice(), info);
            }
        }
    }
    void FinalizeSourceColorTargetViews()
    {
        NN_LOG("Finalize SourceColorTargetView\n");
        for (int layer = 0; layer < LayerCount; layer++)
        {
            for (int miplevel = 0; miplevel < MipLevels; miplevel++)
            {
                g_SourceColorTargetView[layer][miplevel].Finalize(g_GraphicsFramework.GetDevice());
            }
        }
    }

    nn::gfx::TextureView g_SourceTextureView;
    void InitializeSourceTextureView(nn::gfx::ImageFormat imageFormat)
    {
        NN_LOG("Initialize SourceTextureView\n");
        {
            nn::gfx::TextureView::InfoType info;
            info.SetDefault();
            info.SetTexturePtr(&g_SourceTexture);
            info.SetImageFormat(imageFormat);
            info.SetImageDimension(nn::gfx::ImageDimension_2dArray);
            info.EditSubresourceRange().EditMipRange().SetMipCount(MipLevels);
            info.EditSubresourceRange().EditArrayRange().SetArrayLength(LayerCount);
            g_SourceTextureView.Initialize(g_GraphicsFramework.GetDevice(),info);
        }
    }
    void FinalizeSourceTextureView()
    {
        NN_LOG("Finalize SourceTextureView\n");
        g_SourceTextureView.Finalize(g_GraphicsFramework.GetDevice());
    }

    nn::gfx::Texture g_DestTexture;
    nn::gfx::util::TextureCompressorTargetInfo g_TextureCompressorTargetTextureInfo;
    void InitializeDestTexture(nn::gfx::ImageFormat imageFormat)
    {
        nn::gfx::Texture::InfoType info;
        NN_LOG("Initialize DestTexture\n");
        {
            info.SetDefault();
            info.SetGpuAccessFlags(nn::gfx::GpuAccess_Texture);
            info.SetWidth(DestTextureWidth);
            info.SetHeight(DestTextureHeight);
            info.SetImageFormat(imageFormat);
            info.SetMipCount(MipLevels);
            info.SetArrayLength(LayerCount);
        }

        ptrdiff_t offset = 0;
        size_t size = 0;
        if(NN_STATIC_CONDITION(nn::gfx::Texture::IsMemoryPoolRequired))
        {
            size = nn::gfx::Texture::CalculateMipDataSize(g_GraphicsFramework.GetDevice(), info);
            size_t alignment = nn::gfx::Texture::CalculateMipDataAlignment(g_GraphicsFramework.GetDevice(), info);

            offset = -1;
            offset = g_GraphicsFramework.AllocatePoolMemory(
                nnt::gfx::GraphicsFramework::MemoryPoolType::MemoryPoolType_RenderTarget,
                size, alignment);
            NN_ASSERT_NOT_EQUAL(offset, -1);

            g_DestTexture.Initialize(
                g_GraphicsFramework.GetDevice(),
                info,
                g_GraphicsFramework.GetMemoryPool(nnt::gfx::GraphicsFramework::MemoryPoolType::MemoryPoolType_RenderTarget),
                offset, size);
        }
        else
        {
            g_DestTexture.Initialize(g_GraphicsFramework.GetDevice(),info,NULL,0,0);
        }

        // 圧縮処理クラスの初期化
        NN_LOG("Initialize TextureCompressor\n");
        g_TextureCompressorTargetTextureInfo.SetTexture(info,&g_DestTexture);
        g_TextureCompressor.Initialize(g_GraphicsFramework.GetDevice());

        nn::gfx::Shader* pShader = GetTextureCompressionShader(imageFormat);
        g_TextureCompressor.RegisterCompressionShader(imageFormat,pShader);
    }
    void FinalizeDestTexture()
    {
        NN_LOG("Finalize TextureCompressor\n");
        g_TextureCompressor.Finalize();

        NN_LOG("Finalize DestTexture\n");
        g_DestTexture.Finalize(g_GraphicsFramework.GetDevice());
    }

    nn::gfx::TextureView g_DestTextureView;
    void InitializeDestTextureView(nn::gfx::ImageFormat imageFormat)
    {
        NN_LOG("Initialize DestTextureView\n");
        nn::gfx::TextureView::InfoType info;
        info.SetDefault();
        info.SetTexturePtr(&g_DestTexture);
        info.SetImageFormat(imageFormat);
        info.SetImageDimension(nn::gfx::ImageDimension_2dArray);
        info.EditSubresourceRange().EditMipRange().SetMipCount(MipLevels);
        info.EditSubresourceRange().EditArrayRange().SetArrayLength(LayerCount);
        g_DestTextureView.Initialize(g_GraphicsFramework.GetDevice(),info);
    }
    void FinalizeDestTextureView()
    {
        NN_LOG("Finalize DestTextureView\n");
        g_DestTextureView.Finalize(g_GraphicsFramework.GetDevice());
    }


    void InitializeTriangle()
    {
        Triangle::InitializeVertexBuffer();
        Triangle::InitializeIndexBuffer();
    }
    void FinalizeTriangle()
    {
        Triangle::FinalizeVertexBuffer();
        Triangle::FinalizeIndexBuffer();
    }

    nn::gfx::ViewportScissorState   g_RenderTriangleViewportScissorStates[MipLevels];
    void InitializeRenderTriangleViewportScissor()
    {
        nn::gfx::ViewportScissorState::InfoType info;
        for (int miplevel = 0; miplevel < MipLevels; miplevel++)
        {
            info.SetDefault();
            info.SetScissorEnabled(true);

            nn::gfx::ViewportStateInfo viewportInfo;
            {
                viewportInfo.SetDefault();
                viewportInfo.SetWidth(static_cast<float>(std::max(DestTextureWidth >> miplevel, 1)));
                viewportInfo.SetHeight(static_cast<float>(std::max(DestTextureHeight >> miplevel, 1)));
            }

            nn::gfx::ScissorStateInfo scissorInfo;
            {
                scissorInfo.SetDefault();
                scissorInfo.SetWidth(std::max(DestTextureWidth >> miplevel, 1));
                scissorInfo.SetHeight(std::max(DestTextureHeight >> miplevel, 1));
            }

            info.SetViewportStateInfoArray(&viewportInfo, 1);
            info.SetScissorStateInfoArray(&scissorInfo, 1);
            g_RenderTriangleViewportScissorStates[miplevel].Initialize(g_GraphicsFramework.GetDevice(), info);
        }
    }
    void FinalizeRenderTriangleViewportScissor()
    {
        for (int miplevel = 0; miplevel < MipLevels; miplevel++)
        {
            g_RenderTriangleViewportScissorStates[miplevel].Finalize(g_GraphicsFramework.GetDevice());
        }
    }

    nn::gfx::VertexState g_TriangleVertexState;
    void InitializeRenderTriangleVertexState()
    {
        nn::gfx::VertexState::InfoType vertexStateInfo;

        nn::gfx::VertexAttributeStateInfo vertexAttributeStateInfo[Triangle::VertexLayout::End];
        {
            vertexAttributeStateInfo[0].SetDefault();
            vertexAttributeStateInfo[0].SetBufferIndex(0);
            vertexAttributeStateInfo[0].SetFormat(nn::gfx::AttributeFormat_32_32_32_Float);
            vertexAttributeStateInfo[0].SetOffset(Triangle::Offset[0]);
            vertexAttributeStateInfo[0].SetShaderSlot(Triangle::VertexShaderSlot::Position);
        }
        {
            vertexAttributeStateInfo[1].SetDefault();
            vertexAttributeStateInfo[1].SetBufferIndex(0);
            vertexAttributeStateInfo[1].SetFormat(nn::gfx::AttributeFormat_32_32_32_Float);
            vertexAttributeStateInfo[1].SetOffset(Triangle::Offset[1]);
            vertexAttributeStateInfo[1].SetShaderSlot(Triangle::VertexShaderSlot::Color);
        }

        nn::gfx::VertexBufferStateInfo vertexBufferStateInfo;
        {
            vertexBufferStateInfo.SetDefault();
            vertexBufferStateInfo.SetStride(Triangle::VertexStride);
        }


        // VertexStateInfo
        {
            vertexStateInfo.SetDefault();
            vertexStateInfo.SetVertexAttributeStateInfoArray(vertexAttributeStateInfo, Triangle::End);
            vertexStateInfo.SetVertexBufferStateInfoArray(&vertexBufferStateInfo, 1);
        }

         size_t size = nn::gfx::VertexState::GetRequiredMemorySize(vertexStateInfo);
         void* memory = g_GraphicsFramework.AllocateMemory(size,nn::gfx::VertexState::RequiredMemoryInfo_Alignment);
         g_TriangleVertexState.SetMemory(memory,size);

        nn::gfx::Shader* pVertexShader = g_TriangleResShader.pResShaderContainer->GetResShaderVariation(0)->GetResShaderProgram(g_TriangleResShader.codeType)->GetShader();
        g_TriangleVertexState.Initialize(g_GraphicsFramework.GetDevice(), vertexStateInfo, pVertexShader);
    }
    void FinalizeRenderTriangleVertexState()
    {
        g_TriangleVertexState.Finalize(g_GraphicsFramework.GetDevice());
    }


    void InitializeRenderTrianglePipeline()
    {
        InitializeResShaderFile(&g_TriangleResShader,RenderTriangleShaderBinaryFilePath);
        InitializeRenderTriangleVertexState();
    }
    void FinalizeRenderTrianglePipeline()
    {
        FinalizeRenderTriangleVertexState();
        FinalizeResShaderFile(&g_TriangleResShader);
    }


    void InitializeRenderTriangleRoutine()
    {
        NN_LOG("Initialize RenderTriangleRoutine\n");

        InitializeTriangle();
        InitializeRenderTriangleViewportScissor();
        InitializeRenderTrianglePipeline();
    }
    void FinalizeRenderTriangleRoutine()
    {
        NN_LOG("Finalize RenderTriangleRoutine\n");

        FinalizeRenderTrianglePipeline();
        FinalizeRenderTriangleViewportScissor();
        FinalizeTriangle();
    }


    nn::gfx::ViewportScissorState g_SourceViewportScissorState[MipLevels * LayerCount];
    nn::gfx::ViewportScissorState g_DestViewportScissorState[MipLevels * LayerCount];
    void InitializeRectangleViewportScissorState(nn::gfx::ViewportScissorState* pViewportScissorState,int originX,int originY,int width,int height)
    {
        nn::gfx::ViewportScissorState::InfoType info;
        nn::gfx::ScissorStateInfo scissorInfo;
        nn::gfx::ViewportStateInfo viewportInfo;

        const int imageWidth = ((width + MarginX) / LayerCount) - MarginX;
        const int imageHeight = ((height + MarginY) / MipLevels) - MarginY;
        NN_ASSERT_GREATER(imageWidth, 0);
        NN_ASSERT_GREATER(imageHeight, 0);

        int p = 0;
        for (int layer = 0; layer < LayerCount; layer++)
        {
            int x = originX + layer * (imageWidth + MarginX);
            for (int miplevel = 0; miplevel < MipLevels; miplevel++)
            {
                int y = originY + miplevel * (imageHeight + MarginY);

                //viewport
                {
                    viewportInfo.SetDefault();
                    viewportInfo.SetOriginX(static_cast<float>(x));
                    viewportInfo.SetOriginY(static_cast<float>(y));
                    viewportInfo.SetWidth(static_cast<float>(imageWidth));
                    viewportInfo.SetHeight(static_cast<float>(imageHeight));
                    viewportInfo.SetMinDepth(.0f);
                    viewportInfo.SetMaxDepth(1.0f);
                }

                //scissor
                {
                    scissorInfo.SetDefault();
                    scissorInfo.SetOriginX(x);
                    scissorInfo.SetOriginY(y);
                    scissorInfo.SetWidth(imageWidth);
                    scissorInfo.SetHeight(imageHeight);
                }

                info.SetDefault();
                info.SetScissorEnabled(true);
                info.SetViewportStateInfoArray(&viewportInfo,1);
                info.SetScissorStateInfoArray(&scissorInfo,1);

                pViewportScissorState[p].Initialize(g_GraphicsFramework.GetDevice(),info);
                p++;
            }
        }
    }
    void FinalizeRectangleViewportScissorState(nn::gfx::ViewportScissorState* pViewportScissorState)
    {
        int p = 0;
        for (int layer = 0; layer < LayerCount; layer++)
        {
            for (int miplevel = 0; miplevel < MipLevels; miplevel++)
            {
                pViewportScissorState[p].Finalize(g_GraphicsFramework.GetDevice());
                p++;
            }
        }
    }


    void InitializeRenderRectanglePipeline()
    {
        InitializeResShaderFile(&g_TextureResShader,RenderTextureShaderBinaryFilePath);

        // ViewportScissorState
        InitializeRectangleViewportScissorState(
            g_SourceViewportScissorState,
            0,0,
            ScreenWidhth / 2,ScreenHeight
        );
        InitializeRectangleViewportScissorState(
            g_DestViewportScissorState,
            ScreenWidhth / 2,0,
            ScreenWidhth / 2,ScreenHeight
        );
    }
    void FinalizeRenderRectanglePipeLine()
    {
        FinalizeRectangleViewportScissorState(g_DestViewportScissorState);
        FinalizeRectangleViewportScissorState(g_SourceViewportScissorState);
        FinalizeResShaderFile(&g_TextureResShader);
    }

    void InitializeRenderTexture()
    {
        NN_LOG("Initialize RenderTextureRoutine\n");

        InitializeRenderRectanglePipeline();
    }
    void FinalizeRenderTexture()
    {
        NN_LOG("Finalize RenderTextureRoutine\n");

        FinalizeRenderRectanglePipeLine();
    }


    void InitializeGfxObjects(nn::gfx::ImageFormat srcFormat,nn::gfx::ImageFormat dstFormat)
    {
        NN_LOG("Initialize GfxObjects\n");

        InitializeSourceTextures(srcFormat);
        InitializeSourceColorTargetViews(srcFormat);
        InitializeSourceTextureView(srcFormat);

        NN_LOG("Initialize TextureCompressionBC1 Shader\n");
        InitializeResShaderFile(&g_TextureCompressionShader,TextureCompressionShaderBinaryFilePath);

        InitializeDestTexture(dstFormat);
        InitializeDestTextureView(dstFormat);
        InitializeRenderTriangleRoutine();
        InitializeRenderTexture();
    }

    void FinalizeGfxObjects()
    {
        FinalizeRenderTexture();
        FinalizeRenderTriangleRoutine();
        FinalizeDestTextureView();
        FinalizeDestTexture();

        FinalizeResShaderFile(&g_TextureCompressionShader);
        NN_LOG("Finalize TextureCompressionBC1 Shader\n");

        FinalizeSourceTextureView();
        FinalizeSourceColorTargetViews();
        FinalizeSourceTextures();

        NN_LOG("Finalize GfxObjects\n");
    }


    void MakeRenderTriangleCommand(nn::gfx::CommandBuffer* pCommandBuffer)
    {
        nn::gfx::GpuAddress gpuAddress;

        //gfx states
        pCommandBuffer->SetBlendState(g_GraphicsFramework.GetBlendState(nnt::gfx::GraphicsFramework::BlendStateType_Disabled));
        pCommandBuffer->SetDepthStencilState(g_GraphicsFramework.GetDepthStencilState(nnt::gfx::GraphicsFramework::DepthStencilStateType_Disabled));
        pCommandBuffer->SetRasterizerState(g_GraphicsFramework.GetRasterizerState(nnt::gfx::GraphicsFramework::RasterizerStateType_FillSolid_CullNone));

        {
            nn::gfx::Shader* pShader = g_TriangleResShader.pResShaderContainer->GetResShaderVariation(0)->GetResShaderProgram(g_TriangleResShader.codeType)->GetShader();
            pCommandBuffer->SetShader(pShader, nn::gfx::ShaderStageBit_All);
            pCommandBuffer->SetVertexState(&g_TriangleVertexState);


            Triangle::vertexBuffer.GetGpuAddress(&gpuAddress);
            pCommandBuffer->SetVertexBuffer(0, gpuAddress, Triangle::VertexStride, sizeof(Triangle::VertexData));
        }

        Triangle::indexBuffer.GetGpuAddress(&gpuAddress);
        pCommandBuffer->DrawIndexed(nn::gfx::PrimitiveTopology_TriangleList,
            nn::gfx::IndexFormat_Uint32, gpuAddress, sizeof(Triangle::IndexData) / sizeof(Triangle::IndexData[0]), 0);
    }

    void MakeRenderRectangleCommand(nn::gfx::CommandBuffer* pCommandBuffer,nn::gfx::Shader* pShader,nn::gfx::TextureView* pTextureView,int layer,int miplevel)
    {
        nn::gfx::GpuAddress gpuAddress;

        pCommandBuffer->SetBlendState(g_GraphicsFramework.GetBlendState(nnt::gfx::GraphicsFramework::BlendStateType_Alpha));
        pCommandBuffer->SetDepthStencilState(g_GraphicsFramework.GetDepthStencilState(nnt::gfx::GraphicsFramework::DepthStencilStateType_Disabled));
        pCommandBuffer->SetRasterizerState(g_GraphicsFramework.GetRasterizerState(nnt::gfx::GraphicsFramework::RasterizerStateType_FillSolid_CullNone));


        nn::gfx::DescriptorPool* pTextureDescriptorPool = g_pTransientTextureDescriptorAllocator->GetDescriptorPool();
        int textureSlotIndex = g_pTransientTextureDescriptorAllocator->Allocate();
        {
            pTextureDescriptorPool->BeginUpdate();
            pTextureDescriptorPool->SetTextureView(textureSlotIndex,pTextureView);
            pTextureDescriptorPool->EndUpdate();
        }

        nn::gfx::DescriptorPool* pSamplerDescriptorPool = g_pTransientSamplerDescriptorAllocator->GetDescriptorPool();
        int samplerSlotIndex = g_pTransientSamplerDescriptorAllocator->Allocate();
        {
            pSamplerDescriptorPool->BeginUpdate();
            pSamplerDescriptorPool->SetSampler(samplerSlotIndex,&g_SamplerMinMagMipPointEdgeClamp);
            pSamplerDescriptorPool->EndUpdate();
        }

        nn::gfx::DescriptorSlot textureDescriptorSlot;
        nn::gfx::DescriptorSlot samplerDescriptorSlot;
        {
            g_pTransientTextureDescriptorAllocator->GetDescriptorPool()->GetDescriptorSlot(&textureDescriptorSlot,textureSlotIndex);
            g_pTransientSamplerDescriptorAllocator->GetDescriptorPool()->GetDescriptorSlot(&samplerDescriptorSlot,samplerSlotIndex);
        }

        pCommandBuffer->SetTextureAndSampler(0,nn::gfx::ShaderStage_Pixel,textureDescriptorSlot,samplerDescriptorSlot);
        pCommandBuffer->SetShader(pShader, nn::gfx::ShaderStageBit_All);

        // miplevelとレイヤーは15ビットずつあれば十分とする
        NN_ASSERT_RANGE(layer,0,0x8000);
        NN_ASSERT_RANGE(miplevel,0,0x8000);

        int texInfo = layer & 0x7fff;
        texInfo |= ( (miplevel << 15) & 0xffff8000);
        texInfo <<= 2;

        pCommandBuffer->Draw(nn::gfx::PrimitiveTopology_TriangleStrip,4,texInfo,1,0);
    }


    void RenderToSourceTexture(nn::gfx::CommandBuffer* pCommandBuffer,float alpha)
    {
        for (int layer = 0; layer < LayerCount; layer++)
        {
            for (int miplevel = 0; miplevel < MipLevels; miplevel++)
            {
                float clearR = 0.2f + (0.8f / MipLevels) * miplevel;
                float clearG = 0.2f + (0.8f / LayerCount) * layer;
                float clearB = 1.0f;

                nn::gfx::ColorTargetView* pSourceColorTargetView = &g_SourceColorTargetView[layer][miplevel];
                pCommandBuffer->ClearColor(pSourceColorTargetView, clearR, clearG, clearB, alpha, NULL);
                pCommandBuffer->SetRenderTargets(1, &pSourceColorTargetView, NULL);
                pCommandBuffer->SetViewportScissorState(&g_RenderTriangleViewportScissorStates[miplevel]);

                MakeRenderTriangleCommand(pCommandBuffer);
            }
        }
    }

    void RenderToSourceTexture(nn::gfx::CommandBuffer* pCommandBuffer)
    {
        RenderToSourceTexture(pCommandBuffer,1.0f);
    }

    void RenderToDestTexture(nn::gfx::CommandBuffer* pCommandBuffer)
    {
        nn::gfx::DescriptorPool* pTextureDescriptorPool = g_pTransientTextureDescriptorAllocator->GetDescriptorPool();
        int textureSlotIndex = g_pTransientTextureDescriptorAllocator->Allocate();
        {
            pTextureDescriptorPool->BeginUpdate();
            pTextureDescriptorPool->SetTextureView(textureSlotIndex,&g_SourceTextureView);
            pTextureDescriptorPool->EndUpdate();
        }
        nn::gfx::DescriptorSlot textureDescriptorSlot;
        pTextureDescriptorPool->GetDescriptorSlot(&textureDescriptorSlot,textureSlotIndex);

        g_TextureCompressor.MakeCommand(pCommandBuffer,textureDescriptorSlot,g_SamplerDescriptorSlot,g_TextureCompressorTargetTextureInfo);
    }


    void RenderSourceTexture(nn::gfx::CommandBuffer* pCommandBuffer)
    {
        nn::gfx::Shader* pShader = g_TextureResShader.pResShaderContainer->GetResShaderVariation(0)->GetResShaderProgram(g_TriangleResShader.codeType)->GetShader();

        int p = 0;
        for (int layer = 0; layer < LayerCount; layer++)
        {
            for (int miplevel = 0; miplevel < MipLevels; miplevel++)
            {
                pCommandBuffer->SetViewportScissorState(&g_SourceViewportScissorState[p]);
                MakeRenderRectangleCommand(pCommandBuffer,pShader,&g_SourceTextureView,layer,miplevel);
                p++;
            }
        }
    }

    void RenderDestTexture(nn::gfx::CommandBuffer* pCommandBuffer)
    {
        nn::gfx::Shader* pShader = g_TextureResShader.pResShaderContainer->GetResShaderVariation(0)->GetResShaderProgram(g_TriangleResShader.codeType)->GetShader();

        int p = 0;
        for (int layer = 0; layer < LayerCount; layer++)
        {
            for (int miplevel = 0; miplevel < MipLevels; miplevel++)
            {
                pCommandBuffer->SetViewportScissorState(&g_DestViewportScissorState[p]);
                MakeRenderRectangleCommand(pCommandBuffer, pShader, &g_DestTextureView,layer,miplevel);
                p++;
            }
        }
    }

    void InitializeSampler()
    {
        int samplerSlot = g_GraphicsFramework.AllocateDescriptorSlot(nn::gfx::DescriptorPoolType_Sampler,1);
        {
            nn::gfx::Sampler::InfoType info;
            info.SetDefault();
            info.SetFilterMode(nn::gfx::FilterMode_MinPoint_MagPoint_MipPoint);
            info.SetAddressU(nn::gfx::TextureAddressMode_ClampToEdge);
            info.SetAddressV(nn::gfx::TextureAddressMode_ClampToEdge);
            info.SetAddressW(nn::gfx::TextureAddressMode_ClampToEdge);
            g_SamplerMinMagMipPointEdgeClamp.Initialize(g_GraphicsFramework.GetDevice(),info);
        }

        g_GraphicsFramework.SetSamplerToDescriptorPool(samplerSlot,&g_SamplerMinMagMipPointEdgeClamp);
        nn::gfx::DescriptorPool* pSamplerDescriptorPool = g_GraphicsFramework.GetDescriptorPool(nn::gfx::DescriptorPoolType_Sampler);
        pSamplerDescriptorPool->GetDescriptorSlot(&g_SamplerDescriptorSlot,samplerSlot);
    }


    void MountContents()
    {
        static const size_t MountRomCacheBufferSize = 4 * 1024;
        static char s_MountRomCacheBuffer[MountRomCacheBufferSize];
        size_t mountRomCacheUseSize = 0;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::QueryMountRomCacheSize(&mountRomCacheUseSize));
        ASSERT_LE(mountRomCacheUseSize, MountRomCacheBufferSize);

        NN_LOG("Mount Contents\n");
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::MountRom("Contents", s_MountRomCacheBuffer, MountRomCacheBufferSize));
        nn::fs::DirectoryEntry entryList[100];
        nn::fs::DirectoryHandle h = {};
        nn::fs::OpenDirectory(&h, "Contents:/", nn::fs::OpenDirectoryMode_All);
        int64_t n = 0;
        nn::fs::ReadDirectory(&n, entryList, h, 100);
        NN_LOG("%d entry found.\n", static_cast<int>(n));
        for(int64_t i = 0; i < n; i++)
        {
            auto& e =entryList[i];
            NN_LOG("  %s%s\n", e.name, (e.directoryEntryType == nn::fs::DirectoryEntryType_Directory ? "/" : ""));
        }
        nn::fs::CloseDirectory(h);
    }

}

TEST(TextureCompression,Bc1_Unorm)
{
    const nn::gfx::ImageFormat srcFormat = nn::gfx::ImageFormat_B8_G8_R8_A8_Unorm;
    const nn::gfx::ImageFormat dstFormat = nn::gfx::ImageFormat_Bc1_Unorm;
    const int excuteFrame = 60 * 1;

    MountContents();

    NN_LOG("Initialize GraphicsFramework\n");
    nnt::gfx::GraphicsFramework::FrameworkInfo fwInfo;
    {
        fwInfo.SetDefault();
        fwInfo.SetMemoryPoolSize(nnt::gfx::GraphicsFramework::MemoryPoolType_RenderTarget,64 * 1024 * 1024);
    }
    g_GraphicsFramework.Initialize(fwInfo);

    InitializeSampler();
    {
        NN_LOG("Initialize DescriptorAllocatorHolder\n");
        g_TransientTextureDescriptorAllocatorHolder.Initialize(
            g_GraphicsFramework.GetDescriptorPool(nn::gfx::DescriptorPoolType_TextureView),
            g_GraphicsFramework.AllocateDescriptorSlot(nn::gfx::DescriptorPoolType_TextureView,1024),
            1024);
        g_pTransientTextureDescriptorAllocator = g_TransientTextureDescriptorAllocatorHolder.Get();

        g_TransientSamplerDescriptorAllocatorHolder.Initialize(
            g_GraphicsFramework.GetDescriptorPool(nn::gfx::DescriptorPoolType_Sampler),
            g_GraphicsFramework.AllocateDescriptorSlot(nn::gfx::DescriptorPoolType_Sampler,1024),
            1024);
        g_pTransientSamplerDescriptorAllocator = g_TransientSamplerDescriptorAllocatorHolder.Get();
    }
    g_pTransientTextureDescriptorAllocator->FillHistory();
    g_pTransientSamplerDescriptorAllocator->FillHistory();


    InitializeGfxObjects(srcFormat,dstFormat);


    for (int frame = 0; frame < excuteFrame; frame++)
    {
        // テクスチャ取得と vsync 待ち
        g_GraphicsFramework.AcquireTexture(0);
        g_GraphicsFramework.WaitDisplaySync(0, nn::TimeSpan::FromSeconds(2));

        nn::gfx::ColorTargetView* pTarget = g_GraphicsFramework.GetColorTargetView();

        g_pTransientTextureDescriptorAllocator->Free();
        g_pTransientSamplerDescriptorAllocator->Free();
        g_pTransientTextureDescriptorAllocator->Begin();
        g_pTransientSamplerDescriptorAllocator->Begin();

        g_GraphicsFramework.BeginFrame(0);
        {
            nn::gfx::CommandBuffer* rootCommandBuffer = g_GraphicsFramework.GetRootCommandBuffer(0);

            // 三角形を圧縮元に描画する
            {
                RenderToSourceTexture(rootCommandBuffer);

                rootCommandBuffer->SetTextureStateTransition(
                    &g_SourceTexture, NULL /* pRange */,
                    nn::gfx::TextureState_ColorTarget,
                    nn::gfx::PipelineStageBit_RenderTarget,
                    nn::gfx::TextureState_ShaderRead,
                    nn::gfx::PipelineStageBit_PixelShader);
            }

            // 圧縮先に圧縮元を描画する
            {
                RenderToDestTexture(rootCommandBuffer);

                rootCommandBuffer->SetTextureStateTransition(
                    &g_DestTexture,
                    NULL /* pRange */,
                    nn::gfx::TextureState_ColorTarget,
                    nn::gfx::PipelineStageBit_RenderTarget,
                    nn::gfx::TextureState_ShaderRead,
                    nn::gfx::PipelineStageBit_PixelShader
                    );
            }

             //バックバッファをクリア・描画対象に
            rootCommandBuffer->ClearColor(pTarget, .05f, .05f, .05f, 1.0f, NULL);
            rootCommandBuffer->SetRenderTargets(1, &pTarget, NULL);

            RenderSourceTexture(rootCommandBuffer);
            RenderDestTexture(rootCommandBuffer);

        }
        g_GraphicsFramework.EndFrame(0);

        g_GraphicsFramework.ExecuteCommand(0);
        g_GraphicsFramework.QueuePresentTexture(1);
        g_GraphicsFramework.WaitGpuSync(0, nn::TimeSpan::FromSeconds(2));

        g_pTransientTextureDescriptorAllocator->End();
        g_pTransientSamplerDescriptorAllocator->End();
    }

    FinalizeGfxObjects();

    NN_LOG("Finalize DescriptorAllocatorHolder\n");
    g_TransientTextureDescriptorAllocatorHolder.Finalize();
    g_SamplerMinMagMipPointEdgeClamp.Finalize(g_GraphicsFramework.GetDevice());
    g_TransientSamplerDescriptorAllocatorHolder.Finalize();

    NN_LOG("Finalize GraphicsFramework\n");
    g_GraphicsFramework.Finalize();

    //コンテンツをアンマウントする
    nn::fs::Unmount("Contents");

    SUCCEED();
}

TEST(TextureCompression,Bc1_Unorm_UseAlpha)
{
    const nn::gfx::ImageFormat srcFormat = nn::gfx::ImageFormat_B8_G8_R8_A8_Unorm;
    const nn::gfx::ImageFormat dstFormat = nn::gfx::ImageFormat_Bc1_Unorm;
    const int excuteFrame = 60 * 1;

    MountContents();

    NN_LOG("Initialize GraphicsFramework\n");
    nnt::gfx::GraphicsFramework::FrameworkInfo fwInfo;
    {
        fwInfo.SetDefault();
        fwInfo.SetMemoryPoolSize(nnt::gfx::GraphicsFramework::MemoryPoolType_RenderTarget,64 * 1024 * 1024);
    }
    g_GraphicsFramework.Initialize(fwInfo);
    InitializeSampler();

    {
        NN_LOG("Initialize DescriptorAllocatorHolder\n");
        g_TransientTextureDescriptorAllocatorHolder.Initialize(
            g_GraphicsFramework.GetDescriptorPool(nn::gfx::DescriptorPoolType_TextureView),
            g_GraphicsFramework.AllocateDescriptorSlot(nn::gfx::DescriptorPoolType_TextureView,1024),
            1024);
        g_pTransientTextureDescriptorAllocator = g_TransientTextureDescriptorAllocatorHolder.Get();

        g_TransientSamplerDescriptorAllocatorHolder.Initialize(
            g_GraphicsFramework.GetDescriptorPool(nn::gfx::DescriptorPoolType_Sampler),
            g_GraphicsFramework.AllocateDescriptorSlot(nn::gfx::DescriptorPoolType_Sampler,1024),
            1024);
        g_pTransientSamplerDescriptorAllocator = g_TransientSamplerDescriptorAllocatorHolder.Get();
    }
    g_pTransientTextureDescriptorAllocator->FillHistory();
    g_pTransientSamplerDescriptorAllocator->FillHistory();


    InitializeGfxObjects(srcFormat,dstFormat);

    for (int frame = 0; frame < excuteFrame; frame++)
    {
        // テクスチャ取得と vsync 待ち
        g_GraphicsFramework.AcquireTexture(0);
        g_GraphicsFramework.WaitDisplaySync(0, nn::TimeSpan::FromSeconds(2));

        nn::gfx::ColorTargetView* pTarget = g_GraphicsFramework.GetColorTargetView();

        g_pTransientTextureDescriptorAllocator->Free();
        g_pTransientSamplerDescriptorAllocator->Free();
        g_pTransientTextureDescriptorAllocator->Begin();
        g_pTransientSamplerDescriptorAllocator->Begin();

        g_GraphicsFramework.BeginFrame(0);
        {
            nn::gfx::CommandBuffer* rootCommandBuffer = g_GraphicsFramework.GetRootCommandBuffer(0);

            // 三角形を圧縮元に描画する
            {
                RenderToSourceTexture(rootCommandBuffer,.0f);

                rootCommandBuffer->SetTextureStateTransition(
                    &g_SourceTexture, NULL /* pRange */,
                    nn::gfx::TextureState_ColorTarget,
                    nn::gfx::PipelineStageBit_RenderTarget,
                    nn::gfx::TextureState_ShaderRead,
                    nn::gfx::PipelineStageBit_PixelShader);
            }

            // 圧縮先に圧縮元を描画する
            {
                RenderToDestTexture(rootCommandBuffer);

                rootCommandBuffer->SetTextureStateTransition(
                    &g_DestTexture,
                    NULL /* pRange */,
                    nn::gfx::TextureState_ColorTarget,
                    nn::gfx::PipelineStageBit_RenderTarget,
                    nn::gfx::TextureState_ShaderRead,
                    nn::gfx::PipelineStageBit_PixelShader
                    );
            }

            //スキャンバッファをクリア・描画対象に
            rootCommandBuffer->ClearColor(pTarget, .05f, .05f, .05f, 1.0f, NULL);
            rootCommandBuffer->SetRenderTargets(1, &pTarget, NULL);

            RenderSourceTexture(rootCommandBuffer);
            RenderDestTexture(rootCommandBuffer);

        }
        g_GraphicsFramework.EndFrame(0);

        g_GraphicsFramework.ExecuteCommand(0);
        g_GraphicsFramework.QueuePresentTexture(1);
        g_GraphicsFramework.WaitGpuSync(0, nn::TimeSpan::FromSeconds(2));

        g_pTransientTextureDescriptorAllocator->End();
        g_pTransientSamplerDescriptorAllocator->End();
    }

    FinalizeGfxObjects();

    NN_LOG("Finalize DescriptorAllocatorHolder\n");
    g_TransientTextureDescriptorAllocatorHolder.Finalize();
    g_TransientSamplerDescriptorAllocatorHolder.Finalize();

    g_SamplerMinMagMipPointEdgeClamp.Finalize(g_GraphicsFramework.GetDevice());
    NN_LOG("Finalize GraphicsFramework\n");
    g_GraphicsFramework.Finalize();

    //コンテンツをアンマウントする
    nn::fs::Unmount("Contents");

    SUCCEED();
}

TEST(TextureCompression,Bc1_Srgb)
{
    const nn::gfx::ImageFormat srcFormat = nn::gfx::ImageFormat_B8_G8_R8_A8_UnormSrgb;
    const nn::gfx::ImageFormat dstFormat = nn::gfx::ImageFormat_Bc1_UnormSrgb;
    const int excuteFrame = 60 * 1;

    MountContents();

    NN_LOG("Initialize GraphicsFramework\n");
    nnt::gfx::GraphicsFramework::FrameworkInfo fwInfo;
    {
        fwInfo.SetDefault();
        fwInfo.SetMemoryPoolSize(nnt::gfx::GraphicsFramework::MemoryPoolType_RenderTarget,64 * 1024 * 1024);
    }
    g_GraphicsFramework.Initialize(fwInfo);
    InitializeSampler();

    {
        NN_LOG("Initialize DescriptorAllocatorHolder\n");
        g_TransientTextureDescriptorAllocatorHolder.Initialize(
            g_GraphicsFramework.GetDescriptorPool(nn::gfx::DescriptorPoolType_TextureView),
            g_GraphicsFramework.AllocateDescriptorSlot(nn::gfx::DescriptorPoolType_TextureView,1024),
            1024);
        g_pTransientTextureDescriptorAllocator = g_TransientTextureDescriptorAllocatorHolder.Get();

        g_TransientSamplerDescriptorAllocatorHolder.Initialize(
            g_GraphicsFramework.GetDescriptorPool(nn::gfx::DescriptorPoolType_Sampler),
            g_GraphicsFramework.AllocateDescriptorSlot(nn::gfx::DescriptorPoolType_Sampler,1024),
            1024);
        g_pTransientSamplerDescriptorAllocator = g_TransientSamplerDescriptorAllocatorHolder.Get();
    }
    g_pTransientTextureDescriptorAllocator->FillHistory();
    g_pTransientSamplerDescriptorAllocator->FillHistory();


    InitializeGfxObjects(srcFormat,dstFormat);


    for (int frame = 0; frame < excuteFrame; frame++)
    {
        // テクスチャ取得と vsync 待ち
        g_GraphicsFramework.AcquireTexture(0);
        g_GraphicsFramework.WaitDisplaySync(0, nn::TimeSpan::FromSeconds(2));

        nn::gfx::ColorTargetView* pTarget = g_GraphicsFramework.GetColorTargetView();

        g_pTransientTextureDescriptorAllocator->Free();
        g_pTransientSamplerDescriptorAllocator->Free();
        g_pTransientTextureDescriptorAllocator->Begin();
        g_pTransientSamplerDescriptorAllocator->Begin();

        g_GraphicsFramework.BeginFrame(0);
        {
            nn::gfx::CommandBuffer* rootCommandBuffer = g_GraphicsFramework.GetRootCommandBuffer(0);

            // 三角形を圧縮元に描画する
            {
                RenderToSourceTexture(rootCommandBuffer);

                rootCommandBuffer->SetTextureStateTransition(
                    &g_SourceTexture, NULL /* pRange */,
                    nn::gfx::TextureState_ColorTarget,
                    nn::gfx::PipelineStageBit_RenderTarget,
                    nn::gfx::TextureState_ShaderRead,
                    nn::gfx::PipelineStageBit_PixelShader);
            }

            // 圧縮先に圧縮元を描画する
            {
                RenderToDestTexture(rootCommandBuffer);

                rootCommandBuffer->SetTextureStateTransition(
                    &g_DestTexture,
                    NULL /* pRange */,
                    nn::gfx::TextureState_ColorTarget,
                    nn::gfx::PipelineStageBit_RenderTarget,
                    nn::gfx::TextureState_ShaderRead,
                    nn::gfx::PipelineStageBit_PixelShader
                    );
            }

             //バックバッファをクリア・描画対象に
            rootCommandBuffer->ClearColor(pTarget, .05f, .05f, .05f, 1.0f, NULL);
            rootCommandBuffer->SetRenderTargets(1, &pTarget, NULL);

            RenderSourceTexture(rootCommandBuffer);
            RenderDestTexture(rootCommandBuffer);

        }
        g_GraphicsFramework.EndFrame(0);

        g_GraphicsFramework.ExecuteCommand(0);
        g_GraphicsFramework.QueuePresentTexture(1);
        g_GraphicsFramework.WaitGpuSync(0, nn::TimeSpan::FromSeconds(2));

        g_pTransientTextureDescriptorAllocator->End();
        g_pTransientSamplerDescriptorAllocator->End();
    }

    FinalizeGfxObjects();

    NN_LOG("Finalize DescriptorAllocatorHolder\n");
    g_TransientTextureDescriptorAllocatorHolder.Finalize();
    g_TransientSamplerDescriptorAllocatorHolder.Finalize();

    g_SamplerMinMagMipPointEdgeClamp.Finalize(g_GraphicsFramework.GetDevice());
    NN_LOG("Finalize GraphicsFramework\n");
    g_GraphicsFramework.Finalize();

    //コンテンツをアンマウントする
    nn::fs::Unmount("Contents");
    SUCCEED();
}

TEST(TextureCompression,Bc1_Srgb_UseAlpha)
{
    const nn::gfx::ImageFormat srcFormat = nn::gfx::ImageFormat_B8_G8_R8_A8_UnormSrgb;
    const nn::gfx::ImageFormat dstFormat = nn::gfx::ImageFormat_Bc1_UnormSrgb;
    const int excuteFrame = 60 * 1;

    MountContents();

    NN_LOG("Initialize GraphicsFramework\n");
    nnt::gfx::GraphicsFramework::FrameworkInfo fwInfo;
    {
        fwInfo.SetDefault();
        fwInfo.SetMemoryPoolSize(nnt::gfx::GraphicsFramework::MemoryPoolType_RenderTarget,64 * 1024 * 1024);
    }
    g_GraphicsFramework.Initialize(fwInfo);
    InitializeSampler();

    {
        NN_LOG("Initialize DescriptorAllocatorHolder\n");
        g_TransientTextureDescriptorAllocatorHolder.Initialize(
            g_GraphicsFramework.GetDescriptorPool(nn::gfx::DescriptorPoolType_TextureView),
            g_GraphicsFramework.AllocateDescriptorSlot(nn::gfx::DescriptorPoolType_TextureView,1024),
            1024);
        g_pTransientTextureDescriptorAllocator = g_TransientTextureDescriptorAllocatorHolder.Get();

        g_TransientSamplerDescriptorAllocatorHolder.Initialize(
            g_GraphicsFramework.GetDescriptorPool(nn::gfx::DescriptorPoolType_Sampler),
            g_GraphicsFramework.AllocateDescriptorSlot(nn::gfx::DescriptorPoolType_Sampler,1024),
            1024);
        g_pTransientSamplerDescriptorAllocator = g_TransientSamplerDescriptorAllocatorHolder.Get();
    }
    g_pTransientTextureDescriptorAllocator->FillHistory();
    g_pTransientSamplerDescriptorAllocator->FillHistory();


    InitializeGfxObjects(srcFormat,dstFormat);


    for (int frame = 0; frame < excuteFrame; frame++)
    {
        // テクスチャ取得と vsync 待ち
        g_GraphicsFramework.AcquireTexture(0);
        g_GraphicsFramework.WaitDisplaySync(0, nn::TimeSpan::FromSeconds(2));

        nn::gfx::ColorTargetView* pTarget = g_GraphicsFramework.GetColorTargetView();

        g_pTransientTextureDescriptorAllocator->Free();
        g_pTransientSamplerDescriptorAllocator->Free();
        g_pTransientTextureDescriptorAllocator->Begin();
        g_pTransientSamplerDescriptorAllocator->Begin();

        g_GraphicsFramework.BeginFrame(0);
        {
            nn::gfx::CommandBuffer* rootCommandBuffer = g_GraphicsFramework.GetRootCommandBuffer(0);

            // 三角形を圧縮元に描画する
            {
                RenderToSourceTexture(rootCommandBuffer,.0f);

                rootCommandBuffer->SetTextureStateTransition(
                    &g_SourceTexture, NULL /* pRange */,
                    nn::gfx::TextureState_ColorTarget,
                    nn::gfx::PipelineStageBit_RenderTarget,
                    nn::gfx::TextureState_ShaderRead,
                    nn::gfx::PipelineStageBit_PixelShader);
            }

            // 圧縮先に圧縮元を描画する
            {
                RenderToDestTexture(rootCommandBuffer);

                rootCommandBuffer->SetTextureStateTransition(
                    &g_DestTexture,
                    NULL /* pRange */,
                    nn::gfx::TextureState_ColorTarget,
                    nn::gfx::PipelineStageBit_RenderTarget,
                    nn::gfx::TextureState_ShaderRead,
                    nn::gfx::PipelineStageBit_PixelShader
                    );
            }

             //バックバッファをクリア・描画対象に
            rootCommandBuffer->ClearColor(pTarget, .05f, .05f, .05f, 1.0f, NULL);
            rootCommandBuffer->SetRenderTargets(1, &pTarget, NULL);

            RenderSourceTexture(rootCommandBuffer);
            RenderDestTexture(rootCommandBuffer);

        }
        g_GraphicsFramework.EndFrame(0);

        g_GraphicsFramework.ExecuteCommand(0);
        g_GraphicsFramework.QueuePresentTexture(1);
        g_GraphicsFramework.WaitGpuSync(0, nn::TimeSpan::FromSeconds(2));

        g_pTransientTextureDescriptorAllocator->End();
        g_pTransientSamplerDescriptorAllocator->End();
    }

    FinalizeGfxObjects();

    NN_LOG("Finalize DescriptorAllocatorHolder\n");
    g_TransientTextureDescriptorAllocatorHolder.Finalize();
    g_TransientSamplerDescriptorAllocatorHolder.Finalize();

    g_SamplerMinMagMipPointEdgeClamp.Finalize(g_GraphicsFramework.GetDevice());

    NN_LOG("Finalize GraphicsFramework\n");
    g_GraphicsFramework.Finalize();

    //コンテンツをアンマウントする
    nn::fs::Unmount("Contents");
    SUCCEED();
}

TEST(TextureCompression,Bc2_Unorm)
{
    const nn::gfx::ImageFormat srcFormat = nn::gfx::ImageFormat_B8_G8_R8_A8_Unorm;
    const nn::gfx::ImageFormat dstFormat = nn::gfx::ImageFormat_Bc2_Unorm;
    const int excuteFrame = 60 * 2;

    MountContents();

    NN_LOG("Initialize GraphicsFramework\n");
    nnt::gfx::GraphicsFramework::FrameworkInfo fwInfo;
    {
        fwInfo.SetDefault();
        fwInfo.SetMemoryPoolSize(nnt::gfx::GraphicsFramework::MemoryPoolType_RenderTarget,64 * 1024 * 1024);
    }
    g_GraphicsFramework.Initialize(fwInfo);
    InitializeSampler();

    {
        NN_LOG("Initialize DescriptorAllocatorHolder\n");
        g_TransientTextureDescriptorAllocatorHolder.Initialize(
            g_GraphicsFramework.GetDescriptorPool(nn::gfx::DescriptorPoolType_TextureView),
            g_GraphicsFramework.AllocateDescriptorSlot(nn::gfx::DescriptorPoolType_TextureView,1024),
            1024);
        g_pTransientTextureDescriptorAllocator = g_TransientTextureDescriptorAllocatorHolder.Get();

        g_TransientSamplerDescriptorAllocatorHolder.Initialize(
            g_GraphicsFramework.GetDescriptorPool(nn::gfx::DescriptorPoolType_Sampler),
            g_GraphicsFramework.AllocateDescriptorSlot(nn::gfx::DescriptorPoolType_Sampler,1024),
            1024);
        g_pTransientSamplerDescriptorAllocator = g_TransientSamplerDescriptorAllocatorHolder.Get();
    }
    g_pTransientTextureDescriptorAllocator->FillHistory();
    g_pTransientSamplerDescriptorAllocator->FillHistory();


    InitializeGfxObjects(srcFormat,dstFormat);


    for (int frame = 0; frame < excuteFrame; frame++)
    {
        // テクスチャ取得と vsync 待ち
        g_GraphicsFramework.AcquireTexture(0);
        g_GraphicsFramework.WaitDisplaySync(0, nn::TimeSpan::FromSeconds(2));

        nn::gfx::ColorTargetView* pTarget = g_GraphicsFramework.GetColorTargetView();

        g_pTransientTextureDescriptorAllocator->Free();
        g_pTransientSamplerDescriptorAllocator->Free();
        g_pTransientTextureDescriptorAllocator->Begin();
        g_pTransientSamplerDescriptorAllocator->Begin();

        g_GraphicsFramework.BeginFrame(0);
        {
            nn::gfx::CommandBuffer* rootCommandBuffer = g_GraphicsFramework.GetRootCommandBuffer(0);

            // 三角形を圧縮元に描画する
            {
                float alpha = static_cast<float>(frame) / static_cast<float>(excuteFrame);
                RenderToSourceTexture(rootCommandBuffer,alpha);

                rootCommandBuffer->SetTextureStateTransition(
                    &g_SourceTexture, NULL /* pRange */,
                    nn::gfx::TextureState_ColorTarget,
                    nn::gfx::PipelineStageBit_RenderTarget,
                    nn::gfx::TextureState_ShaderRead,
                    nn::gfx::PipelineStageBit_PixelShader);
            }

            // 圧縮先に圧縮元を描画する
            {
                RenderToDestTexture(rootCommandBuffer);

                rootCommandBuffer->SetTextureStateTransition(
                    &g_DestTexture,
                    NULL /* pRange */,
                    nn::gfx::TextureState_ColorTarget,
                    nn::gfx::PipelineStageBit_RenderTarget,
                    nn::gfx::TextureState_ShaderRead,
                    nn::gfx::PipelineStageBit_PixelShader
                    );
            }

             //バックバッファをクリア・描画対象に
            rootCommandBuffer->ClearColor(pTarget, .05f, .05f, .05f, 1.0f, NULL);
            rootCommandBuffer->SetRenderTargets(1, &pTarget, NULL);

            RenderSourceTexture(rootCommandBuffer);
            RenderDestTexture(rootCommandBuffer);

        }
        g_GraphicsFramework.EndFrame(0);

        g_GraphicsFramework.ExecuteCommand(0);
        g_GraphicsFramework.QueuePresentTexture(1);
        g_GraphicsFramework.WaitGpuSync(0, nn::TimeSpan::FromSeconds(2));

        g_pTransientTextureDescriptorAllocator->End();
        g_pTransientSamplerDescriptorAllocator->End();
    }

    FinalizeGfxObjects();

    NN_LOG("Finalize DescriptorAllocatorHolder\n");
    g_TransientTextureDescriptorAllocatorHolder.Finalize();
    g_TransientSamplerDescriptorAllocatorHolder.Finalize();

    g_SamplerMinMagMipPointEdgeClamp.Finalize(g_GraphicsFramework.GetDevice());

    NN_LOG("Finalize GraphicsFramework\n");
    g_GraphicsFramework.Finalize();

    //コンテンツをアンマウントする
    nn::fs::Unmount("Contents");
    SUCCEED();
}

TEST(TextureCompression,Bc2_Srgb)
{
    const nn::gfx::ImageFormat srcFormat = nn::gfx::ImageFormat_B8_G8_R8_A8_UnormSrgb;
    const nn::gfx::ImageFormat dstFormat = nn::gfx::ImageFormat_Bc2_UnormSrgb;
    const int excuteFrame = 60 * 2;

    MountContents();

    NN_LOG("Initialize GraphicsFramework\n");
    nnt::gfx::GraphicsFramework::FrameworkInfo fwInfo;
    {
        fwInfo.SetDefault();
        fwInfo.SetMemoryPoolSize(nnt::gfx::GraphicsFramework::MemoryPoolType_RenderTarget,64 * 1024 * 1024);
    }
    g_GraphicsFramework.Initialize(fwInfo);
    InitializeSampler();

    {
        NN_LOG("Initialize DescriptorAllocatorHolder\n");
        g_TransientTextureDescriptorAllocatorHolder.Initialize(
            g_GraphicsFramework.GetDescriptorPool(nn::gfx::DescriptorPoolType_TextureView),
            g_GraphicsFramework.AllocateDescriptorSlot(nn::gfx::DescriptorPoolType_TextureView,1024),
            1024);
        g_pTransientTextureDescriptorAllocator = g_TransientTextureDescriptorAllocatorHolder.Get();

        g_TransientSamplerDescriptorAllocatorHolder.Initialize(
            g_GraphicsFramework.GetDescriptorPool(nn::gfx::DescriptorPoolType_Sampler),
            g_GraphicsFramework.AllocateDescriptorSlot(nn::gfx::DescriptorPoolType_Sampler,1024),
            1024);
        g_pTransientSamplerDescriptorAllocator = g_TransientSamplerDescriptorAllocatorHolder.Get();
    }
    g_pTransientTextureDescriptorAllocator->FillHistory();
    g_pTransientSamplerDescriptorAllocator->FillHistory();


    InitializeGfxObjects(srcFormat,dstFormat);


    for (int frame = 0; frame < excuteFrame; frame++)
    {
        // テクスチャ取得と vsync 待ち
        g_GraphicsFramework.AcquireTexture(0);
        g_GraphicsFramework.WaitDisplaySync(0, nn::TimeSpan::FromSeconds(2));

        nn::gfx::ColorTargetView* pTarget = g_GraphicsFramework.GetColorTargetView();

        g_pTransientTextureDescriptorAllocator->Free();
        g_pTransientSamplerDescriptorAllocator->Free();
        g_pTransientTextureDescriptorAllocator->Begin();
        g_pTransientSamplerDescriptorAllocator->Begin();

        g_GraphicsFramework.BeginFrame(0);
        {
            float alpha = static_cast<float>(frame) / static_cast<float>(excuteFrame);
            nn::gfx::CommandBuffer* rootCommandBuffer = g_GraphicsFramework.GetRootCommandBuffer(0);

            // 三角形を圧縮元に描画する
            {
                RenderToSourceTexture(rootCommandBuffer,alpha);

                rootCommandBuffer->SetTextureStateTransition(
                    &g_SourceTexture, NULL /* pRange */,
                    nn::gfx::TextureState_ColorTarget,
                    nn::gfx::PipelineStageBit_RenderTarget,
                    nn::gfx::TextureState_ShaderRead,
                    nn::gfx::PipelineStageBit_PixelShader);
            }

            // 圧縮先に圧縮元を描画する
            {
                RenderToDestTexture(rootCommandBuffer);

                rootCommandBuffer->SetTextureStateTransition(
                    &g_DestTexture,
                    NULL /* pRange */,
                    nn::gfx::TextureState_ColorTarget,
                    nn::gfx::PipelineStageBit_RenderTarget,
                    nn::gfx::TextureState_ShaderRead,
                    nn::gfx::PipelineStageBit_PixelShader
                    );
            }

             //バックバッファをクリア・描画対象に
            rootCommandBuffer->ClearColor(pTarget, .05f, .05f, .05f, 1.0f, NULL);
            rootCommandBuffer->SetRenderTargets(1, &pTarget, NULL);

            RenderSourceTexture(rootCommandBuffer);
            RenderDestTexture(rootCommandBuffer);

        }
        g_GraphicsFramework.EndFrame(0);

        g_GraphicsFramework.ExecuteCommand(0);
        g_GraphicsFramework.QueuePresentTexture(1);
        g_GraphicsFramework.WaitGpuSync(0, nn::TimeSpan::FromSeconds(2));

        g_pTransientTextureDescriptorAllocator->End();
        g_pTransientSamplerDescriptorAllocator->End();
    }

    FinalizeGfxObjects();

    NN_LOG("Finalize DescriptorAllocatorHolder\n");
    g_TransientTextureDescriptorAllocatorHolder.Finalize();
    g_TransientSamplerDescriptorAllocatorHolder.Finalize();

    g_SamplerMinMagMipPointEdgeClamp.Finalize(g_GraphicsFramework.GetDevice());

    NN_LOG("Finalize GraphicsFramework\n");
    g_GraphicsFramework.Finalize();

    //コンテンツをアンマウントする
    nn::fs::Unmount("Contents");
    SUCCEED();
}

TEST(TextureCompression,Bc3_Unorm)
{
    const nn::gfx::ImageFormat srcFormat = nn::gfx::ImageFormat_B8_G8_R8_A8_Unorm;
    const nn::gfx::ImageFormat dstFormat = nn::gfx::ImageFormat_Bc3_Unorm;
    const int excuteFrame = 60 * 2;

    MountContents();

    NN_LOG("Initialize GraphicsFramework\n");
    nnt::gfx::GraphicsFramework::FrameworkInfo fwInfo;
    {
        fwInfo.SetDefault();
        fwInfo.SetMemoryPoolSize(nnt::gfx::GraphicsFramework::MemoryPoolType_RenderTarget,64 * 1024 * 1024);
    }
    g_GraphicsFramework.Initialize(fwInfo);
    InitializeSampler();

    {
        NN_LOG("Initialize DescriptorAllocatorHolder\n");
        g_TransientTextureDescriptorAllocatorHolder.Initialize(
            g_GraphicsFramework.GetDescriptorPool(nn::gfx::DescriptorPoolType_TextureView),
            g_GraphicsFramework.AllocateDescriptorSlot(nn::gfx::DescriptorPoolType_TextureView,1024),
            1024);
        g_pTransientTextureDescriptorAllocator = g_TransientTextureDescriptorAllocatorHolder.Get();

        g_TransientSamplerDescriptorAllocatorHolder.Initialize(
            g_GraphicsFramework.GetDescriptorPool(nn::gfx::DescriptorPoolType_Sampler),
            g_GraphicsFramework.AllocateDescriptorSlot(nn::gfx::DescriptorPoolType_Sampler,1024),
            1024);
        g_pTransientSamplerDescriptorAllocator = g_TransientSamplerDescriptorAllocatorHolder.Get();
    }
    g_pTransientTextureDescriptorAllocator->FillHistory();
    g_pTransientSamplerDescriptorAllocator->FillHistory();


    InitializeGfxObjects(srcFormat,dstFormat);


    for (int frame = 0; frame < excuteFrame; frame++)
    {
        // テクスチャ取得と vsync 待ち
        g_GraphicsFramework.AcquireTexture(0);
        g_GraphicsFramework.WaitDisplaySync(0, nn::TimeSpan::FromSeconds(2));

        nn::gfx::ColorTargetView* pTarget = g_GraphicsFramework.GetColorTargetView();

        g_pTransientTextureDescriptorAllocator->Free();
        g_pTransientSamplerDescriptorAllocator->Free();
        g_pTransientTextureDescriptorAllocator->Begin();
        g_pTransientSamplerDescriptorAllocator->Begin();

        g_GraphicsFramework.BeginFrame(0);
        {
            nn::gfx::CommandBuffer* rootCommandBuffer = g_GraphicsFramework.GetRootCommandBuffer(0);

            // 三角形を圧縮元に描画する
            {
                float alpha = static_cast<float>(frame) / static_cast<float>(excuteFrame);
                RenderToSourceTexture(rootCommandBuffer,alpha);

                rootCommandBuffer->SetTextureStateTransition(
                    &g_SourceTexture, NULL /* pRange */,
                    nn::gfx::TextureState_ColorTarget,
                    nn::gfx::PipelineStageBit_RenderTarget,
                    nn::gfx::TextureState_ShaderRead,
                    nn::gfx::PipelineStageBit_PixelShader);
            }

            // 圧縮先に圧縮元を描画する
            {
                RenderToDestTexture(rootCommandBuffer);

                rootCommandBuffer->SetTextureStateTransition(
                    &g_DestTexture,
                    NULL /* pRange */,
                    nn::gfx::TextureState_ColorTarget,
                    nn::gfx::PipelineStageBit_RenderTarget,
                    nn::gfx::TextureState_ShaderRead,
                    nn::gfx::PipelineStageBit_PixelShader
                    );
            }

             //バックバッファをクリア・描画対象に
            rootCommandBuffer->ClearColor(pTarget, .05f, .05f, .05f, 1.0f, NULL);
            rootCommandBuffer->SetRenderTargets(1, &pTarget, NULL);

            RenderSourceTexture(rootCommandBuffer);
            RenderDestTexture(rootCommandBuffer);

        }
        g_GraphicsFramework.EndFrame(0);

        g_GraphicsFramework.ExecuteCommand(0);
        g_GraphicsFramework.QueuePresentTexture(1);
        g_GraphicsFramework.WaitGpuSync(0, nn::TimeSpan::FromSeconds(2));

        g_pTransientTextureDescriptorAllocator->End();
        g_pTransientSamplerDescriptorAllocator->End();
    }

    FinalizeGfxObjects();

    NN_LOG("Finalize DescriptorAllocatorHolder\n");
    g_TransientTextureDescriptorAllocatorHolder.Finalize();
    g_TransientSamplerDescriptorAllocatorHolder.Finalize();

    g_SamplerMinMagMipPointEdgeClamp.Finalize(g_GraphicsFramework.GetDevice());

    NN_LOG("Finalize GraphicsFramework\n");
    g_GraphicsFramework.Finalize();

    //コンテンツをアンマウントする
    nn::fs::Unmount("Contents");
    SUCCEED();
}

TEST(TextureCompression,Bc3_Srgb)
{
    const nn::gfx::ImageFormat srcFormat = nn::gfx::ImageFormat_B8_G8_R8_A8_UnormSrgb;
    const nn::gfx::ImageFormat dstFormat = nn::gfx::ImageFormat_Bc3_UnormSrgb;
    const int excuteFrame = 60 * 2;

    MountContents();

    NN_LOG("Initialize GraphicsFramework\n");
    nnt::gfx::GraphicsFramework::FrameworkInfo fwInfo;
    {
        fwInfo.SetDefault();
        fwInfo.SetMemoryPoolSize(nnt::gfx::GraphicsFramework::MemoryPoolType_RenderTarget,64 * 1024 * 1024);
    }
    g_GraphicsFramework.Initialize(fwInfo);
    InitializeSampler();

    {
        NN_LOG("Initialize DescriptorAllocatorHolder\n");
        g_TransientTextureDescriptorAllocatorHolder.Initialize(
            g_GraphicsFramework.GetDescriptorPool(nn::gfx::DescriptorPoolType_TextureView),
            g_GraphicsFramework.AllocateDescriptorSlot(nn::gfx::DescriptorPoolType_TextureView,1024),
            1024);
        g_pTransientTextureDescriptorAllocator = g_TransientTextureDescriptorAllocatorHolder.Get();

        g_TransientSamplerDescriptorAllocatorHolder.Initialize(
            g_GraphicsFramework.GetDescriptorPool(nn::gfx::DescriptorPoolType_Sampler),
            g_GraphicsFramework.AllocateDescriptorSlot(nn::gfx::DescriptorPoolType_Sampler,1024),
            1024);
        g_pTransientSamplerDescriptorAllocator = g_TransientSamplerDescriptorAllocatorHolder.Get();
    }
    g_pTransientTextureDescriptorAllocator->FillHistory();
    g_pTransientSamplerDescriptorAllocator->FillHistory();


    InitializeGfxObjects(srcFormat,dstFormat);


    for (int frame = 0; frame < excuteFrame; frame++)
    {
        // テクスチャ取得と vsync 待ち
        g_GraphicsFramework.AcquireTexture(0);
        g_GraphicsFramework.WaitDisplaySync(0, nn::TimeSpan::FromSeconds(2));

        nn::gfx::ColorTargetView* pTarget = g_GraphicsFramework.GetColorTargetView();

        g_pTransientTextureDescriptorAllocator->Free();
        g_pTransientSamplerDescriptorAllocator->Free();
        g_pTransientTextureDescriptorAllocator->Begin();
        g_pTransientSamplerDescriptorAllocator->Begin();

        g_GraphicsFramework.BeginFrame(0);
        {
            float alpha = static_cast<float>(frame) / static_cast<float>(excuteFrame);
            nn::gfx::CommandBuffer* rootCommandBuffer = g_GraphicsFramework.GetRootCommandBuffer(0);

            // 三角形を圧縮元に描画する
            {
                RenderToSourceTexture(rootCommandBuffer,alpha);

                rootCommandBuffer->SetTextureStateTransition(
                    &g_SourceTexture, NULL /* pRange */,
                    nn::gfx::TextureState_ColorTarget,
                    nn::gfx::PipelineStageBit_RenderTarget,
                    nn::gfx::TextureState_ShaderRead,
                    nn::gfx::PipelineStageBit_PixelShader);
            }

            // 圧縮先に圧縮元を描画する
            {
                RenderToDestTexture(rootCommandBuffer);

                rootCommandBuffer->SetTextureStateTransition(
                    &g_DestTexture,
                    NULL /* pRange */,
                    nn::gfx::TextureState_ColorTarget,
                    nn::gfx::PipelineStageBit_RenderTarget,
                    nn::gfx::TextureState_ShaderRead,
                    nn::gfx::PipelineStageBit_PixelShader
                    );
            }

             //バックバッファをクリア・描画対象に
            rootCommandBuffer->ClearColor(pTarget, .05f, .05f, .05f, 1.0f, NULL);
            rootCommandBuffer->SetRenderTargets(1, &pTarget, NULL);

            RenderSourceTexture(rootCommandBuffer);
            RenderDestTexture(rootCommandBuffer);

        }
        g_GraphicsFramework.EndFrame(0);

        g_GraphicsFramework.ExecuteCommand(0);
        g_GraphicsFramework.QueuePresentTexture(1);
        g_GraphicsFramework.WaitGpuSync(0, nn::TimeSpan::FromSeconds(2));

        g_pTransientTextureDescriptorAllocator->End();
        g_pTransientSamplerDescriptorAllocator->End();
    }

    FinalizeGfxObjects();

    NN_LOG("Finalize DescriptorAllocatorHolder\n");
    g_TransientTextureDescriptorAllocatorHolder.Finalize();
    g_TransientSamplerDescriptorAllocatorHolder.Finalize();

    g_SamplerMinMagMipPointEdgeClamp.Finalize(g_GraphicsFramework.GetDevice());

    NN_LOG("Finalize GraphicsFramework\n");
    g_GraphicsFramework.Finalize();

    //コンテンツをアンマウントする
    nn::fs::Unmount("Contents");
    SUCCEED();
}

TEST(TextureCompression,Bc4_Unorm)
{
    const nn::gfx::ImageFormat srcFormat = nn::gfx::ImageFormat_R32_Float;
    const nn::gfx::ImageFormat dstFormat = nn::gfx::ImageFormat_Bc4_Unorm;
    const int excuteFrame = 60 * 2;

    MountContents();

    NN_LOG("Initialize GraphicsFramework\n");
    nnt::gfx::GraphicsFramework::FrameworkInfo fwInfo;
    {
        fwInfo.SetDefault();
        fwInfo.SetMemoryPoolSize(nnt::gfx::GraphicsFramework::MemoryPoolType_RenderTarget,64 * 1024 * 1024);
    }
    g_GraphicsFramework.Initialize(fwInfo);
    InitializeSampler();

    {
        NN_LOG("Initialize DescriptorAllocatorHolder\n");
        g_TransientTextureDescriptorAllocatorHolder.Initialize(
            g_GraphicsFramework.GetDescriptorPool(nn::gfx::DescriptorPoolType_TextureView),
            g_GraphicsFramework.AllocateDescriptorSlot(nn::gfx::DescriptorPoolType_TextureView,1024),
            1024);
        g_pTransientTextureDescriptorAllocator = g_TransientTextureDescriptorAllocatorHolder.Get();

        g_TransientSamplerDescriptorAllocatorHolder.Initialize(
            g_GraphicsFramework.GetDescriptorPool(nn::gfx::DescriptorPoolType_Sampler),
            g_GraphicsFramework.AllocateDescriptorSlot(nn::gfx::DescriptorPoolType_Sampler,1024),
            1024);
        g_pTransientSamplerDescriptorAllocator = g_TransientSamplerDescriptorAllocatorHolder.Get();
    }
    g_pTransientTextureDescriptorAllocator->FillHistory();
    g_pTransientSamplerDescriptorAllocator->FillHistory();


    InitializeGfxObjects(srcFormat,dstFormat);


    for (int frame = 0; frame < excuteFrame; frame++)
    {
        // テクスチャ取得と vsync 待ち
        g_GraphicsFramework.AcquireTexture(0);
        g_GraphicsFramework.WaitDisplaySync(0, nn::TimeSpan::FromSeconds(2));

        nn::gfx::ColorTargetView* pTarget = g_GraphicsFramework.GetColorTargetView();

        g_pTransientTextureDescriptorAllocator->Free();
        g_pTransientSamplerDescriptorAllocator->Free();
        g_pTransientTextureDescriptorAllocator->Begin();
        g_pTransientSamplerDescriptorAllocator->Begin();

        g_GraphicsFramework.BeginFrame(0);
        {
            nn::gfx::CommandBuffer* rootCommandBuffer = g_GraphicsFramework.GetRootCommandBuffer(0);

            // 三角形を圧縮元に描画する
            {
                RenderToSourceTexture(rootCommandBuffer,1.0f);

                rootCommandBuffer->SetTextureStateTransition(
                    &g_SourceTexture, NULL /* pRange */,
                    nn::gfx::TextureState_ColorTarget,
                    nn::gfx::PipelineStageBit_RenderTarget,
                    nn::gfx::TextureState_ShaderRead,
                    nn::gfx::PipelineStageBit_PixelShader);
            }

            // 圧縮先に圧縮元を描画する
            {
                RenderToDestTexture(rootCommandBuffer);

                rootCommandBuffer->SetTextureStateTransition(
                    &g_DestTexture,
                    NULL /* pRange */,
                    nn::gfx::TextureState_ColorTarget,
                    nn::gfx::PipelineStageBit_RenderTarget,
                    nn::gfx::TextureState_ShaderRead,
                    nn::gfx::PipelineStageBit_PixelShader
                    );
            }

             //バックバッファをクリア・描画対象に
            rootCommandBuffer->ClearColor(pTarget, .05f, .05f, .05f, 1.0f, NULL);
            rootCommandBuffer->SetRenderTargets(1, &pTarget, NULL);

            RenderSourceTexture(rootCommandBuffer);
            RenderDestTexture(rootCommandBuffer);

        }
        g_GraphicsFramework.EndFrame(0);

        g_GraphicsFramework.ExecuteCommand(0);
        g_GraphicsFramework.QueuePresentTexture(1);
        g_GraphicsFramework.WaitGpuSync(0, nn::TimeSpan::FromSeconds(2));

        g_pTransientTextureDescriptorAllocator->End();
        g_pTransientSamplerDescriptorAllocator->End();
    }

    FinalizeGfxObjects();

    NN_LOG("Finalize DescriptorAllocatorHolder\n");
    g_TransientTextureDescriptorAllocatorHolder.Finalize();
    g_TransientSamplerDescriptorAllocatorHolder.Finalize();

    g_SamplerMinMagMipPointEdgeClamp.Finalize(g_GraphicsFramework.GetDevice());

    NN_LOG("Finalize GraphicsFramework\n");
    g_GraphicsFramework.Finalize();

    //コンテンツをアンマウントする
    nn::fs::Unmount("Contents");
    SUCCEED();
}

TEST(TextureCompression,Bc4_Snorm)
{
    const nn::gfx::ImageFormat srcFormat = nn::gfx::ImageFormat_R32_Float;
    const nn::gfx::ImageFormat dstFormat = nn::gfx::ImageFormat_Bc4_Snorm;
    const int excuteFrame = 60 * 2;

    MountContents();

    NN_LOG("Initialize GraphicsFramework\n");
    nnt::gfx::GraphicsFramework::FrameworkInfo fwInfo;
    {
        fwInfo.SetDefault();
        fwInfo.SetMemoryPoolSize(nnt::gfx::GraphicsFramework::MemoryPoolType_RenderTarget,64 * 1024 * 1024);
    }
    g_GraphicsFramework.Initialize(fwInfo);
    InitializeSampler();

    {
        NN_LOG("Initialize DescriptorAllocatorHolder\n");
        g_TransientTextureDescriptorAllocatorHolder.Initialize(
            g_GraphicsFramework.GetDescriptorPool(nn::gfx::DescriptorPoolType_TextureView),
            g_GraphicsFramework.AllocateDescriptorSlot(nn::gfx::DescriptorPoolType_TextureView,1024),
            1024);
        g_pTransientTextureDescriptorAllocator = g_TransientTextureDescriptorAllocatorHolder.Get();

        g_TransientSamplerDescriptorAllocatorHolder.Initialize(
            g_GraphicsFramework.GetDescriptorPool(nn::gfx::DescriptorPoolType_Sampler),
            g_GraphicsFramework.AllocateDescriptorSlot(nn::gfx::DescriptorPoolType_Sampler,1024),
            1024);
        g_pTransientSamplerDescriptorAllocator = g_TransientSamplerDescriptorAllocatorHolder.Get();
    }
    g_pTransientTextureDescriptorAllocator->FillHistory();
    g_pTransientSamplerDescriptorAllocator->FillHistory();


    InitializeGfxObjects(srcFormat,dstFormat);


    for (int frame = 0; frame < excuteFrame; frame++)
    {
        // テクスチャ取得と vsync 待ち
        g_GraphicsFramework.AcquireTexture(0);
        g_GraphicsFramework.WaitDisplaySync(0, nn::TimeSpan::FromSeconds(2));

        nn::gfx::ColorTargetView* pTarget = g_GraphicsFramework.GetColorTargetView();

        g_pTransientTextureDescriptorAllocator->Free();
        g_pTransientSamplerDescriptorAllocator->Free();
        g_pTransientTextureDescriptorAllocator->Begin();
        g_pTransientSamplerDescriptorAllocator->Begin();

        g_GraphicsFramework.BeginFrame(0);
        {
            nn::gfx::CommandBuffer* rootCommandBuffer = g_GraphicsFramework.GetRootCommandBuffer(0);

            // 三角形を圧縮元に描画する
            {
                RenderToSourceTexture(rootCommandBuffer,1.0f);

                rootCommandBuffer->SetTextureStateTransition(
                    &g_SourceTexture, NULL /* pRange */,
                    nn::gfx::TextureState_ColorTarget,
                    nn::gfx::PipelineStageBit_RenderTarget,
                    nn::gfx::TextureState_ShaderRead,
                    nn::gfx::PipelineStageBit_PixelShader);
            }

            // 圧縮先に圧縮元を描画する
            {
                RenderToDestTexture(rootCommandBuffer);

                rootCommandBuffer->SetTextureStateTransition(
                    &g_DestTexture,
                    NULL /* pRange */,
                    nn::gfx::TextureState_ColorTarget,
                    nn::gfx::PipelineStageBit_RenderTarget,
                    nn::gfx::TextureState_ShaderRead,
                    nn::gfx::PipelineStageBit_PixelShader
                    );
            }

             //バックバッファをクリア・描画対象に
            rootCommandBuffer->ClearColor(pTarget, .05f, .05f, .05f, 1.0f, NULL);
            rootCommandBuffer->SetRenderTargets(1, &pTarget, NULL);

            RenderSourceTexture(rootCommandBuffer);
            RenderDestTexture(rootCommandBuffer);

        }
        g_GraphicsFramework.EndFrame(0);

        g_GraphicsFramework.ExecuteCommand(0);
        g_GraphicsFramework.QueuePresentTexture(1);
        g_GraphicsFramework.WaitGpuSync(0, nn::TimeSpan::FromSeconds(2));

        g_pTransientTextureDescriptorAllocator->End();
        g_pTransientSamplerDescriptorAllocator->End();
    }

    FinalizeGfxObjects();

    NN_LOG("Finalize DescriptorAllocatorHolder\n");
    g_TransientTextureDescriptorAllocatorHolder.Finalize();
    g_TransientSamplerDescriptorAllocatorHolder.Finalize();

    g_SamplerMinMagMipPointEdgeClamp.Finalize(g_GraphicsFramework.GetDevice());

    NN_LOG("Finalize GraphicsFramework\n");
    g_GraphicsFramework.Finalize();

    //コンテンツをアンマウントする
    nn::fs::Unmount("Contents");
    SUCCEED();
}

TEST(TextureCompression,Bc5_Unorm)
{
    const nn::gfx::ImageFormat srcFormat = nn::gfx::ImageFormat_R32_G32_Float;
    const nn::gfx::ImageFormat dstFormat = nn::gfx::ImageFormat_Bc5_Unorm;
    const int excuteFrame = 60 * 2;

    MountContents();

    NN_LOG("Initialize GraphicsFramework\n");
    nnt::gfx::GraphicsFramework::FrameworkInfo fwInfo;
    {
        fwInfo.SetDefault();
        fwInfo.SetMemoryPoolSize(nnt::gfx::GraphicsFramework::MemoryPoolType_RenderTarget,64 * 1024 * 1024);
    }
    g_GraphicsFramework.Initialize(fwInfo);
    InitializeSampler();

    {
        NN_LOG("Initialize DescriptorAllocatorHolder\n");
        g_TransientTextureDescriptorAllocatorHolder.Initialize(
            g_GraphicsFramework.GetDescriptorPool(nn::gfx::DescriptorPoolType_TextureView),
            g_GraphicsFramework.AllocateDescriptorSlot(nn::gfx::DescriptorPoolType_TextureView,1024),
            1024);
        g_pTransientTextureDescriptorAllocator = g_TransientTextureDescriptorAllocatorHolder.Get();

        g_TransientSamplerDescriptorAllocatorHolder.Initialize(
            g_GraphicsFramework.GetDescriptorPool(nn::gfx::DescriptorPoolType_Sampler),
            g_GraphicsFramework.AllocateDescriptorSlot(nn::gfx::DescriptorPoolType_Sampler,1024),
            1024);
        g_pTransientSamplerDescriptorAllocator = g_TransientSamplerDescriptorAllocatorHolder.Get();
    }
    g_pTransientTextureDescriptorAllocator->FillHistory();
    g_pTransientSamplerDescriptorAllocator->FillHistory();


    InitializeGfxObjects(srcFormat,dstFormat);


    for (int frame = 0; frame < excuteFrame; frame++)
    {
        // テクスチャ取得と vsync 待ち
        g_GraphicsFramework.AcquireTexture(0);
        g_GraphicsFramework.WaitDisplaySync(0, nn::TimeSpan::FromSeconds(2));

        nn::gfx::ColorTargetView* pTarget = g_GraphicsFramework.GetColorTargetView();

        g_pTransientTextureDescriptorAllocator->Free();
        g_pTransientSamplerDescriptorAllocator->Free();
        g_pTransientTextureDescriptorAllocator->Begin();
        g_pTransientSamplerDescriptorAllocator->Begin();

        g_GraphicsFramework.BeginFrame(0);
        {
            nn::gfx::CommandBuffer* rootCommandBuffer = g_GraphicsFramework.GetRootCommandBuffer(0);

            // 三角形を圧縮元に描画する
            {
                RenderToSourceTexture(rootCommandBuffer,1.0f);

                rootCommandBuffer->SetTextureStateTransition(
                    &g_SourceTexture, NULL /* pRange */,
                    nn::gfx::TextureState_ColorTarget,
                    nn::gfx::PipelineStageBit_RenderTarget,
                    nn::gfx::TextureState_ShaderRead,
                    nn::gfx::PipelineStageBit_PixelShader);
            }

            // 圧縮先に圧縮元を描画する
            {
                RenderToDestTexture(rootCommandBuffer);

                rootCommandBuffer->SetTextureStateTransition(
                    &g_DestTexture,
                    NULL /* pRange */,
                    nn::gfx::TextureState_ColorTarget,
                    nn::gfx::PipelineStageBit_RenderTarget,
                    nn::gfx::TextureState_ShaderRead,
                    nn::gfx::PipelineStageBit_PixelShader
                    );
            }

             //バックバッファをクリア・描画対象に
            rootCommandBuffer->ClearColor(pTarget, .05f, .05f, .05f, 1.0f, NULL);
            rootCommandBuffer->SetRenderTargets(1, &pTarget, NULL);

            RenderSourceTexture(rootCommandBuffer);
            RenderDestTexture(rootCommandBuffer);

        }
        g_GraphicsFramework.EndFrame(0);

        g_GraphicsFramework.ExecuteCommand(0);
        g_GraphicsFramework.QueuePresentTexture(1);
        g_GraphicsFramework.WaitGpuSync(0, nn::TimeSpan::FromSeconds(2));

        g_pTransientTextureDescriptorAllocator->End();
        g_pTransientSamplerDescriptorAllocator->End();
    }

    FinalizeGfxObjects();

    NN_LOG("Finalize DescriptorAllocatorHolder\n");
    g_TransientTextureDescriptorAllocatorHolder.Finalize();
    g_TransientSamplerDescriptorAllocatorHolder.Finalize();

    g_SamplerMinMagMipPointEdgeClamp.Finalize(g_GraphicsFramework.GetDevice());

    NN_LOG("Finalize GraphicsFramework\n");
    g_GraphicsFramework.Finalize();

    //コンテンツをアンマウントする
    nn::fs::Unmount("Contents");
    SUCCEED();
}

TEST(TextureCompression,Bc5_Snorm)
{
    const nn::gfx::ImageFormat srcFormat = nn::gfx::ImageFormat_R32_G32_Float;
    const nn::gfx::ImageFormat dstFormat = nn::gfx::ImageFormat_Bc5_Snorm;
    const int excuteFrame = 60 * 2;

    MountContents();

    NN_LOG("Initialize GraphicsFramework\n");
    nnt::gfx::GraphicsFramework::FrameworkInfo fwInfo;
    {
        fwInfo.SetDefault();
        fwInfo.SetMemoryPoolSize(nnt::gfx::GraphicsFramework::MemoryPoolType_RenderTarget,64 * 1024 * 1024);
    }
    g_GraphicsFramework.Initialize(fwInfo);
    InitializeSampler();

    {
        NN_LOG("Initialize DescriptorAllocatorHolder\n");
        g_TransientTextureDescriptorAllocatorHolder.Initialize(
            g_GraphicsFramework.GetDescriptorPool(nn::gfx::DescriptorPoolType_TextureView),
            g_GraphicsFramework.AllocateDescriptorSlot(nn::gfx::DescriptorPoolType_TextureView,1024),
            1024);
        g_pTransientTextureDescriptorAllocator = g_TransientTextureDescriptorAllocatorHolder.Get();

        g_TransientSamplerDescriptorAllocatorHolder.Initialize(
            g_GraphicsFramework.GetDescriptorPool(nn::gfx::DescriptorPoolType_Sampler),
            g_GraphicsFramework.AllocateDescriptorSlot(nn::gfx::DescriptorPoolType_Sampler,1024),
            1024);
        g_pTransientSamplerDescriptorAllocator = g_TransientSamplerDescriptorAllocatorHolder.Get();
    }
    g_pTransientTextureDescriptorAllocator->FillHistory();
    g_pTransientSamplerDescriptorAllocator->FillHistory();


    InitializeGfxObjects(srcFormat,dstFormat);


    for (int frame = 0; frame < excuteFrame; frame++)
    {
        // テクスチャ取得と vsync 待ち
        g_GraphicsFramework.AcquireTexture(0);
        g_GraphicsFramework.WaitDisplaySync(0, nn::TimeSpan::FromSeconds(2));

        nn::gfx::ColorTargetView* pTarget = g_GraphicsFramework.GetColorTargetView();

        g_pTransientTextureDescriptorAllocator->Free();
        g_pTransientSamplerDescriptorAllocator->Free();
        g_pTransientTextureDescriptorAllocator->Begin();
        g_pTransientSamplerDescriptorAllocator->Begin();

        g_GraphicsFramework.BeginFrame(0);
        {
            nn::gfx::CommandBuffer* rootCommandBuffer = g_GraphicsFramework.GetRootCommandBuffer(0);

            // 三角形を圧縮元に描画する
            {
                RenderToSourceTexture(rootCommandBuffer,1.0f);

                rootCommandBuffer->SetTextureStateTransition(
                    &g_SourceTexture, NULL /* pRange */,
                    nn::gfx::TextureState_ColorTarget,
                    nn::gfx::PipelineStageBit_RenderTarget,
                    nn::gfx::TextureState_ShaderRead,
                    nn::gfx::PipelineStageBit_PixelShader);
            }

            // 圧縮先に圧縮元を描画する
            {
                RenderToDestTexture(rootCommandBuffer);

                rootCommandBuffer->SetTextureStateTransition(
                    &g_DestTexture,
                    NULL /* pRange */,
                    nn::gfx::TextureState_ColorTarget,
                    nn::gfx::PipelineStageBit_RenderTarget,
                    nn::gfx::TextureState_ShaderRead,
                    nn::gfx::PipelineStageBit_PixelShader
                    );
            }

             //バックバッファをクリア・描画対象に
            rootCommandBuffer->ClearColor(pTarget, .05f, .05f, .05f, 1.0f, NULL);
            rootCommandBuffer->SetRenderTargets(1, &pTarget, NULL);

            RenderSourceTexture(rootCommandBuffer);
            RenderDestTexture(rootCommandBuffer);

        }
        g_GraphicsFramework.EndFrame(0);

        g_GraphicsFramework.ExecuteCommand(0);
        g_GraphicsFramework.QueuePresentTexture(1);
        g_GraphicsFramework.WaitGpuSync(0, nn::TimeSpan::FromSeconds(2));

        g_pTransientTextureDescriptorAllocator->End();
        g_pTransientSamplerDescriptorAllocator->End();
    }

    FinalizeGfxObjects();

    NN_LOG("Finalize DescriptorAllocatorHolder\n");
    g_TransientTextureDescriptorAllocatorHolder.Finalize();
    g_TransientSamplerDescriptorAllocatorHolder.Finalize();

    g_SamplerMinMagMipPointEdgeClamp.Finalize(g_GraphicsFramework.GetDevice());

    NN_LOG("Finalize GraphicsFramework\n");
    g_GraphicsFramework.Finalize();

    //コンテンツをアンマウントする
    nn::fs::Unmount("Contents");
    SUCCEED();
}

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    nnt::gfx::GraphicsFramework::InitializeGraphicsSystem(GraphicsSystemMemorySize);
    auto ret = RUN_ALL_TESTS();
    nnt::Exit(ret);
}
