﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <nn/nn_Common.h>
#include <nn/os.h>
#include <nn/gfx.h>
#include <nn/gfx/util/gfx_ObjectHolder.h>
#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/nnt_Argument.h>

#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
    #include <nv/nv_MemoryManagement.h>
#endif

#if NN_GFX_IS_TARGET_NVN && defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON )
    #include <nvnTool/nvnTool_GlslcInterface.h>
#endif

NN_PRAGMA_PUSH_WARNINGS
NN_DISABLE_WARNING_DEPRECATED_DECLARATIONS

namespace
{
    enum
    {
        BlendTargetCountMax = 7,        // TODO: gfx のアサート条件が修正されたら 8 に戻す
        VertexAttributeCountMax = 16,   // nvn の制約
        VertexBufferCountMax = 16,      // nvn の制約
        ViewportCountMax = 16,          // 適当
        DescriptorTableCountMax = 16,   // 適当
        DynamicDescriptorCountMax = 16, // 適当
    };

    const size_t GraphicsSystemMemorySize = 8 * 1024 * 1024;

    // Vertex shader
    const char* const g_pVsGlsl =
        "#version 430\n"
        "out gl_PerVertex\n"
        "{\n"
        "    vec4 gl_Position;\n"
        "};\n"
        "in vec4 i_Position;\n"
        "in vec2 i_TexCoord;\n"
        "out vec2 texCoord;\n"
        "void main()\n"
        "{\n"
        "    gl_Position = i_Position;\n"
        "    texCoord = i_TexCoord.xy;\n"
        "}\n";

    // Compute Shader
    const char* const g_pCsGlsl =
        "#version 430\n"
        "layout( local_size_x = 64, local_size_y = 1, local_size_z = 1 ) in;\n"
        "layout( std140 ) buffer ExpBuf\n"
        "{\n"
        "    ivec4 result[ 64 ];\n"
        "};\n"
        "void main()\n"
        "{\n"
        "    result[ gl_LocalInvocationIndex ] *= result[ gl_LocalInvocationIndex ];\n"
        "}\n";

#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
    //------------------------------------------------------------------------------
    // グラフィックスシステム用メモリ割り当て・破棄関数
    //------------------------------------------------------------------------------
    static void* NvAllocateFunction(size_t size, size_t alignment, void* userPtr)
    {
        NN_UNUSED(userPtr);
        return aligned_alloc(alignment, size);
    }
    static void NvFreeFunction(void* addr, void* userPtr)
    {
        NN_UNUSED(userPtr);
        free(addr);
    }
    static void* NvReallocateFunction(void* addr, size_t newSize, void* userPtr)
    {
        NN_UNUSED(userPtr);
        return realloc(addr, newSize);
    }
#endif
#if NN_GFX_IS_TARGET_NVN && defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON )
    //------------------------------------------------------------------------------
    // GLSLC 用メモリ割り当て・破棄関数
    //------------------------------------------------------------------------------
    static void* GlslcAllocateFunction(size_t size, size_t alignment, void* userPtr)
    {
        NN_UNUSED(userPtr);
        return aligned_alloc(alignment, size);
    }
    static void GlslcFreeFunction(void* addr, void* userPtr)
    {
        NN_UNUSED(userPtr);
        free(addr);
    }
    static void* GlslcReallocateFunction(void* addr, size_t newSize, void* userPtr)
    {
        NN_UNUSED(userPtr);
        return realloc(addr, newSize);
    }
#endif
}

class TestGfxUtilObjectHolder : public ::testing::Test
{
protected:
    TestGfxUtilObjectHolder()
    {
        nn::gfx::Device::InfoType deviceInfo;
        deviceInfo.SetDefault();
        deviceInfo.SetApiVersion( nn::gfx::ApiMajorVersion, nn::gfx::ApiMinorVersion );

        m_Device.Initialize(deviceInfo);
    }

    ~TestGfxUtilObjectHolder()
    {
        m_Device.Finalize();
    }

    nn::gfx::Device m_Device;
};

TEST_F(TestGfxUtilObjectHolder, BlendStateHolder)
{
    nn::gfx::BlendTargetStateInfo blendTargetStateInfoArray[BlendTargetCountMax];
    for(int i = 0; i < BlendTargetCountMax; ++i)
    {
        blendTargetStateInfoArray[i].SetDefault();
    }

    nn::gfx::BlendState::InfoType info;
    info.SetDefault();

    //
    // 正常系
    //

    {
        info.SetBlendTargetStateInfoArray(blendTargetStateInfoArray, 0);

        nn::gfx::util::BlendStateHolder<0> holder;
        holder.Initialize(&m_Device, info);

        nn::gfx::BlendState* object = holder.Get();
        NN_UNUSED(object);

        const nn::gfx::util::BlendStateHolder<0>* pHolder = &holder;
        const nn::gfx::BlendState* cObject = pHolder->Get();
        NN_UNUSED(cObject);

        holder.Finalize(&m_Device);
    }

    {
        info.SetBlendTargetStateInfoArray(blendTargetStateInfoArray, 1);

        nn::gfx::util::BlendStateHolder<1> holder;
        holder.Initialize(&m_Device, info);

        holder.Finalize(&m_Device);
    }

    {
        info.SetBlendTargetStateInfoArray(blendTargetStateInfoArray, BlendTargetCountMax);

        nn::gfx::util::BlendStateHolder<BlendTargetCountMax> holder;
        holder.Initialize(&m_Device, info);

        holder.Finalize(&m_Device);
    }

    //
    // 異常系
    //

    {
        info.SetBlendTargetStateInfoArray(blendTargetStateInfoArray, 1);

        nn::gfx::util::BlendStateHolder<0> holder;
        EXPECT_DEATH_IF_SUPPORTED(holder.Initialize(&m_Device, info), "");
    }
}

TEST_F(TestGfxUtilObjectHolder, VertexStateHolder)
{
    nn::gfx::VertexAttributeStateInfo vertexAttributeStateInfoArray[VertexAttributeCountMax];
    for(int i = 0; i < VertexAttributeCountMax; ++i)
    {
        vertexAttributeStateInfoArray[i].SetDefault();
        vertexAttributeStateInfoArray[i].SetFormat( nn::gfx::AttributeFormat_32_32_32_Float );
        vertexAttributeStateInfoArray[i].SetShaderSlot(i);
    }

    nn::gfx::VertexBufferStateInfo vertexBufferStateInfoArray[VertexBufferCountMax];
    for(int i = 0; i < VertexBufferCountMax; ++i)
    {
        vertexBufferStateInfoArray[i].SetDefault();
    }

    nn::gfx::VertexState::InfoType info;
    info.SetDefault();

    //
    // 正常系
    //

    {
        info.SetVertexAttributeStateInfoArray(vertexAttributeStateInfoArray, 0);
        info.SetVertexBufferStateInfoArray(vertexBufferStateInfoArray, 0);

        nn::gfx::util::VertexStateHolder<0, 0> holder;
        holder.Initialize(&m_Device, info, nullptr);

        nn::gfx::VertexState* object = holder.Get();
        NN_UNUSED(object);

        const nn::gfx::util::VertexStateHolder<0, 0>* pHolder = &holder;
        const nn::gfx::VertexState* cObject = pHolder->Get();
        NN_UNUSED(cObject);

        holder.Finalize(&m_Device);
    }

    {
        info.SetVertexAttributeStateInfoArray(vertexAttributeStateInfoArray, 1);
        info.SetVertexBufferStateInfoArray(vertexBufferStateInfoArray, 1);

        nn::gfx::util::VertexStateHolder<1, 1> holder;
        holder.Initialize(&m_Device, info, nullptr);

        holder.Finalize(&m_Device);
    }

    {
        info.SetVertexAttributeStateInfoArray(vertexAttributeStateInfoArray, VertexAttributeCountMax);
        info.SetVertexBufferStateInfoArray(vertexBufferStateInfoArray, VertexBufferCountMax);

        nn::gfx::util::VertexStateHolder<VertexAttributeCountMax, VertexBufferCountMax> holder;
        holder.Initialize(&m_Device, info, nullptr);

        holder.Finalize(&m_Device);
    }

    //
    // 異常系
    //

    {
        info.SetVertexAttributeStateInfoArray(vertexAttributeStateInfoArray, 1);
        info.SetVertexBufferStateInfoArray(vertexBufferStateInfoArray, 1);

        nn::gfx::util::VertexStateHolder<0, 0> holder;
        EXPECT_DEATH_IF_SUPPORTED(holder.Initialize(&m_Device, info, nullptr), "");
    }
}

TEST_F(TestGfxUtilObjectHolder, ViewportScissorStateHolder)
{
    nn::gfx::ViewportStateInfo viewportStateInfoArray[ViewportCountMax];
    for(int i = 0; i < ViewportCountMax; ++i)
    {
        viewportStateInfoArray[i].SetDefault();
    }

    nn::gfx::ViewportScissorState::InfoType info;
    info.SetDefault();

    //
    // 正常系
    //

    {
        info.SetViewportStateInfoArray(viewportStateInfoArray, 1);

        nn::gfx::util::ViewportScissorStateHolder<1> holder;
        holder.Initialize(&m_Device, info);

        nn::gfx::ViewportScissorState* object = holder.Get();
        NN_UNUSED(object);

        const nn::gfx::util::ViewportScissorStateHolder<1>* pHolder = &holder;
        const nn::gfx::ViewportScissorState* cObject = pHolder->Get();
        NN_UNUSED(cObject);

        holder.Finalize(&m_Device);
    }

    {
        info.SetViewportStateInfoArray(viewportStateInfoArray, 2);

        nn::gfx::util::ViewportScissorStateHolder<2> holder;
        holder.Initialize(&m_Device, info);

        holder.Finalize(&m_Device);
    }

    {
        info.SetViewportStateInfoArray(viewportStateInfoArray, ViewportCountMax);

        nn::gfx::util::ViewportScissorStateHolder<ViewportCountMax> holder;
        holder.Initialize(&m_Device, info);

        holder.Finalize(&m_Device);
    }

    //
    // 異常系
    //

    {
        info.SetViewportStateInfoArray(viewportStateInfoArray, 2);

        nn::gfx::util::ViewportScissorStateHolder<1> holder;
        EXPECT_DEATH_IF_SUPPORTED(holder.Initialize(&m_Device, info), "");
    }
}

TEST_F(TestGfxUtilObjectHolder, GraphicsPipelineHolder)
{
    nn::gfx::ShaderCode shaderCode;
    shaderCode.codeSize = static_cast< uint32_t >( strlen( g_pVsGlsl ) );
    shaderCode.pCode = g_pVsGlsl;

    nn::gfx::Shader::InfoType shaderInfo;
    shaderInfo.SetDefault();
    shaderInfo.SetShaderCodePtr( nn::gfx::ShaderStage_Vertex, &shaderCode );
    shaderInfo.SetSourceFormat( nn::gfx::ShaderSourceFormat_Glsl );
    shaderInfo.SetCodeType( nn::gfx::ShaderCodeType_Source );

    nn::gfx::BlendTargetStateInfo blendTargetStateInfoArray[BlendTargetCountMax];
    for(int i = 0; i < BlendTargetCountMax; ++i)
    {
        blendTargetStateInfoArray[i].SetDefault();
    }

    nn::gfx::VertexAttributeStateInfo vertexAttributeStateInfoArray[VertexAttributeCountMax];
    for(int i = 0; i < VertexAttributeCountMax; ++i)
    {
        vertexAttributeStateInfoArray[i].SetDefault();
        vertexAttributeStateInfoArray[i].SetFormat( nn::gfx::AttributeFormat_32_32_32_Float );
        vertexAttributeStateInfoArray[i].SetShaderSlot(i);
    }

    nn::gfx::VertexBufferStateInfo vertexBufferStateInfoArray[VertexBufferCountMax];
    for(int i = 0; i < VertexBufferCountMax; ++i)
    {
        vertexBufferStateInfoArray[i].SetDefault();
    }

    nn::gfx::Shader shader;
    nn::gfx::RasterizerState::InfoType rasterizerStateInfo;
    nn::gfx::DepthStencilState::InfoType depthStencilStateInfo;
    nn::gfx::RenderTargetStateInfo renderTargetStateInfo;
    nn::gfx::BlendState::InfoType blendStateInfo;
    nn::gfx::VertexState::InfoType vertexStateIInfo;

    shader.Initialize(&m_Device, shaderInfo);
    rasterizerStateInfo.SetDefault();
    depthStencilStateInfo.SetDefault();
    renderTargetStateInfo.SetDefault();
    blendStateInfo.SetDefault();
    vertexStateIInfo.SetDefault();

    nn::gfx::Pipeline::GraphicsInfoType info;
    info.SetDefault();
    info.SetShaderPtr(&shader);
    info.SetRasterizerStateInfo(&rasterizerStateInfo);
    info.SetDepthStencilStateInfo(&depthStencilStateInfo);
    info.SetRenderTargetStateInfo(&renderTargetStateInfo);

    //
    // 正常系
    //

    {
        blendStateInfo.SetBlendTargetStateInfoArray(blendTargetStateInfoArray, 0);

        vertexStateIInfo.SetVertexAttributeStateInfoArray(vertexAttributeStateInfoArray, 0);
        vertexStateIInfo.SetVertexBufferStateInfoArray(vertexBufferStateInfoArray, 0);

        info.SetBlendStateInfo(&blendStateInfo);
        info.SetVertexStateInfo(&vertexStateIInfo);

        nn::gfx::util::GraphicsPipelineHolder<0, 0, 0> holder;
        holder.Initialize(&m_Device, info);

        nn::gfx::Pipeline* object = holder.Get();
        NN_UNUSED(object);

        const nn::gfx::util::GraphicsPipelineHolder<0, 0, 0>* pHolder = &holder;
        const nn::gfx::Pipeline* cObject = pHolder->Get();
        NN_UNUSED(cObject);

        holder.Finalize(&m_Device);
    }

    {
        blendStateInfo.SetBlendTargetStateInfoArray(blendTargetStateInfoArray, 1);

        vertexStateIInfo.SetVertexAttributeStateInfoArray(vertexAttributeStateInfoArray, 1);
        vertexStateIInfo.SetVertexBufferStateInfoArray(vertexBufferStateInfoArray, 1);

        info.SetBlendStateInfo(&blendStateInfo);
        info.SetVertexStateInfo(&vertexStateIInfo);

        nn::gfx::util::GraphicsPipelineHolder<1, 1, 1> holder;
        holder.Initialize(&m_Device, info);

        holder.Finalize(&m_Device);
    }

    {
        blendStateInfo.SetBlendTargetStateInfoArray(blendTargetStateInfoArray, BlendTargetCountMax);

        vertexStateIInfo.SetVertexAttributeStateInfoArray(
            vertexAttributeStateInfoArray,
            VertexAttributeCountMax);
        vertexStateIInfo.SetVertexBufferStateInfoArray(
            vertexBufferStateInfoArray,
            VertexBufferCountMax);

        info.SetBlendStateInfo(&blendStateInfo);
        info.SetVertexStateInfo(&vertexStateIInfo);

        nn::gfx::util::GraphicsPipelineHolder<
            BlendTargetCountMax,
            VertexAttributeCountMax,
            VertexBufferCountMax> holder;
        holder.Initialize(&m_Device, info);

        holder.Finalize(&m_Device);
    }

    //
    // 異常系
    //

    {
        blendStateInfo.SetBlendTargetStateInfoArray(blendTargetStateInfoArray, 1);

        vertexStateIInfo.SetVertexAttributeStateInfoArray(vertexAttributeStateInfoArray, 1);
        vertexStateIInfo.SetVertexBufferStateInfoArray(vertexBufferStateInfoArray, 1);

        info.SetBlendStateInfo(&blendStateInfo);
        info.SetVertexStateInfo(&vertexStateIInfo);

        {
            nn::gfx::util::GraphicsPipelineHolder<0, 1, 1> holder;
            EXPECT_DEATH_IF_SUPPORTED(holder.Initialize(&m_Device, info), "");
        }

        {
            nn::gfx::util::GraphicsPipelineHolder<1, 0, 1> holder;
            EXPECT_DEATH_IF_SUPPORTED(holder.Initialize(&m_Device, info), "");
        }

        {
            nn::gfx::util::GraphicsPipelineHolder<1, 1, 0> holder;
            EXPECT_DEATH_IF_SUPPORTED(holder.Initialize(&m_Device, info), "");
        }
    }

    shader.Finalize(&m_Device);
} // NOLINT(impl/function_size)

TEST_F(TestGfxUtilObjectHolder, ComputePipelineHolder)
{
    nn::gfx::ShaderCode shaderCode;
    shaderCode.codeSize = static_cast< uint32_t >( strlen( g_pCsGlsl ) );
    shaderCode.pCode = g_pCsGlsl;

    nn::gfx::Shader::InfoType shaderInfo;
    shaderInfo.SetDefault();
    shaderInfo.SetShaderCodePtr( nn::gfx::ShaderStage_Compute, &shaderCode );
    shaderInfo.SetSourceFormat( nn::gfx::ShaderSourceFormat_Glsl );
    shaderInfo.SetCodeType( nn::gfx::ShaderCodeType_Source );

    nn::gfx::Shader shader;
    shader.Initialize(&m_Device, shaderInfo);

    nn::gfx::Pipeline::ComputeInfoType info;
    info.SetDefault();
    info.SetShaderPtr(&shader);

    {
        nn::gfx::util::ComputePipelineHolder holder;
        holder.Initialize(&m_Device, info);

        nn::gfx::Pipeline* object = holder.Get();
        NN_UNUSED(object);

        const nn::gfx::util::ComputePipelineHolder* pHolder = &holder;
        const nn::gfx::Pipeline* cObject = pHolder->Get();
        NN_UNUSED(cObject);

        holder.Finalize(&m_Device);
    }

    shader.Finalize(&m_Device);
}

TEST_F(TestGfxUtilObjectHolder, RootSignatureHolder)
{
    // 最大数は DescriptorTableCountMax と同じにしておく
    nn::gfx::DescriptorRangeInfo descriptorRangeInfoArray[DescriptorTableCountMax];
    for(int i = 0; i < DescriptorTableCountMax; ++i)
    {
        descriptorRangeInfoArray[i].SetDefault();
    }

    nn::gfx::DescriptorTableInfo descriptorTableInfoArray[DescriptorTableCountMax];
    for(int i = 0; i < DescriptorTableCountMax; ++i)
    {
        descriptorTableInfoArray[i].SetDefault();
        // インデックスに合わせて Range の数を変えます
        descriptorTableInfoArray[i].SetDescriptorRangeInfoArray(descriptorRangeInfoArray, i + 1);
    }

    nn::gfx::DynamicDescriptorInfo dynamicDescriptorInfoArray[DynamicDescriptorCountMax];
    for(int i = 0; i < DynamicDescriptorCountMax; ++i)
    {
        dynamicDescriptorInfoArray[i].SetDefault();
    }

    nn::gfx::RootSignature::InfoType info;
    info.SetDefault();

    //
    // 正常系
    //

    {
        info.SetDescriptorTableInfoArray(descriptorTableInfoArray, 0);
        info.SetDynamicDescriptorInfoArray(dynamicDescriptorInfoArray, 0);

        nn::gfx::util::RootSignatureHolder<0, 0, 0> holder;
        holder.Initialize(&m_Device, info);

        nn::gfx::RootSignature* object = holder.Get();
        NN_UNUSED(object);

        const nn::gfx::util::RootSignatureHolder<0, 0, 0>* pHolder = &holder;
        const nn::gfx::RootSignature* cObject = pHolder->Get();
        NN_UNUSED(cObject);

        holder.Finalize(&m_Device);
    }

    {
        info.SetDescriptorTableInfoArray(descriptorTableInfoArray, 1);
        info.SetDynamicDescriptorInfoArray(dynamicDescriptorInfoArray, 1);

        nn::gfx::util::RootSignatureHolder<1, 1, 1> holder;
        holder.Initialize(&m_Device, info);

        holder.Finalize(&m_Device);
    }

    {
        info.SetDescriptorTableInfoArray(descriptorTableInfoArray, DescriptorTableCountMax);
        info.SetDynamicDescriptorInfoArray(dynamicDescriptorInfoArray, DynamicDescriptorCountMax);

        nn::gfx::util::RootSignatureHolder<
            DescriptorTableCountMax,
            DynamicDescriptorCountMax,
            (1 + DescriptorTableCountMax) * (DescriptorTableCountMax / 2)   // (1 + 2 + ... + n) = (1 + n) * (n / 2)
        > holder;
        holder.Initialize(&m_Device, info);

        holder.Finalize(&m_Device);
    }

    //
    // 異常系
    //

    {
        info.SetDescriptorTableInfoArray(descriptorTableInfoArray, 1);
        info.SetDynamicDescriptorInfoArray(dynamicDescriptorInfoArray, 1);

        {
            nn::gfx::util::RootSignatureHolder<0, 1, 1> holder;
            EXPECT_DEATH_IF_SUPPORTED(holder.Initialize(&m_Device, info), "");
        }

        {
            nn::gfx::util::RootSignatureHolder<1, 0, 1> holder;
            EXPECT_DEATH_IF_SUPPORTED(holder.Initialize(&m_Device, info), "");
        }

        {
            nn::gfx::util::RootSignatureHolder<1, 1, 0> holder;
            EXPECT_DEATH_IF_SUPPORTED(holder.Initialize(&m_Device, info), "");
        }
    }
}

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
    // グラフィックスシステムのためのメモリ周りの初期化を行います。
    {
        nv::SetGraphicsAllocator(NvAllocateFunction, NvFreeFunction, NvReallocateFunction, NULL);
        nv::SetGraphicsDevtoolsAllocator(NvAllocateFunction, NvFreeFunction, NvReallocateFunction, NULL);
        nv::InitializeGraphics(malloc(GraphicsSystemMemorySize), GraphicsSystemMemorySize);
    }
#endif

#if NN_GFX_IS_TARGET_NVN && defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON )
    // GLSLC のためのメモリアロケータを設定します。
    glslcSetAllocator(GlslcAllocateFunction, GlslcFreeFunction, GlslcReallocateFunction, NULL);
#endif

    nn::gfx::Initialize();

    auto ret = RUN_ALL_TESTS();

    nnt::Exit(ret);
}

NN_PRAGMA_POP_WARNINGS
