﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include <nn/nn_Log.h>
#include <nn/util/util_Color.h>
#include <nn/os/os_Tick.h>
#include <nn/os/os_Random.h>

#include "testGfxUtil_Application.h"
#include "testGfxUtil_ControllerInput.h"

#include "gfxUtilGpuBenchmark_JsonStreamer.h"
#include "gfxUtilGpuBenchmark_Property.h"
#include "gfxUtilGpuBenchmark_PlatformId.h"
#include "gfxUtilGpuBenchmark_Factory.h"
#include "gfxUtilGpuBenchmark_GpuBenchmark.h"
#include "gfxUtilGpuBenchmark_ResourceAllocator.h"



namespace {

enum MenuItem
{
    MenuItem_RepeatCount,
    MenuItem_SelectBenchmark,
    MenuItem_BenchmarkPropertyFirst,
};

enum SelectionDirection
{
    SelectionDirection_Left,
    SelectionDirection_Right
};

struct BenchmarkSavedConfiguration
{
    static const int                        BENCHMARK_PROPERTY_MAX = 16;

    bool        saved;
    int64_t     targetDurationInMs;
    int         propertyValue[BENCHMARK_PROPERTY_MAX];
    int         propertyCount;
};


class InteractiveModeMenuData
{
public:

    static const int                        WarmUpCount = 4;
    static const int                        RunForDurationRetryCount = 4;
    static const nn::TimeSpan               DefaultTargetDuration;
    static const nn::TimeSpan               MinTargetDuration;
    static const nn::TimeSpan               MaxTargetDuration;

    bool                                    mustExit;
    int                                     menuIndex;
    nn::TimeSpan                            benchmarkTargetDuration;
    int                                     benchmarkRepeatCount;
    nnt::gfx::util::GpuBenchmark*           pGpuBenchmark;

    BenchmarkSavedConfiguration             benchmarkSavedConfiguration[nnt::gfx::util::BenchmarkType_Max];


    const char*                             outputFilePath;

    static const int                        DistributionBinCount = 24;

    ApplicationGpuBenchmarkResult           resultMean;
    ApplicationGpuBenchmarkResult           resultStandardDeviation;
    ApplicationGpuBenchmarkResult           distributionBin[DistributionBinCount];
    ApplicationGpuBenchmarkResult           distributionMax;
    ApplicationGpuBenchmarkResult           distributionMin;
    ApplicationGpuBenchmarkResult           distributionCount;

    ApplicationGpuBenchmarkValidation       resultValidation;


public:
    InteractiveModeMenuData()
        : mustExit(false)
        , menuIndex(MenuItem_SelectBenchmark)
        , benchmarkTargetDuration(DefaultTargetDuration)
        , benchmarkRepeatCount(0)
        , pGpuBenchmark(nullptr)
        , benchmarkSavedConfiguration()
        , resultValidation()
    {}
};

const nn::TimeSpan InteractiveModeMenuData::DefaultTargetDuration = nn::TimeSpan::FromMilliSeconds(7);
const nn::TimeSpan InteractiveModeMenuData::MinTargetDuration = nn::TimeSpan::FromMilliSeconds(0);
const nn::TimeSpan InteractiveModeMenuData::MaxTargetDuration = nn::TimeSpan::FromMilliSeconds(500);


bool isBenchmarkIgnored(int benchmarkType)
{
    NN_UNUSED(benchmarkType);
    //if (benchmarkType == BenchmarkType_Fillrate)
    //    return true;

    return false;
}

void ClearValidationData(InteractiveModeMenuData* pMenuData)
{
    memset(&pMenuData->resultMean, 0, sizeof(pMenuData->resultMean));
    memset(&pMenuData->resultStandardDeviation, 0, sizeof(pMenuData->resultStandardDeviation));
    memset(pMenuData->distributionBin, 0, sizeof(pMenuData->distributionBin));
    memset(&pMenuData->distributionMax, 0, sizeof(pMenuData->distributionMax));
    memset(&pMenuData->distributionMin, 0, sizeof(pMenuData->distributionMin));
    memset(&pMenuData->distributionCount, 0, sizeof(pMenuData->distributionCount));

    pMenuData->resultValidation.Clear();
}

void UpdateBenchmarkData(InteractiveModeMenuData* pMenuData, ApplicationTestData* pTestData)
{
    const int powerUpdateIgnoreCount = 32;

    if (pMenuData->benchmarkTargetDuration.GetNanoSeconds() == 0)
    {
        pTestData->previousRunGpuTimeElapsed = pMenuData->benchmarkTargetDuration;
        pMenuData->benchmarkRepeatCount = 1;
    }
    else
    {
        RunGpuBenchmarkForDurationWithRetry(
            &pTestData->previousRunGpuTimeElapsed, &pMenuData->benchmarkRepeatCount,
            pTestData, pMenuData->pGpuBenchmark,
            InteractiveModeMenuData::WarmUpCount, pMenuData->benchmarkTargetDuration,
            InteractiveModeMenuData::RunForDurationRetryCount);
    }

    RecordGpuBenchmarkCommandList(
        pTestData, pMenuData->pGpuBenchmark,
        InteractiveModeMenuData::WarmUpCount, pMenuData->benchmarkRepeatCount);
    ResetHistory(pTestData);
    ClearValidationData(pMenuData);

    pTestData->platformMeasurementTracker.ResetAndIgnoreNextUpdates(powerUpdateIgnoreCount);
}

void SaveBenchmarkConfiguration(InteractiveModeMenuData* pMenuData, nnt::gfx::util::GpuBenchmark* pGpuBenchmark)
{
    nnt::gfx::util::BenchmarkType benchmarkType = pGpuBenchmark->GetType();

    BenchmarkSavedConfiguration* pBenchmarkSavedConfiguration =
        &pMenuData->benchmarkSavedConfiguration[benchmarkType];

    int propertyCount = pGpuBenchmark->GetPropertyCount();
    NN_ASSERT((propertyCount + 1) <= BenchmarkSavedConfiguration::BENCHMARK_PROPERTY_MAX);

    for (int i = 0; i < propertyCount; ++i)
    {
        pBenchmarkSavedConfiguration->propertyValue[i] = pGpuBenchmark->GetPropertyByIndex(i)->Get();
    }

    pBenchmarkSavedConfiguration->propertyCount = propertyCount;
    pBenchmarkSavedConfiguration->targetDurationInMs = pMenuData->benchmarkTargetDuration.GetMilliSeconds();
    pBenchmarkSavedConfiguration->saved = true;
}

bool RestoreBenchmarkConfigurationIfExists(
    InteractiveModeMenuData* pMenuData, nnt::gfx::util::BenchmarkType benchmarkType,
    int const** ppPropertyValueArray, int* pPropertyValueArraySize)
{
    BenchmarkSavedConfiguration* pBenchmarkSavedConfiguration =
        &pMenuData->benchmarkSavedConfiguration[benchmarkType];

    if (!pBenchmarkSavedConfiguration->saved)
        return false;

    *ppPropertyValueArray = pBenchmarkSavedConfiguration->propertyValue;
    *pPropertyValueArraySize = pBenchmarkSavedConfiguration->propertyCount;

    pMenuData->benchmarkTargetDuration = nn::TimeSpan::FromMilliSeconds(pBenchmarkSavedConfiguration->targetDurationInMs);
    return true;
}


int64_t GetDurationIncrementFromValue(int64_t value)
{
    if (value < 20)
        return 1;

    if (value < 100)
        return 5;

    return 20;
}

void UpdateSelection(InteractiveModeMenuData* pMenuData, ApplicationTestData* pTestData, SelectionDirection direction)
{
    if (pMenuData->menuIndex == MenuItem_SelectBenchmark)
    {
        int previousBenchmarkIndex = pMenuData->pGpuBenchmark->GetType();
        int currentBenchmarkIndex = previousBenchmarkIndex;

        if (direction == SelectionDirection_Left)
        {
            do
            {
                if (currentBenchmarkIndex == 0)
                    currentBenchmarkIndex = nnt::gfx::util::BenchmarkType_Max;
                currentBenchmarkIndex = currentBenchmarkIndex - 1;

            } while (isBenchmarkIgnored(currentBenchmarkIndex));
        }
        else
        {
            do
            {
                currentBenchmarkIndex = currentBenchmarkIndex + 1;
                if (currentBenchmarkIndex == nnt::gfx::util::BenchmarkType_Max)
                    currentBenchmarkIndex = 0;

            } while (isBenchmarkIgnored(currentBenchmarkIndex));
        }

        pTestData->benchmarkQueue.Flush();
        pTestData->benchmarkQueue.Sync();

        SaveBenchmarkConfiguration(pMenuData, pMenuData->pGpuBenchmark);

        FinalizeBenchmark(pTestData, pMenuData->pGpuBenchmark);
        pMenuData->pGpuBenchmark = nullptr;

        nnt::gfx::util::BenchmarkType currentBenchmarkType = static_cast<nnt::gfx::util::BenchmarkType>(currentBenchmarkIndex);
        const char* testName = GetBenchmarkNameFromType(currentBenchmarkType);
        NN_LOG("New benchmark type: %s\n", testName);

        const int* propertyConfigurationArray = nullptr;
        int propertyConfigurationArraySize = 0;
        RestoreBenchmarkConfigurationIfExists(
            pMenuData, currentBenchmarkType,
            &propertyConfigurationArray, &propertyConfigurationArraySize);

        pMenuData->pGpuBenchmark = InitializeBenchmarkFromConfiguration(
            pTestData, testName,
            propertyConfigurationArray, propertyConfigurationArraySize);

        UpdateBenchmarkData(pMenuData, pTestData);
    }
    else if (pMenuData->menuIndex == MenuItem_RepeatCount)
    {
        bool durationChanged = false;

        if (direction == SelectionDirection_Left)
        {
            if (pMenuData->benchmarkTargetDuration > InteractiveModeMenuData::MinTargetDuration)
            {
                durationChanged = true;
                int64_t currentMs = pMenuData->benchmarkTargetDuration.GetMilliSeconds();
                int64_t increment = GetDurationIncrementFromValue(currentMs);
                pMenuData->benchmarkTargetDuration = nn::TimeSpan::FromMilliSeconds(currentMs - increment);
            }
        }
        else
        {

            if (pMenuData->benchmarkTargetDuration < InteractiveModeMenuData::MaxTargetDuration)
            {
                durationChanged = true;
                int64_t currentMs = pMenuData->benchmarkTargetDuration.GetMilliSeconds();
                int64_t increment = GetDurationIncrementFromValue(currentMs);
                pMenuData->benchmarkTargetDuration = nn::TimeSpan::FromMilliSeconds(currentMs + increment);
            }
        }

        if (durationChanged)
        {
            UpdateBenchmarkData(pMenuData, pTestData);
        }
    }
    else
    {
#if defined(NN_SDK_BUILD_DEBUG)
        pTestData->pResourceAllocator->PopAndCompareMemoryPoolAllocatorStatus();
#endif
        int propertyIndex = pMenuData->menuIndex - MenuItem_BenchmarkPropertyFirst;
        nnt::gfx::util::GpuBenchmark* pBenchmark = pMenuData->pGpuBenchmark;
        nnt::gfx::util::GpuBenchmarkPropertyHolder* pProperty = pBenchmark->GetPropertyByIndex(propertyIndex);
        NN_ASSERT(pProperty != nullptr);

        pBenchmark->FinalizeGfxObjects(pTestData->pResourceAllocator, pTestData->pDevice);
        if (direction == SelectionDirection_Left)
            pProperty->Decrease();
        else
            pProperty->Increase();
        pBenchmark->InitializeGfxObjects(pTestData->pResourceAllocator, pTestData->pDevice);

        UpdateBenchmarkData(pMenuData, pTestData);

#if defined(NN_SDK_BUILD_DEBUG)
        pTestData->pResourceAllocator->PushMemoryPoolAllocatorStatus();
        pTestData->pResourceAllocator->UpdateMemoryPoolAllocatorMaxUsage();
#endif

    }
}

void SetRandomConfiguration(InteractiveModeMenuData* pMenuData, ApplicationTestData* pTestData)
{
#if defined(NN_SDK_BUILD_DEBUG)
    pTestData->pResourceAllocator->PopAndCompareMemoryPoolAllocatorStatus();
#endif

    nnt::gfx::util::GpuBenchmark* pBenchmark = pMenuData->pGpuBenchmark;


    pBenchmark->FinalizeGfxObjects(pTestData->pResourceAllocator, pTestData->pDevice);


    int propertyCount = pBenchmark->GetPropertyCount();

    for (int propertyIndex = 0; propertyIndex < propertyCount; ++propertyIndex)
    {
        nnt::gfx::util::GpuBenchmarkPropertyHolder* pProperty = pBenchmark->GetPropertyByIndex(propertyIndex);

        int newValue = 0;
        int randomValue = 0;
        nn::os::GenerateRandomBytes(&randomValue, 1);

        switch (pProperty->GetType())
        {
        case nnt::gfx::util::PropertyType_Enumeration:
            {
                int count = pProperty->ToEnum()->GetCount();
                newValue = randomValue % count;
            }
            break;
        case nnt::gfx::util::PropertyType_IntegerRange:
            {
                int step = pProperty->ToIntegerRange()->GetStep();
                int min = pProperty->ToIntegerRange()->GetMin();
                int max = pProperty->ToIntegerRange()->GetMax();
                int range = max - min;
                int stepCount = (range / step) + 1;
                int stepIndex = randomValue % stepCount;
                newValue = min + (stepIndex * step);
            }
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }

        pProperty->Set(newValue);
    }

    pBenchmark->InitializeGfxObjects(pTestData->pResourceAllocator, pTestData->pDevice);

    UpdateBenchmarkData(pMenuData, pTestData);

#if defined(NN_SDK_BUILD_DEBUG)
    pTestData->pResourceAllocator->PushMemoryPoolAllocatorStatus();
    pTestData->pResourceAllocator->UpdateMemoryPoolAllocatorMaxUsage();
#endif
}

void UpdateInput(
    InteractiveModeMenuData* pMenuData,
    ApplicationTestData* pTestData,
    nnt::gfx::util::json::Document* pJsonDocument)
{
    unsigned int buttonMask = UpdateControllerInput(pTestData->pHostContext);
    int menuCount = pMenuData->pGpuBenchmark->GetPropertyCount() + MenuItem_BenchmarkPropertyFirst;

    if (buttonMask & UserInput_Exit)
    {
        pMenuData->mustExit = true;
    }

    if (buttonMask & UserInput_Up)
    {
        if (pMenuData->menuIndex == 0)
        {
            pMenuData->menuIndex = menuCount - 1;
        }
        else
        {
            pMenuData->menuIndex = pMenuData->menuIndex - 1;
        }
    }

    if (buttonMask & UserInput_Down)
    {
        pMenuData->menuIndex = pMenuData->menuIndex + 1;
        if (pMenuData->menuIndex >= menuCount)
        {
            pMenuData->menuIndex = 0;
        }
    }

    if (buttonMask & UserInput_Left)
    {
        pTestData->benchmarkQueue.Flush();
        pTestData->benchmarkQueue.Sync();

        UpdateSelection(pMenuData, pTestData, SelectionDirection_Left);
    }

    if (buttonMask & UserInput_Right)
    {
        pTestData->benchmarkQueue.Flush();
        pTestData->benchmarkQueue.Sync();

        UpdateSelection(pMenuData, pTestData, SelectionDirection_Right);
    }

    if (buttonMask & UserInput_Validate)
    {
        const int TestIdBufferSize = 128;
        char testIdBuffer[TestIdBufferSize];
        int testIdIndex = 0;
        do
        {
            MakeTestId(testIdBuffer, TestIdBufferSize, pMenuData->pGpuBenchmark);
            int platformIdLength = static_cast<int>(strlen(testIdBuffer));
            if (testIdIndex > 0)
            {
                snprintf(
                    testIdBuffer + platformIdLength, TestIdBufferSize - platformIdLength,
                    "%d", testIdIndex);
            }
            testIdIndex++;
        } while (nnt::gfx::util::json::HasTestWithId(pJsonDocument, testIdBuffer));

        CreateTestCaseData(
            pTestData, pMenuData->pGpuBenchmark,
            InteractiveModeMenuData::WarmUpCount, pMenuData->benchmarkRepeatCount,
            pJsonDocument, testIdBuffer);

        if (TestFilePathMountPointExist(pMenuData->outputFilePath))
        {
            nnt::gfx::util::json::SaveToFile(pJsonDocument, pMenuData->outputFilePath);
        }
    }

    if (buttonMask & UserInput_L)
    {
        pTestData->benchmarkQueue.Flush();
        pTestData->benchmarkQueue.Sync();

        SetRandomConfiguration(pMenuData, pTestData);

    }
}

} // anonymous namespace

void PrintDistribution(
    nn::gfx::util::DebugFontTextWriter* pDebugFontTextWriter,
    const ApplicationGpuBenchmarkResult* pDistributionCount,
    const ApplicationGpuBenchmarkResult* pDistributionBin, int distributionBinCount)
{
    const int bufferCount = 1024;
    char buffer[bufferCount];
    int bufferOffset = 0;

    for (int valueIndex = 0; valueIndex < ApplicationGpuBenchmarkResult::ValueCount; ++valueIndex)
    {
        bufferOffset = snprintf(buffer, bufferCount, "%3d: ", static_cast<int>(pDistributionCount->value[valueIndex]));

        for (int binIndex = 0; binIndex < distributionBinCount; ++binIndex)
        {
            uint64_t count = pDistributionBin[binIndex].value[valueIndex];
            bufferOffset += snprintf(buffer + bufferOffset, bufferCount - bufferOffset, "%3d ", static_cast<int>(count));
        }
        bufferOffset += snprintf(buffer + bufferOffset, bufferCount - bufferOffset, "\n");
        NN_ASSERT(bufferOffset < bufferCount);
        pDebugFontTextWriter->Print(buffer);
    }
}

void PrintPowerUsage(ApplicationTestData* pTestData)
{
    pTestData->debugFontWriter.Print("Average Power Usage Cpu:%4d Gpu:%4d Ddr:%4d VsysAp:%4d\n",
        pTestData->platformMeasurementTracker.GetAverage(PlatformMeasuringPoint_PowerCpu),
        pTestData->platformMeasurementTracker.GetAverage(PlatformMeasuringPoint_PowerGpu),
        pTestData->platformMeasurementTracker.GetAverage(PlatformMeasuringPoint_PowerDdr),
        pTestData->platformMeasurementTracker.GetAverage(PlatformMeasuringPoint_PowerVsysAp));
    pTestData->debugFontWriter.Print("Average temperature internal:%4d external:%4d\n",
        pTestData->platformMeasurementTracker.GetAverage(PlatformMeasuringPoint_ThermalSensorInternal),
        pTestData->platformMeasurementTracker.GetAverage(PlatformMeasuringPoint_ThermalSensorExternal));
}

void PrintBenchmarkInformation(ApplicationTestData* pTestData, InteractiveModeMenuData* pMenuData)
{
    char buffer[256];
    snprintf(
        buffer, sizeof(buffer),
        "Target duration:%2d ms (RepeatCount: %d)\n",
        static_cast<int>(pMenuData->benchmarkTargetDuration.GetMilliSeconds()),
        pMenuData->benchmarkRepeatCount);
    PrintWithSelectionMarker(&pTestData->debugFontWriter, pMenuData->menuIndex == MenuItem_RepeatCount, buffer);

    PrintBenchmarkInformation(
        &pTestData->debugFontWriter,
        pMenuData->pGpuBenchmark,
        pMenuData->menuIndex - MenuItem_BenchmarkPropertyFirst);
}

void PrintBenchmarkResults(ApplicationTestData* pTestData, InteractiveModeMenuData* pMenuData, nn::TimeSpan frameDuration)
{
    ApplicationGpuBenchmarkResult displayResult;
    ComputeAverageFromHistory(&displayResult, pTestData);

    pTestData->debugFontWriter.Print("\nBenchmark Results:\n");
    pMenuData->pGpuBenchmark->PrintResults(
        displayResult.GetCpuTimeElapsed(), displayResult.GetGpuTimeElapsed(),
        pMenuData->benchmarkRepeatCount, &pTestData->debugFontWriter);

    pTestData->debugFontWriter.Print("frame time: %06llu test gpu time: %06llu\n",
        frameDuration.GetMicroSeconds(),
        pTestData->previousRunGpuTimeElapsed.GetMicroSeconds());

    double cpuMissPercentage = std::numeric_limits<double>::infinity();
    double gpuMissPercentage = std::numeric_limits<double>::infinity();

    if (pMenuData->resultValidation.testCount > 0)
    {
        cpuMissPercentage = pMenuData->resultValidation.GetCpuMissPercentage();
        gpuMissPercentage = pMenuData->resultValidation.GetGpuMissPercentage();
    }

    pTestData->debugFontWriter.Print("[CPU] %d result(s) did not validate (%3.2f%%)\n",
        pMenuData->resultValidation.cpuMiss, cpuMissPercentage);
    pTestData->debugFontWriter.Print("[GPU] %d result(s) did not validate (%3.2f%%)\n",
        pMenuData->resultValidation.gpuMiss, gpuMissPercentage);

    pTestData->debugFontWriter.Print("\n\n------------------------------------\n");
    pTestData->debugFontWriter.Print("                             CPU          GPU\n");
    pTestData->debugFontWriter.Print("Average:            %12lu %12lu\n",
        pMenuData->resultMean.byName.cpuTime, pMenuData->resultMean.byName.gpuTime);
    pTestData->debugFontWriter.Print("Standard Deviation: %12lu %12lu\n",
        pMenuData->resultStandardDeviation.byName.cpuTime, pMenuData->resultStandardDeviation.byName.gpuTime);

    pTestData->debugFontWriter.Print("Distribution\n");
    PrintDistribution(&pTestData->debugFontWriter, &pMenuData->distributionCount, pMenuData->distributionBin, pMenuData->DistributionBinCount);
}

void DoInteractiveMode(
    ApplicationTestData* pTestData,
    nnt::gfx::util::BenchmarkType initialBenchmarkType,
    const char* outputFilePath)
{
    InitializeControllerInput();

    nnt::gfx::util::json::Document* pJsonDocument = nullptr;

    nnt::gfx::util::json::Create(&pJsonDocument, nullptr);

    InteractiveModeMenuData menuData;

    menuData.mustExit = false;
    menuData.menuIndex = MenuItem_SelectBenchmark;
    menuData.benchmarkTargetDuration = InteractiveModeMenuData::DefaultTargetDuration;
    menuData.outputFilePath = outputFilePath;

    const char* testName = GetBenchmarkNameFromType(initialBenchmarkType);
    menuData.pGpuBenchmark = InitializeBenchmarkFromConfiguration(pTestData, testName, nullptr, 0);

    UpdateBenchmarkData(&menuData, pTestData);

    nn::os::Tick frameStartTime = nn::os::GetSystemTick();

    int computeValidationValueHistoryCount = ApplicationTestData::BenchmarkResultHistoryLength + 16;

    while (!menuData.mustExit)
    {
        UpdateInput(&menuData, pTestData, pJsonDocument);

        RunGpuBenchmarkCommandList(pTestData);

        pTestData->debugFontWriter.SetFixedWidthEnabled(true);
        pTestData->debugFontWriter.SetFixedWidth(11);
        pTestData->debugFontWriter.SetCursor(0.0f, 0.0f);
        pTestData->debugFontWriter.SetScale(1.0f, 1.0f);
        pTestData->debugFontWriter.SetTextColor(nn::util::Color4u8::White());

        PrintPowerUsage(pTestData);
        PrintBenchmarkInformation(pTestData, &menuData);

        nn::os::Tick frameEndTime = nn::os::GetSystemTick();
        nn::TimeSpan frameDuration = (frameEndTime - frameStartTime).ToTimeSpan();
        frameStartTime = frameEndTime;

        if (pTestData->benchmarkResultHistoryCount == computeValidationValueHistoryCount)
        {
            ComputeStandardDeviationFromHistory(
                &menuData.resultMean, &menuData.resultStandardDeviation,
                pTestData);
            FillDistributionBinFromHistory(
                pTestData, menuData.distributionBin, menuData.DistributionBinCount,
                &menuData.distributionCount, &menuData.distributionMin, &menuData.distributionMax);
        }

        if (pTestData->benchmarkResultHistoryCount > computeValidationValueHistoryCount)
        {
            // ２倍の標準偏差の中に、95.4％の成功確率
            const int validationFactor = 2;
            UpdateResultValidationStandardDeviation(
                &menuData.resultValidation,
                GetPreviousResult(pTestData),
                &menuData.resultMean, &menuData.resultStandardDeviation, validationFactor);
        }

        PrintBenchmarkResults(pTestData, &menuData, frameDuration);

        DrawFrame(pTestData, menuData.pGpuBenchmark);
    }

    pTestData->benchmarkQueue.Flush();
    pTestData->benchmarkQueue.Sync();

    FinalizeBenchmark(pTestData, menuData.pGpuBenchmark);
    menuData.pGpuBenchmark = nullptr;

    nnt::gfx::util::json::Finalize(pJsonDocument);
    pJsonDocument = nullptr;

    FinalizeControllerInput();
}

