﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/fs/fs_File.h>
#include <nn/fs/fs_FileSystem.h>
#include <nn/gfx/gfx_Buffer.h>
#include <nn/gfx/gfx_MemoryPool.h>
#include <nn/gfx/gfx_Shader.h>

#include "gfxUtilGpuBenchmark_ResShader.h"
#include "gfxUtilGpuBenchmark_ResourceAllocator.h"

namespace nnt { namespace gfx { namespace util {

void InitializeResShader(
    ResShader* pResShaderType, const void* pShaderDataBuffer, size_t shaderDataBufferSize,
    ResourceAllocator* pResourceAllocator, nn::gfx::Device* pDevice)
{
    nn::gfx::MemoryPool* pMemoryPool = pResourceAllocator->GetMemoryPool(MemoryPoolType_Shader);
    nn::gfx::Buffer buffer;
    void* pShaderBinary = nullptr;
    {
        size_t alignment = reinterpret_cast<const nn::gfx::ResShaderFile*>(pShaderDataBuffer)->ToData().fileHeader.GetAlignment();
        pResShaderType->binarySize = shaderDataBufferSize;



        nn::gfx::Buffer::InfoType info;
        {
            info.SetDefault();
            info.SetGpuAccessFlags(nn::gfx::GpuAccess_Read);
            info.SetSize(pResShaderType->binarySize);
        }

        pResShaderType->binaryOffset = -1;

        if (NN_STATIC_CONDITION(nn::gfx::Buffer::IsMemoryPoolRequired))
        {
            pResShaderType->binaryOffset = pResourceAllocator->AllocatePoolMemory(
                MemoryPoolType_Shader, pResShaderType->binarySize, alignment);
            NN_ASSERT_NOT_EQUAL(pResShaderType->binaryOffset, -1);


            buffer.Initialize(pDevice, info,
                pMemoryPool,
                pResShaderType->binaryOffset, pResShaderType->binarySize);
        }
        else
        {
            buffer.Initialize(pDevice, info, nullptr, 0, 0);
        }

        pShaderBinary = buffer.Map();
        std::memcpy(pShaderBinary, pShaderDataBuffer, shaderDataBufferSize);
    }

    pResShaderType->pResShaderFile = nn::gfx::ResShaderFile::ResCast(pShaderBinary);
    pResShaderType->pResShaderContainer = pResShaderType->pResShaderFile->GetShaderContainer();
    NN_ASSERT_NOT_NULL(pResShaderType->pResShaderContainer);
    size_t containerOffset =
        pResShaderType->binaryOffset +
        nn::util::BytePtr(pResShaderType->pResShaderFile).Distance(pResShaderType->pResShaderContainer);
    pResShaderType->pResShaderContainer->Initialize(
        pDevice, pMemoryPool, containerOffset, pResShaderType->binarySize);

    nn::gfx::ResShaderVariation* pVariation = pResShaderType->pResShaderContainer->GetResShaderVariation(0);
    NN_ASSERT_NOT_NULL(pVariation);

    nn::gfx::ShaderInitializeResult shaderResult = nn::gfx::ShaderInitializeResult_Success;
    pResShaderType->codeType = nn::gfx::ShaderCodeType_Binary;
    nn::gfx::ResShaderProgram* pResShaderProgram = pVariation->GetResShaderProgram(pResShaderType->codeType);
    if (pResShaderProgram)
    {
        shaderResult = pResShaderProgram->Initialize(pDevice);
    }

    if (!pResShaderProgram || shaderResult != nn::gfx::ShaderInitializeResult_Success)
    {
        pResShaderType->codeType = nn::gfx::ShaderCodeType_Source;
        pResShaderProgram = pVariation->GetResShaderProgram(pResShaderType->codeType);
        if (pResShaderProgram)
        {
            shaderResult = pResShaderProgram->Initialize(pDevice);
        }
    }
    NN_ASSERT(pResShaderProgram && shaderResult == nn::gfx::ShaderInitializeResult_Success);

    int variationCount = pResShaderType->pResShaderContainer->GetShaderVariationCount();
    for (int idxVariation = 1; idxVariation < variationCount; idxVariation++)
    {
        pVariation = pResShaderType->pResShaderContainer->GetResShaderVariation(idxVariation);
        NN_ASSERT_NOT_NULL(pVariation);
        shaderResult = pVariation->GetResShaderProgram(pResShaderType->codeType)->Initialize(pDevice);
        NN_ASSERT_EQUAL(shaderResult, nn::gfx::ShaderInitializeResult_Success);
    }

    buffer.Unmap();
    buffer.FlushMappedRange(0, shaderDataBufferSize);
    buffer.Finalize(pDevice);

#if NN_GFX_IS_TARGET_NVN
    size_t maxScratchMemory = nn::gfx::NvnGetMaxRecommendedScratchMemorySize(pDevice, &pResShaderType->pResShaderFile, 1);
    NN_ASSERT(maxScratchMemory == 0);
#endif
}

void FinalizeResShader(
    ResShader* pResShaderType,
    ResourceAllocator* pResourceAllocator, nn::gfx::Device* pDevice)
{
    int variationCount = pResShaderType->pResShaderContainer->GetShaderVariationCount();
    for (int idxVariation = 0; idxVariation < variationCount; idxVariation++)
    {
        nn::gfx::ResShaderVariation* pVariation = pResShaderType->pResShaderContainer->GetResShaderVariation(idxVariation);
        pVariation->GetResShaderProgram(pResShaderType->codeType)->Finalize(pDevice);
    }

    pResShaderType->pResShaderContainer->Finalize(pDevice);

    if (pResShaderType->binaryOffset != -1)
    {
        pResourceAllocator->FreePoolMemory(MemoryPoolType_Shader, pResShaderType->binaryOffset);
    }

    memset(pResShaderType, 0, sizeof(ResShader));
}

} } } // namespace nnt { namespace gfx { namespace util {
