﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "gfxUtilGpuBenchmark_Property.h"
#include "gfxUtilGpuBenchmark_ResourceAllocator.h"

namespace nnt { namespace gfx { namespace util {

namespace
{

const char* g_PropertyTypeName[PropertyType_Max]
{
    "Invalid",          // PropertyType_Invalid
    "Enumeration",      // PropertyType_Enumeration
    "IntegerRange",     // PropertyType_IntegerRange
};

} // anonymous namespace

void* AllocateSetterGetterEnumContext(ResourceAllocator* pAllocator, size_t size, size_t alignment)
{
    return pAllocator->AllocateMemory(size, alignment);
}

void FreeSetterGetterEnumContext(ResourceAllocator* pAllocator, void* pMemory)
{
    pAllocator->FreeMemory(pMemory);
}


const char* GetPropertyTypeName(PropertyType type)
{
    NN_ASSERT(type < PropertyType_Max);
    return g_PropertyTypeName[type];
}

GpuBenchmarkPropertyDefinitionEnum::GpuBenchmarkPropertyDefinitionEnum(const char* name, const char** pValueNameArray, int valueNameArraySize)
{
    m_Name = name;
    m_pValueNameArray = pValueNameArray;
    m_ValueNameArraySize = valueNameArraySize;

#if defined(NN_SDK_BUILD_DEBUG)
    for (int i = 0; i < m_ValueNameArraySize; ++i)
    {
        for (int j = i + 1; j < m_ValueNameArraySize; ++j)
        {
            NN_ASSERT(strcmp(m_pValueNameArray[i], m_pValueNameArray[j]) != 0);
        }
    }
#endif
}

GpuBenchmarkPropertyDefinitionEnum::~GpuBenchmarkPropertyDefinitionEnum()
{
    m_pValueNameArray = nullptr;
    m_ValueNameArraySize = 0;
}

const char* GpuBenchmarkPropertyDefinitionEnum::GetName() const
{
    return m_Name;
}

int GpuBenchmarkPropertyDefinitionEnum::GetCount() const
{
    return m_ValueNameArraySize;
}

const char* GpuBenchmarkPropertyDefinitionEnum::GetElementNameAt(int index) const
{
    NN_ASSERT(index < GetCount());
    return m_pValueNameArray[index];
}

int GpuBenchmarkPropertyDefinitionEnum::FindElementIndexFromName(const char* lookupName) const
{
    for (int index = 0; index < GetCount(); ++index)
    {
        const char* elementName = m_pValueNameArray[index];
        if (strcmp(elementName, lookupName) == 0)
            return index;
    }
    return -1;
}


GpuBenchmarkPropertyDefinitionIntegerRange::GpuBenchmarkPropertyDefinitionIntegerRange(const char* name, int min, int max, int step)
{
    NN_ASSERT(max >= min);
    NN_ASSERT(step > 0);
    NN_ASSERT(((max - min) % step) == 0);

    m_Name  = name;
    m_Min   = min;
    m_Max   = max;
    m_Step  = step;
}

GpuBenchmarkPropertyDefinitionIntegerRange::~GpuBenchmarkPropertyDefinitionIntegerRange()
{
    m_Name  = nullptr;
    m_Min   = 0;
    m_Max   = 0;
    m_Step  = 0;
}

const char* GpuBenchmarkPropertyDefinitionIntegerRange::GetName() const
{
    return m_Name;
}


int GpuBenchmarkPropertyDefinitionIntegerRange::GetMin() const
{
    return m_Min;
}

int GpuBenchmarkPropertyDefinitionIntegerRange::GetMax() const
{
    return m_Max;
}

int GpuBenchmarkPropertyDefinitionIntegerRange::GetStep() const
{
    return m_Step;
}

bool GpuBenchmarkPropertyDefinitionIntegerRange::IsValidValue(int value) const
{
    if (value < m_Min)
        return false;
    if (value > m_Max)
        return false;

    int offset = value - m_Min;
    if ((offset % m_Step) != 0)
        return false;

    return true;
}


GpuBenchmarkPropertyHolder::GpuBenchmarkPropertyHolder()
: pContext(nullptr)
, pGetterFunc(nullptr)
, pSetterFunc(nullptr)
, type(PropertyType_Invalid)
{
}

GpuBenchmarkPropertyHolder::~GpuBenchmarkPropertyHolder()
{
    pContext = nullptr;
    pGetterFunc = nullptr;
    pSetterFunc = nullptr;
    type = PropertyType_Invalid;
}

const char* GpuBenchmarkPropertyHolder::GetName() const
{
    PropertyType propertyType = GetType();

    switch (propertyType)
    {
    case PropertyType_Enumeration:
        return ToEnum()->GetName();

    case PropertyType_IntegerRange:
        return ToIntegerRange()->GetName();

    case PropertyType_Invalid:
    default:
        NN_UNEXPECTED_DEFAULT;
        break;
    }
}

void GpuBenchmarkPropertyHolder::Increase()
{
    PropertyType propertyType = GetType();

    switch (propertyType)
    {
    case PropertyType_Enumeration:
        {
            int count = ToEnum()->GetCount();
            int nextIndex = Get() + 1;
            NN_ASSERT(nextIndex > 0);
            if (nextIndex == count)
                nextIndex = 0;
            Set(nextIndex);
        }
        break;

    case PropertyType_IntegerRange:
        {
            int value = Get();
            if (value >= ToIntegerRange()->GetMax())
            {
                Set(ToIntegerRange()->GetMin());
            }
            else
            {
                Set(value + ToIntegerRange()->GetStep());
            }
        }
        break;

    case PropertyType_Invalid:
    default:
        NN_UNEXPECTED_DEFAULT;
        break;
    }
}

void GpuBenchmarkPropertyHolder::Decrease()
{
    PropertyType propertyType = GetType();

    switch (propertyType)
    {
    case PropertyType_Enumeration:
        {
            int count = ToEnum()->GetCount();
            int nextIndex = Get() - 1;
            if (nextIndex < 0)
                nextIndex = static_cast<int>(count - 1);
            Set(nextIndex);
        }
        break;

    case PropertyType_IntegerRange:
        {
            int value = Get();
            if (value <= ToIntegerRange()->GetMin())
            {
                Set(ToIntegerRange()->GetMax());
            }
            else
            {
                Set(value - ToIntegerRange()->GetStep());
            }
        }
        break;

    case PropertyType_Invalid:
    default:
        NN_UNEXPECTED_DEFAULT;
        break;
    }
}

} } } // namespace nnt { namespace gfx { namespace util {
