﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/gfx/gfx_GpuAddress.h>
#include <nn/gfx/util/gfx_DebugFontTextWriter.h>

#include "gfxUtilGpuBenchmark_GpuBenchmark_Template.h"

#include "gfxUtilGpuBenchmark_ResHelpers.h"
#include "gfxUtilGpuBenchmark_PropertyMacros.h"

namespace nnt { namespace gfx { namespace util {

const char* GpuBenchmarkTemplate::ClassName = "TemplateName";

GpuBenchmarkTemplate::GpuBenchmarkTemplate()
{
}

GpuBenchmarkTemplate::~GpuBenchmarkTemplate()
{
}

void GpuBenchmarkTemplate::Initialize(ResourceAllocator* pResourceAllocator)
{
    NN_UNUSED(pResourceAllocator);

    // Dummy0
    GpuBenchmarkPropertyHolder* pPropertyDummy0  = m_PropertyArray.Get(Property_Dummy0);
    BENCHMARK_PROPERTY_INTEGER_RANGE_DEFINITION(
        pPropertyDummy0, "Dummy0",
        GpuBenchmarkTemplate::GetDummy0,
        GpuBenchmarkTemplate::SetDummy0,
        0, 10, 1);

    // Dummy1
    GpuBenchmarkPropertyHolder* pPropertyDummy1 = m_PropertyArray.Get(Property_Dummy1);
    BENCHMARK_PROPERTY_ENUM_DEFINITION(
        pPropertyDummy1, "Dummy1",
        Dummy1Value, pResourceAllocator,
        GpuBenchmarkTemplate::GetDummy1,
        GpuBenchmarkTemplate::SetDummy1,
        "Dummy1Value_0", Dummy1Value_0,
        "Dummy1Value_1", Dummy1Value_1);

    NN_ABORT();
}

void GpuBenchmarkTemplate::Finalize(ResourceAllocator* pResourceAllocator)
{
    NN_UNUSED(pResourceAllocator);

    for (int i = 0; i < m_PropertyArray.GetCount(); ++i)
    {
        m_PropertyArray.Get(i)->Finalize();
    }

    NN_ABORT();
}

void GpuBenchmarkTemplate::InitializeGfxObjects(ResourceAllocator* pResourceAllocator, nn::gfx::Device* pDevice)
{
    GpuBenchmark::InitializeGfxObjects(pResourceAllocator, pDevice);

    m_OutputCopyBufferSize = InitializeColorRenderTarget(
        &m_RenderTexture, &m_OutputCopyBuffer,
        &m_RenderTextureColorTargetView, &m_ViewportScissorState,
        m_RenderSize, m_RenderSize, m_RenderFormat, m_TileMode,
        pResourceAllocator, pDevice);

    InitializeResShader(&m_ResShader, NULL, 0, pResourceAllocator, pDevice);

    InitializeFullScreenQuadVertexBuffer(&m_VertexBuffer, pResourceAllocator, pDevice);
    NN_ABORT();
}

void GpuBenchmarkTemplate::FinalizeGfxObjects(ResourceAllocator* pResourceAllocator, nn::gfx::Device* pDevice)
{
    FinalizeBuffer(&m_VertexBuffer, pResourceAllocator, pDevice);
    FinalizeResShader(&m_ResShader, pResourceAllocator, pDevice);
    FinalizeColorRenderTarget(
        &m_RenderTexture, &m_OutputCopyBuffer,
        &m_RenderTextureColorTargetView, &m_ViewportScissorState,
        pResourceAllocator, pDevice);
    NN_ABORT();

    GpuBenchmark::FinalizeGfxObjects(pResourceAllocator, pDevice);
}

void GpuBenchmarkTemplate::PreBenchmark(nn::gfx::CommandBuffer* pTestCommandBuffer)
{
    nn::gfx::ColorTargetView* pTestTarget = &m_RenderTextureColorTargetView;

    pTestCommandBuffer->ClearColor(pTestTarget, 0.5f, 0.5f, 0.5f, 0.5f, nullptr);

    pTestCommandBuffer->SetRenderTargets(1, &pTestTarget, nullptr);
    pTestCommandBuffer->SetViewportScissorState(&m_ViewportScissorState);

    const int shaderVariationIndex = 0;
    nn::gfx::Shader* pShader = m_ResShader.pResShaderContainer->GetResShaderVariation(shaderVariationIndex)->GetResShaderProgram(m_ResShader.codeType)->GetShader();
    pTestCommandBuffer->SetShader(pShader, nn::gfx::ShaderStageBit_All);

    nn::gfx::GpuAddress vertexBufferGpuAddress;
    m_VertexBuffer.GetGpuAddress(&vertexBufferGpuAddress);
    pTestCommandBuffer->SetVertexBuffer(0, vertexBufferGpuAddress, sizeof(DefaultVertex), g_RectangleVertexBufferDataSize);

    NN_ABORT();
}

void GpuBenchmarkTemplate::DoBenchmark(nn::gfx::CommandBuffer* pTestCommandBuffer, int runCount)
{
    NN_UNUSED(pTestCommandBuffer);
    NN_UNUSED(runCount);
    NN_ABORT();
}


void GpuBenchmarkTemplate::PrintResults(nn::TimeSpan cpuTimeElapsed, nn::TimeSpan gpuTimeElapsed, int runCount, nn::gfx::util::DebugFontTextWriter* pDebugFontTextWriter)
{
    NN_ASSERT(runCount > 0);

    uint64_t gpuTimeElapsedValueInNs = static_cast<uint64_t>(gpuTimeElapsed.GetNanoSeconds());
    uint64_t cpuTimeElapsedValueInNs = static_cast<uint64_t>(cpuTimeElapsed.GetNanoSeconds());

    uint64_t gpuTimeElapsedAvgValueInNs = gpuTimeElapsedValueInNs / static_cast<uint64_t>(runCount);
    uint64_t cpuTimeElapsedAvgValueInNs = cpuTimeElapsedValueInNs / static_cast<uint64_t>(runCount);

    pDebugFontTextWriter->Print("gpu time (ns): %8lu\n", gpuTimeElapsedAvgValueInNs);
    pDebugFontTextWriter->Print("cpu time (ns): %8lu\n", cpuTimeElapsedAvgValueInNs);
    pDebugFontTextWriter->Print("total gpu time (ns): %12lu\n", gpuTimeElapsedValueInNs);
    pDebugFontTextWriter->Print("total cpu time (ns): %12lu\n", cpuTimeElapsedValueInNs);
    NN_ABORT();
}

void GpuBenchmarkTemplate::CopyResultToBuffer(nn::gfx::CommandBuffer* pCommandBuffer)
{
    int renderSize = m_RenderSize;

    nn::gfx::BufferTextureCopyRegion bufferTextureCopyRegion;
    bufferTextureCopyRegion.SetDefault();
    bufferTextureCopyRegion.SetBufferImageHeight(renderSize);
    bufferTextureCopyRegion.SetBufferImageWidth(renderSize);
    bufferTextureCopyRegion.EditTextureCopyRegion().SetDefault();
    bufferTextureCopyRegion.EditTextureCopyRegion().SetWidth(renderSize);
    bufferTextureCopyRegion.EditTextureCopyRegion().SetHeight(renderSize);
    bufferTextureCopyRegion.EditTextureCopyRegion().EditSubresource().SetDefault();

    pCommandBuffer->InvalidateMemory(nn::gfx::GpuAccess_Texture);
    pCommandBuffer->CopyImageToBuffer(&m_OutputCopyBuffer, &m_RenderTexture, bufferTextureCopyRegion);
    pCommandBuffer->InvalidateMemory(nn::gfx::GpuAccess_Write);
}

void GpuBenchmarkTemplate::MapResultBuffer(void** pOutBuffer, size_t* pOutBufferSize)
{
    *pOutBuffer = m_OutputCopyBuffer.Map();
    *pOutBufferSize = m_OutputCopyBufferSize;
}

void GpuBenchmarkTemplate::UnmapResultBuffer()
{
    m_OutputCopyBuffer.Unmap();
}

const char* GpuBenchmarkTemplate::GetName() const
{
    NN_ABORT();
    return ClassName;
}

BenchmarkType GpuBenchmarkTemplate::GetType() const
{
    NN_ABORT();
    return BenchmarkType_Invalid;
}

int GpuBenchmarkTemplate::GetPropertyCount() const
{
    return m_PropertyArray.GetCount();
}

int GpuBenchmarkTemplate::FillPropertyList(const GpuBenchmarkPropertyHolder** ppDestinationArray, int destinationArrayMaxSize) const
{
    return m_PropertyArray.FillPropertyList(ppDestinationArray, destinationArrayMaxSize);
}

int GpuBenchmarkTemplate::FillPropertyList(GpuBenchmarkPropertyHolder** ppDestinationArray, int destinationArrayMaxSize)
{
    return m_PropertyArray.FillPropertyList(ppDestinationArray, destinationArrayMaxSize);
}

GpuBenchmarkPropertyHolder* GpuBenchmarkTemplate::FindPropertyByName(const char* propertyName)
{
    return m_PropertyArray.FindPropertyByName(propertyName);
}

GpuBenchmarkPropertyHolder* GpuBenchmarkTemplate::GetPropertyByIndex(int index)
{
    return m_PropertyArray.Get(index);
}

} } } // namespace nnt { namespace gfx { namespace util {
