﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "nn/gfx/gfx_GpuAddress.h"
#include "nn/gfx/util/gfx_DebugFontTextWriter.h"

#include "gfxUtilGpuBenchmark_GpuBenchmarkBlending.h"

#include "gfxUtilGpuBenchmark_ResourceAllocator.h"
#include "gfxUtilGpuBenchmark_ResHelpers.h"
#include "gfxUtilGpuBenchmark_PropertyMacros.h"
#include "gfxUtilGpuBenchmark_ComputeBlendingShaderVariationIndex.h"

#include "gfxUtilGpuBenchmark_BuiltinBlendingShader.h"

#if defined(NN_GFXUTIL_GPUBENCHMARK_BLENDING_DEBUG)
#include "gfxUtilGpuBenchmark_ComputeRenderQuadShaderVariationIndex.h"
#include "gfxUtilGpuBenchmark_BuiltinRenderQuadShader.h"
#endif

#if 0

#define BLEND_SOURCE_FACTOR_LIST                                                                        \
        "Zero",                       nn::gfx::BlendFactor_Zero,                                        \
        "One",                        nn::gfx::BlendFactor_One,                                         \
        "SourceColor",                nn::gfx::BlendFactor_SourceColor,                                 \
        "OneMinusSourceColor",        nn::gfx::BlendFactor_OneMinusSourceColor,                         \
        "DestinationColor",           nn::gfx::BlendFactor_DestinationColor,                            \
        "OneMinusDestinationColor",   nn::gfx::BlendFactor_OneMinusDestinationColor,                    \
        "SourceAlpha",                nn::gfx::BlendFactor_SourceAlpha,                                 \
        "OneMinusSourceAlpha",        nn::gfx::BlendFactor_OneMinusSourceAlpha,                         \
        "DestinationAlpha",           nn::gfx::BlendFactor_DestinationAlpha,                            \
        "OneMinusDestinationAlpha",   nn::gfx::BlendFactor_OneMinusDestinationAlpha,                    \
        "ConstantColor",              nn::gfx::BlendFactor_ConstantColor,                               \
        "OneMinusConstantColor",      nn::gfx::BlendFactor_OneMinusConstantColor,                       \
        "ConstantAlpha",              nn::gfx::BlendFactor_ConstantAlpha,                               \
        "OneMinusConstantAlpha",      nn::gfx::BlendFactor_OneMinusConstantAlpha,                       \
        "SourceAlphaSaturate",        nn::gfx::BlendFactor_SourceAlphaSaturate,                         \
        "Source1Color",               nn::gfx::BlendFactor_Source1Color,                                \
        "OneMinusSource1Color",       nn::gfx::BlendFactor_OneMinusSource1Color,                        \
        "Source1Alpha",               nn::gfx::BlendFactor_Source1Alpha,                                \
        "OneMinusSource1Alpha",       nn::gfx::BlendFactor_OneMinusSource1Alpha                         \


#define BLEND_DESTINATION_FACTOR_LIST BLEND_SOURCE_FACTOR_LIST

#endif

#if 0

#define BLEND_DESTINATION_FACTOR_LIST                                                                   \
        "OneMinusDestinationColor",   nn::gfx::BlendFactor_OneMinusDestinationColor,                    \
        "DestinationAlpha",           nn::gfx::BlendFactor_DestinationAlpha                             \

#define BLEND_SOURCE_FACTOR_LIST BLEND_DESTINATION_FACTOR_LIST

#endif

#if 1

#define BLEND_SOURCE_FACTOR_LIST                                                                        \
        "Zero",                       nn::gfx::BlendFactor_Zero,                                        \
        "SourceColor",                nn::gfx::BlendFactor_SourceColor,                                 \
        "DestinationColor",           nn::gfx::BlendFactor_DestinationColor,                            \
        "SourceAlpha",                nn::gfx::BlendFactor_SourceAlpha,                                 \
        "DestinationAlpha",           nn::gfx::BlendFactor_DestinationAlpha,                            \
        "ConstantColor",              nn::gfx::BlendFactor_ConstantColor,                               \
        "ConstantAlpha",              nn::gfx::BlendFactor_ConstantAlpha,                               \
        "SourceAlphaSaturate",        nn::gfx::BlendFactor_SourceAlphaSaturate,                         \
        "OneMinusSource1Color",       nn::gfx::BlendFactor_OneMinusSource1Color,                        \
        "OneMinusSource1Alpha",       nn::gfx::BlendFactor_OneMinusSource1Alpha                         \

#define BLEND_DESTINATION_FACTOR_LIST                                                                   \
        "One",                        nn::gfx::BlendFactor_One,                                         \
        "OneMinusSourceColor",        nn::gfx::BlendFactor_OneMinusSourceColor,                         \
        "OneMinusDestinationColor",   nn::gfx::BlendFactor_OneMinusDestinationColor,                    \
        "OneMinusSourceAlpha",        nn::gfx::BlendFactor_OneMinusSourceAlpha,                         \
        "OneMinusDestinationAlpha",   nn::gfx::BlendFactor_OneMinusDestinationAlpha,                    \
        "OneMinusConstantColor",      nn::gfx::BlendFactor_OneMinusConstantColor,                       \
        "OneMinusConstantAlpha",      nn::gfx::BlendFactor_OneMinusConstantAlpha,                       \
        "Source1Color",               nn::gfx::BlendFactor_Source1Color,                                \
        "Source1Alpha",               nn::gfx::BlendFactor_Source1Alpha                                 \

#endif


namespace nnt { namespace gfx { namespace util {

const char* GpuBenchmarkBlending::ClassName = "Blending";

GpuBenchmarkBlending::GpuBenchmarkBlending()
: m_RenderFormat(nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm)
, m_BlendFunction(nn::gfx::BlendFunction_Add)
, m_BlendDestinationFactor(nn::gfx::BlendFactor_OneMinusSourceAlpha)
, m_BlendSourceFactor(nn::gfx::BlendFactor_SourceAlpha)
, m_RenderTexture()
, m_RenderTextureColorTargetView()
, m_ViewportScissorState()
, m_BlendState()
, m_ResShader()
, m_VertexBuffer()
#if defined(NN_GFXUTIL_GPUBENCHMARK_BLENDING_DEBUG)
, m_RenderQuadResShader()
, m_RenderTextureView()
, m_RenderTextureViewDescriptorSlotIndex(-1)
, m_RenderTextureViewDescriptorSlot()
#endif
{
}

GpuBenchmarkBlending::~GpuBenchmarkBlending()
{
}

void GpuBenchmarkBlending::Initialize(ResourceAllocator* pResourceAllocator)
{
    // RenderFormat
    GpuBenchmarkPropertyHolder* pPropertyRenderFormat = m_PropertyArray.Get(Property_RenderFormat);
    BENCHMARK_PROPERTY_ENUM_DEFINITION(
        pPropertyRenderFormat, "RenderFormat",
        nn::gfx::ImageFormat, pResourceAllocator,
        GpuBenchmarkBlending::GetRenderFormat,
        GpuBenchmarkBlending::SetRenderFormat,
        "ImageFormat_R8_G8_B8_A8_Unorm", nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm,
        "ImageFormat_R16_G16_B16_A16_Float", nn::gfx::ImageFormat_R16_G16_B16_A16_Float);

    // BlendFunction
    GpuBenchmarkPropertyHolder* pPropertyBlendFunction = m_PropertyArray.Get(Property_BlendFunction);
    BENCHMARK_PROPERTY_ENUM_DEFINITION(
        pPropertyBlendFunction, "BlendFunction",
        nn::gfx::BlendFunction, pResourceAllocator,
        GpuBenchmarkBlending::GetBlendFunction,
        GpuBenchmarkBlending::SetBlendFunction,
        "Disabled",         nn::gfx::BlendFunction_End,
        "Add",              nn::gfx::BlendFunction_Add,
        "Subtract",         nn::gfx::BlendFunction_Subtract,
        "ReverseSubtract",  nn::gfx::BlendFunction_ReverseSubtract,
        "Min",              nn::gfx::BlendFunction_Min,
        "Max",              nn::gfx::BlendFunction_Max);

    // BlendDestinationFactor
    GpuBenchmarkPropertyHolder* pPropertyBlendDestinationFactor = m_PropertyArray.Get(Property_BlendDestinationFactor);
    BENCHMARK_PROPERTY_ENUM_DEFINITION(
        pPropertyBlendDestinationFactor, "BlendDestinationFactor",
        nn::gfx::BlendFactor, pResourceAllocator,
        GpuBenchmarkBlending::GetBlendDestinationFactor,
        GpuBenchmarkBlending::SetBlendDestinationFactor,
        BLEND_DESTINATION_FACTOR_LIST);

    // BlendSourceFactor
    GpuBenchmarkPropertyHolder* pPropertyBlendSourceFactor = m_PropertyArray.Get(Property_BlendSourceFactor);
    BENCHMARK_PROPERTY_ENUM_DEFINITION(
        pPropertyBlendSourceFactor, "BlendSourceFactor",
        nn::gfx::BlendFactor, pResourceAllocator,
        GpuBenchmarkBlending::GetBlendSourceFactor,
        GpuBenchmarkBlending::SetBlendSourceFactor,
        BLEND_SOURCE_FACTOR_LIST);

    m_DrawIndex = 0;

    SetBlendFunction(nn::gfx::BlendFunction_Add);
    SetBlendDestinationFactor(nn::gfx::BlendFactor_One);
    SetBlendSourceFactor(nn::gfx::BlendFactor_SourceAlpha);
}

void GpuBenchmarkBlending::Finalize(ResourceAllocator* pResourceAllocator)
{
    NN_UNUSED(pResourceAllocator);

    for (int i = 0; i < m_PropertyArray.GetCount(); ++i)
    {
        m_PropertyArray.Get(i)->Finalize();
    }
}

void GpuBenchmarkBlending::InitializeGfxObjects(ResourceAllocator* pResourceAllocator, nn::gfx::Device* pDevice)
{
    GpuBenchmark::InitializeGfxObjects(pResourceAllocator, pDevice);

    static bool disableCompression = true;

    if (disableCompression)
    {
        m_OutputCopyBufferSize = InitializeColorRenderTargetDisableFrameBufferCompression(
            &m_RenderTexture, &m_OutputCopyBuffer,
            &m_RenderTextureColorTargetView, &m_ViewportScissorState,
            m_RenderSize, m_RenderSize, m_RenderFormat, m_TileMode,
            pResourceAllocator, pDevice);
    }
    else
    {
        m_OutputCopyBufferSize = InitializeColorRenderTarget(
            &m_RenderTexture, &m_OutputCopyBuffer,
            &m_RenderTextureColorTargetView, &m_ViewportScissorState,
            m_RenderSize, m_RenderSize, m_RenderFormat, m_TileMode,
            pResourceAllocator, pDevice);
    }

    bool blendEnabled = (m_BlendFunction != nn::gfx::BlendFunction_End);
    nn::gfx::BlendFunction blendFunction = (m_BlendFunction == nn::gfx::BlendFunction_End) ? nn::gfx::BlendFunction_Add : m_BlendFunction;

    nn::gfx::BlendTargetStateInfo targetInfo;
    targetInfo.SetDefault();
    targetInfo.SetChannelMask(nn::gfx::ChannelMask_Red | nn::gfx::ChannelMask_Green | nn::gfx::ChannelMask_Blue | nn::gfx::ChannelMask_Alpha);
    targetInfo.SetBlendEnabled(blendEnabled);
    targetInfo.SetColorBlendFunction(blendFunction);
    targetInfo.SetDestinationColorBlendFactor(m_BlendDestinationFactor);
    targetInfo.SetSourceColorBlendFactor(m_BlendSourceFactor);

    nn::gfx::BlendState::InfoType blendInfo;
    blendInfo.SetDefault();
    blendInfo.SetBlendTargetStateInfoArray(&targetInfo, 1);
    blendInfo.SetBlendConstant(0.5f, 0.5f, 0.5f, 0.5f);
    InitializeBlendState(&m_BlendState, blendInfo, pResourceAllocator, pDevice);

    InitializeResShader(
        &m_ResShader, g_BlendingShaderData, sizeof(g_BlendingShaderData),
        pResourceAllocator, pDevice);

    InitializeFullScreenQuadVertexBuffer(&m_VertexBuffer, pResourceAllocator, pDevice);

#if defined(NN_GFXUTIL_GPUBENCHMARK_BLENDING_DEBUG)
    InitializeResShader(
        &m_RenderQuadResShader, g_RenderQuadShaderData, sizeof(g_RenderQuadShaderData),
        pResourceAllocator, pDevice);

    nn::gfx::TextureView::InfoType renderTextureViewInfo;
    {
        renderTextureViewInfo.SetDefault();
        renderTextureViewInfo.SetTexturePtr(&m_RenderTexture);
        renderTextureViewInfo.SetImageDimension(nn::gfx::ImageDimension_2d);
        renderTextureViewInfo.SetImageFormat(m_RenderFormat);
        renderTextureViewInfo.EditSubresourceRange().EditMipRange().SetMipCount(1);
    }
    m_RenderTextureView.Initialize(pDevice, renderTextureViewInfo);

    m_RenderTextureViewDescriptorSlotIndex = pResourceAllocator->AllocateAndSetTextureViewToDescriptorPool(
            &m_RenderTextureView, &m_RenderTextureViewDescriptorSlot);

    nn::gfx::Sampler::InfoType sourceTextureSamplerInfo;
    sourceTextureSamplerInfo.SetDefault();
    sourceTextureSamplerInfo.SetFilterMode(nn::gfx::FilterMode_MinPoint_MagPoint_MipPoint);
    sourceTextureSamplerInfo.SetAddressU(nn::gfx::TextureAddressMode_Repeat);
    sourceTextureSamplerInfo.SetAddressV(nn::gfx::TextureAddressMode_Repeat);
    sourceTextureSamplerInfo.SetAddressW(nn::gfx::TextureAddressMode_Repeat);

    m_DebugTextureSampler.Initialize(pDevice, sourceTextureSamplerInfo);
    m_DebugTextureSamplerSlotIndex = pResourceAllocator->AllocateAndSetSamplerToDescriptorPool(
        &m_DebugTextureSampler, &m_DebugTextureSamplerDescriptorSlot);
#endif
}

void GpuBenchmarkBlending::FinalizeGfxObjects(ResourceAllocator* pResourceAllocator, nn::gfx::Device* pDevice)
{
#if defined(NN_GFXUTIL_GPUBENCHMARK_BLENDING_DEBUG)
    pResourceAllocator->FreeDescriptorSlots(nn::gfx::DescriptorPoolType_Sampler, m_DebugTextureSamplerSlotIndex);
    m_DebugTextureSamplerSlotIndex = -1;
    m_DebugTextureSampler.Finalize(pDevice);


    pResourceAllocator->FreeDescriptorSlots(nn::gfx::DescriptorPoolType_TextureView, m_RenderTextureViewDescriptorSlotIndex);
    m_RenderTextureViewDescriptorSlotIndex = -1;
    m_RenderTextureView.Finalize(pDevice);

    FinalizeResShader(&m_RenderQuadResShader, pResourceAllocator, pDevice);
#endif

    FinalizeBuffer(&m_VertexBuffer, pResourceAllocator, pDevice);
    FinalizeResShader(&m_ResShader, pResourceAllocator, pDevice);
    FinalizeBlendState(&m_BlendState, pResourceAllocator, pDevice);
    FinalizeColorRenderTarget(
        &m_RenderTexture, &m_OutputCopyBuffer,
        &m_RenderTextureColorTargetView, &m_ViewportScissorState,
        pResourceAllocator, pDevice);

    GpuBenchmark::FinalizeGfxObjects(pResourceAllocator, pDevice);
}

void GpuBenchmarkBlending::PreBenchmark(nn::gfx::CommandBuffer* pTestCommandBuffer)
{
    nn::gfx::ColorTargetView* pTestTarget = &m_RenderTextureColorTargetView;
    pTestCommandBuffer->ClearColor(pTestTarget, 0.5f, 0.5f, 0.5f, 1.0f, nullptr);
    pTestCommandBuffer->SetRenderTargets(1, &pTestTarget, nullptr);
    pTestCommandBuffer->SetViewportScissorState(&m_ViewportScissorState);

    pTestCommandBuffer->SetBlendState(&m_BlendState);

    int shaderVariationIndex = ComputeBlendingShaderVariationIndex(0);
    nn::gfx::ResShaderVariation* pResShaderVariation =
        m_ResShader.pResShaderContainer->GetResShaderVariation(shaderVariationIndex);
    nn::gfx::Shader* pShaderTest = pResShaderVariation->GetResShaderProgram(m_ResShader.codeType)->GetShader();
    pTestCommandBuffer->SetShader(pShaderTest, nn::gfx::ShaderStageBit_All);

    nn::gfx::GpuAddress vertexBufferGpuAddress;
    m_VertexBuffer.GetGpuAddress(&vertexBufferGpuAddress);
    pTestCommandBuffer->SetVertexBuffer(
        0, vertexBufferGpuAddress,
        sizeof(DefaultVertex), g_RectangleVertexBufferDataSize);

    m_DrawIndex = 0;
}

void GpuBenchmarkBlending::DoBenchmark(nn::gfx::CommandBuffer* pTestCommandBuffer, int runCount)
{
    NN_ASSERT(runCount > 0);

    for (int i = 0; i < runCount; ++i)
    {
        int colorIntensity = (m_DrawIndex & 0x01) == 0 ? 15 : 0;

        pTestCommandBuffer->Draw(
            nn::gfx::PrimitiveTopology_TriangleStrip, 4,
            (colorIntensity << 3));

        m_DrawIndex++;
    }
}

void GpuBenchmarkBlending::RenderDebug(nn::gfx::CommandBuffer* pTestCommandBuffer)
{
#if defined(NN_GFXUTIL_GPUBENCHMARK_BLENDING_DEBUG)
    int shaderVariationIndex = ComputeRenderQuadShaderVariationIndex(1, 1); // instance + blit texture
    nn::gfx::Shader* pShader = m_RenderQuadResShader.pResShaderContainer->GetResShaderVariation(shaderVariationIndex)->GetResShaderProgram(m_RenderQuadResShader.codeType)->GetShader();
    pTestCommandBuffer->SetShader(pShader, nn::gfx::ShaderStageBit_All);

    pTestCommandBuffer->SetTextureAndSampler(
        0, nn::gfx::ShaderStage_Pixel,
        m_RenderTextureViewDescriptorSlot,
        m_DebugTextureSamplerDescriptorSlot);

    int scale = 3;
    int offsetX = 1;
    int offsetY = 2;

    pTestCommandBuffer->Draw(
        nn::gfx::PrimitiveTopology_TriangleStrip, 4,
        (scale << 8) | (offsetX << 16) | (offsetY << 24), 1, 0);
#else
    NN_UNUSED(pTestCommandBuffer);
#endif
}


void GpuBenchmarkBlending::PrintResults(nn::TimeSpan cpuTimeElapsed, nn::TimeSpan gpuTimeElapsed, int runCount, nn::gfx::util::DebugFontTextWriter* pDebugFontTextWriter)
{
    NN_ASSERT(runCount > 0);

    uint64_t pixelCount = static_cast<uint64_t>(runCount) * static_cast<uint64_t>(m_RenderSize * m_RenderSize);

    uint64_t gpuTimeElapsedValueInNs = static_cast<uint64_t>(gpuTimeElapsed.GetNanoSeconds());
    uint64_t cpuTimeElapsedValueInNs = static_cast<uint64_t>(cpuTimeElapsed.GetNanoSeconds());

    uint64_t gpuTimeElapsedAvgValueInNs = gpuTimeElapsedValueInNs / static_cast<uint64_t>(runCount);
    uint64_t cpuTimeElapsedAvgValueInNs = cpuTimeElapsedValueInNs / static_cast<uint64_t>(runCount);

    uint64_t gpuFillrateAvgPixPerSec = 0;
    if (gpuTimeElapsed.GetNanoSeconds() > 0)
    {
        gpuFillrateAvgPixPerSec = (pixelCount * 1000000000LLU) / gpuTimeElapsedValueInNs;
    }

    pDebugFontTextWriter->Print("fillRateAvgPixPerSec:%12lu\n", gpuFillrateAvgPixPerSec);
    pDebugFontTextWriter->Print("gpu time (ns): %8lu\n", gpuTimeElapsedAvgValueInNs);
    pDebugFontTextWriter->Print("cpu time (ns): %8lu\n", cpuTimeElapsedAvgValueInNs);
    pDebugFontTextWriter->Print("total gpu time (ns): %12lu\n", gpuTimeElapsedValueInNs);
    pDebugFontTextWriter->Print("total cpu time (ns): %12lu\n", cpuTimeElapsedValueInNs);
}

void GpuBenchmarkBlending::CopyResultToBuffer(nn::gfx::CommandBuffer* pCommandBuffer)
{
    int renderSize = m_RenderSize;

    nn::gfx::BufferTextureCopyRegion bufferTextureCopyRegion;
    bufferTextureCopyRegion.SetDefault();
    bufferTextureCopyRegion.SetBufferImageHeight(renderSize);
    bufferTextureCopyRegion.SetBufferImageWidth(renderSize);
    bufferTextureCopyRegion.EditTextureCopyRegion().SetDefault();
    bufferTextureCopyRegion.EditTextureCopyRegion().SetWidth(renderSize);
    bufferTextureCopyRegion.EditTextureCopyRegion().SetHeight(renderSize);
    bufferTextureCopyRegion.EditTextureCopyRegion().EditSubresource().SetDefault();

    pCommandBuffer->InvalidateMemory(nn::gfx::GpuAccess_Texture);
    pCommandBuffer->CopyImageToBuffer(&m_OutputCopyBuffer, &m_RenderTexture, bufferTextureCopyRegion);
    pCommandBuffer->InvalidateMemory(nn::gfx::GpuAccess_Write);
}

void GpuBenchmarkBlending::MapResultBuffer(void** pOutBuffer, size_t* pOutBufferSize)
{
    *pOutBuffer = m_OutputCopyBuffer.Map();
    *pOutBufferSize = m_OutputCopyBufferSize;
}

void GpuBenchmarkBlending::UnmapResultBuffer()
{
    m_OutputCopyBuffer.Unmap();
}

const char* GpuBenchmarkBlending::GetName() const
{
    return ClassName;
}

BenchmarkType GpuBenchmarkBlending::GetType() const
{
    return BenchmarkType_Blending;
}

int GpuBenchmarkBlending::GetPropertyCount() const
{
    return m_PropertyArray.GetCount();
}

int GpuBenchmarkBlending::FillPropertyList(const GpuBenchmarkPropertyHolder** ppDestinationArray, int destinationArrayMaxSize) const
{
    return m_PropertyArray.FillPropertyList(ppDestinationArray, destinationArrayMaxSize);
}

int GpuBenchmarkBlending::FillPropertyList(GpuBenchmarkPropertyHolder** ppDestinationArray, int destinationArrayMaxSize)
{
    return m_PropertyArray.FillPropertyList(ppDestinationArray, destinationArrayMaxSize);
}

GpuBenchmarkPropertyHolder* GpuBenchmarkBlending::FindPropertyByName(const char* propertyName)
{
    return m_PropertyArray.FindPropertyByName(propertyName);
}

GpuBenchmarkPropertyHolder* GpuBenchmarkBlending::GetPropertyByIndex(int index)
{
    return m_PropertyArray.Get(index);
}

} } } // namespace nnt { namespace gfx { namespace util {
