﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdio>
#include <cstring>

#include <gfx/demo.h>

#include <nnt.h>
#include <nnt/nnt_Argument.h>

// ---- - Surface Information

#define SURFACE_WIDTH  (DEMOColorBufferInfo.GetWidth())
#define SURFACE_HEIGHT  (DEMOColorBufferInfo.GetHeight())

static const char TEXTURE_FILES[] = "textures/filter/noa";

static int g_ShaderFileIdx = 0;
#if NN_GFX_IS_TARGET_GX
#include <cafe/gx2.h>

// ----- GX2 Shader information

static const char * const SHADER_FILE_NAME[] =
{
    "shaders/filter/filter",
    "shaders/filter/filterHlslcc",
};
#else

static const char* const SHADER_FILE_NAME[] =
{
    "shaders/filter/filter",
    "shaders/filter/filterHlslcc",
};
#endif

static DEMOGfxShader myShader;
static DEMOGfxTexture myTexture;

static nn::gfx::Sampler mySampler;
static nn::gfx::DescriptorSlot mySamplerDescriptor;

static nn::gfx::ViewportScissorState fullViewport;
static void* fullViewportMem;

static DEMOGfxBuffer vertexBuffer;
static DEMOGfxBuffer indexBuffer;
static DEMOGfxBuffer uniformBuffer;

static nn::gfx::RasterizerState myRasterizerState;
static nn::gfx::VertexState myVertexState;
static nn::gfx::BlendState myBlendState;
static void* myBlendStateMem;
static void* myVertexStateMem;

static nn::gfx::DepthStencilState myDepthState;

static u32 samplerLoc;

static u32 ub_inpLoc;

static const int DEMO_MIN_ALIGNMENT = 4;

// ----- Model Vertex Data

#define countof(buf) (sizeof(buf)/sizeof(buf[0]))

static const int BUFFER_IDX = 0;

static f32 modelVtxs[20] =
{
    -0.9f, -0.9f,  0.0f,  1.0f,
     0.9f, -0.9f,  1.0f,  1.0f,
     0.9f,  0.9f,  1.0f,  0.0f,
    -0.9f,  0.9f,  0.0f,  0.0f
};

static const int QUAD_VTX_STRIDE = (sizeof(f32) * 2 + sizeof(f32) * 2);
static const int QUAD_POS_OFFSET = 0;
static const int QUAD_TEXCOORD_OFFSET = 2;

// ----- Quad Triangles

static u32 quadTrianglesIdxs[] = {0, 1, 2, 2, 3, 0};

// -----
static int filterSize;

// Prototype
static int SceneInit(void);
static int SceneDraw(void);
static void SceneDone(void);

// initialize a rasterizer state
static void SetupRasterizer(nn::gfx::RasterizerState* raster)
{
    nn::gfx::RasterizerStateInfo info;

    info.SetDefault();
    info.SetCullMode(nn::gfx::CullMode_None);
    info.SetDepthClipEnabled(false);
    info.SetFillMode(nn::gfx::FillMode_Solid);

    raster->Initialize(&DEMODevice, info);
}

// The init function for the rendering portions of this app
static int SceneInit()
{
    filterSize = 1;
    DEMOGfxLoadShadersFromFile(&myShader, 0, SHADER_FILE_NAME[g_ShaderFileIdx]);

    // Texture Sampler lookup
    samplerLoc = myShader.GetInterfaceSlot( nn::gfx::ShaderStage_Pixel,nn::gfx::ShaderInterfaceType_Sampler, "s_texture");

    // Uniform location lookup
    ub_inpLoc = myShader.GetInterfaceSlot( nn::gfx::ShaderStage_Pixel,nn::gfx::ShaderInterfaceType_ConstantBuffer, "ub_inp");

    // position setup
    DEMOGfxInitShaderAttribute(&myShader, "a_position", 0, QUAD_POS_OFFSET * sizeof(f32), nn::gfx::AttributeFormat_32_32_Float);
    DEMOGfxInitShaderAttribute(&myShader, "a_texCoord", 0, QUAD_TEXCOORD_OFFSET * sizeof(f32), nn::gfx::AttributeFormat_32_32_Float);

    DEMOGfxInitShaderVertexBuffer(&myShader, 0, QUAD_VTX_STRIDE, 0);

    // set up vertex and index buffers
    vertexBuffer.Initialize( sizeof( modelVtxs ), modelVtxs, nn::gfx::GpuAccess_VertexBuffer | nn::gfx::GpuAccess_Read, 0 );
    indexBuffer.Initialize( sizeof( quadTrianglesIdxs ), quadTrianglesIdxs, nn::gfx::GpuAccess_IndexBuffer | nn::gfx::GpuAccess_Read, 0 );
    uniformBuffer.Initialize( sizeof( u32 ), NULL, nn::gfx::GpuAccess_ConstantBuffer | nn::gfx::GpuAccess_Read, 0 );

    // Texture copied
    myTexture.Initialize( TEXTURE_FILES );

    // sampler setting
    DEMOGfxInitSampler(&mySampler, &mySamplerDescriptor, nn::gfx::TextureAddressMode_ClampToEdge, nn::gfx::FilterMode_MinLinear_MagLinear_MipLinear, nn::gfx::ComparisonFunction_Always);

    // viewport setting
    DEMOGfxSetViewportScissorState(&fullViewport, &fullViewportMem, 0.0, 0.0, float(SURFACE_WIDTH), float(SURFACE_HEIGHT), 0.0, 1.0, float(SURFACE_HEIGHT), false);

    // set up rasterizer state
    SetupRasterizer(&myRasterizerState);

    // set up vertex state
    {
        nn::gfx::VertexStateInfo info;
        info.SetDefault();
        info.SetVertexAttributeStateInfoArray(myShader.vertexAttributes, myShader.attribCount);
        info.SetVertexBufferStateInfoArray(myShader.vertexBuffers, myShader.bufferCount);
        size_t memsize = nn::gfx::VertexState::GetRequiredMemorySize(info);
        size_t memalign = nn::gfx::VertexState::RequiredMemoryInfo_Alignment;
        if (memalign < DEMO_MIN_ALIGNMENT)
        {
            memalign = DEMO_MIN_ALIGNMENT;
        }
        if ( memsize )
        {
            myVertexStateMem = DEMOAllocEx( memsize, memalign );
            myVertexState.SetMemory( myVertexStateMem, memsize );
        }
        myVertexState.Initialize( &DEMODevice, info, myShader.GetShader() );
    }
    // set up blend state
    {
        nn::gfx::BlendTargetStateInfo blendTargets[1];

        blendTargets[0].SetDefault();
        blendTargets[0].SetBlendEnabled(false);

        nn::gfx::BlendStateInfo info;
        info.SetDefault();
        info.SetDualSourceBlendEnabled(false);
        info.SetIndependentBlendEnabled(false);
        info.SetLogicOperationEnabled(false);
        info.SetBlendTargetStateInfoArray(blendTargets, 1);

        size_t size = nn::gfx::BlendState::GetRequiredMemorySize(info);
        size_t align = nn::gfx::BlendState::RequiredMemoryInfo_Alignment;
        if (align < DEMO_MIN_ALIGNMENT)
        {
            align = DEMO_MIN_ALIGNMENT;
        }
        myBlendStateMem = DEMOAllocEx(size, align);
        myBlendState.SetMemory(myBlendStateMem, size);
        myBlendState.Initialize(&DEMODevice, info);
    }
    {
        nn::gfx::DepthStencilStateInfo info;

        info.SetDefault();
        info.SetDepthTestEnabled(false);
        info.SetDepthWriteEnabled(false);
        myDepthState.Initialize(&DEMODevice, info);
    }
    return 1;
}

// The draw function for the rendering portions of this app
static int SceneDraw()
{
    //static f32 mOffset[4] = { 0.0f, 0.0f, 0.0f, 0.0f };

    DEMOGfxBeforeRender();

    nn::gfx::ColorTargetView* pCurrentScanBuffer = DEMOGetColorBufferView();

    // Set Viewport
    DEMOCommandBuffer.SetViewportScissorState(&fullViewport);
    DEMOCommandBuffer.ClearColor(pCurrentScanBuffer, 0.6f, 0.2f, 0.1f, 1.0f, NULL);
    DEMOCommandBuffer.ClearDepthStencil(&DEMODepthBufferView, 0.0f, 0, nn::gfx::DepthStencilClearMode_DepthStencil, NULL);

#if NN_GFX_IS_TARGET_GX
    GX2SetShaderMode( GX2_SHADER_MODE_UNIFORM_BLOCK );
#endif
    DEMOCommandBuffer.SetRasterizerState(&myRasterizerState);
    DEMOCommandBuffer.SetVertexState(&myVertexState);
    DEMOCommandBuffer.SetBlendState(&myBlendState);
    DEMOCommandBuffer.SetDepthStencilState(&myDepthState);

    // Set Shaders
    DEMOCommandBuffer.SetShader( myShader.GetShader(), nn::gfx::ShaderStageBit_All );

    // set up the color/depth buffers
    const nn::gfx::ColorTargetView* renderTargets[1];
    renderTargets[0] = pCurrentScanBuffer;
    DEMOCommandBuffer.SetRenderTargets(1, renderTargets, &DEMODepthBufferView);

    // Bind vertex buffer
    DEMOCommandBuffer.SetVertexBuffer( BUFFER_IDX, vertexBuffer.gpuAddress, QUAD_VTX_STRIDE, vertexBuffer.size );

    // Set Texture
    DEMOCommandBuffer.SetDescriptorPool(&DEMOTextureDescriptorPool);
    DEMOCommandBuffer.SetDescriptorPool(&DEMOSamplerDescriptorPool);

    DEMOCommandBuffer.SetTextureAndSampler(samplerLoc, nn::gfx::ShaderStage_Pixel, myTexture.GetDescriptorSlot(0), mySamplerDescriptor);

    // set uniform values
    filterSize++;
    if (filterSize > 64)
    {
        filterSize = 1;
    }
    u32 *u_count = uniformBuffer.Map< u32 >();
    {
        *u_count = filterSize;
    }

#if NN_GFX_IS_TARGET_GX
    GX2EndianSwap( u_count, sizeof(u32));
#endif

    uniformBuffer.Unmap();
    DEMOCommandBuffer.SetConstantBuffer(ub_inpLoc, nn::gfx::ShaderStage_Pixel, uniformBuffer.gpuAddress, uniformBuffer.size);

    DEMOCommandBuffer.DrawIndexed(nn::gfx::PrimitiveTopology_TriangleList, nn::gfx::IndexFormat_Uint32,
        indexBuffer.gpuAddress, 6, 0);


    DEMOGfxDoneRender();

    return 1; // 0 makes it exit
}

static void SceneDone()
{
    fullViewport.Finalize(&DEMODevice);
    DEMOFree(fullViewportMem);
    mySampler.Finalize(&DEMODevice);

    vertexBuffer.Finalize();
    indexBuffer.Finalize();
    uniformBuffer.Finalize();

    myRasterizerState.Finalize(&DEMODevice);
    myVertexState.Finalize(&DEMODevice);
    myDepthState.Finalize(&DEMODevice);
    myBlendState.Finalize(&DEMODevice);
    DEMOFree(myBlendStateMem);
    DEMOFree(myVertexStateMem);
}

//extern "C" void nnMain()
TEST(GfxTextureFilter, Run)
{
    int argc = nnt::GetHostArgc();
    char** argv = nnt::GetHostArgv();

    DEMOInit();
    DEMOTestInit(argc, argv);
    DEMOGfxInit(argc, argv);
    DEMOFontInit();
    DEMOTestIsUseHlslccGlsl() ? g_ShaderFileIdx = 1 : g_ShaderFileIdx = 0;
    SceneInit();

    while (DEMOIsRunning())
    {
        SceneDraw();
    }

    // clean up
    SceneDone();

    // Free shaders
    DEMOGfxFreeShaders(&myShader);

    // Free textures
    myTexture.Finalize();

    DEMOFontShutdown();
    DEMOTestShutdown();
    DEMOGfxShutdown();
    DEMOShutdown();

    SUCCEED();
}

