﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdio>
#include <cstring>
#include <algorithm>

#ifdef WIN32
#include <filesystem>
#endif
#include <stdint.h> // This is C++11, not supported by Cafe: #include <cstdint>
#include <nn/vi.h>
#include <nn/gfx.h>

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/init.h>

#include <nnt/nnt_Argument.h>

#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON )
    #include <nn/tma/tma.h>
#endif
#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_WIN32 )
# ifndef WIN32_LEAN_AND_MEAN
    #define WIN32_LEAN_AND_MEAN
# endif
# ifndef NOMINMAX
    #define NOMINMAX
# endif
    #include <nn/nn_Windows.h>
#endif

#if NN_GFX_IS_TARGET_GX
    #include <cafe/gx2.h>
    #include "simpleTriangleShaders_Cafe.h"
#else
    #include <nn/fs.h>
    #include <nn/hws/hws_Message.h>
    #include <nn/os.h>

static const char pixelShader_string[] =
"#version 330\n"
"layout(location = 0) out vec4 o_Color;\n"
"in vec2 texCoord;\n"
"uniform sampler2D tex;\n"
"layout(std140) uniform Mat\n"
"{\n"
"    vec4 color;\n"
"};\n"
"void main() {\n"
"    vec4 tex_color = texture(tex, texCoord);\n"
"    o_Color = color + tex_color;\n"
"}\n";

static const char vertexShader_string[] =
"#version 330\n"
"out gl_PerVertex {\n"
"    vec4 gl_Position;\n"
"};\n"
"\n"
"in vec4 i_Position;\n"
"in vec2 i_TexCoord;\n"
"out vec2 texCoord;\n"
"void main() {\n"
"    gl_Position = i_Position;\n"
"    texCoord = i_TexCoord.xy;\n"
"}\n";
#endif
#if defined(NN_BUILD_CONFIG_SPEC_NX)
#include <nv/nv_MemoryManagement.h>
#endif

static const int RENDER_WIDTH = 1280;
static const int RENDER_HEIGHT = 720;

namespace {

    // set g_UseTriStrips to use triangle strips
    static const bool g_UseTriStrips = true;

    // set g_IsPipelineMode to true for pipelines
    static const bool g_IsPipelineMode = true;

    template< typename T >
    T* AlignedAlloc(void** ppMemory, size_t size, size_t alignment = 1)
    {
        *ppMemory = reinterpret_cast<void*>((
            reinterpret_cast<uintptr_t>(*ppMemory) + alignment - 1) & ~(alignment - 1));
        T* ret = static_cast<T*>(*ppMemory);
        *ppMemory = static_cast<uint8_t*>(*ppMemory) + size;
        return ret;
    }

    // 頂点
    float triVertices[] =
    {
        -0.5f, -0.5f, 0.0f, 0.0f, 0.0f,
        -0.5f, 0.5f, 0.0f, 0.0f, 1.0f,
        0.5f, -0.5f, 0.0f, 1.0f, 0.0f
    };

    float quadVertices[] =
    {
        -0.75f, -0.75f, 0.0f, 0.0f, 0.0f,
        -0.75f, +0.75f, 0.0f, 0.0f, 1.0f,
        +0.75f, -0.75f, 0.0f, 1.0f, 0.0f,
        +0.75f, +0.75f, 0.0f, 1.0f, 1.0f,
    };

    // インデクス
    int indices[] =
    {
        0, 1, 2, 3, 2, 1
    };

    int g_BaseSamplerPoolSlot = 256;
    int g_BaseTextureViewPoolSlot = 256;

    //-----------------------------------------------------------------------------
    // メモリ
    nn::gfx::MemoryPool g_memPool;
    uint8_t* g_memPoolUnalignedBase;
    uint8_t* g_memPoolBase;
    static const size_t s_poolSize = 8 * 1024 * 1024;
    static size_t s_offset = 0;

    size_t AllocOffsetAligned(size_t size, size_t align)
    {
        size_t base, last;

        base = (s_offset + (align - 1)) & (~(align - 1));
        last = base + size;
        NN_SDK_ASSERT(last < s_poolSize);
        s_offset = last;
        return base;
    }

    void* AllocPtrAligned(size_t size, size_t align)
    {
        size_t base = AllocOffsetAligned(size, align);
        return (void*)(g_memPoolBase + base);
    }

    nn::gfx::MemoryPool g_memPoolInvisible;
    uint8_t* g_memPoolUnalignedBaseInvisible;
    uint8_t* g_memPoolBaseInvisible;
    static const size_t s_poolSizeInvisible = 64 * 1024 * 1024;
    static size_t s_offsetInvisible = 0;

    size_t AllocOffsetAlignedInvisible(size_t size, size_t align)
    {
        size_t base, last;

        base = (s_offsetInvisible + (align - 1)) & (~(align - 1));
        last = base + size;
        NN_SDK_ASSERT(last < s_poolSizeInvisible);
        s_offsetInvisible = last;
        return base;
    }

    // レイヤを初期化
    nn::vi::Display* g_pDisplay;
    nn::vi::Layer* g_pLayer;
    void InitializeLayer()
    {
        nn::Result result = nn::vi::OpenDefaultDisplay( &g_pDisplay );
        NN_ASSERT( result.IsSuccess() );
        NN_UNUSED( result );

        result = nn::vi::CreateLayer( &g_pLayer, g_pDisplay );
        NN_ASSERT( result.IsSuccess() );

        result = nn::vi::SetLayerScalingMode( g_pLayer, nn::vi::ScalingMode_FitToLayer );
        NN_ASSERT( result.IsSuccess() );
}

    // デバイスを初期化
    nn::gfx::Device g_Device;
    void InitDevice()
    {
        nn::gfx::Device::InfoType info;
        info.SetDefault();
        info.SetApiVersion(nn::gfx::ApiMajorVersion, nn::gfx::ApiMinorVersion);
        g_Device.Initialize(info);
    }

    // スワップチェーンを初期化
    nn::gfx::SwapChain g_SwapChain;
    static const int TOTAL_SWAP_BUFFERS = 2;
    void InitSwapChain()
    {
        nn::gfx::SwapChain::InfoType info;

        info.SetDefault();
        info.SetLayer(g_pLayer);
        info.SetWidth(RENDER_WIDTH);
        info.SetHeight(RENDER_HEIGHT);
        info.SetFormat(nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm);
        info.SetBufferCount(TOTAL_SWAP_BUFFERS);
        if (NN_STATIC_CONDITION(nn::gfx::SwapChain::IsMemoryPoolRequired))
        {
            size_t memSize = nn::gfx::SwapChain::CalculateScanBufferSize(&g_Device, info);
            size_t off = AllocOffsetAlignedInvisible(memSize, nn::gfx::SwapChain::GetScanBufferAlignment(&g_Device, info));
            g_SwapChain.Initialize(&g_Device, info, &g_memPoolInvisible, off, memSize);
        }
        else
        {
            g_SwapChain.Initialize(&g_Device, info, NULL, 0, 0);
        }
    }

    // キューを初期化
    nn::gfx::Queue g_Queue;
    void InitQueue()
    {
        nn::gfx::Queue::InfoType info;
        info.SetDefault();
        info.SetCapability(nn::gfx::QueueCapability_Graphics);
        g_Queue.Initialize(&g_Device, info);
    }

    // コマンドバッファを初期化
    nn::gfx::CommandBuffer g_commandBuffer;
    nn::gfx::CommandBuffer g_commandBuffer2;
    void InitCommandBuffer(nn::gfx::CommandBuffer* cmdbuf)
    {
        nn::gfx::CommandBuffer::InfoType info;
        info.SetDefault();
        info.SetQueueCapability(nn::gfx::QueueCapability_Graphics);
        info.SetCommandBufferType(nn::gfx::CommandBufferType_Direct);
        cmdbuf->Initialize(&g_Device, info);

        size_t size = 256 * 1024;
        size_t alignment = nn::gfx::CommandBuffer::GetCommandMemoryAlignment(&g_Device);
        size_t off = AllocOffsetAligned(size, alignment);
        void* ptr = AllocPtrAligned(64 * 1024, alignment);

        cmdbuf->Reset();
        cmdbuf->AddCommandMemory(&g_memPool, off, size);
        cmdbuf->AddControlMemory(ptr, 64 * 1024);
    }

    nn::gfx::DescriptorPool g_TextureDescriptorPool;
    nn::gfx::DescriptorPool g_SamplerDescriptorPool;
    void InitDescriptorPools()
    {
        {
            nn::gfx::DescriptorPool::InfoType info;
            info.SetDefault();
            info.SetDescriptorPoolType(nn::gfx::DescriptorPoolType_Sampler);
            info.SetSlotCount(g_BaseSamplerPoolSlot + 1);

            size_t alignment = nn::gfx::DescriptorPool::GetDescriptorPoolAlignment(&g_Device, info);
            size_t size = nn::gfx::DescriptorPool::CalculateDescriptorPoolSize(&g_Device, info);
            size_t off = AllocOffsetAligned(size, alignment);
            g_SamplerDescriptorPool.Initialize(&g_Device, info, &g_memPool, off, size);
        }

    {
        nn::gfx::DescriptorPool::InfoType info;
        info.SetDefault();
        info.SetDescriptorPoolType(nn::gfx::DescriptorPoolType_TextureView);
        info.SetSlotCount(g_BaseTextureViewPoolSlot + 3);

        size_t alignment = nn::gfx::DescriptorPool::GetDescriptorPoolAlignment(&g_Device, info);
        size_t size = nn::gfx::DescriptorPool::CalculateDescriptorPoolSize(&g_Device, info);
        size_t off = AllocOffsetAligned(size, alignment);
        g_TextureDescriptorPool.Initialize(&g_Device, info, &g_memPool, off, size);
    }
    }

    // ビューポートシザーを初期化
    nn::gfx::ViewportScissorState g_viewportScissor;
    void InitViewport()
    {
        nn::gfx::ViewportScissorState::InfoType info;
        info.SetDefault();
        info.SetScissorEnabled(true);
        nn::gfx::ViewportStateInfo viewportInfo;
        {
            viewportInfo.SetDefault();
            viewportInfo.SetWidth(static_cast<float>(RENDER_WIDTH));
            viewportInfo.SetHeight(static_cast<float>(RENDER_HEIGHT));
        }
        nn::gfx::ScissorStateInfo scissorInfo;
        {
            scissorInfo.SetDefault();
            scissorInfo.SetWidth(RENDER_WIDTH);
            scissorInfo.SetHeight(RENDER_HEIGHT);
        }
        info.SetViewportStateInfoArray(&viewportInfo, 1);
        info.SetScissorStateInfoArray(&scissorInfo, 1);
        g_viewportScissor.Initialize(&g_Device, info);
    }

    // 頂点シェーダを初期化
    nn::gfx::Shader* g_pShader;
#if NN_GFX_IS_TARGET_GX
    nn::gfx::Shader g_Shader;
    void InitShader()
    {
        nn::gfx::Shader::InfoType info;
        info.SetDefault();
        info.SetCodeType(nn::gfx::ShaderCodeType_Binary);
        info.SetShaderCodePtr( nn::gfx::ShaderStage_Vertex, &SimpleTriangleShaders_Cafe_VS);
        info.SetShaderCodePtr( nn::gfx::ShaderStage_Pixel, &SimpleTriangleShaders_Cafe_PS);
        g_Shader.Initialize( &g_Device, info );
        g_pShader = &g_Shader;
    }
#else
    nn::gfx::Shader g_Shader;
    void InitShader()
    {
        static nn::gfx::ShaderCode vertexCode, pixelCode;

        vertexCode.codeSize = (uint32_t)strlen(vertexShader_string);
        vertexCode.pCode = vertexShader_string;
        pixelCode.codeSize = (uint32_t)strlen(pixelShader_string);
        pixelCode.pCode = pixelShader_string;
        nn::gfx::Shader::InfoType info;
        info.SetDefault();
        info.SetCodeType(nn::gfx::ShaderCodeType_Source);
        info.SetShaderCodePtr(nn::gfx::ShaderStage_Vertex, &vertexCode);
        info.SetShaderCodePtr(nn::gfx::ShaderStage_Pixel, &pixelCode);
        NN_ASSERT(g_Shader.Initialize(&g_Device, info) == nn::gfx::ShaderInitializeResult_Success);
        g_pShader = &g_Shader;
    }
#endif

    // パイプラインを初期化
    nn::gfx::Pipeline g_pipeline1;
    nn::gfx::Pipeline g_pipeline2;

    void InitPipeline(nn::gfx::Pipeline* pipeline)
    {
        nn::gfx::Pipeline::InfoType info;
        nn::gfx::BlendStateInfo blendStateInfo;
        nn::gfx::DepthStencilStateInfo depthStencilStateInfo;
        nn::gfx::RenderTargetStateInfo renderTargetStateInfo;
        nn::gfx::RasterizerStateInfo rasterizerStateInfo;
        nn::gfx::VertexStateInfo vertexStateInfo;

        info.SetDefault();

        nn::gfx::ColorTargetStateInfo colorTargetInfo[1];
        colorTargetInfo[0].SetDefault();
        colorTargetInfo[0].SetFormat(nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm);
        renderTargetStateInfo.SetDefault();
        renderTargetStateInfo.SetDepthStencilFormat(nn::gfx::ImageFormat_D16_Unorm);
        renderTargetStateInfo.SetColorTargetStateInfoArray(colorTargetInfo, 1);

        info.SetShaderPtr(g_pShader);

        rasterizerStateInfo.SetDefault();
        rasterizerStateInfo.SetCullMode(nn::gfx::CullMode_None);
        rasterizerStateInfo.SetScissorEnabled(true);
        blendStateInfo.SetDefault();
        nn::gfx::BlendTargetStateInfo blendTargetInfo[1];
        {
            blendTargetInfo[0].SetDefault();
        }
        blendStateInfo.SetBlendTargetStateInfoArray(blendTargetInfo, 1);

        depthStencilStateInfo.SetDefault();
        depthStencilStateInfo.SetDepthTestEnabled(true);
        depthStencilStateInfo.SetDepthWriteEnabled(true);
        nn::gfx::VertexAttributeStateInfo attribs[2];
        {
            attribs[0].SetDefault();
            attribs[0].SetNamePtr("i_Position");
            attribs[0].SetBufferIndex(0);
            attribs[0].SetFormat(nn::gfx::AttributeFormat_32_32_32_Float);
            attribs[0].SetOffset(0);
        }
    {
        attribs[1].SetDefault();
        attribs[1].SetNamePtr("i_TexCoord");
        attribs[1].SetBufferIndex(0);
        attribs[1].SetFormat(nn::gfx::AttributeFormat_32_32_Float);
        attribs[1].SetOffset(sizeof(float) * 3);
    }
    nn::gfx::VertexBufferStateInfo buffer;
    {
        buffer.SetDefault();
        buffer.SetStride(sizeof(float) * 5);
    }
    vertexStateInfo.SetDefault();
    vertexStateInfo.SetVertexAttributeStateInfoArray(attribs, 2);
    vertexStateInfo.SetVertexBufferStateInfoArray(&buffer, 1);

    info.SetBlendStateInfo(&blendStateInfo);
    info.SetDepthStencilStateInfo(&depthStencilStateInfo);
    info.SetRasterizerStateInfo(&rasterizerStateInfo);
    info.SetRenderTargetStateInfo(&renderTargetStateInfo);
    info.SetVertexStateInfo(&vertexStateInfo);

    size_t dataSize = nn::gfx::Pipeline::GetRequiredMemorySize(info);
    void* ptr = AllocPtrAligned(dataSize, nn::gfx::Pipeline::RequiredMemoryInfo_Alignment);

    pipeline->SetMemory(ptr, dataSize);
    pipeline->Initialize(&g_Device, info);
    }

    // Initialize the rasterizer state
    nn::gfx::RasterizerState g_rasterizerState;
    static void InitRasterizerState()
    {
        nn::gfx::RasterizerState::InfoType info;
        info.SetDefault();
        info.SetCullMode(nn::gfx::CullMode_None);
        info.SetScissorEnabled(true);
        info.SetDepthClipEnabled(false);
        g_rasterizerState.Initialize(&g_Device, info);
    }

    // Initialize the blend state
    nn::gfx::BlendState g_blendState;
    static void InitBlendState()
    {
        nn::gfx::BlendState::InfoType info;
        info.SetDefault();
        nn::gfx::BlendTargetStateInfo targetInfo;
        {
            targetInfo.SetDefault();
        };
        info.SetBlendTargetStateInfoArray(&targetInfo, 1);
        size_t size = nn::gfx::BlendState::GetRequiredMemorySize(info);
        uint8_t* data = (uint8_t*)AllocPtrAligned(size, nn::gfx::BlendState::RequiredMemoryInfo_Alignment);
        g_blendState.SetMemory(data, size);
        g_blendState.Initialize(&g_Device, info);
    }

    // Initialize the depth stencil state
    nn::gfx::DepthStencilState g_depthStencilState;
    static void InitDepthStencilState()
    {
        nn::gfx::DepthStencilState::InfoType info;
        info.SetDefault();
        info.SetDepthTestEnabled(true);
        info.SetDepthWriteEnabled(true);
        g_depthStencilState.Initialize(&g_Device, info);
    }

    // Initialize the vertex state
    nn::gfx::VertexState g_vertexState;
    static void InitVertexState()
    {
        nn::gfx::VertexState::InfoType info;
        info.SetDefault();
        uint16_t stride = sizeof(float) * 5;
        nn::gfx::VertexAttributeStateInfo attribs[2];
        {
            attribs[0].SetDefault();
            attribs[0].SetNamePtr("i_Position");
            attribs[0].SetBufferIndex(0);
            attribs[0].SetFormat(nn::gfx::AttributeFormat_32_32_32_Float);
            attribs[0].SetOffset(0);
        }
    {
        attribs[1].SetDefault();
        attribs[1].SetNamePtr("i_TexCoord");
        attribs[1].SetBufferIndex(0);
        attribs[1].SetFormat(nn::gfx::AttributeFormat_32_32_Float);
        attribs[1].SetOffset(sizeof(float) * 3);
    }
    nn::gfx::VertexBufferStateInfo buffer;
    {
        buffer.SetDefault();
        buffer.SetStride(stride);
    }
    info.SetVertexAttributeStateInfoArray(attribs, 2);
    info.SetVertexBufferStateInfoArray(&buffer, 1);
    size_t size = nn::gfx::VertexState::GetRequiredMemorySize(info);
    uint8_t* data = (uint8_t*)AllocPtrAligned(size, nn::gfx::VertexState::RequiredMemoryInfo_Alignment);
    g_vertexState.SetMemory(data, size);
    g_vertexState.Initialize(&g_Device, info, g_pShader);
    }

    // 頂点用のバッファを初期化
    nn::gfx::Buffer g_triVertexBuffer;
    nn::gfx::Buffer g_quadVertexBuffer;

    struct BufferView
    {
        nn::gfx::GpuAddress gpuAddress;
        size_t size;
    };

    void InitBuffer(nn::gfx::GpuAccess flags, nn::gfx::Buffer* buffer, void* data, size_t size, BufferView* view)
    {
        nn::gfx::Buffer::InfoType info;

        info.SetDefault();
        info.SetSize(size);
        info.SetGpuAccessFlags(flags);
        if (NN_STATIC_CONDITION(nn::gfx::Buffer::IsMemoryPoolRequired))
        {
            size_t off = AllocOffsetAligned(size, nn::gfx::Buffer::GetBufferAlignment(&g_Device, info));
            buffer->Initialize(&g_Device, info, &g_memPool, off, size);
        }
        else
        {
            buffer->Initialize(&g_Device, info, NULL, 0, 0);
        }

        if (data)
        {
            void* alignedVertices = buffer->Map();
            memcpy(alignedVertices, data, size);
            buffer->FlushMappedRange( 0, size );
            buffer->Unmap();
        }

        view->size = size;
        buffer->GetGpuAddress(&view->gpuAddress);
    }

    // 頂点バッファビューを初期化
    BufferView g_triVertexBufferView;
    BufferView g_quadVertexBufferView;

    void InitVertexBuffer(nn::gfx::Buffer* vertBuffer, float* origVertices, int length, BufferView *view)
    {
        size_t size = sizeof(float) * 5 * length;

        InitBuffer(nn::gfx::GpuAccess_VertexBuffer, vertBuffer, origVertices, size, view);
    }


    // インデクス用のバッファを初期化
    nn::gfx::Buffer g_indexBuffer;
    BufferView g_indexBufferView;

    void InitIndexBuffer()
    {
        size_t size = sizeof(indices);
        InitBuffer(nn::gfx::GpuAccess_IndexBuffer, &g_indexBuffer, indices, size, &g_indexBufferView);
    }

    // カラーバッファ用のテクスチャを初期化
    nn::gfx::Texture g_colorBuffer1;
    nn::gfx::TextureView g_colorTextureView1;
    nn::gfx::ColorTargetView g_colorBufferView1;
    nn::gfx::DescriptorSlot g_colorTextureSlot1;

    nn::gfx::Texture g_colorBuffer2;
    nn::gfx::TextureView g_colorTextureView2;
    nn::gfx::ColorTargetView g_colorBufferView2;

    void InitColorBuffer(nn::gfx::Texture* colorBuffer, nn::gfx::TextureView* colorTextureView, nn::gfx::ColorTargetView* colorBufferView)
    {
        nn::gfx::Texture::InfoType info;
        info.SetDefault();
        info.SetWidth(RENDER_WIDTH);
        info.SetHeight(RENDER_HEIGHT);

        info.SetGpuAccessFlags(nn::gfx::GpuAccess_ColorBuffer | nn::gfx::GpuAccess_Read | nn::gfx::GpuAccess_Write);
        info.SetImageStorageDimension(nn::gfx::ImageStorageDimension_2d);
        info.SetImageFormat(nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm);
        info.SetMipCount(1);

        size_t size = nn::gfx::Texture::CalculateMipDataSize(&g_Device, info);
        size_t off = AllocOffsetAlignedInvisible(size, nn::gfx::Texture::CalculateMipDataAlignment(&g_Device, info));
        colorBuffer->Initialize(&g_Device, info, &g_memPoolInvisible, off, size);

        nn::gfx::TextureView::InfoType vinfo;
        vinfo.SetDefault();
        vinfo.SetImageDimension(nn::gfx::ImageDimension_2d);
        vinfo.SetImageFormat(nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm);
        vinfo.SetTexturePtr(colorBuffer);
        colorTextureView->Initialize(&g_Device, vinfo);

        nn::gfx::ColorTargetView::InfoType cinfo;
        cinfo.SetDefault();
        cinfo.SetImageDimension(nn::gfx::ImageDimension_2d);
        cinfo.SetImageFormat(nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm);
        cinfo.SetTexturePtr(colorBuffer);
        colorBufferView->Initialize(&g_Device, cinfo);
    }

    // 深度ステンシルバッファ用のテクスチャを初期化
    nn::gfx::Texture g_depthStencilBuffer;
    void InitDepthBuffer()
    {
        nn::gfx::Texture::InfoType info;
        info.SetDefault();
        info.SetWidth( RENDER_WIDTH );
        info.SetHeight( RENDER_HEIGHT );
        info.SetGpuAccessFlags(nn::gfx::GpuAccess_DepthStencil | nn::gfx::GpuAccess_Read | nn::gfx::GpuAccess_Write);
        info.SetImageStorageDimension(nn::gfx::ImageStorageDimension_2d);
        info.SetImageFormat(nn::gfx::ImageFormat_D16_Unorm);
        size_t size = nn::gfx::Texture::CalculateMipDataSize(&g_Device, info);
        size_t off = AllocOffsetAlignedInvisible(size, nn::gfx::Texture::CalculateMipDataAlignment(&g_Device, info));
        g_depthStencilBuffer.Initialize(&g_Device, info, &g_memPoolInvisible, off, size);
    }
    // 深度ステンシルビューを初期化
    nn::gfx::DepthStencilView g_depthStencilView;
    void InitDepthBufferView()
    {
        nn::gfx::DepthStencilView::InfoType info;
        info.SetDefault();
        info.SetImageDimension(nn::gfx::ImageDimension_2d);
        info.SetTexturePtr(&g_depthStencilBuffer);
        g_depthStencilView.Initialize(&g_Device, info);
    }

    // 定数バッファ用のバッファを初期化
    nn::gfx::Buffer g_constantBuffer;
    BufferView g_constantBufferView;

    void InitConstantBuffer()
    {
        InitBuffer(nn::gfx::GpuAccess_ConstantBuffer, &g_constantBuffer, NULL, sizeof(float) * 4, &g_constantBufferView);
    }

    const int TEX_WIDTH = 256;
    const int TEX_HEIGHT = 256;

    // テクスチャを初期化
    nn::gfx::Texture g_texture;
    void InitTextureBuffer()
    {
        nn::gfx::Buffer initData;
        {
            nn::gfx::Buffer::InfoType info;
            size_t size = TEX_WIDTH * TEX_HEIGHT * 4;

            info.SetDefault();
            info.SetGpuAccessFlags( nn::gfx::GpuAccess_Read );
            info.SetSize( size );
            // FIXME: Cafe requires the buffer to be aligned on a 256 byte boundary for image copy
            size_t off = AllocOffsetAligned( size, std::max( static_cast< size_t >( 256 ), nn::gfx::Buffer::GetBufferAlignment( &g_Device, info ) ) );
            initData.Initialize( &g_Device, info, &g_memPool, off, size );

            uint8_t* pixels = initData.Map< uint8_t >();

            // The only reason this works on Cafe is because pitch = 256.
            // make the first 8 lines all white
            for (int x = 0; x < TEX_WIDTH * 8 * 4; ++x)
            {
                pixels[x] = 0xff;
            }
            for (int y = 8; y < TEX_HEIGHT; ++y)
            {
                for (int x = 0; x < TEX_WIDTH; ++x)
                {
                    uint8_t scaley = ( uint8_t )y / 16;
                    uint8_t scalex = ( uint8_t )x / 16;
                    bool checker = 0 != ((scalex ^ scaley) & 1);
                    pixels[x * 4 + y * TEX_WIDTH * 4 + 0] = checker ? (0xFF / (scaley + 1)) : 0x00;
                    pixels[x * 4 + y * TEX_WIDTH * 4 + 1] = checker ? 0x00 : (0xFF / (scaley + 1));
                    pixels[x * 4 + y * TEX_WIDTH * 4 + 2] = 0x00;
                    pixels[x * 4 + y * TEX_WIDTH * 4 + 3] = 0xFF;
                }
            }

            initData.FlushMappedRange( 0, size );
            initData.Unmap();
        }

        {
            nn::gfx::Texture::InfoType info;
            info.SetDefault();
            info.SetGpuAccessFlags( nn::gfx::GpuAccess_Texture );
            info.SetWidth( TEX_WIDTH );
            info.SetHeight( TEX_HEIGHT );
            info.SetImageStorageDimension( nn::gfx::ImageStorageDimension_2d );
            info.SetImageFormat( nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm );
            info.SetTileMode( nn::gfx::TileMode_Optimal );
            size_t size = nn::gfx::Texture::CalculateMipDataSize( &g_Device, info );
            size_t off = AllocOffsetAlignedInvisible( size, nn::gfx::Texture::CalculateMipDataAlignment( &g_Device, info ) );
            g_texture.Initialize(&g_Device, info, &g_memPoolInvisible, off, size);
        }

        // Copy over
        nn::gfx::TextureCopyRegion dstRegion;
        dstRegion.SetDefault();
        dstRegion.SetWidth( TEX_WIDTH );
        dstRegion.SetHeight( TEX_HEIGHT );
        g_commandBuffer.Begin();
        g_commandBuffer.CopyBufferToImage( &g_texture, dstRegion, &initData, 0 );
        g_commandBuffer.End();
        g_Queue.ExecuteCommand( &g_commandBuffer, NULL );

        initData.Finalize( &g_Device );
    };

    // テクスチャビューを初期化
    nn::gfx::TextureView g_textureView;
    nn::gfx::DescriptorSlot g_textureSlot;
    void InitTextureBufferView()
    {
        nn::gfx::TextureView::InfoType info;
        info.SetDefault();
        info.SetImageDimension(nn::gfx::ImageDimension_2d);
        info.SetImageFormat(nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm);
        info.SetTexturePtr(&g_texture);
        info.SetChannelMapping(nn::gfx::ChannelMapping_Red, nn::gfx::ChannelMapping_Green,
            nn::gfx::ChannelMapping_Blue, nn::gfx::ChannelMapping_Alpha);
        g_textureView.Initialize(&g_Device, info);
    }

    // サンプラを初期化
    nn::gfx::Sampler g_sampler;
    nn::gfx::DescriptorSlot g_samplerSlot;
    void InitSampler()
    {
        nn::gfx::Sampler::InfoType info;
        info.SetDefault();
        //info.SetFilterMode( nn::gfx::FilterMode_MinLinear_MagLinear_MipPoint );
        info.SetFilterMode(nn::gfx::FilterMode_MinPoint_MagPoint_MipPoint);
        info.SetAddressU(nn::gfx::TextureAddressMode_Repeat);
        info.SetAddressV(nn::gfx::TextureAddressMode_Repeat);
        info.SetAddressW(nn::gfx::TextureAddressMode_Repeat);
        g_sampler.Initialize(&g_Device, info);

        // Register the sampler
        g_SamplerDescriptorPool.BeginUpdate();
        g_SamplerDescriptorPool.SetSampler(g_BaseSamplerPoolSlot + 0, &g_sampler);
        g_SamplerDescriptorPool.EndUpdate();
        g_SamplerDescriptorPool.GetDescriptorSlot(&g_samplerSlot, g_BaseSamplerPoolSlot + 0);
    }

    // 同期フェンスを初期化
    nn::gfx::Fence g_fence;
    nn::gfx::Fence g_fence2;

    nn::gfx::CommandBuffer g_PreambleCommandBuffers[TOTAL_SWAP_BUFFERS];
    void InitializePreambleCommandBuffers()
    {
        nn::gfx::CommandBuffer::InfoType info;
        info.SetDefault();
        info.SetQueueCapability(nn::gfx::QueueCapability_Graphics);
        info.SetCommandBufferType(nn::gfx::CommandBufferType_Direct);
        for (int index = 0; index < TOTAL_SWAP_BUFFERS; index++)
        {
            g_PreambleCommandBuffers[index].Initialize(&g_Device, info);

            g_PreambleCommandBuffers[index].Reset();
            size_t off = AllocOffsetAligned(8 * 1024, nn::gfx::CommandBuffer::GetCommandMemoryAlignment(&g_Device));

            g_PreambleCommandBuffers[index].AddCommandMemory(&g_memPool, off, 8 * 1024);
            void* cmdptr = AllocPtrAligned(8 * 256, nn::gfx::CommandBuffer::GetControlMemoryAlignment(&g_Device));
            g_PreambleCommandBuffers[index].AddControlMemory(cmdptr, 8 * 256);
            {
                g_PreambleCommandBuffers[index].Begin();
                nn::gfx::ColorTargetView* pColorTargetView[1];
                pColorTargetView[0] = g_SwapChain.AcquireNextScanBufferView();
                g_PreambleCommandBuffers[index].ClearColor(pColorTargetView[0], 0.5f, 0.1f, 0.1f, 1.0f, NULL);
                g_PreambleCommandBuffers[index].ClearDepthStencil(&g_depthStencilView, 1.0f, 0, nn::gfx::DepthStencilClearMode_DepthStencil, NULL);
#if NN_GFX_IS_TARGET_GX
                GX2SetShaderMode(GX2_SHADER_MODE_UNIFORM_BLOCK);
#endif
                g_PreambleCommandBuffers[index].SetRenderTargets(1, pColorTargetView, &g_depthStencilView);
                g_PreambleCommandBuffers[index].End();

                g_Queue.Present(&g_SwapChain, 1); // FIXME: This is the only way to advance the current scan buffer pointer to get the other view.
            }
        }
    }


    void Cleanup()
    {
        // 各オブジェクトを破棄
        g_fence.Finalize(&g_Device);
        g_fence2.Finalize(&g_Device);
        g_sampler.Finalize(&g_Device);
        g_textureView.Finalize(&g_Device);
        g_texture.Finalize(&g_Device);
        g_constantBuffer.Finalize(&g_Device);
        g_depthStencilView.Finalize(&g_Device);
        g_depthStencilBuffer.Finalize(&g_Device);
        g_colorBufferView1.Finalize(&g_Device);
        g_colorTextureView1.Finalize(&g_Device);
        g_colorBuffer1.Finalize(&g_Device);
        g_colorBufferView2.Finalize(&g_Device);
        g_colorTextureView2.Finalize(&g_Device);
        g_colorBuffer2.Finalize(&g_Device);
        g_indexBuffer.Finalize(&g_Device);
        g_triVertexBuffer.Finalize(&g_Device);
        g_quadVertexBuffer.Finalize(&g_Device);
        if (NN_STATIC_CONDITION(g_IsPipelineMode))
        {
            g_pipeline1.Finalize(&g_Device);
            g_pipeline2.Finalize(&g_Device);
        }
        else
        {
            g_rasterizerState.Finalize(&g_Device);
            g_blendState.Finalize(&g_Device);
            g_depthStencilState.Finalize(&g_Device);
            g_vertexState.Finalize(&g_Device);
        }
        g_TextureDescriptorPool.Finalize(&g_Device);
        g_SamplerDescriptorPool.Finalize(&g_Device);
        g_pShader->Finalize(&g_Device);
        g_viewportScissor.Finalize(&g_Device);
        for (int index = 0; index < TOTAL_SWAP_BUFFERS; index++)
        {
            g_PreambleCommandBuffers[index].Finalize(&g_Device);
        }
        g_commandBuffer.Finalize(&g_Device);
        g_commandBuffer2.Finalize(&g_Device);
        g_SwapChain.Finalize(&g_Device);
        g_Queue.Finalize(&g_Device);
        g_memPoolInvisible.Finalize(&g_Device);
        g_memPool.Finalize(&g_Device);
        g_Device.Finalize();

        // ライブラリを終了
        nn::gfx::Finalize();

        delete[] g_memPoolUnalignedBaseInvisible;
        delete[] g_memPoolUnalignedBase;
    }

    int g_slotMat;
    int g_slotTex;

    void BuildCommandBuffer()
    {
        // コマンドを生成
        g_commandBuffer.Begin();
        {
            nn::gfx::ColorTargetView* pTarget[1];
            pTarget[0] = &g_colorBufferView1;

            g_commandBuffer.SetDescriptorPool( &g_TextureDescriptorPool );
            g_commandBuffer.SetDescriptorPool( &g_SamplerDescriptorPool );
            g_commandBuffer.ClearColor(&g_colorBufferView1, 0.1f, 0.1f, 0.7f, 1.0f, NULL);
            g_commandBuffer.ClearDepthStencil(&g_depthStencilView, 1.0f, 0, nn::gfx::DepthStencilClearMode_DepthStencil, NULL);

#if NN_GFX_IS_TARGET_GX
            GX2SetShaderMode( GX2_SHADER_MODE_UNIFORM_BLOCK );
#endif
            g_commandBuffer.SetRenderTargets(1, pTarget, &g_depthStencilView);

            g_commandBuffer.SetViewportScissorState(&g_viewportScissor);

            if (NN_STATIC_CONDITION(g_IsPipelineMode))
            {
                g_commandBuffer.SetPipeline(&g_pipeline1);
            }
            else
            {
                g_commandBuffer.SetShader(g_pShader, nn::gfx::ShaderStageBit_All);
                g_commandBuffer.SetRasterizerState(&g_rasterizerState);
                g_commandBuffer.SetBlendState(&g_blendState);
                g_commandBuffer.SetDepthStencilState(&g_depthStencilState);
                g_commandBuffer.SetVertexState(&g_vertexState);
            }

            // Guarantee that the internal GPU caches are invalidated for our inputs
            g_commandBuffer.InvalidateMemory(nn::gfx::GpuAccess_Texture | nn::gfx::GpuAccess_IndexBuffer | nn::gfx::GpuAccess_ConstantBuffer | nn::gfx::GpuAccess_VertexBuffer);

            g_commandBuffer.SetConstantBuffer(g_slotMat, nn::gfx::ShaderStage_Pixel, g_constantBufferView.gpuAddress, g_constantBufferView.size);
            g_commandBuffer.SetTextureAndSampler(g_slotTex, nn::gfx::ShaderStage_Pixel, g_textureSlot, g_samplerSlot);

            g_commandBuffer.SetVertexBuffer(0, g_triVertexBufferView.gpuAddress, sizeof(float) * 5, g_triVertexBufferView.size);
            g_commandBuffer.Draw(nn::gfx::PrimitiveTopology_TriangleList, 3, 0);
        }
        g_commandBuffer.End();
    }

    void BuildCommandBuffer2()
    {
        // コマンドを生成
        g_commandBuffer2.Begin();
        {
            g_commandBuffer2.SetDescriptorPool( &g_TextureDescriptorPool );
            g_commandBuffer2.SetDescriptorPool( &g_SamplerDescriptorPool );
            g_commandBuffer2.SetViewportScissorState(&g_viewportScissor);

            if (NN_STATIC_CONDITION(g_IsPipelineMode))
            {
                g_commandBuffer2.SetPipeline(&g_pipeline2);
            }
            else
            {
                g_commandBuffer2.SetShader(g_pShader, nn::gfx::ShaderStageBit_All);
                g_commandBuffer2.SetRasterizerState(&g_rasterizerState);
                g_commandBuffer2.SetBlendState(&g_blendState);
                g_commandBuffer2.SetDepthStencilState(&g_depthStencilState);
                g_commandBuffer2.SetVertexState(&g_vertexState);
            }

            // Guarantee that the internal GPU caches are invalidated for our inputs
            g_commandBuffer2.InvalidateMemory(nn::gfx::GpuAccess_Texture | nn::gfx::GpuAccess_IndexBuffer | nn::gfx::GpuAccess_ConstantBuffer | nn::gfx::GpuAccess_VertexBuffer);

            g_commandBuffer2.SetConstantBuffer(g_slotMat, nn::gfx::ShaderStage_Pixel, g_constantBufferView.gpuAddress, g_constantBufferView.size);
            g_commandBuffer2.SetTextureAndSampler(g_slotTex, nn::gfx::ShaderStage_Pixel, g_colorTextureSlot1, g_samplerSlot);

            g_commandBuffer2.SetVertexBuffer(0, g_quadVertexBufferView.gpuAddress, sizeof(float) * 5, g_quadVertexBufferView.size);
            if ( NN_STATIC_CONDITION( g_UseTriStrips ) )
            {
                g_commandBuffer2.DrawIndexed(nn::gfx::PrimitiveTopology_TriangleStrip, nn::gfx::IndexFormat_Uint32,
                    g_indexBufferView.gpuAddress, 4, 0, 1, 0);
            }
            else
            {
                g_commandBuffer2.DrawIndexed(nn::gfx::PrimitiveTopology_TriangleList, nn::gfx::IndexFormat_Uint32,
                    g_indexBufferView.gpuAddress, 6, 0, 1, 0);
            }
        }
        g_commandBuffer2.End();
    }

};

void* Allocate(size_t size)
{
    return malloc(size);
}

void Deallocate(void* ptr, size_t)
{
    free(ptr);
}

#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
//------------------------------------------------------------------------------
// グラフィックスシステム用メモリ割り当て・破棄関数
//------------------------------------------------------------------------------
static void* NvAllocateFunction(size_t size, size_t alignment, void* userPtr)
{
    NN_UNUSED(userPtr);
    return aligned_alloc(alignment, size);
}
static void NvFreeFunction(void* addr, void* userPtr)
{
    NN_UNUSED(userPtr);
    free(addr);
}
static void* NvReallocateFunction(void* addr, size_t newSize, void* userPtr)
{
    NN_UNUSED(userPtr);
    return realloc(addr, newSize);
}
#endif

void InitializeFs()
{
#if !NN_GFX_IS_TARGET_GX
    nn::Result result;

    result = nn::fs::MountHostRoot();
    NN_ASSERT( result.IsSuccess() );

    const size_t FS_MAX_PATH = 512;
    char contentsPath[ FS_MAX_PATH ];
    char** argv = nnt::GetHostArgv();
    char* path;

    NN_ASSERT_NOT_NULL( argv );
    NN_ASSERT_NOT_NULL( argv[0] );
    NN_ASSERT_GREATER( FS_MAX_PATH, std::strlen( argv[ 0 ] ) );

    // Find the root of the SDK
    std::strcpy(contentsPath, argv[0]);
    path = std::strstr(contentsPath, "\\Tests\\Outputs");
    NN_ASSERT_NOT_NULL(path);
    *path = '\0';
    strcat(contentsPath, "\\Externals\\TestBinaries\\Gfx\\Prebuilt\\testGfx_RenderToTexture");
#if NN_GFX_IS_TARGET_NVN
#if defined(NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON)
    std::strcat(contentsPath, "\\NX");
#else
    std::strcat(contentsPath, "\\NXWin32");
#endif
#elif NN_GFX_IS_TARGET_GL
    std::strcat(contentsPath, "\\Generic");
#elif NN_GFX_IS_TARGET_GX
    std::strcat(contentsPath, "\\Cafe");
#endif
    result = nn::fs::MountHost( "Contents", contentsPath );

    NN_ASSERT( result.IsSuccess() );
    NN_UNUSED( result );
#endif
}

#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) || defined ( NN_BUILD_CONFIG_OS_SUPPORTS_WIN32 )
//-----------------------------------------------------------------------------
// nninitStartup() is invoked before calling nnMain().
//
extern "C" void nninitStartup()
{
#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON )
#if defined( NN_BUILD_CONFIG_ADDRESS_SUPPORTS_32 )
    const size_t MallocMemorySize = 256 * 1024 * 1024;
#else
    const size_t MallocMemorySize = 512 * 1024 * 1024;
#endif

    nn::Result result = nn::os::SetMemoryHeapSize(2 * MallocMemorySize);
    uintptr_t address;
    result = nn::os::AllocateMemoryBlock(&address, MallocMemorySize);
    NN_ASSERT(result.IsSuccess());
    nn::init::InitializeAllocator(reinterpret_cast<void*>(address), MallocMemorySize);
#endif

    nn::fs::SetAllocator(Allocate, Deallocate);
}
#endif

//
//  Main Function
//  メイン関数です。
//
extern "C" void nnMain()
{
#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON)  && defined(NN_BUILD_CONFIG_SPEC_NX)
    const size_t GraphicsSystemMemorySize = 8 * 1024 * 1024;
    nv::SetGraphicsAllocator(NvAllocateFunction, NvFreeFunction, NvReallocateFunction, NULL);
    nv::SetGraphicsDevtoolsAllocator(NvAllocateFunction, NvFreeFunction, NvReallocateFunction, NULL);
    nv::InitializeGraphics(Allocate(GraphicsSystemMemorySize), GraphicsSystemMemorySize);
#endif

    nn::vi::Initialize();
    InitializeLayer();

    nn::gfx::Initialize();

    InitDevice();

    // ライブラリを初期化
    InitializeFs();

    // メモリ
    {
        nn::gfx::MemoryPoolInfo meminfo;
        meminfo.SetDefault();
        meminfo.SetMemoryPoolProperty( nn::gfx::MemoryPoolProperty_CpuUncached | nn::gfx::MemoryPoolProperty_GpuUncached );

        size_t alignment = nn::gfx::MemoryPool::GetPoolMemoryAlignment( &g_Device, meminfo );
        g_memPoolUnalignedBase = static_cast< uint8_t* >( malloc( s_poolSize + alignment ) );
        g_memPoolBase = reinterpret_cast< uint8_t* >( nn::util::BytePtr( g_memPoolUnalignedBase ).AlignUp( alignment ).Get() );
        meminfo.SetPoolMemory(g_memPoolBase, s_poolSize);

        g_memPool.Initialize(&g_Device, meminfo);

        // Setup the invisible pool for textures and swap buffers
        meminfo.SetMemoryPoolProperty( nn::gfx::MemoryPoolProperty_Compressible | nn::gfx::MemoryPoolProperty_CpuInvisible | nn::gfx::MemoryPoolProperty_GpuCached );

        g_memPoolUnalignedBaseInvisible = static_cast<uint8_t*>(malloc(s_poolSizeInvisible + alignment));
        g_memPoolBaseInvisible = reinterpret_cast< uint8_t* >( nn::util::BytePtr( g_memPoolUnalignedBaseInvisible ).AlignUp( alignment ).Get() );
        meminfo.SetPoolMemory(g_memPoolBaseInvisible, s_poolSizeInvisible);

        g_memPoolInvisible.Initialize(&g_Device, meminfo);
    }

    InitSwapChain();
    InitQueue();
    InitCommandBuffer(&g_commandBuffer);
    InitCommandBuffer(&g_commandBuffer2);
    InitViewport();
    InitShader();
    InitDescriptorPools();
    g_slotMat = g_pShader->GetInterfaceSlot(nn::gfx::ShaderStage_Pixel, nn::gfx::ShaderInterfaceType_ConstantBuffer, "Mat");
    g_slotTex = g_pShader->GetInterfaceSlot(nn::gfx::ShaderStage_Pixel, nn::gfx::ShaderInterfaceType_Sampler, "tex");
    if (NN_STATIC_CONDITION(g_IsPipelineMode))
    {
        InitPipeline(&g_pipeline1);
        InitPipeline(&g_pipeline2);
    }
    else
    {
        InitRasterizerState();
        InitBlendState();
        InitDepthStencilState();
        InitVertexState();
    }

    InitVertexBuffer(&g_triVertexBuffer, triVertices, 3, &g_triVertexBufferView);
    InitVertexBuffer(&g_quadVertexBuffer, quadVertices, 4, &g_quadVertexBufferView);
    InitIndexBuffer();
    InitColorBuffer(&g_colorBuffer1, &g_colorTextureView1, &g_colorBufferView1);
    InitColorBuffer(&g_colorBuffer2, &g_colorTextureView2, &g_colorBufferView2);

    InitDepthBuffer();
    InitDepthBufferView();
    InitConstantBuffer();
    InitTextureBuffer();
    InitTextureBufferView();
    InitSampler();

    // Register the texture views
    g_TextureDescriptorPool.BeginUpdate();
    g_TextureDescriptorPool.SetTextureView(g_BaseTextureViewPoolSlot + 0, &g_textureView);
    g_TextureDescriptorPool.SetTextureView(g_BaseTextureViewPoolSlot + 1, &g_colorTextureView1);
    g_TextureDescriptorPool.SetTextureView(g_BaseTextureViewPoolSlot + 2, &g_colorTextureView2);
    g_TextureDescriptorPool.EndUpdate();

    g_TextureDescriptorPool.GetDescriptorSlot(&g_textureSlot, g_BaseTextureViewPoolSlot + 0);
    g_TextureDescriptorPool.GetDescriptorSlot(&g_colorTextureSlot1, g_BaseTextureViewPoolSlot + 1);

    {
        nn::gfx::Fence::InfoType info;
        info.SetDefault();
        g_fence.Initialize(&g_Device, info);
        g_fence2.Initialize(&g_Device, info);
    }

    BuildCommandBuffer();
    BuildCommandBuffer2();
    InitializePreambleCommandBuffers();

    // 毎フレームのレンダリング
    for (int frame = 0; frame < 200; ++frame)
    {
        // 定数バッファを更新
        float* pConstantBuffer = g_constantBuffer.Map< float >();
        {
            float varColor = (frame & 0xFF) * (1.0f / 256.0f);
            pConstantBuffer[0] = 0.0f;
            pConstantBuffer[1] = 0.0f;
            pConstantBuffer[2] = varColor * 0.5f;
            pConstantBuffer[3] = 0.0f;
        }
#ifdef CAFE
        GX2EndianSwap( pConstantBuffer, sizeof( float ) * 4 );
#endif // CAFE
        g_constantBuffer.FlushMappedRange( 0, sizeof( float ) * 4 );
        g_constantBuffer.Unmap();

        // コマンドの実行
        g_Queue.ExecuteCommand(&g_commandBuffer, &g_fence);
        g_Queue.Flush();

        g_Queue.ExecuteCommand(&g_PreambleCommandBuffers[frame % TOTAL_SWAP_BUFFERS], NULL);
        g_Queue.ExecuteCommand(&g_commandBuffer2, &g_fence2);
        g_Queue.Flush();


        // FIXME: This sequence forces an acquire flag to be set
        nn::gfx::ColorTargetView* pColorTargetView = g_SwapChain.AcquireNextScanBufferView();
        g_Queue.CopyToScanBuffer(&g_SwapChain, pColorTargetView);

#ifdef WIN32
        MSG msg;
        while (PeekMessageA(&msg, nullptr, 0, 0, PM_REMOVE))
        {
            if (msg.message == WM_QUIT)
            {
                Cleanup();
                return;
            }
            TranslateMessage(&msg);
            DispatchMessageA(&msg);
        }

#endif
        g_fence.Sync(nn::TimeSpan::FromNanoSeconds(1000 * 1000 * 1000 / 60));
        g_Queue.Present(&g_SwapChain, 1);
    }

    Cleanup();

    nn::vi::DestroyLayer(g_pLayer);
    nn::vi::CloseDisplay(g_pDisplay);
} //NOLINT(impl/function_size)

