﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdio>
#include <cstring>
#include <cmath>

#include <gfx/demo.h>

#ifdef CAFE
#include <cafe/gx2ut.h>
#endif

#include <nnt.h>
#include <nnt/nnt_Argument.h>

static const int NUM_FORMATS = 4;

// ----- Surface Information

static const int RENDERTARGET_WIDTH = 120;
static const int RENDERTARGET_HEIGHT = 80;

#define SURFACE_WIDTH  (DEMOColorBufferInfo.GetWidth())
#define SURFACE_HEIGHT  (DEMOColorBufferInfo.GetHeight())

// ----- Shader information
static int g_ShaderFileIdx = 0;
static const char * const SHADER_FILE_1ST[] =
{
    "shaders/renderToDepth/transform",
    "shaders/renderToDepth/transformHlslcc",
};
static const char * const SHADER_FILE_2ND[] =
{
    "shaders/renderToDepth/texture2D",
    "shaders/renderToDepth/texture2DHlslcc",
};
static const char * const SHADER_FILE_3RD[] =
{
    "shaders/renderToDepth/texture2Duint",
    "shaders/renderToDepth/texture2DuintHlslcc",
};


typedef struct _ImageData
{
    nn::gfx::Texture texture;
    nn::gfx::TextureView depthTextureView;
    nn::gfx::DescriptorSlot depthTextureSlot;
    nn::gfx::TextureView stencilTextureView;
    nn::gfx::DescriptorSlot stencilTextureSlot;
    nn::gfx::DepthStencilView depthView;
    DEMOGfxMemPool* pPool;
    bool hasStencilView;
} ImageData;

typedef struct _FirstPassUniformStatic
{
    Mtx44 viewMtx;
    Mtx44 projMtx;
} FirstPassUniformStatic;

typedef struct _FirstPassUniformDynamic
{
    Mtx44 modelMtx;
} FirstPassUniformDynamic;

static struct _FirstPass
{
    DEMOGfxPipeline pipeline;

    nn::gfx::ViewportScissorState viewportScissor;
    void* pViewportScissorData;

    // Vertex Buffers
    int posLoc;
    int clrLoc;

    // Constant Buffers
    int uniformStaticBlockLoc;
    DEMOGfxBuffer uniformStaticBuffer;
    int uniformDynamicBlockLoc;
    DEMOGfxBuffer uniformDynamicBuffer[NUM_FORMATS];

    // -- Initial Data -- //
    int numAttrib;
    int numIndices;
    f32 posInitData[ 9 ];
    f32 clrInitData[ 9 ];
    u32 idxInitData[ 3 ];

    // -- Attribute Buffers -- //
    DEMOGfxBuffer positionBuffer;
    DEMOGfxBuffer colorBuffer;
    DEMOGfxBuffer indexBuffer;

    // -- Texture Setting -- //
    ImageData *pMyDepthBuffer;
} FirstPass;

static struct _SecondPass
{
    DEMOGfxPipeline pipeline;
    DEMOGfxPipeline pipeline2;

    // -- Shader Location -- //
    int posLoc;
    int texcoordLoc;
    int textureLoc;
    int textureLoc2;
    int offsetLoc;

    // -- Initial Data -- //
    int numAttrib;
    int numIndices;
    f32 posInitData[ 12 ];
    f32 texInitData[ 8 ];
    u32 idxInitData[ 4 ];

    // -- Attribute Buffers -- //
    DEMOGfxBuffer positionBuffer;
    DEMOGfxBuffer textureCoordinateBuffer;
    DEMOGfxBuffer indexBuffer;

    // -- Texture Setting -- //
    ImageData *pMyTextureBuffer;
    nn::gfx::Sampler mySampler;
    nn::gfx::DescriptorSlot mySamplerSlot;

    nn::gfx::ViewportScissorState depthViewports[NUM_FORMATS];
    nn::gfx::ViewportScissorState stencilViewports[NUM_FORMATS];
    void* pDepthViewportsData[NUM_FORMATS];
    void* pStencilViewportsData[NUM_FORMATS];

} SecondPass;

static ImageData DepthBuffers[ NUM_FORMATS ];

static nn::gfx::ImageFormat DepthFormats[ NUM_FORMATS ] = {
    nn::gfx::ImageFormat_D16_Unorm,
    nn::gfx::ImageFormat_D32_Float,
    nn::gfx::ImageFormat_D24_Unorm_S8_Uint,
    nn::gfx::ImageFormat_D32_Float_S8_Uint_X24
};

static f32 stvalue = 0.0f;
static u32 frame = 0;

static float rotateY;

// Prototype
static int SceneInit();
static int SceneDraw();
static void PrintInfo();

// The init function for the rendering portions of this app
static int SceneInit()
{
    rotateY = 0.0f;

    // ----------------- //
    // First pass setup
    // ----------------- //

    // geometry setup
    FirstPass.posInitData[ 0 ] = -0.4f; FirstPass.posInitData[ 1 ] = -0.4f;
    FirstPass.posInitData[ 2 ] = 0.0f; FirstPass.posInitData[ 3 ] = 0.4f;
    FirstPass.posInitData[ 4 ] = -0.4f; FirstPass.posInitData[ 5 ] = 0.0f;
    FirstPass.posInitData[ 6 ] = 0.0f; FirstPass.posInitData[ 7 ] = 0.4f;
    FirstPass.posInitData[ 8 ] = 0.0f;

    FirstPass.clrInitData[ 0 ] = 1.0f; FirstPass.clrInitData[ 1 ] = 0.0f;
    FirstPass.clrInitData[ 2 ] = 0.0f; FirstPass.clrInitData[ 3 ] = 0.0f;
    FirstPass.clrInitData[ 4 ] = 1.0f; FirstPass.clrInitData[ 5 ] = 0.0f;
    FirstPass.clrInitData[ 6 ] = 0.0f; FirstPass.clrInitData[ 7 ] = 0.0f;
    FirstPass.clrInitData[ 8 ] = 1.0f;

    FirstPass.idxInitData[ 0 ] = 0;
    FirstPass.idxInitData[ 1 ] = 1;
    FirstPass.idxInitData[ 2 ] = 2;

    FirstPass.numAttrib = 2;
    FirstPass.numIndices = 3;

    // Set the pipeline defaults
    FirstPass.pipeline.SetDefaults();

    DEMOGfxLoadShadersFromFile(&FirstPass.pipeline.shaders, 0, SHADER_FILE_1ST[g_ShaderFileIdx]);

    // Attribute Location Lookup
    FirstPass.posLoc = FirstPass.pipeline.shaders.GetInterfaceSlot( nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_Input, "a_position" );
    FirstPass.clrLoc = FirstPass.pipeline.shaders.GetInterfaceSlot( nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_Input, "a_color" );

    // Uniform Location Lookup
    FirstPass.uniformStaticBlockLoc = FirstPass.pipeline.shaders.GetInterfaceSlot( nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_ConstantBuffer, "ub_static" );
    FirstPass.uniformDynamicBlockLoc = FirstPass.pipeline.shaders.GetInterfaceSlot( nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_ConstantBuffer, "ub_dynamic" );

    // Attribute Format Setup
    DEMOGfxInitShaderAttribute( &FirstPass.pipeline.shaders, "a_position", FirstPass.posLoc, 0, nn::gfx::AttributeFormat_32_32_32_Float );
    DEMOGfxInitShaderAttribute( &FirstPass.pipeline.shaders, "a_color", FirstPass.clrLoc, 0, nn::gfx::AttributeFormat_32_32_32_Float );
    DEMOGfxInitShaderVertexBuffer( &FirstPass.pipeline.shaders, FirstPass.posLoc, sizeof( float ) * 3, 0 );
    DEMOGfxInitShaderVertexBuffer( &FirstPass.pipeline.shaders, FirstPass.clrLoc, sizeof( float ) * 3, 0 );

    // Vertex position buffer
    FirstPass.positionBuffer.Initialize( sizeof( FirstPass.posInitData ), FirstPass.posInitData,
        nn::gfx::GpuAccess_VertexBuffer, 0 );

    // Vertex color buffer
    FirstPass.colorBuffer.Initialize( sizeof( FirstPass.clrInitData ), FirstPass.clrInitData,
        nn::gfx::GpuAccess_VertexBuffer, 0 );

    // Index buffer
    FirstPass.indexBuffer.Initialize( sizeof( FirstPass.idxInitData ), FirstPass.idxInitData,
        nn::gfx::GpuAccess_IndexBuffer, 0 );

    // Setup a static uniform block
    {
        float projmat[ 4 ][ 4 ] = { { 1.0f, 0.0f, 0.0f, 0.0f },
            { 0.0f, 1.0f, 0.0f, 0.0f },
            { 0.0f, 0.0f, 1.0f, 0.0f },
            { 0.0f, 0.0f, 0.0f, 1.0f } };
        float viewmat[ 4 ][ 4 ] = { { 1.0f, 0.0f, 0.0f, 0.0f },
            { 0.0f, 1.0f, 0.0f, 0.0f },
            { 0.0f, 0.0f, 1.0f, 0.0f },
            { 0.0f, 0.0f, 0.0f, 1.0f } };

        FirstPass.uniformStaticBuffer.Initialize( sizeof( FirstPassUniformStatic ), NULL,
            nn::gfx::GpuAccess_ConstantBuffer, 0 );
        FirstPassUniformStatic* data = FirstPass.uniformStaticBuffer.Map< FirstPassUniformStatic >();

        memcpy( &data->viewMtx, viewmat, sizeof( viewmat ) );
        memcpy( &data->projMtx, projmat, sizeof( projmat ) );
#ifdef CAFE
        GX2EndianSwap( data, sizeof( *data ) );
#endif
        FirstPass.uniformStaticBuffer.Unmap();
    }

    // Setup the dynamic uniforms
    for ( int i = 0; i < 2; i++ )
    {
        FirstPass.uniformDynamicBuffer[ i ].Initialize( sizeof( FirstPassUniformDynamic ), NULL,
            nn::gfx::GpuAccess_ConstantBuffer, 0 );
    }

    // Initialize depth & texture buffers
    for ( u32 i = 0; i < NUM_FORMATS; i++ )
    {

        // Setup the Depth buffer with a depth view
        DEMOGfxSetupTextureBuffer( &DepthBuffers[ i ].texture, &DepthBuffers[ i ].depthTextureView, &DepthBuffers[ i ].depthTextureSlot, NULL,
            &DepthBuffers[ i ].depthView, &DepthBuffers[ i ].pPool,
            RENDERTARGET_WIDTH, RENDERTARGET_HEIGHT, 1, 1, nn::gfx::ImageDimension_2d,
            DepthFormats[ i ],
            nn::gfx::DepthStencilFetchMode_DepthComponent, 0 );

        // Setup a Stencil view if necessary
        if ( DepthFormats[ i ] == nn::gfx::ImageFormat_D24_Unorm_S8_Uint ||
             DepthFormats[ i ] == nn::gfx::ImageFormat_D32_Float_S8_Uint_X24 )
        {
            DepthBuffers[ i ].hasStencilView = true;
            DEMOGfxSetupTextureView( &DepthBuffers[ i ].texture, &DepthBuffers[ i ].stencilTextureView, &DepthBuffers[ i ].stencilTextureSlot,
                nn::gfx::ImageDimension_2d, DepthFormats[i], nn::gfx::DepthStencilFetchMode_StencilIndex );
        }
        else
        {
            DepthBuffers[ i ].hasStencilView = false;
        }
    }

    // Set up the sampler object
    DEMOGfxInitSampler( &SecondPass.mySampler, &SecondPass.mySamplerSlot, nn::gfx::TextureAddressMode_ClampToEdge,
        nn::gfx::FilterMode_MinPoint_MagPoint_MipPoint, nn::gfx::ComparisonFunction_Always );

    // Finalize the Pipeline
    FirstPass.pipeline.blendTargetStateCount = 1;
    FirstPass.pipeline.colorTargetStateCount = 1;
    FirstPass.pipeline.blendTargetStateInfoArray[ 0 ].SetDefault();
    FirstPass.pipeline.blendTargetStateInfoArray[ 0 ].SetChannelMask( 0 ); // Disable color writes
    FirstPass.pipeline.colorTargetStateInfoArray[ 0 ].SetDefault();

    FirstPass.pipeline.depthStencilStateInfo.SetStencilTestEnabled( true );
    FirstPass.pipeline.depthStencilStateInfo.EditFrontStencilStateInfo().SetComparisonFunction( nn::gfx::ComparisonFunction_Always );
    FirstPass.pipeline.depthStencilStateInfo.EditFrontStencilStateInfo().SetDepthPassOperation( nn::gfx::StencilOperation_Replace );
    FirstPass.pipeline.depthStencilStateInfo.EditFrontStencilStateInfo().SetDepthFailOperation( nn::gfx::StencilOperation_Replace );
    FirstPass.pipeline.depthStencilStateInfo.EditFrontStencilStateInfo().SetStencilFailOperation( nn::gfx::StencilOperation_Replace );
    FirstPass.pipeline.depthStencilStateInfo.EditFrontStencilStateInfo().SetStencilRef( 0xFF );
    FirstPass.pipeline.depthStencilStateInfo.EditBackStencilStateInfo().SetComparisonFunction( nn::gfx::ComparisonFunction_Always );
    FirstPass.pipeline.depthStencilStateInfo.EditBackStencilStateInfo().SetDepthPassOperation( nn::gfx::StencilOperation_Replace );
    FirstPass.pipeline.depthStencilStateInfo.EditBackStencilStateInfo().SetDepthFailOperation( nn::gfx::StencilOperation_Replace );
    FirstPass.pipeline.depthStencilStateInfo.EditBackStencilStateInfo().SetStencilFailOperation( nn::gfx::StencilOperation_Replace );
    FirstPass.pipeline.depthStencilStateInfo.EditBackStencilStateInfo().SetStencilRef( 0xFF );
    FirstPass.pipeline.depthStencilStateInfo.SetStencilReadMask( 0xFF );
    FirstPass.pipeline.depthStencilStateInfo.SetStencilWriteMask( 0xFF );

    FirstPass.pipeline.Initialize( &DEMODevice );

    // Setup the viewport
    DEMOGfxSetViewportScissorState( &FirstPass.viewportScissor, &FirstPass.pViewportScissorData,
        0.0f, 0.0f,
        static_cast< float >( RENDERTARGET_WIDTH ),
        static_cast< float >( RENDERTARGET_HEIGHT ),
        0.0f, 1.0f,
        static_cast< float >( RENDERTARGET_HEIGHT ),
        true );

    // ------------------ //
    // Second pass setup
    // ------------------ //

    SecondPass.posInitData[ 0 ] = -0.6f; SecondPass.posInitData[ 1 ] = -0.8f;
    SecondPass.posInitData[ 2 ] = 0.0f; SecondPass.posInitData[ 3 ] = 0.6f;
    SecondPass.posInitData[ 4 ] = -0.8f; SecondPass.posInitData[ 5 ] = 0.0f;
    SecondPass.posInitData[ 6 ] = 0.6f; SecondPass.posInitData[ 7 ] = 0.8f;
    SecondPass.posInitData[ 8 ] = 0.0f; SecondPass.posInitData[ 9 ] = -0.6f;
    SecondPass.posInitData[ 10 ] = 0.8f; SecondPass.posInitData[ 11 ] = 0.0f;

    SecondPass.texInitData[ 0 ] = 0.0f; SecondPass.texInitData[ 1 ] = 1.0f;
    SecondPass.texInitData[ 2 ] = 1.0f; SecondPass.texInitData[ 3 ] = 1.0f;
    SecondPass.texInitData[ 4 ] = 1.0f; SecondPass.texInitData[ 5 ] = 0.0f;
    SecondPass.texInitData[ 6 ] = 0.0f; SecondPass.texInitData[ 7 ] = 0.0f;

    SecondPass.idxInitData[ 0 ] = 0; SecondPass.idxInitData[ 1 ] = 1;
    SecondPass.idxInitData[ 2 ] = 3; SecondPass.idxInitData[ 3 ] = 2;
    SecondPass.numAttrib = 2;
    SecondPass.numIndices = 4;

    // Initialize the pipelines
    SecondPass.pipeline.SetDefaults();
    SecondPass.pipeline2.SetDefaults();

    DEMOGfxLoadShadersFromFile( &SecondPass.pipeline.shaders, 0, SHADER_FILE_2ND[g_ShaderFileIdx]);

    // Attribute Location Lookup
    SecondPass.posLoc = SecondPass.pipeline.shaders.GetInterfaceSlot( nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_Input, "a_position" );
    SecondPass.texcoordLoc = SecondPass.pipeline.shaders.GetInterfaceSlot( nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_Input, "a_texCoord" );
    SecondPass.textureLoc = SecondPass.pipeline.shaders.GetInterfaceSlot( nn::gfx::ShaderStage_Pixel, nn::gfx::ShaderInterfaceType_Sampler, "s_texture" );

    // Uniform Location lookup
    SecondPass.offsetLoc = SecondPass.pipeline.shaders.GetInterfaceSlot( nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_ConstantBuffer, "u_offset" );

    // Attribute Format Setup
    DEMOGfxInitShaderAttribute( &SecondPass.pipeline.shaders, "a_position", SecondPass.posLoc, 0, nn::gfx::AttributeFormat_32_32_32_Float );
    DEMOGfxInitShaderAttribute( &SecondPass.pipeline.shaders, "a_texCoord", SecondPass.texcoordLoc, 0, nn::gfx::AttributeFormat_32_32_Float );
    DEMOGfxInitShaderVertexBuffer( &SecondPass.pipeline.shaders, SecondPass.posLoc, sizeof( float ) * 3, 0 );
    DEMOGfxInitShaderVertexBuffer( &SecondPass.pipeline.shaders, SecondPass.texcoordLoc, sizeof( float ) * 2, 0 );

    SecondPass.pipeline.blendTargetStateCount = 1;
    SecondPass.pipeline.colorTargetStateCount = 1;
    SecondPass.pipeline.blendTargetStateInfoArray[ 0 ].SetDefault();
    SecondPass.pipeline.colorTargetStateInfoArray[ 0 ].SetDefault();
    SecondPass.pipeline.colorTargetStateInfoArray[ 0 ].SetFormat( DEMOColorBufferInfo.GetImageFormat() );

    SecondPass.pipeline.Initialize( &DEMODevice );

    // Vertex position buffer
    SecondPass.positionBuffer.Initialize( sizeof( SecondPass.posInitData ), SecondPass.posInitData,
        nn::gfx::GpuAccess_VertexBuffer, 0 );

    // Vertex texcoord buffer
    SecondPass.textureCoordinateBuffer.Initialize( sizeof( SecondPass.texInitData ), SecondPass.texInitData,
        nn::gfx::GpuAccess_VertexBuffer, 0 );

    // Index buffer
    SecondPass.indexBuffer.Initialize( sizeof( SecondPass.idxInitData ), SecondPass.idxInitData,
        nn::gfx::GpuAccess_IndexBuffer, 0 );

    // ------------------ //
    // Third part setup
    // ------------------ //
    DEMOGfxLoadShadersFromFile(&SecondPass.pipeline2.shaders, 0, SHADER_FILE_3RD[g_ShaderFileIdx]);

    // Attribute Format Setup
    DEMOGfxInitShaderAttribute( &SecondPass.pipeline2.shaders, "a_position", SecondPass.posLoc, 0, nn::gfx::AttributeFormat_32_32_32_Float );
    DEMOGfxInitShaderAttribute( &SecondPass.pipeline2.shaders, "a_texCoord", SecondPass.texcoordLoc, 0, nn::gfx::AttributeFormat_32_32_Float );
    DEMOGfxInitShaderVertexBuffer( &SecondPass.pipeline2.shaders, SecondPass.posLoc, sizeof( float ) * 3, 0 );
    DEMOGfxInitShaderVertexBuffer( &SecondPass.pipeline2.shaders, SecondPass.texcoordLoc, sizeof( float ) * 2, 0 );

    SecondPass.pipeline2.blendTargetStateCount = 1;
    SecondPass.pipeline2.colorTargetStateCount = 1;
    SecondPass.pipeline2.blendTargetStateInfoArray[ 0 ].SetDefault();
    SecondPass.pipeline2.colorTargetStateInfoArray[ 0 ].SetDefault();
    SecondPass.pipeline2.colorTargetStateInfoArray[ 0 ].SetFormat( DEMOColorBufferInfo.GetImageFormat() );

    SecondPass.pipeline2.Initialize( &DEMODevice );

    // Grab the second texture location since it may differ
    SecondPass.textureLoc2 = SecondPass.pipeline2.shaders.GetInterfaceSlot( nn::gfx::ShaderStage_Pixel, nn::gfx::ShaderInterfaceType_Sampler, "s_texture" );

    // Setup the viewports for displaying

    for ( int i = 0; i < NUM_FORMATS; i++ )
    {
        float width = SURFACE_WIDTH / 3.0f;
        float height = SURFACE_HEIGHT / 2.0f;
        float xo;
        float yo;

        switch ( i )
        {
            case 0:
            {
                xo = float( SURFACE_WIDTH / 3 * 0 );
                yo = float( SURFACE_HEIGHT / 2 * 0 );
                break;
            }

            case 1:
            {
                xo = float( SURFACE_WIDTH / 3 * 0 );
                yo = float( SURFACE_HEIGHT / 2 * 1 );
                break;
            }
            case 2:
            {
                xo = float( SURFACE_WIDTH / 3 * 1 );
                yo = float( SURFACE_HEIGHT / 2 * 0 );
                break;
            }

            default:
            {
                xo = float( SURFACE_WIDTH / 3 * 1 );
                yo = float( SURFACE_HEIGHT / 2 * 1 );
                break;
            }
        }

        DEMOGfxSetViewportScissorState( &SecondPass.depthViewports[ i ],
            &SecondPass.pDepthViewportsData[ i ], xo, yo, width, height, 0.0f, 1.0f,
            static_cast< float >( SURFACE_HEIGHT ), false );

        switch ( i )
        {
            case 0:
            {
                xo = float( SURFACE_WIDTH / 3 * 2 );
                yo = float( SURFACE_HEIGHT / 2 * 0 );
                break; // not shown
            }

            case 1:
            {
                xo = float( SURFACE_WIDTH / 3 * 2 );
                yo = float( SURFACE_HEIGHT / 2 * 1 );
                break; // not shown
            }

            case 2:
            {
                xo = float( SURFACE_WIDTH / 3 * 2 );
                yo = float( SURFACE_HEIGHT / 2 * 0 );
                break;
            }

            default:
            case 3:
            {
                xo = float( SURFACE_WIDTH / 3 * 2 );
                yo = float( SURFACE_HEIGHT / 2 * 1 );
                break;
            }
        }

        DEMOGfxSetViewportScissorState( &SecondPass.stencilViewports[ i ],
            &SecondPass.pStencilViewportsData[ i ], xo, yo, width, height, 0.0f, 1.0f,
            static_cast< float >( SURFACE_HEIGHT ), false );
    }

    return 1;
} // NOLINT(impl/function_size)

// The draw function for the rendering portions of this app
static int SceneDraw()
{
    DEMOGfxBeforeRender();

    // Clear each depth-stencil buffer
    for ( u32 i = 0; i<NUM_FORMATS; i++ )
    {

        DEMOCommandBuffer.ClearDepthStencil( &DepthBuffers[ i ].depthView, 1.0f, static_cast< int >( stvalue ),
            nn::gfx::DepthStencilClearMode_DepthStencil, NULL );
    }

#if NN_GFX_IS_TARGET_GX
    GX2SetShaderMode( GX2_SHADER_MODE_UNIFORM_BLOCK );
#endif
    // ------------------------------- //
    //   1ST PASS (RENDERING TEXTURE)
    // ------------------------------- //
    DEMOCommandBuffer.SetViewportScissorState( &FirstPass.viewportScissor );

    DEMOCommandBuffer.SetPipeline( &FirstPass.pipeline.pipeline );

    // Bind position buffer
    DEMOCommandBuffer.SetVertexBuffer( FirstPass.posLoc, FirstPass.positionBuffer.gpuAddress, sizeof( float ) * 3, FirstPass.positionBuffer.size );

    // Bind color buffer
    DEMOCommandBuffer.SetVertexBuffer( FirstPass.clrLoc, FirstPass.colorBuffer.gpuAddress, sizeof( float ) * 3, FirstPass.colorBuffer.size );

    float modelmat[ 4 ][ 4 ] = { { 1.5f, 0.0f, 0.0f, -0.5f },
        { 0.0f, 1.5f, 0.0f, 0.0f },
        { 0.0f, 0.0f, 1.0f, 0.0f },
        { 0.0f, 0.0f, 0.0f, 1.0f } };
    DEMOCommandBuffer.SetConstantBuffer( FirstPass.uniformStaticBlockLoc, nn::gfx::ShaderStage_Vertex,
        FirstPass.uniformStaticBuffer.gpuAddress, FirstPass.uniformStaticBuffer.size );

    // Update the dynamic uniform blocks
    FirstPassUniformDynamic* pBuf = FirstPass.uniformDynamicBuffer[ 0 ].Map< FirstPassUniformDynamic >();
    modelmat[ 0 ][ 0 ] = cos( rotateY );  modelmat[ 2 ][ 0 ] = -sin( rotateY );
    modelmat[ 0 ][ 2 ] = sin( rotateY );  modelmat[ 2 ][ 2 ] = cos( rotateY );
    modelmat[ 0 ][ 3 ] = -0.5f;
#if NN_GFX_IS_TARGET_D3D
    // GLとD3DのZclipの範囲が異なるため、調整します。
    modelmat[ 2 ][ 3 ] = 0.5f;
#endif
    memcpy( pBuf, modelmat, sizeof( modelmat ) );
#ifdef CAFE
    GX2EndianSwap( pBuf, sizeof( *pBuf ) );
#endif
    FirstPass.uniformDynamicBuffer[ 0 ].Unmap();

    pBuf = FirstPass.uniformDynamicBuffer[ 1 ].Map< FirstPassUniformDynamic >();
    modelmat[ 0 ][ 0 ] = cos( -rotateY / 2.0f );  modelmat[ 2 ][ 0 ] = -sin( -rotateY / 2.0f );
    modelmat[ 0 ][ 2 ] = sin( -rotateY / 2.0f );  modelmat[ 2 ][ 2 ] = cos( -rotateY / 2.0f );
    modelmat[ 0 ][ 3 ] = 0.5f;
#if NN_GFX_IS_TARGET_D3D
    // GLとD3DのZclipの範囲が異なるため、調整します。
    modelmat[ 2 ][ 3 ] = 0.5f;
#endif
    memcpy( pBuf, modelmat, sizeof( modelmat ) );
#ifdef CAFE
    GX2EndianSwap( pBuf, sizeof( *pBuf ) );
#endif
    FirstPass.uniformDynamicBuffer[ 1 ].Unmap();

    // Draw to each depth-stencil buffer
    for ( u32 i = 0; i<NUM_FORMATS; i++ )
    {

        FirstPass.pMyDepthBuffer = &DepthBuffers[ i ];

        DEMOCommandBuffer.SetRenderTargets( 0, NULL, &FirstPass.pMyDepthBuffer->depthView );

        if ( frame < 3000 )
        {
            frame++;
            stvalue += 0.1f;
            if ( frame > 2550 ) stvalue = 0;
        }
        else
        {
            frame = 0;
        }

        // -----------------------
        // draw first triangle

        // Set Uniform Model Matrix
        DEMOCommandBuffer.SetConstantBuffer( FirstPass.uniformDynamicBlockLoc, nn::gfx::ShaderStage_Vertex,
            FirstPass.uniformDynamicBuffer[ 0 ].gpuAddress, FirstPass.uniformDynamicBuffer[ 0 ].size );

        DEMOCommandBuffer.DrawIndexed( nn::gfx::PrimitiveTopology_TriangleList, nn::gfx::IndexFormat_Uint32,
            FirstPass.indexBuffer.gpuAddress, FirstPass.numIndices, 0 );

        // -----------------------
        // draw second triangle

        // Set Uniform Model Matrix
        DEMOCommandBuffer.SetConstantBuffer( FirstPass.uniformDynamicBlockLoc, nn::gfx::ShaderStage_Vertex,
            FirstPass.uniformDynamicBuffer[ 1 ].gpuAddress, FirstPass.uniformDynamicBuffer[ 1 ].size );

        DEMOCommandBuffer.DrawIndexed( nn::gfx::PrimitiveTopology_TriangleList, nn::gfx::IndexFormat_Uint32,
            FirstPass.indexBuffer.gpuAddress, FirstPass.numIndices, 0 );

        DEMOCommandBuffer.FlushMemory( nn::gfx::GpuAccess_DepthStencil );
        DEMOCommandBuffer.InvalidateMemory( nn::gfx::GpuAccess_Texture );
    }

#ifdef CAFE
    // Need to expand HiZ
    GX2UTSetExpandDepthState( GX2_ENABLE );
    GX2UTExpandDepthBufferOp( reinterpret_cast< GX2DepthBuffer* >( &DepthBuffers[ 0 ].depthView.ToData()->gx2DepthBuffer ) );
    GX2UTSetExpandDepthState( GX2_DISABLE );

    GX2UTSetExpandDepthState( GX2_ENABLE );
    GX2UTExpandDepthBufferOp( reinterpret_cast< GX2DepthBuffer* >( &DepthBuffers[ 1 ].depthView.ToData()->gx2DepthBuffer ) );
    GX2UTSetExpandDepthState( GX2_DISABLE );

    GX2UTSetConvertDepthState( GX2_ENABLE );
    GX2UTConvertDepthBufferToTextureSurfaceOp(
        reinterpret_cast< GX2DepthBuffer* >( &(DepthBuffers[ 2 ].depthView.ToData()->gx2DepthBuffer) ),
        &(reinterpret_cast< GX2Texture* >( &(DepthBuffers[ 2 ].depthTextureView.ToData()->gx2Texture) )->surface), 0, 0 );
    GX2UTSetConvertDepthState( GX2_DISABLE );

    GX2UTSetConvertDepthState( GX2_ENABLE );
    GX2UTConvertDepthBufferToTextureSurfaceOp(
        reinterpret_cast< GX2DepthBuffer* >( &(DepthBuffers[ 3 ].depthView.ToData()->gx2DepthBuffer) ),
        &(reinterpret_cast< GX2Texture* >( &(DepthBuffers[ 3 ].depthTextureView.ToData()->gx2Texture) )->surface), 0, 0 );
    GX2UTSetConvertDepthState( GX2_DISABLE );

    DEMOCommandBuffer.FlushMemory( nn::gfx::GpuAccess_DepthStencil );
    DEMOCommandBuffer.InvalidateMemory( nn::gfx::GpuAccess_Texture );
#endif

    // ------------------------------- //
    //   2ND PASS (VISUALIZE TEXTURE)
    // ------------------------------- //

    nn::gfx::ColorTargetView* pCurrentScanBuffer = DEMOGetColorBufferView();
    DEMOCommandBuffer.ClearColor( pCurrentScanBuffer, 0.1f, 0.4f, 0.8f, 1.0f, NULL );
    DEMOCommandBuffer.ClearDepthStencil( &DEMODepthBufferView, 1.0f, 0, nn::gfx::DepthStencilClearMode_DepthStencil, NULL );

#if NN_GFX_IS_TARGET_GX
    GX2SetShaderMode( GX2_SHADER_MODE_UNIFORM_BLOCK );
#endif
    // Set to use regular demo color/depth buffers
    DEMOGfxSetDefaultRenderTarget();
    DEMOGfxSetDefaultViewportScissor();

    DEMOCommandBuffer.SetPipeline( &SecondPass.pipeline.pipeline );

    // Bind position buffer
    DEMOCommandBuffer.SetVertexBuffer( SecondPass.posLoc, SecondPass.positionBuffer.gpuAddress,
        sizeof( float ) * 3, SecondPass.positionBuffer.size );

    // Bind texcoord buffer
    DEMOCommandBuffer.SetVertexBuffer( SecondPass.texcoordLoc, SecondPass.textureCoordinateBuffer.gpuAddress,
        sizeof( float ) * 2, SecondPass.textureCoordinateBuffer.size );

    for ( u32 i = 0; i<NUM_FORMATS; i++ )
    {

        DEMOCommandBuffer.SetViewportScissorState( &SecondPass.depthViewports[ i ] );

        SecondPass.pMyTextureBuffer = &DepthBuffers[ i ];

        DEMOCommandBuffer.SetTextureAndSampler( SecondPass.textureLoc, nn::gfx::ShaderStage_Pixel,
            SecondPass.pMyTextureBuffer->depthTextureSlot, SecondPass.mySamplerSlot );


        // -----------------------
        // draw first quad
        DEMOCommandBuffer.DrawIndexed( nn::gfx::PrimitiveTopology_TriangleStrip, nn::gfx::IndexFormat_Uint32,
            SecondPass.indexBuffer.gpuAddress, SecondPass.numIndices, 0 );
    }

    // Setup the Stenciling pipeline
    DEMOCommandBuffer.SetPipeline( &SecondPass.pipeline2.pipeline );

    for ( u32 i = 0; i<NUM_FORMATS; i++ )
    {

        if ( !DepthBuffers[i].hasStencilView ) continue;

        DEMOCommandBuffer.SetViewportScissorState( &SecondPass.stencilViewports[ i ] );

        SecondPass.pMyTextureBuffer = &DepthBuffers[ i ];

        DEMOCommandBuffer.SetTextureAndSampler( SecondPass.textureLoc2, nn::gfx::ShaderStage_Pixel,
            SecondPass.pMyTextureBuffer->stencilTextureSlot, SecondPass.mySamplerSlot );

        // -----------------------
        // draw first quad
        DEMOCommandBuffer.DrawIndexed( nn::gfx::PrimitiveTopology_TriangleStrip, nn::gfx::IndexFormat_Uint32,
            SecondPass.indexBuffer.gpuAddress, SecondPass.numIndices, 0 );

    }

    PrintInfo();

    // -----------------------
    // DEMODoneRender
    DEMOGfxDoneRender();

    rotateY += 0.06f;

    return 1; // 0 makes it exit
} // NOLINT(impl/function_size)

// Print Informations
static void PrintInfo()
{
    // Set Demo Font state
    DEMOGfxSetDefaultViewportScissor();

    // Set Font Size
    DEMOFontSetGridSize( 120, 36 );

    DEMOFontPrintf( 8, 16.25, "GX2_SURFACE_FORMAT_TCD_R16_UNORM" );
    DEMOFontPrintf( 8, 18.75, "GX2_SURFACE_FORMAT_TCD_R32_FLOAT" );

    DEMOFontPrintf( 45, 16.25, "GX2_SURFACE_FORMAT_T_R24_UNORM_X8" );
    DEMOFontPrintf( 45, 18.75, "GX2_SURFACE_FORMAT_T_R32_FLOAT_X8_X24" );

    DEMOFontPrintf( 85, 16.25, "GX2_SURFACE_FORMAT_T_X24_G8_UINT" );
    DEMOFontPrintf( 85, 18.75, "GX2_SURFACE_FORMAT_T_X32_G8_UINT_X24" );
}

//extern "C" void nnMain()
TEST(GfxRenderToDepth, Run)
{
    int argc = nnt::GetHostArgc();
    char** argv = nnt::GetHostArgv();

    DEMOInit();
    DEMOTestInit( argc, argv );
    DEMOGfxInit( argc, argv );
    DEMOTestIsUseHlslccGlsl() ? g_ShaderFileIdx = 1 : g_ShaderFileIdx = 0;

    // Initialize Demo Font
    DEMOFontInit();

    SceneInit();

    stvalue = 0.0f;
    frame = 0;

    while ( DEMOIsRunning() )
    {
        SceneDraw();
    }

    // Free the viewport
    FirstPass.viewportScissor.Finalize( &DEMODevice );
    DEMOGfxFreeMEM2( FirstPass.pViewportScissorData );
    for ( int i = 0; i < NUM_FORMATS; i++ )
    {
        SecondPass.depthViewports[ i ].Finalize( &DEMODevice );
        DEMOGfxFreeMEM2( SecondPass.pDepthViewportsData[ i ] );

        SecondPass.stencilViewports[ i ].Finalize( &DEMODevice );
        DEMOGfxFreeMEM2( SecondPass.pStencilViewportsData[ i ] );
    }

    // Free the pipelines
    FirstPass.pipeline.Finalize( &DEMODevice );
    SecondPass.pipeline.Finalize( &DEMODevice );
    SecondPass.pipeline2.Finalize( &DEMODevice );

    // Free the vertex/index buffers
    FirstPass.positionBuffer.Finalize();
    FirstPass.colorBuffer.Finalize();
    FirstPass.indexBuffer.Finalize();
    SecondPass.positionBuffer.Finalize();
    SecondPass.textureCoordinateBuffer.Finalize();
    SecondPass.indexBuffer.Finalize();

    // Free the uniforms
    FirstPass.uniformStaticBuffer.Finalize();
    FirstPass.uniformDynamicBuffer[ 0 ].Finalize();
    FirstPass.uniformDynamicBuffer[ 1 ].Finalize();

    // Free the textures
    for ( int i = 0; i<NUM_FORMATS; i++ )
    {
        DepthBuffers[ i ].depthView.Finalize( &DEMODevice );
        if ( DepthBuffers[ i ].hasStencilView )
        {
            DepthBuffers[ i ].stencilTextureView.Finalize( &DEMODevice );
        }
        DepthBuffers[ i ].depthTextureView.Finalize( &DEMODevice );
        DepthBuffers[ i ].texture.Finalize( &DEMODevice );
        DepthBuffers[ i ].pPool->Finalize();
    }

    // Release the sampler
    SecondPass.mySampler.Finalize( &DEMODevice );

    DEMOFontShutdown();
    DEMOTestShutdown();
    DEMOGfxShutdown();
    DEMOShutdown();

    SUCCEED();
}
