﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/////////////////////////////////////////////////////////////////////////////
//
// latticeDL.cpp
//
// Draws like Wii's smp-onetri demo using display lists.
//
//////////////////////////////////////////////////////////////////////////////

#include <cstdio>
#include <cstring>
#include <cmath>

#include <gfx/demo.h>

#include <nnt.h>
#include <nnt/nnt_Argument.h>

static const char * const SHADER_FILE = "shaders/latticeDl/transform";

DEMOGfxPipeline g_AppPipeline;


// Constant Buffers
static int staticMatrixLoc;
static int dynamicMatrixLoc;

//---------------------------------------------------------------------------*
//  Model Data
//---------------------------------------------------------------------------*/
static const int STRUT_LN = 130;     /// long side of strut
static const int STRUT_SD = 4;       /// short side of strut
static const int JOINT_SD = 10;      /// joint is a cube

/// Vertices
static f32 modelVtxs[] =
{
    //      x             y             z          No:Id
    -STRUT_SD, STRUT_SD, -STRUT_SD,   // 0:0
    //   r          g          b          a
    0.5f, 0.5f, 0.42f, 1.0f,
    STRUT_SD, STRUT_SD, -STRUT_SD,   // 1:1
    0.5f, 0.5f, 0.42f, 1.0f,
    STRUT_SD, STRUT_SD, -STRUT_LN,   // 2:2
    0.5f, 0.5f, 0.42f, 1.0f,
    -STRUT_SD, STRUT_SD, -STRUT_SD,   // 0:3
    0.5f, 0.5f, 0.42f, 1.0f,
    STRUT_SD, STRUT_SD, -STRUT_LN,   // 2:4
    0.5f, 0.5f, 0.42f, 1.0f,
    -STRUT_SD, STRUT_SD, -STRUT_LN,   // 3:5
    0.5f, 0.5f, 0.42f, 1.0f,

    STRUT_SD, STRUT_SD, -STRUT_SD,   // 1:6
    0.3f, 0.3f, 0.3f, 1.0f,
    STRUT_SD, -STRUT_SD, -STRUT_SD,   // 4:7
    0.3f, 0.3f, 0.3f, 1.0f,
    STRUT_SD, -STRUT_SD, -STRUT_LN,   // 5:8
    0.3f, 0.3f, 0.3f, 1.0f,
    STRUT_SD, STRUT_SD, -STRUT_SD,   // 1:9
    0.3f, 0.3f, 0.3f, 1.0f,
    STRUT_SD, -STRUT_SD, -STRUT_LN,   // 5:10
    0.3f, 0.3f, 0.3f, 1.0f,
    STRUT_SD, STRUT_SD, -STRUT_LN,   // 4:11
    0.3f, 0.3f, 0.3f, 1.0f,

    STRUT_SD, STRUT_SD, -STRUT_SD,   // 1:12
    0.5f, 0.5f, 0.42f, 1.0f,
    STRUT_SD, STRUT_SD, STRUT_SD,   // 6:13
    0.5f, 0.5f, 0.42f, 1.0f,
    STRUT_LN, STRUT_SD, STRUT_SD,   // 7:14
    0.5f, 0.5f, 0.42f, 1.0f,
    STRUT_SD, STRUT_SD, -STRUT_SD,   // 1:15
    0.5f, 0.5f, 0.42f, 1.0f,
    STRUT_LN, STRUT_SD, STRUT_SD,   // 7:16
    0.5f, 0.5f, 0.42f, 1.0f,
    STRUT_LN, STRUT_SD, -STRUT_SD,   // 8:17
    0.5f, 0.5f, 0.42f, 1.0f,

    STRUT_SD, STRUT_SD, STRUT_SD,   // 6:18
    0.2f, 0.2f, 0.2f, 1.0f,
    STRUT_SD, -STRUT_SD, STRUT_SD,   //10:19
    0.2f, 0.2f, 0.2f, 1.0f,
    STRUT_LN, STRUT_SD, STRUT_SD,   // 7:20
    0.2f, 0.2f, 0.2f, 1.0f,
    STRUT_SD, -STRUT_SD, STRUT_SD,   //10:21
    0.2f, 0.2f, 0.2f, 1.0f,
    STRUT_LN, -STRUT_SD, STRUT_SD,   // 9:22
    0.2f, 0.2f, 0.2f, 1.0f,
    STRUT_LN, STRUT_SD, STRUT_SD,   // 7:23
    0.2f, 0.2f, 0.2f, 1.0f,


    STRUT_SD, STRUT_SD, -STRUT_SD,   //  1:24
    0.3f, 0.3f, 0.3f, 1.0f,
    STRUT_SD, STRUT_LN, STRUT_SD,   // 11:25
    0.3f, 0.3f, 0.3f, 1.0f,
    STRUT_SD, STRUT_SD, STRUT_SD,   //  6:26
    0.3f, 0.3f, 0.3f, 1.0f,
    STRUT_SD, STRUT_SD, -STRUT_SD,   //  1:27
    0.3f, 0.3f, 0.3f, 1.0f,
    STRUT_SD, STRUT_LN, -STRUT_SD,   // 12:29
    0.3f, 0.3f, 0.3f, 1.0f,
    STRUT_SD, STRUT_LN, STRUT_SD,   // 11:28
    0.3f, 0.3f, 0.3f, 1.0f,

    STRUT_SD, STRUT_SD, STRUT_SD,   //  6:30
    0.2f, 0.2f, 0.2f, 1.0f,
    STRUT_SD, STRUT_LN, STRUT_SD,   // 11:31
    0.2f, 0.2f, 0.2f, 1.0f,
    -STRUT_SD, STRUT_SD, STRUT_SD,   // 14:32
    0.2f, 0.2f, 0.2f, 1.0f,
    -STRUT_SD, STRUT_SD, STRUT_SD,   // 14:33
    0.2f, 0.2f, 0.2f, 1.0f,
    STRUT_SD, STRUT_LN, STRUT_SD,   // 11:34
    0.2f, 0.2f, 0.2f, 1.0f,
    -STRUT_SD, STRUT_LN, STRUT_SD,   // 13:35
    0.2f, 0.2f, 0.2f, 1.0f,

    -JOINT_SD, JOINT_SD, -JOINT_SD,   // 20:36
    0.5f, 0.5f, 0.42f, 1.0f,
    -JOINT_SD, JOINT_SD, JOINT_SD,   // 21:38
    0.5f, 0.5f, 0.42f, 1.0f,
    JOINT_SD, JOINT_SD, -JOINT_SD,   // 22:37
    0.5f, 0.5f, 0.42f, 1.0f,
    JOINT_SD, JOINT_SD, -JOINT_SD,   // 20:39
    0.5f, 0.5f, 0.42f, 1.0f,
    -JOINT_SD, JOINT_SD, JOINT_SD,   // 22:40
    0.5f, 0.5f, 0.42f, 1.0f,
    JOINT_SD, JOINT_SD, JOINT_SD,   // 23:41
    0.5f, 0.5f, 0.42f, 1.0f,

    JOINT_SD, JOINT_SD, -JOINT_SD,   // 21:42
    0.3f, 0.3f, 0.3f, 1.0f,
    JOINT_SD, JOINT_SD, JOINT_SD,   // 23:43
    0.3f, 0.3f, 0.3f, 1.0f,
    JOINT_SD, -JOINT_SD, -JOINT_SD,   // 24:44
    0.3f, 0.3f, 0.3f, 1.0f,
    JOINT_SD, JOINT_SD, JOINT_SD,   // 23:45
    0.3f, 0.3f, 0.3f, 1.0f,
    JOINT_SD, -JOINT_SD, JOINT_SD,   // 25:46
    0.3f, 0.3f, 0.3f, 1.0f,
    JOINT_SD, -JOINT_SD, -JOINT_SD,   // 24:47
    0.3f, 0.3f, 0.3f, 1.0f,

    -JOINT_SD, JOINT_SD, JOINT_SD,   // 22:48
    0.2f, 0.2f, 0.2f, 1.0f,
    -JOINT_SD, -JOINT_SD, JOINT_SD,   // 26:49
    0.2f, 0.2f, 0.2f, 1.0f,
    JOINT_SD, JOINT_SD, JOINT_SD,   // 23:50
    0.2f, 0.2f, 0.2f, 1.0f,
    JOINT_SD, JOINT_SD, JOINT_SD,   // 23:51
    0.2f, 0.2f, 0.2f, 1.0f,
    -JOINT_SD, -JOINT_SD, JOINT_SD,   // 26:52
    0.2f, 0.2f, 0.2f, 1.0f,
    JOINT_SD, -JOINT_SD, JOINT_SD,   // 25:53
    0.2f, 0.2f, 0.2f, 1.0f,

};
static DEMOGfxBuffer modelVertexBuffer;

//static const int MODEL_VTX_COUNT = 54 * 3;
static const int MODEL_VTX_SIZE = ( sizeof( modelVtxs ) );
static const int MODEL_VTX_STRIDE = ( sizeof( f32 ) * 3 + sizeof( f32 ) * 4 );
static const int MODEL_POS_OFFSET = 0;
static const int MODEL_CLR_OFFSET = 3;


// ----------------------------------------------------
//
//            13----------- 11
//             |          /|
//             |         / |
//             |        /  |
//             |        12 |
//             |        |  |
//             |        |  |
//             |        |  |
//             |        |  |
//            14--------|-- 6--------------- 7
//                      | /|                /|
//                      |/ |               / |
//                      |  |              /  |
//         0 ----------- 1----------------8  |
//         /           /|  | 10              |
//        /           / |  ------------------ 9
//       /           /  |
//      /           /   |
//     /           /   /4
//   3 ------------ 2 /
//                |  /
//                | /
//                |/
//                5
//
//       22 ------------ 23
//        /|             /|
//       / |            / |
//      /  |           /  |
//     /   |          /   |
//  20 --------------- 21 |
//         |          |   |
//       26 ----------|--- 25
//                    |  /
//                    | /
//                    |/
//                    24
//  y
//   ^  ^ z
//   | /
//   |/
//   ------> x
//
//---------------------------------------------------------

/// Indices
static const int MODEL_IDX_COUNT = 54;
static const int MODEL_IDX_SIZE = ( sizeof( u32 ) * MODEL_IDX_COUNT );

static u32 modelIdxs[] = {
    0, 1, 2, 3, 4, 5, // STRUT1 #0
    6, 7, 8, 9, 10, 11, // STRUT1 #1
    12, 13, 14, 15, 16, 17, // STRUT2 #2
    18, 19, 20, 21, 22, 23, // STRUT2 #3
    24, 25, 26, 27, 28, 29, // STRUT3 #4
    30, 31, 32, 33, 34, 35, // STRUT3 #5

    36, 37, 38, 39, 40, 41, // JOINT #6
    42, 43, 44, 45, 46, 47, // JOINT #7
    48, 49, 50, 51, 52, 53, // JOINT #8
};
static DEMOGfxBuffer modelIndexBuffer;

//static const int NUM_ATTRIB = 2;
static const int BUFFER_IDX = 0;

// -----

// Matricies
typedef struct _StaticMatrix
{
    Mtx44   viewMtx44;
    Mtx44   projMtx44;
} StaticMatrix;

static DEMOGfxBuffer g_StaticUniforms;

typedef struct _DynamicMatrix
{
    Mtx44   modelMtx44;
} DynamicMatrix;
static size_t alignedDynamicSize = 0;

static f32 s_animTranZ;
static u32 s_ticks;

//---------------------------------------------------------------------------*
//  Display list data
//---------------------------------------------------------------------------*/
static const int MODEL_DL_POOL_SIZE = ( 2 * 1024 * 1024 );
static const int MODEL_DL_SIZE = ( 128 * 1024 );
static const int NUM_MODEL_DLS = ( MODEL_DL_POOL_SIZE / MODEL_DL_SIZE );
static void* pControlMemory = NULL;
static const int CONTROL_MEMORY_SIZE = 1024 * 1024;
static size_t sControlChunk = 0;
static DEMOGfxMemPool* pCommandMemory = NULL;
static nn::gfx::CommandBuffer commandBuffer;
static u32 modelDLCount = 0;  // current display list

typedef struct _UniformView
{
    nn::gfx::GpuAddress gpuAddress[ 2 ];
    int size;
} UniformView;

// Have one uniform block per frame to hold all of the data
static DEMOGfxBuffer dynamicUniforms[ 2 ];

typedef struct _DisplayListInfo
{
    nn::gfx::Fence* pFence;
    UniformView* pView;
    int maxDynamicUniforms;
    u32   offset;
    u32   maxSize;  // maximum size of the display list buffer in bytes
    u32   dlSize;   // actual size of the display list in bytes
} DisplayListInfo;

static nn::gfx::Fence frameFences[ 2 ];
static int s_FrameCount = 0;

static DisplayListInfo modelDLInfo[ NUM_MODEL_DLS ];

// Viewport/Scissor
static nn::gfx::ViewportScissorState g_ViewportScissor;
static void* g_pViewportScissorData;

// Prototype
static void CameraInit( Mtx44 resultProjMtx44, Mtx44 resultViewMtx44 );
static int SceneInit();
static int SceneDraw();
static void ModelTick( Mtx44 resultMtx44, s32 offsetX, s32 offsetY, s32 offsetZ );
static void AnimTick();
static void PrintInfo();
static DisplayListInfo* GetModelDL();
static void InitModelDLPool();


static void OutOfCommandMemoryEventCallback(
    nn::gfx::CommandBuffer* pCommandBuffer, const nn::gfx::OutOfMemoryEventArg& )
{
    DisplayListInfo* pModelDL = GetModelDL();
    pCommandBuffer->AddCommandMemory( pCommandMemory->GetPool(),
        pCommandMemory->GetBaseOffset() + pModelDL->offset, pModelDL->maxSize );
}

static void OutOfControlMemoryEventCallback(
    nn::gfx::CommandBuffer* pCommandBuffer, const nn::gfx::OutOfMemoryEventArg& )
{
    sControlChunk++;
    if ( sControlChunk > 1 )
    {
        sControlChunk = 0;
    }
    pCommandBuffer->AddControlMemory( nn::util::BytePtr( pControlMemory,
        ( sControlChunk * CONTROL_MEMORY_SIZE / 2 ) ).Get(), CONTROL_MEMORY_SIZE / 2 );
}

// Init function for setting projection matrix
static void CameraInit( Mtx44 resultProjMtx44, Mtx44 resultViewMtx44 )
{
    // row major matricies
    Mtx   lookAtMtx34;

    Vec     up = { 0.20f, 0.97f, 0.0f };
    Vec  objPt = { -110.0f, -70.0f, -190.0f };
    Vec camLoc = { 90.0f, 110.0f, 13.0f };

    f32   pers = 50.0f;
    f32 aspect = ( f32 ) DEMOColorBufferInfo.GetWidth() / ( f32 ) DEMOColorBufferInfo.GetHeight();
    f32  znear = 50.0f;
    f32   zfar = 2000.0f;

    // Compute perspective matrix
    MTXPerspective( resultProjMtx44, pers, aspect, znear, zfar );

    // Compute lookAt matrix
    MTXLookAt( lookAtMtx34, &camLoc, &up, &objPt );
    MTX34To44( lookAtMtx34, resultViewMtx44 );
}

// Update Animation
static void AnimTick()
{
    u32    animSteps = STRUT_LN;
    f32    animLoopBack = ( f32 ) STRUT_LN;
    f32    animStepFwd = animLoopBack / ( f32 ) animSteps;

    s_animTranZ += animStepFwd;

    if ( ( s_ticks % animSteps ) == 0 )
        s_animTranZ = animLoopBack;

    s_ticks++;
}

// Update Model function for setting model matrix
static void ModelTick( Mtx44 resultMtx44, s32 offsetX, s32 offsetY, s32 offsetZ )
{
    MTX44Identity( resultMtx44 );
    resultMtx44[ 0 ][ 3 ] = offsetX + 2.0f * STRUT_LN;
    resultMtx44[ 1 ][ 3 ] = static_cast< f32 >( offsetY );
    resultMtx44[ 2 ][ 3 ] = offsetZ + s_animTranZ;
}

// DL Info: Initialize the model display list memory pool
static void InitModelDLPool()
{
    u32 idx;

    // DL Info: Allocate the memory used for display lists
    pCommandMemory = DEMOGfxSharedPool->AllocSubPool( MODEL_DL_POOL_SIZE, nn::gfx::CommandBuffer::GetCommandMemoryAlignment( &DEMODevice ) );
    {
        nn::gfx::CommandBuffer::InfoType cbInfo;
        cbInfo.SetDefault();
        cbInfo.SetQueueCapability( nn::gfx::QueueCapability_Graphics );
        cbInfo.SetCommandBufferType( nn::gfx::CommandBufferType_Direct );

        commandBuffer.Initialize( &DEMODevice, cbInfo );
    }

    ASSERT( pCommandMemory != NULL );

    pControlMemory = DEMOGfxAllocMEM2( CONTROL_MEMORY_SIZE, nn::gfx::CommandBuffer::GetControlMemoryAlignment( &DEMODevice ) );
    ASSERT( pControlMemory != NULL );
    commandBuffer.AddControlMemory( pControlMemory, CONTROL_MEMORY_SIZE / 2 );

    // determine the alignment
    {
        size_t alignment = 1;
        nn::gfx::Buffer::InfoType info;
        info.SetDefault();
        info.SetGpuAccessFlags( nn::gfx::GpuAccess_ConstantBuffer );
        info.SetSize( sizeof( DynamicMatrix ) );

        alignment = nn::gfx::Buffer::GetBufferAlignment( &DEMODevice, info );
        alignedDynamicSize = ( sizeof( DynamicMatrix ) + alignment - 1 ) & ( ~( alignment - 1 ) );
    }

    dynamicUniforms[ 0 ].Initialize( 12 * 11 * 16 * alignedDynamicSize, NULL, nn::gfx::GpuAccess_ConstantBuffer, 0 );
    dynamicUniforms[ 1 ].Initialize( 12 * 11 * 16 * alignedDynamicSize, NULL, nn::gfx::GpuAccess_ConstantBuffer, 0 );

    // DL Info: Initialize the display list info structures for each display list
    for ( idx = 0; idx < NUM_MODEL_DLS; idx++ )
    {
        modelDLInfo[ idx ].offset = ( MODEL_DL_SIZE * idx );
        modelDLInfo[ idx ].maxSize = MODEL_DL_SIZE;
        modelDLInfo[ idx ].dlSize = 0;
        modelDLInfo[ idx ].pFence = NULL;
        modelDLInfo[ idx ].maxDynamicUniforms = 12 * 11 * 16; // Total x/y/z of the grid
        modelDLInfo[ idx ].pView = new UniformView[ modelDLInfo[ idx ].maxDynamicUniforms ];
        for ( int uniformId = 0; uniformId < modelDLInfo[ idx ].maxDynamicUniforms; uniformId++ )
        {
            modelDLInfo[ idx ].pView[ uniformId ].gpuAddress[ 0 ] = dynamicUniforms[ 0 ].gpuAddress;
            modelDLInfo[ idx ].pView[ uniformId ].gpuAddress[ 1 ] = dynamicUniforms[ 1 ].gpuAddress;
            modelDLInfo[ idx ].pView[ uniformId ].gpuAddress[ 0 ].Offset( uniformId * alignedDynamicSize );
            modelDLInfo[ idx ].pView[ uniformId ].gpuAddress[ 1 ].Offset( uniformId * alignedDynamicSize );
            modelDLInfo[ idx ].pView[ uniformId ].size = sizeof( DynamicMatrix );
        }
    }

    commandBuffer.SetOutOfCommandMemoryEventCallback( OutOfCommandMemoryEventCallback );
    commandBuffer.SetOutOfControlMemoryEventCallback( OutOfControlMemoryEventCallback );

    for ( int i = 0; i < 2; i++ )
    {
        nn::gfx::Fence::InfoType info;
        info.SetDefault();
        frameFences[ i ].Initialize( &DEMODevice, info );
    }
}

// DL Info: Get a pointer to the display list info structure to use for rendering the model
static DisplayListInfo* GetModelDL()
{
    DisplayListInfo* pDLInfo;

    modelDLCount++;
    if ( modelDLCount >= NUM_MODEL_DLS )
    {
        modelDLCount = 0;
    }
    pDLInfo = &modelDLInfo[ modelDLCount ];

    // Check to see if the command buffer is free to reuse
    if ( modelDLInfo[modelDLCount].dlSize &&
         modelDLInfo[modelDLCount].pFence && !modelDLInfo[ modelDLCount ].pFence->IsSignaled() )
    {
        // DL Info: wait for the HW to finish with the DL
        printf( "Waiting for DL finish...\n" );

        nn::gfx::SyncResult result = modelDLInfo[ modelDLCount ].pFence->Sync( nn::TimeSpan::FromSeconds( 1 ) );
        ASSERT( result == nn::gfx::SyncResult_TimeoutExpired );
        NN_UNUSED( result );

        modelDLInfo[ modelDLCount ].pFence = &frameFences[ s_FrameCount % 2 ];
    }

    return pDLInfo;
}

// The init function for the rendering portions of this app
static int SceneInit()
{

    s_animTranZ = 0;
    s_ticks = 0;

    DEMOGfxLoadShadersFromFile(&g_AppPipeline.shaders, 0, SHADER_FILE);

    // Uniform Location Lookup
    staticMatrixLoc = g_AppPipeline.shaders.GetInterfaceSlot( nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_ConstantBuffer, "ub_static" );
    dynamicMatrixLoc = g_AppPipeline.shaders.GetInterfaceSlot( nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_ConstantBuffer, "ub_dynamic" );

    // position setup
    DEMOGfxInitShaderAttribute( &g_AppPipeline.shaders, "a_position", BUFFER_IDX, MODEL_POS_OFFSET * sizeof( f32 ), nn::gfx::AttributeFormat_32_32_32_Float );
    DEMOGfxInitShaderAttribute( &g_AppPipeline.shaders, "a_color", BUFFER_IDX, MODEL_CLR_OFFSET * sizeof( f32 ), nn::gfx::AttributeFormat_32_32_32_32_Float );
    DEMOGfxInitShaderVertexBuffer( &g_AppPipeline.shaders, BUFFER_IDX, MODEL_VTX_STRIDE, 0 );

    // Vertex buffer invalidation
    modelVertexBuffer.Initialize( MODEL_VTX_SIZE, &modelVtxs[0], nn::gfx::GpuAccess_VertexBuffer, 0 );

    // Index buffer invalidation
    modelIndexBuffer.Initialize( MODEL_IDX_SIZE, &modelIdxs[0], nn::gfx::GpuAccess_IndexBuffer, 0 );

    // Setup the static matrix buffers
    {
        g_StaticUniforms.Initialize( sizeof( StaticMatrix ), NULL, nn::gfx::GpuAccess_ConstantBuffer, 0 );
        StaticMatrix* pMatrix = g_StaticUniforms.Map< StaticMatrix >();
        CameraInit( pMatrix->projMtx44, pMatrix->viewMtx44 );
#ifdef CAFE
        GX2EndianSwap( pMatrix, sizeof( StaticMatrix ) );
#endif
        g_StaticUniforms.Unmap();
    }

    // DL Info: Initialize the model DL pool
    InitModelDLPool();

    g_AppPipeline.SetDefaults();

    g_AppPipeline.colorTargetStateCount = 1;
    g_AppPipeline.blendTargetStateCount = 1;
    g_AppPipeline.blendTargetStateInfoArray[ 0 ].SetDefault();
    g_AppPipeline.colorTargetStateInfoArray[ 0 ].SetDefault();
    g_AppPipeline.colorTargetStateInfoArray[ 0 ].SetFormat( DEMOColorBufferInfo.GetImageFormat() );

    // Setup the pipeline
    g_AppPipeline.depthStencilStateInfo.SetDepthComparisonFunction( nn::gfx::ComparisonFunction_Less );
    g_AppPipeline.Initialize( &DEMODevice );

    // Setup the viewport
    DEMOGfxSetViewportScissorState( &g_ViewportScissor, &g_pViewportScissorData,
        0.0f, 0.0f,
        static_cast< float >( DEMOColorBufferInfo.GetWidth() ),
        static_cast< float >( DEMOColorBufferInfo.GetHeight() ),
        0.0f, 1.0f,
        static_cast< float >( DEMOColorBufferInfo.GetHeight() ), false );

    return 1;
}

static void PrintInfo()
{
    DEMOGfxSetDefaultRenderTarget();
    DEMOGfxSetDefaultViewportScissor();

    // Set Demo Font String
    DEMOFontPrintf( 2, 1, "<Lattice Display List>" );
    DEMOFontPrintf( 3, 2, "- frames:%d", s_ticks );

}

// The draw function for the rendering portions of this app
static int SceneDraw()
{
    u32 dynamicUnformIdx = 0;
    s32 x = 0;
    s32 y = 0;
    s32 z = 0;
    DisplayListInfo* pModelDLInfo;

    DEMOGfxBeforeRender();

    DEMOCommandBuffer.End();
    DEMOQueue.ExecuteCommand( &DEMOCommandBuffer, NULL );
    DEMOQueue.Sync();

    pModelDLInfo = GetModelDL();

    // Reset the command buffer structure and add the command memory.
    commandBuffer.AddCommandMemory( pCommandMemory->GetPool(), pCommandMemory->GetBaseOffset() + pModelDLInfo->offset, pModelDLInfo->maxSize );

    nn::gfx::ColorTargetView* pCurrentScanBuffer = DEMOGetColorBufferView();

    // Start the CommandBuffer
    commandBuffer.Begin();

    // Clear buffers
    commandBuffer.ClearColor( pCurrentScanBuffer, 0.1f, 0.1f, 0.1f, 1.0f, NULL );
    commandBuffer.ClearDepthStencil( &DEMODepthBufferView, 1.0f, 0, nn::gfx::DepthStencilClearMode_DepthStencil, NULL );

#if NN_GFX_IS_TARGET_GX
    GX2SetShaderMode( GX2_SHADER_MODE_UNIFORM_BLOCK );
#endif
    // Setup the render targets
    const nn::gfx::ColorTargetView* colorBuffers[] = { pCurrentScanBuffer };
    commandBuffer.SetRenderTargets( 1, colorBuffers, &DEMODepthBufferView );

    commandBuffer.SetViewportScissorState( &g_ViewportScissor );

    // Set the Pipeline
    commandBuffer.SetPipeline( &g_AppPipeline.pipeline );

    // Update View Projection Matrix Uniforms
    commandBuffer.SetConstantBuffer( staticMatrixLoc, nn::gfx::ShaderStage_Vertex, g_StaticUniforms.gpuAddress, g_StaticUniforms.size );

    // Bind vertex & color buffer
    commandBuffer.SetVertexBuffer( BUFFER_IDX, modelVertexBuffer.gpuAddress, MODEL_VTX_STRIDE, modelVertexBuffer.size );

    // Draw Model
    DynamicMatrix* pMatrix = dynamicUniforms[ s_FrameCount % 2 ].Map< DynamicMatrix >();

    for ( x = -10 * STRUT_LN; x < 2 * STRUT_LN; x += STRUT_LN )
    {
        for ( y = -10 * STRUT_LN; y < STRUT_LN; y += STRUT_LN )
        {
            for ( z = STRUT_LN; z > -15 * STRUT_LN; z -= STRUT_LN )
            {
                // Sanity check
                ASSERT( dynamicUnformIdx < (u32)pModelDLInfo->maxDynamicUniforms );

                // Update Model Matrix Uniform
                ModelTick( pMatrix->modelMtx44, x, y, z );
#if NN_GFX_IS_TARGET_GX
                GX2EndianSwap( pMatrix, sizeof( DynamicMatrix ) );
#endif
                // Set the uniform block
                commandBuffer.SetConstantBuffer( dynamicMatrixLoc, nn::gfx::ShaderStage_Vertex,
                    pModelDLInfo->pView[ dynamicUnformIdx ].gpuAddress[ s_FrameCount % 2 ], pModelDLInfo->pView[ dynamicUnformIdx ].size );

                // Draw 1 Structure
                commandBuffer.DrawIndexed( nn::gfx::PrimitiveTopology_TriangleList, nn::gfx::IndexFormat_Uint32, modelIndexBuffer.gpuAddress, MODEL_IDX_COUNT, 0 );

                dynamicUnformIdx++;
                pMatrix = reinterpret_cast< DynamicMatrix* >( reinterpret_cast< ptrdiff_t >( pMatrix ) + static_cast< ptrdiff_t >( alignedDynamicSize ) );
            }
        }
    }
    dynamicUniforms[ s_FrameCount % 2 ].Unmap();

    // DL Info: Stop adding commands
    commandBuffer.End();
    ASSERT( pModelDLInfo->dlSize <= pModelDLInfo->maxSize );

    // DL Info: Execute the display list by placing it directly in the
    // graphics ring buffer
    DEMOQueue.ExecuteCommand( &commandBuffer, &frameFences[ s_FrameCount % 2 ] );

    // Begin recording for the default command buffer
    DEMOCommandBuffer.Begin();

    // Draw Infomation
    PrintInfo();
    DEMOGfxDoneRender();

    // Update Animation
    AnimTick();

    s_FrameCount++;

    return 1; // 0 makes it exit
}

//extern "C" void nnMain()
TEST(GfxLatticeDl, Run)
{
    int argc = nnt::GetHostArgc();
    char** argv = nnt::GetHostArgv();

    // Init DEMO lib
    DEMOInit();
    DEMOTestInit( argc, argv );
    DEMOGfxInit( argc, argv );
    DEMOFontInit();
    SceneInit();

    while ( DEMOIsRunning() )
    {
        SceneDraw();
    }

    // Free shaders/pipeline
    g_AppPipeline.Finalize( &DEMODevice );

    // Free the vertex/index buffer
    modelVertexBuffer.Finalize();
    modelIndexBuffer.Finalize();

    // Free the static uniform
    g_StaticUniforms.Finalize();

    // Free the viewport
    g_ViewportScissor.Finalize( &DEMODevice );
    DEMOGfxFreeMEM2( g_pViewportScissorData );

    // Free the command buffers
    for ( int i = 0; i < NUM_MODEL_DLS; i++ )
    {
        delete[] modelDLInfo[ i ].pView;
    }
    dynamicUniforms[ 0 ].Finalize();
    dynamicUniforms[ 1 ].Finalize();

    frameFences[ 0 ].Finalize( &DEMODevice );
    frameFences[ 1 ].Finalize( &DEMODevice );
    commandBuffer.Finalize( &DEMODevice );
    pCommandMemory->Finalize();

    DEMOFontShutdown();
    DEMOTestShutdown();
    DEMOGfxShutdown();
    DEMOShutdown();

    SUCCEED();
}
