﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/////////////////////////////////////////////////////////////////////////////
//
// format.cpp
//
// Attribute format testware
//
//////////////////////////////////////////////////////////////////////////////

#include <cstdio>
#include <cstring>

#include <gfx/demo.h>

#include <nnt.h>
#include <nnt/nnt_Argument.h>

#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON )
#include <nn/os.h>
#elif defined( NN_BUILD_CONFIG_OS_SUPPORTS_WIN32 )
#include <nn/os.h> // NOLINT(analysis/include)
#endif

#include "format.h"

//#define USE_PIPELINES

#if NN_GFX_IS_TARGET_GX
#pragma ghs nowarning 1518
#endif

// Grid Information

static const int GRID_MAX_X =  9;
static const int GRID_MAX_Y =  7;
// -- Viewport Info -- //
nn::gfx::ViewportScissorState viewportScissor[GRID_MAX_Y][GRID_MAX_X];
void* pViewportScissorData[ GRID_MAX_Y ][ GRID_MAX_X ];
nn::gfx::ViewportScissorState defaultViewportScissor;
void* pDefaultViewportScissorData;


static const int GRID_W = 126;
static const int GRID_H =  87;

static const int GRID_OFFSET_X = 64;
static const int GRID_OFFSET_Y = 45;

static const int NUM_SHADERS = 2;

// A .gsh file is a GX2 specific file that contains vertex and pixel
// shader data. In this case the vertex shader just passes through
// the position and color without modification. The pixel shader
// outputs the interpolated vertex color.
static int g_ShaderFileIdx = 0;
static const char * const SHADER_FILE_NAME[][NUM_SHADERS] =
{
    {"shaders/formatScaledShader", "shaders/formatIntshader"},
    {"shaders/formatScaledShaderHlslcc", "shaders/formatIntshaderHlslcc"},
};

static const NBitFloatDesc Float16Desc =
{
    16,                                                         // bitSize
    10,                                                         // numFracBits
    5,                                                          // numExpBits
    15,                                                         // signBit
    (1 << 15),                                                  // signMask
    (1 << 10) - 1,                                              // fracMask
    ((1 << 5) - 1) << 10,                                       // expMask
    (1 << (5 - 1)) - 1,                                         // expBias
    (1 << (5 - 1)) - 1,                                         // eMax
    -((1 << (5 - 1)) - 1) + 1,                                  // eMin
    ((((1 << (5 - 1)) - 1) + 127) << 23) | 0x7FE000,            // maxNormal
    ((-((1 << (5 - 1)) - 1) + 1) + 127) << 23,                  // minNormal
    static_cast<u32>((((1 << (5 - 1)) - 1) - 127)) << 23,       // biasDiff
    23 - 10,                                                    // fracBitsDiff
};

static u32 Float32ToFloatN(
    f32                     f,      ///< 32bit float value to convert
    const NBitFloatDesc*    pDesc)  ///< descriptor for the n-bit float
{
    static const u32 FloatExponentBias = 127;
    // 23 bit mantissa on single precision floats
    static const u32 FloatNumMantissaBits = 23;
    // 23 bit mantissa on single precision floats
    static const u32 FloatMantissaMask = 0x007FFFFF;
    // ANDing with this value gives the abs value
    static const u32 FloatSignBitMask = 0x80000000;
    // ANDing with this value gives the abs value
    static const u32 FloatMaskOutSignBit = 0x7FFFFFFF;

    u32 fBits    = (*(reinterpret_cast<u32*>(&f)));
    u32 fAbsBits = fBits & FloatMaskOutSignBit; // absolute value
    u32 sign;
    u32 floatN;
    u32 float32IsNeg = ((fBits & FloatSignBitMask) ? 1 : 0);

    // extract sign bit only if the desired representation will have a sign bit
    if (pDesc->signMask != 0)
    {
        // if the sign mask is non-zero,
        // the destination float is assumed to have a sign bit
        sign = (fBits & FloatSignBitMask) >>
            (pDesc->numFracBits + pDesc->numExpBits + 1);
    } else {
        sign = 0;
    }

    if ((pDesc->signMask == 0) && (float32IsNeg == 1))
    {
        // if the number is negative and the desired
        // representation is not signed, clamp to zero
        floatN = 0;
    } else if (fAbsBits > pDesc->maxNormal) {
        // Not representable by N bit float -> clamp to floatNMax
        floatN = static_cast<u32>(sign |
                                  ((((1 << pDesc->numExpBits) - 2))
                                  << pDesc->numFracBits)
                                  | pDesc->fracMask);
    } else if (fAbsBits < pDesc->minNormal) {
        // Denormalized value
        // Make implicit 1 explicit
        u32 fracBits =
            (fAbsBits & FloatMantissaMask) | (1 << FloatNumMantissaBits);
        s32  nshift = (pDesc->eMin + FloatExponentBias -
             (fAbsBits >> FloatNumMantissaBits));

        if (nshift < 24)
        {
            fAbsBits = fracBits >> nshift;
        }else{
            fAbsBits = 0;
        }

        // Round to zero
        floatN = static_cast<u32>(sign | (fAbsBits >> pDesc->fracBitsDiff));
    } else {
        // Normalize value with Round to zero
        floatN = static_cast<u32>(sign | ((fAbsBits + pDesc->biasDiff)
                                           >> pDesc->fracBitsDiff));
    }

    return floatN;
}

static VtxFmtF32x3 TRIANGLE_POSITION_DATA[] = {
    { { -0.75f, 0.6f, -0.5f } }, { { 0.75f, 0.6f, -0.5f } },
    { { 0.0f, -0.9f, -0.5f } }, { { -0.75f, -0.9f, -0.5f } }
};

static DEMOGfxBuffer trianglePositionBuffer;

static const uint32_t DATA_U32 = 0xffffffff;
static const int DATA_S32 = 0x7fffffff;

static VtxFmtU8x1 TRIANGLE_COLOR_DATA_U8x1_NORM[] = { {0xff}, {0}, {0}, {0} };
static VtxFmtU8x1 TRIANGLE_COLOR_DATA_U8x1_INT[] = { {0}, {0xff}, {0}, {0} };
static VtxFmtU8x1 TRIANGLE_COLOR_DATA_S8x1_NORM[] = { {0}, {0}, {0x7f}, {0} };
static VtxFmtU8x1 TRIANGLE_COLOR_DATA_S8x1_INT[] = { {0x7f}, {0}, {0}, {0} };
static VtxFmtU16x1 TRIANGLE_COLOR_DATA_U16x1_NORM[] = { {0}, {0xffff}, {0}, {0} };
static VtxFmtU16x1 TRIANGLE_COLOR_DATA_U16x1_INT[] = { {0}, {0}, {0xffff}, {0} };
static VtxFmtU16x1 TRIANGLE_COLOR_DATA_S16x1_NORM[] = { {0x7fff}, {0}, {0}, {0} };
static VtxFmtU16x1 TRIANGLE_COLOR_DATA_S16x1_INT[] = { {0}, {0x7fff}, {0}, {0} }; // 10
static VtxFmtU16x1 TRIANGLE_COLOR_DATA_F16x1_FLOAT[4];
static VtxFmtU8x2 TRIANGLE_COLOR_DATA_U8x2_NORM[] =
    { { { 0xff, 0 } }, { { 0, 0xff } }, { { 0, 0 } }, { { 0, 0 } } };
static VtxFmtU8x2 TRIANGLE_COLOR_DATA_U8x2_INT[] =
    { { { 0, 0 } }, { { 0xff, 0 } }, { { 0, 0xff } }, { { 0, 0 } } };
static VtxFmtU8x2 TRIANGLE_COLOR_DATA_S8x2_NORM[] =
    { { { 0, 0x7f } }, { { 0, 0 } }, { { 0x7f, 0 } }, { { 0, 0 } } };
static VtxFmtU8x2 TRIANGLE_COLOR_DATA_S8x2_INT[] =
    { { { 0x7f, 0 } }, { { 0, 0x7f } }, { { 0, 0 } }, { { 0, 0 } } };
static VtxFmtU32x1 TRIANGLE_COLOR_DATA_U32x1_INT[] = { { DATA_U32 }, { 0 }, { 0 }, { 0 } };
static VtxFmtU32x1 TRIANGLE_COLOR_DATA_S32x1_INT[] = { { 0 }, { DATA_S32 }, { 0 }, { 0 } };
static VtxFmtF32x1 TRIANGLE_COLOR_DATA_F32x1_FLOAT[] = { { 0 }, { 0 }, { 1.0f }, { 0 } };
static VtxFmtU16x2 TRIANGLE_COLOR_DATA_U16x2_NORM[] =
    { { { 0, 0 } }, { { 0xffff, 0 } }, { { 0, 0xffff } }, { { 0, 0 } } };
static VtxFmtU16x2 TRIANGLE_COLOR_DATA_U16x2_INT[] =
    { { { 0, 0xffff } }, { { 0, 0 } }, { { 0xffff, 0 } }, { { 0, 0 } } };
static VtxFmtU16x2 TRIANGLE_COLOR_DATA_S16x2_NORM[] = // 20
    { { { 0x7fff, 0 } }, { { 0, 0x7fff } }, { { 0, 0 } }, { { 0, 0 } } };
static VtxFmtU16x2 TRIANGLE_COLOR_DATA_S16x2_INT[] =
    { { { 0, 0 } }, { { 0x7fff, 0 } }, { { 0, 0x7fff } }, { { 0, 0 } } };
static VtxFmtU16x2 TRIANGLE_COLOR_DATA_F16x2_FLOAT[3];
static VtxFmtU8x4 TRIANGLE_COLOR_DATA_U8x4_NORM[] =
    { { { 0, 0xff, 0, 0 } }, { { 0, 0, 0xff, 0 } }, { { 0xff, 0, 0, 0 } }, { { 0, 0xff, 0, 0 } } };
static VtxFmtU8x4 TRIANGLE_COLOR_DATA_U8x4_INT[] =
    { { { 0, 0, 0xff, 0 } }, { { 0xff, 0, 0, 0 } }, { { 0, 0xff, 0, 0 } }, { { 0, 0, 0xff, 0 } } };
static VtxFmtU8x4 TRIANGLE_COLOR_DATA_S8x4_NORM[] =
    { { { 0x7f, 0, 0, 0 } }, { { 0, 0x7f, 0, 0 } }, { { 0, 0, 0x7f, 0 } }, { { 0x7f, 0, 0, 0 } } };
static VtxFmtU8x4 TRIANGLE_COLOR_DATA_S8x4_INT[] =
    { { {0, 0x7f, 0, 0 }}, { {0, 0, 0x7f, 0} }, { {0x7f, 0, 0, 0} } };
static VtxFmtU32x1 TRIANGLE_COLOR_DATA_U10x3_2_NORM[] =
    { { 0x3ff << 0 }, { 0x3ff << 20 }, { 0x3ff << 10 }, { 0x3ff << 0 } };
static VtxFmtU32x1 TRIANGLE_COLOR_DATA_U10x3_2_INT[] =
    { { 0x3ff << 0 }, { 0x3ff << 20 }, { 0x3ff << 10 }, { 0x3ff << 0 } };
static VtxFmtU32x1 TRIANGLE_COLOR_DATA_S10x3_2_NORM[] = // 30
    { { 0x1ff << 0 }, { 0x1ff << 20 }, { 0x1ff << 10 }, { 0x1ff << 0 } };
static VtxFmtU32x1 TRIANGLE_COLOR_DATA_S10x3_2_INT[] =
    { { 0x1ff << 0 }, { 0x1ff << 20 }, { 0x1ff << 10 }, { 0x1ff << 0 } };
static VtxFmtU32x2 TRIANGLE_COLOR_DATA_U32x2_INT[] =
    { { { DATA_U32, 0 } }, { { 0, DATA_U32 } }, { { 0, 0 } }, { { DATA_U32, 0 } } };
static VtxFmtU32x2 TRIANGLE_COLOR_DATA_S32x2_INT[] =
    { { { 0, 0 } }, { { DATA_S32, 0 } }, { { 0, DATA_S32 } }, { { 0, 0 } } };
static VtxFmtF32x2 TRIANGLE_COLOR_DATA_F32x2_FLOAT[] =
    { { { 0.0f, 1.0f } }, { { 0.0f, 0.0f } }, { { 1.0f, 0.0f } }, { { 0.0f, 1.0f } } };
static VtxFmtU16x4 TRIANGLE_COLOR_DATA_U16x4_NORM[] =
    { { { 0, 0, 0xffff, 0 } }, { { 0xffff, 0, 0, 0 } }, { { 0, 0xffff, 0, 0 } }, { { 0, 0, 0xffff, 0 }  } };
static VtxFmtU16x4 TRIANGLE_COLOR_DATA_U16x4_INT[] =
    { { { 0xffff, 0, 0, 0 } }, { { 0, 0xffff, 0, 0 } }, { { 0, 0, 0xffff, 0 } }, { { 0xffff, 0, 0, 0 } } };
static VtxFmtU16x4 TRIANGLE_COLOR_DATA_S16x4_NORM[] =
    { { { 0, 0x7fff, 0, 0 } }, { { 0, 0, 0x7fff, 0 } }, { { 0x7fff, 0, 0, 0 } }, { { 0, 0x7fff, 0, 0 } } };
static VtxFmtU16x4 TRIANGLE_COLOR_DATA_S16x4_INT[] =
    { { { 0, 0, 0x7fff, 0 } }, { { 0x7fff, 0, 0, 0 } }, { { 0, 0x7fff, 0, 0 } }, { { 0, 0, 0x7fff, 0 } } };
static VtxFmtU16x4 TRIANGLE_COLOR_DATA_F16x4_FLOAT[4];
static VtxFmtU32x3 TRIANGLE_COLOR_DATA_U32x3_INT[] = // 40
    { { { DATA_U32, 0, 0 } }, { { 0, DATA_U32, 0 } }, { { 0, 0, DATA_U32 } }, { { DATA_U32, 0, 0 } } };
static VtxFmtU32x3 TRIANGLE_COLOR_DATA_S32x3_INT[] =
    { { { 0, DATA_S32, 0 } }, { { 0, 0, DATA_S32 } }, { { DATA_S32, 0, 0 } }, { { 0, DATA_S32, 0 } } };
static VtxFmtF32x3 TRIANGLE_COLOR_DATA_F32x3_FLOAT[] =
    { { { 0.0f, 0.0f, 1.0f } }, { { 1.0f, 0.0f, 0.0f } }, { { 0.0f, 1.0f, 0.0f } }, { { 0.0f, 0.0f, 1.0f } } };
static VtxFmtU32x4 TRIANGLE_COLOR_DATA_U32x4_INT[] =
    { { { DATA_U32, 0, 0, 0 } }, { { 0, DATA_U32, 0, 0 } }, { { 0, 0, DATA_U32, 0 } }, { { DATA_U32, 0, 0, 0 } } };
static VtxFmtU32x4 TRIANGLE_COLOR_DATA_S32x4_INT[] =
    { { { 0, DATA_S32, 0, 0 } }, { { 0, 0, DATA_S32, 0 } }, { { DATA_S32, 0, 0, 0 } }, { { 0, DATA_S32, 0, 0 } } };
static VtxFmtF32x4 TRIANGLE_COLOR_DATA_F32x4_FLOAT[] = // 45
    { { { 0.0f, 0.0f, 1.0f, 0.0f } }, { { 1.0f, 0.0f, 0.0f, 0.0f } }, { { 0.0f, 1.0f, 0.0f, 0.0f } }, { { 0.0f, 0.0f, 1.0f, 0.0f } } };

static u8 g_CurrentAttribute = 0;

/*static VtxFmtF32x3 TRIANGLE_COLOR_DATA_F32x3[] = {
    {1.0f, 0.0f, 0.0f}, {0.0f, 1.0f, 0.0f}, {0.0f, 0.0f, 1.0f} };*/

static u16 TRIANGLE_INDEX16_DATA[4];
static u32 TRIANGLE_INDEX32_DATA[4];

static DEMOGfxBuffer triangleIndex16Buffer;
static DEMOGfxBuffer triangleIndex32Buffer;

static const u32 INDEX_COUNT = 3; // 4 for quad

typedef struct _AttributeProperty
{
    u32   colorStride;
    u32   size;
    void *bufferData;
    u32   uniformDivisor;
    nn::gfx::AttributeFormat attributeFormat;
    const char *name;

    DEMOGfxBuffer buffer;

    DEMOGfxBuffer uniformBuffer;
    size_t uniformBufferSize;
} AttributeProperty;

static bool FORMAT_IS_INTEGER( nn::gfx::AttributeFormat format )
{
    return ( ( ( (format) & (nn::gfx::TypeFormat_Bits - 1)) == nn::gfx::TypeFormat_Uint) || ( ( (format) & (nn::gfx::TypeFormat_Bits - 1)) == nn::gfx::TypeFormat_Sint) );
}

static const int SCALE_0 =           1;
static const int SCALE_7 =         127;
static const int SCALE_8 =         255;
static const int SCALE_9 =         511;
static const int SCALE_10 =       1023;
static const int SCALE_15 =      32767;
static const int SCALE_16 =      65535;
static const int SCALE_31 = 2147483647;
static const u32 SCALE_32 = 4294967295;

static AttributeProperty g_AttributeProperty[NUM_ATTRIBUTE] =
{
    {1 * sizeof(u8),  sizeof(TRIANGLE_COLOR_DATA_U8x1_NORM), TRIANGLE_COLOR_DATA_U8x1_NORM,      SCALE_0,  nn::gfx::AttributeFormat_8_Unorm,                   "8_UNORM"},
    {1 * sizeof(u8),  sizeof(TRIANGLE_COLOR_DATA_U8x1_INT), TRIANGLE_COLOR_DATA_U8x1_INT,       SCALE_8,  nn::gfx::AttributeFormat_8_Uint,                    "8_UINT"},
    {1 * sizeof(s8),  sizeof(TRIANGLE_COLOR_DATA_S8x1_NORM), TRIANGLE_COLOR_DATA_S8x1_NORM,      SCALE_0,  nn::gfx::AttributeFormat_8_Snorm,                   "8_SNORM"},
    {1 * sizeof(s8),  sizeof(TRIANGLE_COLOR_DATA_S8x1_INT), TRIANGLE_COLOR_DATA_S8x1_INT,       SCALE_7,  nn::gfx::AttributeFormat_8_Sint,                    "8_SINT"},
    {1 * sizeof(u8),  sizeof(TRIANGLE_COLOR_DATA_U8x1_INT), TRIANGLE_COLOR_DATA_U8x1_INT,       SCALE_8,  nn::gfx::AttributeFormat_8_UintToFloat,             "8_UINT_TO_FLOAT"},
    {1 * sizeof(s8),  sizeof(TRIANGLE_COLOR_DATA_S8x1_INT), TRIANGLE_COLOR_DATA_S8x1_INT,       SCALE_7,  nn::gfx::AttributeFormat_8_SintToFloat,             "8_SINT_TO_FLOAT"},

    {1 * sizeof(u16), sizeof(TRIANGLE_COLOR_DATA_U16x1_NORM), TRIANGLE_COLOR_DATA_U16x1_NORM,     SCALE_0,  nn::gfx::AttributeFormat_16_Unorm,                  "16_UNORM"},
    {1 * sizeof(u16), sizeof(TRIANGLE_COLOR_DATA_U16x1_INT), TRIANGLE_COLOR_DATA_U16x1_INT,      SCALE_16, nn::gfx::AttributeFormat_16_Uint,                   "16_UINT"},
    {1 * sizeof(s16), sizeof(TRIANGLE_COLOR_DATA_S16x1_NORM), TRIANGLE_COLOR_DATA_S16x1_NORM,     SCALE_0,  nn::gfx::AttributeFormat_16_Snorm,                  "16_SNORM"},
    {1 * sizeof(s16), sizeof(TRIANGLE_COLOR_DATA_S16x1_INT), TRIANGLE_COLOR_DATA_S16x1_INT,      SCALE_15, nn::gfx::AttributeFormat_16_Sint,                   "16_SINT"},
    {1 * sizeof(u16), sizeof(TRIANGLE_COLOR_DATA_F16x1_FLOAT), TRIANGLE_COLOR_DATA_F16x1_FLOAT,    SCALE_0,  nn::gfx::AttributeFormat_16_Float,                  "16_FLOAT"},
    {1 * sizeof(u16), sizeof(TRIANGLE_COLOR_DATA_U16x1_INT), TRIANGLE_COLOR_DATA_U16x1_INT,      SCALE_16, nn::gfx::AttributeFormat_16_UintToFloat,            "16_UINT_TO_FLOAT"},
    {1 * sizeof(s16), sizeof(TRIANGLE_COLOR_DATA_S16x1_INT), TRIANGLE_COLOR_DATA_S16x1_INT,      SCALE_15, nn::gfx::AttributeFormat_16_SintToFloat,            "16_SINT_TO_FLOAT"},

    {2 * sizeof(u8),  sizeof(TRIANGLE_COLOR_DATA_U8x2_NORM), TRIANGLE_COLOR_DATA_U8x2_NORM,      SCALE_0,  nn::gfx::AttributeFormat_8_8_Unorm,                 "8_8_UNORM"},
    {2 * sizeof(u8),  sizeof(TRIANGLE_COLOR_DATA_U8x2_INT), TRIANGLE_COLOR_DATA_U8x2_INT,       SCALE_8,  nn::gfx::AttributeFormat_8_8_Uint,                  "8_8_UINT"},
    {2 * sizeof(s8),  sizeof(TRIANGLE_COLOR_DATA_S8x2_NORM), TRIANGLE_COLOR_DATA_S8x2_NORM,      SCALE_0,  nn::gfx::AttributeFormat_8_8_Snorm,                 "8_8_SNORM"},
    {2 * sizeof(s8),  sizeof(TRIANGLE_COLOR_DATA_S8x2_INT), TRIANGLE_COLOR_DATA_S8x2_INT,       SCALE_7,  nn::gfx::AttributeFormat_8_8_Sint,                  "8_8_SINT"},
    {2 * sizeof(u8),  sizeof(TRIANGLE_COLOR_DATA_U8x2_INT), TRIANGLE_COLOR_DATA_U8x2_INT,       SCALE_8,  nn::gfx::AttributeFormat_8_8_UintToFloat,           "8_8_UINT_TO_FLOAT"},
    {2 * sizeof(s8),  sizeof(TRIANGLE_COLOR_DATA_S8x2_INT), TRIANGLE_COLOR_DATA_S8x2_INT,       SCALE_7,  nn::gfx::AttributeFormat_8_8_SintToFloat,           "8_8_SINT_TO_FLOAT"},

    {1 * sizeof(u32), sizeof(TRIANGLE_COLOR_DATA_U32x1_INT), TRIANGLE_COLOR_DATA_U32x1_INT,      SCALE_32, nn::gfx::AttributeFormat_32_Uint,                   "32_UINT"},
    {1 * sizeof(s32), sizeof(TRIANGLE_COLOR_DATA_S32x1_INT), TRIANGLE_COLOR_DATA_S32x1_INT,      SCALE_31, nn::gfx::AttributeFormat_32_Sint,                   "32_SINT"},
    {1 * sizeof(f32), sizeof(TRIANGLE_COLOR_DATA_F32x1_FLOAT), TRIANGLE_COLOR_DATA_F32x1_FLOAT,    SCALE_0,  nn::gfx::AttributeFormat_32_Float,                  "32_FLOAT"},

    {2 * sizeof(u16), sizeof(TRIANGLE_COLOR_DATA_U16x2_NORM), TRIANGLE_COLOR_DATA_U16x2_NORM,     SCALE_0,  nn::gfx::AttributeFormat_16_16_Unorm,               "16_16_UNORM"},
    {2 * sizeof(u16), sizeof(TRIANGLE_COLOR_DATA_U16x2_INT), TRIANGLE_COLOR_DATA_U16x2_INT,      SCALE_16, nn::gfx::AttributeFormat_16_16_Uint,                "16_16_UINT"},
    {2 * sizeof(s16), sizeof(TRIANGLE_COLOR_DATA_S16x2_NORM), TRIANGLE_COLOR_DATA_S16x2_NORM,     SCALE_0,  nn::gfx::AttributeFormat_16_16_Snorm,               "16_16_SNORM"},
    {2 * sizeof(s16), sizeof(TRIANGLE_COLOR_DATA_S16x2_INT), TRIANGLE_COLOR_DATA_S16x2_INT,      SCALE_15, nn::gfx::AttributeFormat_16_16_Sint,                "16_16_SINT"},
    {2 * sizeof(u16), sizeof(TRIANGLE_COLOR_DATA_U16x2_INT), TRIANGLE_COLOR_DATA_U16x2_INT,      SCALE_16, nn::gfx::AttributeFormat_16_16_UintToFloat,         "16_16_UINT_TO_FLOAT"},
    {2 * sizeof(s16), sizeof(TRIANGLE_COLOR_DATA_S16x2_INT), TRIANGLE_COLOR_DATA_S16x2_INT,      SCALE_15, nn::gfx::AttributeFormat_16_16_SintToFloat,         "16_16_SINT_TO_FLOAT"},
    {2 * sizeof(u16), sizeof(TRIANGLE_COLOR_DATA_F16x2_FLOAT), TRIANGLE_COLOR_DATA_F16x2_FLOAT,    SCALE_0,  nn::gfx::AttributeFormat_16_16_Float,               "16_16_FLOAT"},

    {4 * sizeof(u8),  sizeof(TRIANGLE_COLOR_DATA_U8x4_NORM), TRIANGLE_COLOR_DATA_U8x4_NORM,      SCALE_0,  nn::gfx::AttributeFormat_8_8_8_8_Unorm,             "8_8_8_8_UNORM"},
    {4 * sizeof(u8),  sizeof(TRIANGLE_COLOR_DATA_U8x4_INT), TRIANGLE_COLOR_DATA_U8x4_INT,       SCALE_8,  nn::gfx::AttributeFormat_8_8_8_8_Uint,              "8_8_8_8_UINT"},
    {4 * sizeof(s8),  sizeof(TRIANGLE_COLOR_DATA_S8x4_NORM), TRIANGLE_COLOR_DATA_S8x4_NORM,      SCALE_0,  nn::gfx::AttributeFormat_8_8_8_8_Snorm,             "8_8_8_8_SNORM"},
    {4 * sizeof(s8),  sizeof(TRIANGLE_COLOR_DATA_S8x4_INT), TRIANGLE_COLOR_DATA_S8x4_INT,       SCALE_7,  nn::gfx::AttributeFormat_8_8_8_8_Sint,              "8_8_8_8_SINT"},
    {4 * sizeof(u8),  sizeof(TRIANGLE_COLOR_DATA_U8x4_INT), TRIANGLE_COLOR_DATA_U8x4_INT,       SCALE_8,  nn::gfx::AttributeFormat_8_8_8_8_UintToFloat,       "8_8_8_8_UINT_TO_FLOAT"},
    {4 * sizeof(s8),  sizeof(TRIANGLE_COLOR_DATA_S8x4_INT), TRIANGLE_COLOR_DATA_S8x4_INT,       SCALE_7,  nn::gfx::AttributeFormat_8_8_8_8_SintToFloat,       "8_8_8_8_SINT_TO_FLOAT"},

    {1 * sizeof(u32), sizeof(TRIANGLE_COLOR_DATA_U10x3_2_NORM), TRIANGLE_COLOR_DATA_U10x3_2_NORM,   SCALE_0,  nn::gfx::AttributeFormat_10_10_10_2_Unorm,          "10_10_10_2_UNORM"},
    {1 * sizeof(u32), sizeof(TRIANGLE_COLOR_DATA_U10x3_2_INT), TRIANGLE_COLOR_DATA_U10x3_2_INT,    SCALE_10, nn::gfx::AttributeFormat_10_10_10_2_Uint,           "10_10_10_2_UINT"},
    {1 * sizeof(u32), sizeof(TRIANGLE_COLOR_DATA_S10x3_2_NORM), TRIANGLE_COLOR_DATA_S10x3_2_NORM,   SCALE_0,  nn::gfx::AttributeFormat_10_10_10_2_Snorm,          "10_10_10_2_SNORM"},
    {1 * sizeof(u32), sizeof(TRIANGLE_COLOR_DATA_S10x3_2_INT), TRIANGLE_COLOR_DATA_S10x3_2_INT,    SCALE_9,  nn::gfx::AttributeFormat_10_10_10_2_Sint,           "10_10_10_2_SINT"},

    {2 * sizeof(u32), sizeof(TRIANGLE_COLOR_DATA_U32x2_INT), TRIANGLE_COLOR_DATA_U32x2_INT,      SCALE_32, nn::gfx::AttributeFormat_32_32_Uint,                "32_32_UINT"},
    {2 * sizeof(s32), sizeof(TRIANGLE_COLOR_DATA_S32x2_INT), TRIANGLE_COLOR_DATA_S32x2_INT,      SCALE_31, nn::gfx::AttributeFormat_32_32_Sint,                "32_32_SINT"},
    {2 * sizeof(f32), sizeof(TRIANGLE_COLOR_DATA_F32x2_FLOAT), TRIANGLE_COLOR_DATA_F32x2_FLOAT,    SCALE_0,  nn::gfx::AttributeFormat_32_32_Float,               "32_32_FLOAT"},

    {4 * sizeof(u16), sizeof(TRIANGLE_COLOR_DATA_U16x4_NORM), TRIANGLE_COLOR_DATA_U16x4_NORM,     SCALE_0,  nn::gfx::AttributeFormat_16_16_16_16_Unorm,         "16_16_16_16_UNORM"},
    {4 * sizeof(u16), sizeof(TRIANGLE_COLOR_DATA_U16x4_INT), TRIANGLE_COLOR_DATA_U16x4_INT,      SCALE_16, nn::gfx::AttributeFormat_16_16_16_16_Uint,          "16_16_16_16_UINT"},
    {4 * sizeof(s16), sizeof(TRIANGLE_COLOR_DATA_S16x4_NORM), TRIANGLE_COLOR_DATA_S16x4_NORM,     SCALE_0,  nn::gfx::AttributeFormat_16_16_16_16_Snorm,         "16_16_16_16_SNORM"},
    {4 * sizeof(s16), sizeof(TRIANGLE_COLOR_DATA_S16x4_INT), TRIANGLE_COLOR_DATA_S16x4_INT,      SCALE_15, nn::gfx::AttributeFormat_16_16_16_16_Sint,          "16_16_16_16_SINT"},
    {4 * sizeof(u16), sizeof(TRIANGLE_COLOR_DATA_F16x4_FLOAT), TRIANGLE_COLOR_DATA_F16x4_FLOAT,    SCALE_0,  nn::gfx::AttributeFormat_16_16_16_16_Float,         "16_16_16_16_FLOAT"},
    {4 * sizeof(u16), sizeof(TRIANGLE_COLOR_DATA_U16x4_INT), TRIANGLE_COLOR_DATA_U16x4_INT,      SCALE_16, nn::gfx::AttributeFormat_16_16_16_16_UintToFloat,   "16_16_16_16_UINT_TO_FLOAT"},
    {4 * sizeof(s16), sizeof(TRIANGLE_COLOR_DATA_S16x4_INT), TRIANGLE_COLOR_DATA_S16x4_INT,      SCALE_15, nn::gfx::AttributeFormat_16_16_16_16_SintToFloat,   "16_16_16_16_SINT_TO_FLOAT"},

    {3 * sizeof(u32), sizeof(TRIANGLE_COLOR_DATA_U32x3_INT), TRIANGLE_COLOR_DATA_U32x3_INT,      SCALE_32, nn::gfx::AttributeFormat_32_32_32_Uint,             "32_32_32_UINT"},
    {3 * sizeof(s32), sizeof(TRIANGLE_COLOR_DATA_S32x3_INT), TRIANGLE_COLOR_DATA_S32x3_INT,      SCALE_31, nn::gfx::AttributeFormat_32_32_32_Sint,             "32_32_32_SINT"},
    {3 * sizeof(f32), sizeof(TRIANGLE_COLOR_DATA_F32x3_FLOAT), TRIANGLE_COLOR_DATA_F32x3_FLOAT,    SCALE_0,  nn::gfx::AttributeFormat_32_32_32_Float,            "32_32_32_FLOAT"},

    {4 * sizeof(u32), sizeof(TRIANGLE_COLOR_DATA_U32x4_INT), TRIANGLE_COLOR_DATA_U32x4_INT,      SCALE_32, nn::gfx::AttributeFormat_32_32_32_32_Uint,          "32_32_32_32_UINT"},
    {4 * sizeof(s32), sizeof(TRIANGLE_COLOR_DATA_S32x4_INT), TRIANGLE_COLOR_DATA_S32x4_INT,      SCALE_31, nn::gfx::AttributeFormat_32_32_32_32_Sint,          "32_32_32_32_SINT"},
    {4 * sizeof(f32), sizeof(TRIANGLE_COLOR_DATA_F32x4_FLOAT), TRIANGLE_COLOR_DATA_F32x4_FLOAT,    SCALE_0,  nn::gfx::AttributeFormat_32_32_32_32_Float,         "32_32_32_32_FLOAT"},
};

// This sample uses the two attributes, position and color, for its vertex
// shader. The position and color data are held in two separate buffers so
// two buffer indices are used and offsets for each attribute are zero.
typedef struct _AttributeF32x3F32x3
{
    u32 attributeCount;

    u32 positionStride;
    int positionBufferIndex;
    u32 positionOffset;

    int colorBufferIndex;
    u32 colorOffset;

} AttributeF32x3F32x3;

static AttributeF32x3F32x3 g_TriangleAttribute =
{
    2, // ATTRIBUTE_COUNT

    3 * sizeof (f32), // POSITION_STRIDE
    0, // POSITION_BUFFER_INDEX
    0, // POSITION_OFFSET

    1, // COLOR_BUFFER_INDEX
    0  // COLOR_OFFSET
};

struct BaseShader
{
    DEMOGfxShader shaders;

    // The register locations where the position and color variables are stored
    // for the vertex shader.
    int positionLocation;
    int colorLocation;

    // The register locations where the divisor uniform is stored
    // for the vertex shader.
    int divisorLocation;

#ifdef USE_PIPELINES
    nn::gfx::Pipeline::InfoType pipelineInfo;

    // These are used by the pipelineInfo so they need to be static
    nn::gfx::BlendTargetStateInfo blendTargets[ 1 ];
    nn::gfx::ColorTargetStateInfo colorTargets[ 1 ];
#endif
};

struct Model
{
    AttributeProperty* pAttribute;
#ifdef USE_PIPELINES
    nn::gfx::Pipeline pipeline;
    void* pipelineData;
#else
    nn::gfx::VertexState vertexState;
    void* vertexStateData;
#endif

    BaseShader* pShader;
};

static BaseShader g_SimpleShader;
static BaseShader g_SimpleIntShader;

#ifndef USE_PIPELINES
static nn::gfx::DepthStencilState g_DepthStencilState;
static nn::gfx::RasterizerState g_RasterizerState;
static nn::gfx::BlendState g_BlendState;
static void* g_pBlendStateData = NULL;
#endif

static Model g_Models[ NUM_ATTRIBUTE ];

static u32 s_frame = 0;

static bool s_PrintForTest = false;

// Prototypes for functions used in this sample.
static bool SceneInit(void);
static void SceneDraw(void);
static void SetupAttribute(AttributeProperty& attr);

static void SetupAttribute(AttributeProperty& attr)
{
    attr.buffer.Initialize( attr.size, attr.bufferData, nn::gfx::GpuAccess_VertexBuffer | nn::gfx::GpuAccess_Read, 0 );

    // Setup the uniform block
    attr.uniformBufferSize = sizeof(int);
#if NN_GFX_IS_TARGET_GX
    GX2EndianSwap( &attr.uniformDivisor, sizeof( u32 ) );
#endif
    attr.uniformBuffer.Initialize( attr.uniformBufferSize, &attr.uniformDivisor, nn::gfx::GpuAccess_ConstantBuffer | nn::gfx::GpuAccess_Read, 0 );
}

static void InitBlendStateInfo( nn::gfx::BlendStateInfo* pBlendStateInfo, int numColorTargets, nn::gfx::BlendTargetStateInfo* pBlendTargets )
{
    pBlendStateInfo->SetDefault();
    for ( int idx = 0; idx < numColorTargets; idx++ )
    {
        pBlendTargets[ idx ].SetDefault();
        pBlendTargets[ idx ].SetChannelMask( nn::gfx::ChannelMask_All );
        pBlendTargets[ idx ].SetBlendEnabled( false );
    }
    pBlendStateInfo->SetBlendTargetStateInfoArray( pBlendTargets, 1 );
}

static void InitDepthStencilStateInfo( nn::gfx::DepthStencilStateInfo* pDepthStencilStateInfo )
{
    pDepthStencilStateInfo->SetDefault();
    pDepthStencilStateInfo->SetDepthTestEnabled( true );
    pDepthStencilStateInfo->SetDepthWriteEnabled( true );
    pDepthStencilStateInfo->SetDepthComparisonFunction( nn::gfx::ComparisonFunction_Less );
    pDepthStencilStateInfo->SetStencilTestEnabled( false );
}

static void InitRasterizerStateInfo( nn::gfx::RasterizerStateInfo* pRasterizerStateInfo )
{
    pRasterizerStateInfo->SetDefault();
    pRasterizerStateInfo->SetCullMode( nn::gfx::CullMode_None );
    pRasterizerStateInfo->SetScissorEnabled( false );
}

static void InitShaders()
{
    // Load shader binary to memory allocated by DEMOGfxLoadAssetFile. This
    // memory must be freed with a call to DEMOFree after shaders are loaded.
    for (int i = 0; i < NUM_SHADERS; i++)
    {
        BaseShader* pShader;

        if (i == 0)
        {
            pShader = &g_SimpleShader;
        }
        else
        {
            pShader = &g_SimpleIntShader;
        }

        DEMOGfxLoadShadersFromFile(&pShader->shaders, 0, SHADER_FILE_NAME[g_ShaderFileIdx][i]);

        // Lookup the attribute locations in the vertex shaders to be used with
        // the fetch shader. The shader author chose the names "position" and
        // "color."
        pShader->positionLocation = pShader->shaders.GetInterfaceSlot( nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_Input, "a_position" );
        pShader->colorLocation    = pShader->shaders.GetInterfaceSlot( nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_Input, "a_color");

        pShader->divisorLocation  = pShader->shaders.GetInterfaceSlot( nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_ConstantBuffer, "u_divisor");

        ASSERT((pShader->positionLocation != (u32)(-1))
               && (pShader->colorLocation != (u32)(-1))
               && (pShader->divisorLocation != (u32)(-1))
               && "Couldn't find the correct vertex shader attributes.");

#ifdef USE_PIPELINES
        // Setup the base pipeline info
        pShader->pipelineInfo.SetDefault();
        InitBlendStateInfo( &pShader->pipelineInfo.EditBlendStateInfo(), 1, &pShader->blendTargets[ 0 ] );
        InitDepthStencilStateInfo( &pShader->pipelineInfo.EditDepthStencilStateInfo() );
        InitRasterizerStateInfo( &pShader->pipelineInfo.EditRasterizerStateInfo() );
        pShader->pipelineInfo.EditPipelineShaderInfo( nn::gfx::ShaderStage_Vertex ).SetShaderPtr( &pShader->vertexShader );
        pShader->pipelineInfo.EditPipelineShaderInfo( nn::gfx::ShaderStage_Pixel ).SetShaderPtr( &pShader->pixelShader );
        pShader->colorTargets[ 0 ].SetDefault();
        pShader->colorTargets[ 0 ].SetFormat( DEMOColorBufferInfo.GetImageFormat() );
        pShader->pipelineInfo.EditRenderTargetStateInfo().SetDepthStencilFormat( DEMODepthBufferInfo.GetImageFormat() );
        pShader->pipelineInfo.EditRenderTargetStateInfo().SetColorTargetStateInfoArray( pShader->colorTargets, 1 );
#endif
    }

#ifndef USE_PIPELINES
    {
        nn::gfx::BlendStateInfo info;
        nn::gfx::BlendTargetStateInfo blendTargetInfo[ 1 ];
        InitBlendStateInfo( &info, 1, blendTargetInfo );
        size_t size = nn::gfx::BlendState::GetRequiredMemorySize( info );
        g_pBlendStateData = DEMOGfxAllocMEM2( size, nn::gfx::BlendState::RequiredMemoryInfo_Alignment );
        g_BlendState.SetMemory( g_pBlendStateData, size );
        g_BlendState.Initialize( &DEMODevice, info );
    }

    {
        nn::gfx::DepthStencilStateInfo info;
        InitDepthStencilStateInfo( &info );
        g_DepthStencilState.Initialize( &DEMODevice, info );
    }

    {
        nn::gfx::RasterizerStateInfo info;
        InitRasterizerStateInfo( &info );
        g_RasterizerState.Initialize( &DEMODevice, info );
    }
#endif

}

#if NN_GFX_IS_TARGET_D3D
static bool IsAttributeAvailable( int idx )
{
    switch ( g_AttributeProperty[ idx ].attributeFormat )
    {
    case nn::gfx::AttributeFormat_8_UintToFloat:
    case nn::gfx::AttributeFormat_8_SintToFloat:
    case nn::gfx::AttributeFormat_8_8_UintToFloat:
    case nn::gfx::AttributeFormat_8_8_SintToFloat:
    case nn::gfx::AttributeFormat_16_UintToFloat:
    case nn::gfx::AttributeFormat_16_SintToFloat:
    case nn::gfx::AttributeFormat_8_8_8_8_UintToFloat:
    case nn::gfx::AttributeFormat_8_8_8_8_SintToFloat:
    case nn::gfx::AttributeFormat_16_16_UintToFloat:
    case nn::gfx::AttributeFormat_16_16_SintToFloat:
    case nn::gfx::AttributeFormat_16_16_16_16_UintToFloat:
    case nn::gfx::AttributeFormat_16_16_16_16_SintToFloat:
    case nn::gfx::AttributeFormat_10_10_10_2_Snorm:
    case nn::gfx::AttributeFormat_10_10_10_2_Sint:
        // These attribute format are not supported in D3D11.
        return false;
    default:
        break;
    }

    return true;
}
#endif

// The initialization function for the rendering portions of this sample.
// It is responsible for allocating the three types of shaders and buffers
// as well as ensuring that data is flushed from the CPU to GPU memory.
static bool SceneInit()
{
    u32 valuef16;

    s_frame = 0;

    valuef16 = Float32ToFloatN(1.0f, &Float16Desc);

    InitShaders();

    // setting index buffers

    TRIANGLE_INDEX16_DATA[ 0 ] = 0;
    TRIANGLE_INDEX16_DATA[ 1 ] = 1;
    TRIANGLE_INDEX16_DATA[ 2 ] = 2;
    TRIANGLE_INDEX16_DATA[ 3 ] = 3;

    triangleIndex16Buffer.Initialize( sizeof( TRIANGLE_INDEX16_DATA ), TRIANGLE_INDEX16_DATA, nn::gfx::GpuAccess_IndexBuffer, 0 );

    TRIANGLE_INDEX32_DATA[ 0 ] = 0;
    TRIANGLE_INDEX32_DATA[ 1 ] = 1;
    TRIANGLE_INDEX32_DATA[ 2 ] = 2;
    TRIANGLE_INDEX32_DATA[ 3 ] = 3;

    triangleIndex32Buffer.Initialize( sizeof( TRIANGLE_INDEX32_DATA ), TRIANGLE_INDEX32_DATA, nn::gfx::GpuAccess_IndexBuffer, 0 );

    // setting position buffer
    trianglePositionBuffer.Initialize( sizeof( TRIANGLE_POSITION_DATA ), TRIANGLE_POSITION_DATA, nn::gfx::GpuAccess_VertexBuffer, 0 );

    // GX2_ATTRIB_FORMAT_16_FLOAT
    TRIANGLE_COLOR_DATA_F16x1_FLOAT[2].color = (u16)valuef16;

    // GX2_ATTRIB_FORMAT_16_16_FLOAT
    TRIANGLE_COLOR_DATA_F16x2_FLOAT[0].color[1] = (u16)valuef16;
    TRIANGLE_COLOR_DATA_F16x2_FLOAT[2].color[0] = (u16)valuef16;

    // GX2_ATTRIB_FORMAT_16_16_16_16_FLOAT
    TRIANGLE_COLOR_DATA_F16x4_FLOAT[0].color[1] = (u16)valuef16;
    TRIANGLE_COLOR_DATA_F16x4_FLOAT[1].color[2] = (u16)valuef16;
    TRIANGLE_COLOR_DATA_F16x4_FLOAT[2].color[0] = (u16)valuef16;

    for (int idx = 0; idx < NUM_ATTRIBUTE; idx++)
    {
#if NN_GFX_IS_TARGET_D3D
        if ( !IsAttributeAvailable( idx ) )
        {
            continue;
        }
#endif

        nn::gfx::VertexAttributeStateInfo attributeStates[ 2 ];
        nn::gfx::VertexBufferStateInfo bufferStates[ 2 ];

        SetupAttribute(g_AttributeProperty[ idx ]);

        g_Models[ idx ].pAttribute = &g_AttributeProperty[ idx ];

        if ( FORMAT_IS_INTEGER( g_AttributeProperty[ idx ].attributeFormat ) )
        {
            g_Models[ idx ].pShader = &g_SimpleIntShader;
        }
        else
        {
            g_Models[ idx ].pShader = &g_SimpleShader;
        }

        bufferStates[ g_TriangleAttribute.positionBufferIndex ].SetDefault();
        bufferStates[ g_TriangleAttribute.positionBufferIndex ].SetStride( g_TriangleAttribute.positionStride );

        bufferStates[ g_TriangleAttribute.colorBufferIndex ].SetDefault();
        bufferStates[ g_TriangleAttribute.colorBufferIndex ].SetStride( g_AttributeProperty[idx].colorStride );

        attributeStates[ g_Models[idx].pShader->positionLocation ].SetDefault();
        attributeStates[ g_Models[idx].pShader->positionLocation ].SetBufferIndex( g_TriangleAttribute.positionBufferIndex );
        attributeStates[ g_Models[idx].pShader->positionLocation ].SetShaderSlot( g_Models[ idx ].pShader->positionLocation );
        attributeStates[ g_Models[idx].pShader->positionLocation ].SetFormat( nn::gfx::AttributeFormat_32_32_32_Float );
        attributeStates[ g_Models[idx].pShader->positionLocation ].SetOffset( g_TriangleAttribute.positionOffset * sizeof(f32) );

        attributeStates[ g_Models[idx].pShader->colorLocation ].SetDefault();
        attributeStates[ g_Models[idx].pShader->colorLocation ].SetBufferIndex( g_TriangleAttribute.colorBufferIndex );
        attributeStates[ g_Models[idx].pShader->colorLocation ].SetShaderSlot( g_Models[idx].pShader->colorLocation );
        attributeStates[ g_Models[idx].pShader->colorLocation ].SetFormat( g_AttributeProperty[idx].attributeFormat );
        attributeStates[ g_Models[idx].pShader->colorLocation ].SetOffset( g_TriangleAttribute.colorOffset );

#ifdef USE_PIPELINES
        g_Models[idx].pShader->pipelineInfo.EditVertexStateInfo().SetVertexBufferStateInfoArray( bufferStates, 2 );
        g_Models[idx].pShader->pipelineInfo.EditVertexStateInfo().SetVertexAttributeStateInfoArray( attributeStates, 2 );

        // pipelines for SimpleIntShader are stored in SimpleShader
        size_t pipelineSize = nn::gfx::Pipeline::GetRequiredMemorySize( g_Models[idx].pShader->pipelineInfo );
        if ( pipelineSize )
        {
            g_Models[idx].pipelineData = DEMOGfxAllocMEM2( pipelineSize, nn::gfx::Pipeline::RequiredMemoryInfo_Alignment );
            g_Models[idx].pipeline.SetMemory( g_Models[idx].pipelineData, pipelineSize );
        }
        else
        {
            g_Models[idx].pipelineData = NULL;
        }
        g_Models[idx].pipeline.Initialize( &DEMODevice, g_Models[idx].pShader->pipelineInfo );
#else
        nn::gfx::VertexStateInfo info;
        info.SetDefault();
        info.SetVertexBufferStateInfoArray( bufferStates, 2 );
        info.SetVertexAttributeStateInfoArray( attributeStates, 2 );
        size_t size = nn::gfx::VertexState::GetRequiredMemorySize( info );
        if ( size )
        {
            g_Models[ idx ].vertexStateData = DEMOGfxAllocMEM2( size, nn::gfx::VertexState::RequiredMemoryInfo_Alignment );
        }
        g_Models[ idx ].vertexState.SetMemory( g_Models[ idx ].vertexStateData, size );
        g_Models[ idx ].vertexState.Initialize( &DEMODevice, info, g_Models[ idx ].pShader->shaders.GetShader() );
#endif
    }

    // Create Grid Viewports
    for ( int y = 0; y < GRID_MAX_Y; y++ )
    {
        for ( int x = 0; x < GRID_MAX_X; x++ )
        {
            // Set Viewport for each Model
            DEMOGfxSetViewportScissorState( &viewportScissor[ y ][ x ], &pViewportScissorData[ y ][ x ],
                static_cast< float >( x * GRID_W + GRID_OFFSET_X ),
#if NN_GFX_IS_TARGET_D3D
                static_cast< float >( DEMOColorBufferInfo.GetHeight() - ( y + 1 ) * GRID_H - GRID_OFFSET_Y ),
#else
                static_cast< float >( y * GRID_H + GRID_OFFSET_Y ),
#endif
                static_cast< float >( GRID_W ),
                static_cast< float >( GRID_H ),
                0.0f,
                1.0f,
                static_cast< float >( DEMOColorBufferInfo.GetHeight() ),
                false );
        }
    }

    // Create default viewport
    DEMOGfxSetViewportScissorState( &defaultViewportScissor, &pDefaultViewportScissorData,
        0.0f, 0.0f,
        static_cast< float >( DEMOColorBufferInfo.GetWidth() ),
        static_cast< float >( DEMOColorBufferInfo.GetHeight() ),
        0.0f, 1.0f,
        static_cast< float >( DEMOColorBufferInfo.GetHeight() ),
        false );

    return true;
} // NOLINT(impl/function_size)

static void SceneCleanup()
{
    trianglePositionBuffer.Finalize();

    for ( int i = 0; i < NUM_ATTRIBUTE; i++ )
    {
#ifdef USE_PIPELINES
        g_Models[i].pipeline.Finalize( &DEMODevice );
        if ( g_Models[i].pipelineData )
        {
            DEMOGfxFreeMEM2( g_Models[i].pipelineData );
        }
#else
        g_Models[ i ].vertexState.Finalize( &DEMODevice );
        if ( g_Models[i].vertexStateData )
        {
            DEMOGfxFreeMEM2( g_Models[i].vertexStateData );
        }
#endif

        // Free the Uniform Block
        g_AttributeProperty[ i ].uniformBuffer.Finalize();

        // Free the color attribute buffer/view
        g_AttributeProperty[ i ].buffer.Finalize();
    }

#ifndef USE_PIPELINES
    g_BlendState.Finalize( &DEMODevice );
    if ( g_pBlendStateData )
    {
        DEMOGfxFreeMEM2( g_pBlendStateData );
    }
    g_DepthStencilState.Finalize( &DEMODevice );
    g_RasterizerState.Finalize( &DEMODevice );
#endif

    // Free Index buffers
    triangleIndex16Buffer.Finalize();
    triangleIndex32Buffer.Finalize();

    // Free shaders
    DEMOGfxFreeShaders( &g_SimpleIntShader.shaders );
    DEMOGfxFreeShaders( &g_SimpleShader.shaders );

    // Delete Grid Viewports
    for ( int y = 0; y < GRID_MAX_Y; y++ )
    {
        for ( int x = 0; x < GRID_MAX_X; x++ )
        {
            viewportScissor[ y ][ x ].Finalize( &DEMODevice );
            if ( pViewportScissorData[ y ][ x ] )
            {
                DEMOGfxFreeMEM2( pViewportScissorData[ y ][ x ] );
            }
        }
    }
    defaultViewportScissor.Finalize( &DEMODevice );
    if ( pDefaultViewportScissorData )
    {
        DEMOGfxFreeMEM2( pDefaultViewportScissorData );
    }
}

static const int INTERVAL_DRAW_MODE = 90;

static void DoDraw()
{
    // 32 bit index draw

    // GX2_PRIMITIVE_QUADS
    if(s_frame % (INTERVAL_DRAW_MODE * 4) < INTERVAL_DRAW_MODE)
    {
        DEMOCommandBuffer.DrawIndexed( nn::gfx::PrimitiveTopology_TriangleList, nn::gfx::IndexFormat_Uint32,
            triangleIndex32Buffer.gpuAddress, INDEX_COUNT, 0, 1, 0 );
    }
    else if(s_frame % (INTERVAL_DRAW_MODE * 4) < INTERVAL_DRAW_MODE * 2)
    {
        DEMOCommandBuffer.DrawIndexed( nn::gfx::PrimitiveTopology_TriangleList, nn::gfx::IndexFormat_Uint16,
            triangleIndex16Buffer.gpuAddress, INDEX_COUNT, 0, 1, 0 );
    }
    else if(s_frame % (INTERVAL_DRAW_MODE * 4) < INTERVAL_DRAW_MODE * 3)
    {
        // Indexed immediate not supported for nn::gfx
        DEMOCommandBuffer.DrawIndexed( nn::gfx::PrimitiveTopology_TriangleList, nn::gfx::IndexFormat_Uint32,
            triangleIndex32Buffer.gpuAddress, INDEX_COUNT, 0, 1, 0 );
    }
    else
    {
        // Indexed immediate not supported for nn::gfx
        DEMOCommandBuffer.DrawIndexed( nn::gfx::PrimitiveTopology_TriangleList, nn::gfx::IndexFormat_Uint16,
            triangleIndex16Buffer.gpuAddress, INDEX_COUNT, 0, 1, 0 );
    }
    // Draw information

    // Set Font Size
    DEMOFontSetGridSize(19,5);
    DEMOFontPrintf(0, 2, g_AttributeProperty[g_CurrentAttribute].name);

    switch(g_AttributeProperty[g_CurrentAttribute].attributeFormat >> nn::gfx::TypeFormat_Bits)
    {
    case nn::gfx::ChannelFormat_R8:
    case nn::gfx::ChannelFormat_R16:
    case nn::gfx::ChannelFormat_R32:
        DEMOFontPrintf(0, 3, "RED");
        break;
    case nn::gfx::ChannelFormat_R4_G4:
    case nn::gfx::ChannelFormat_R8_G8:
    case nn::gfx::ChannelFormat_R16_G16:
    case nn::gfx::ChannelFormat_R32_G32:
        DEMOFontPrintf(0, 3, "RED GREEN");
        break;
    case nn::gfx::ChannelFormat_R5_G6_B5:
    case nn::gfx::ChannelFormat_B5_G6_R5:
    case nn::gfx::ChannelFormat_R9_G9_B9_E5:
    case nn::gfx::ChannelFormat_R11_G11_B10:
    case nn::gfx::ChannelFormat_R10_G11_B11:
    case nn::gfx::ChannelFormat_R32_G32_B32:
        DEMOFontPrintf(0, 3, "RED GREEN BLUE");
        break;
    case nn::gfx::ChannelFormat_R4_G4_B4_A4:
    case nn::gfx::ChannelFormat_R5_G5_B5_A1:
    case nn::gfx::ChannelFormat_R8_G8_B8_A8:
    case nn::gfx::ChannelFormat_B8_G8_R8_A8:
    case nn::gfx::ChannelFormat_R10_G10_B10_A2:
    case nn::gfx::ChannelFormat_R16_G16_B16_A16:
    case nn::gfx::ChannelFormat_R32_G32_B32_A32:
    case nn::gfx::ChannelFormat_A1_B5_G5_R5:
        DEMOFontPrintf(0, 3, "RED GREEN BLUE");
        break;
    default:
        break;
    }
}

static void DrawTitle()
{
    static OSTime lastTime = 0;
    OSTime now = OSGetTime();
    double dtime = OSTicksToMilliseconds(now - lastTime) / 1000.0;
    lastTime = now;

    // Set Font Size
    DEMOFontSetGridSize( 80, 24 );
    if ( s_PrintForTest )
    {
        DEMOFontPrintf( 4, 1, "<Attribute Format Test>" );
    }
    else
    {
        DEMOFontPrintf( 4, 1, "<Attribute Format Test>: %.1f fps", 1.0 / dtime);
    }
    if ( s_frame % ( INTERVAL_DRAW_MODE * 4 ) < INTERVAL_DRAW_MODE )
    {
        DEMOFontPrintf( 30, 1, "32-bit index buffer draw" );
    }
    else if ( s_frame % ( INTERVAL_DRAW_MODE * 4 ) < INTERVAL_DRAW_MODE * 2 )
    {
        DEMOFontPrintf( 30, 1, "16-bit index buffer draw" );
    }
    else if ( s_frame % ( INTERVAL_DRAW_MODE * 4 ) < INTERVAL_DRAW_MODE * 3 )
    {
        DEMOFontPrintf( 30, 1, "32-bit immediate draw" );
    }
    else
    {
        DEMOFontPrintf( 30, 1, "16-bit immediate draw" );
    }


}
// The draw function for the rendering portions of this sample.
static void SceneDraw()
{
    // When using the DEMO library, it is necessary to call
    // DEMOGfxBeforeRender and DEMOGfxDoneRender before and after drawing.
    DEMOGfxBeforeRender();
    nn::gfx::ColorTargetView* pScanBufferView = DEMOGetColorBufferView();

    DEMOCommandBuffer.ClearColor(pScanBufferView, 0.5f, 0.5f, 0.5f, 1.0f, NULL);
    DEMOCommandBuffer.ClearDepthStencil(&DEMODepthBufferView, 1.0f, 0, nn::gfx::DepthStencilClearMode_DepthStencil, NULL);

#if NN_GFX_IS_TARGET_GX
    GX2SetShaderMode( GX2_SHADER_MODE_UNIFORM_BLOCK );
#endif

    DEMOGfxSetDefaultRenderTarget();

    // Init g_CurrentAttribute
    g_CurrentAttribute = 0;

    // Draw Grid Models with each format
    for (int y = 0; y < GRID_MAX_Y; y++)
    {
        for (int x = 0; x < GRID_MAX_X; x++)
        {
#if NN_GFX_IS_TARGET_D3D
            if ( !IsAttributeAvailable( g_CurrentAttribute ) )
            {
                g_CurrentAttribute++;
                continue;
            }
#endif

            // Set Viewport for each Model
            DEMOCommandBuffer.SetViewportScissorState( &viewportScissor[ y ][ x ] );

#ifdef USE_PIPELINES
            DEMOCommandBuffer.SetPipeline( &g_Models[ g_CurrentAttribute ].pipeline );
#else
            DEMOCommandBuffer.SetBlendState( &g_BlendState );
            DEMOCommandBuffer.SetDepthStencilState( &g_DepthStencilState );
            DEMOCommandBuffer.SetRasterizerState( &g_RasterizerState );
            DEMOCommandBuffer.SetVertexState( &g_Models[ g_CurrentAttribute ].vertexState );
            DEMOCommandBuffer.SetShader( g_Models[ g_CurrentAttribute ].pShader->shaders.GetShader(), nn::gfx::ShaderStageBit_All );
#endif

            // This call with set all shaders.
            DEMOCommandBuffer.SetConstantBuffer( g_Models[g_CurrentAttribute].pShader->divisorLocation, nn::gfx::ShaderStage_Vertex, g_AttributeProperty[g_CurrentAttribute].uniformBuffer.gpuAddress,
                g_AttributeProperty[g_CurrentAttribute].uniformBuffer.size );

            DEMOCommandBuffer.SetVertexBuffer( g_TriangleAttribute.positionBufferIndex, trianglePositionBuffer.gpuAddress, g_TriangleAttribute.positionStride,
                trianglePositionBuffer.size );
            DEMOCommandBuffer.SetVertexBuffer( g_TriangleAttribute.colorBufferIndex, g_AttributeProperty[ g_CurrentAttribute ].buffer.gpuAddress, g_AttributeProperty[g_CurrentAttribute].colorStride,
                g_AttributeProperty[ g_CurrentAttribute ].size);

            // This command will actually result in a draw command being issued
            DoDraw();

            g_CurrentAttribute++;
            if(g_CurrentAttribute >= NUM_ATTRIBUTE)
            {
                g_CurrentAttribute = 0;
                break;
            }
        }
    }

    DEMOCommandBuffer.SetViewportScissorState( &defaultViewportScissor );
    DrawTitle();

    // This function will handle presenting the rendered buffer to the screen
    // by swapping the color buffer, setting the new context state,
    // resetting the color buffer, and flushing the pipeline.
    DEMOGfxDoneRender();

    s_frame++;
}

//
//  Main Function
//  メイン関数です。
//

//extern "C" void nnMain()
TEST(GfxFormat, Run)
{
    int argc = nnt::GetHostArgc();
    char** argv = nnt::GetHostArgv();

    // Initialize the DEMO library, DEMO test system, and create the
    // primary display.
    DEMOInit();
    DEMOTestInit(argc, argv);

    // This function will create the appropriate size color and depth
    // buffers, setup the scan buffer, and initialize the viewport and
    // scissor rectangles to be the entire screen.
    DEMOGfxInit(argc, argv);

    // This function will initialize font utility
    DEMOFontInit();

    DEMOTestIsUseHlslccGlsl() ? g_ShaderFileIdx = 1 : g_ShaderFileIdx = 0;

    // Disable printing FPS during test
    for ( int i = 0; i < argc; i++ )
    {
        if ( !strstr(argv[i], "TEST_SELECT" ) )
        {
            s_PrintForTest = true;
        }
    }

    // After the DEMO library is initialize the sample's initialization
    // function can be called.
    SceneInit();

    while (DEMOIsRunning())
    {
        SceneDraw();
    }

    SceneCleanup();

    DEMOFontShutdown();
    DEMOTestShutdown();
    DEMOGfxShutdown();
    DEMOShutdown();

    SUCCEED();
}
