﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
* @examplesource{GfxPrimitiveRenderer.cpp,PageSampleGfxPrimitiveRenderer}
*
* @brief
*  プリミティブ描画のサンプルプログラム
*/

/**
* @page PageSampleGfxPrimitiveRenderer プリミティブの描画
* @tableofcontents
*
* @brief
*  サンプルプログラム GfxPrimitiveRenderer の解説です。
*
* @section PageSampleGfxPrimitiveRenderer_SectionBrief 概要
*  nns::gfx::PrimitiveRenderer を使用してプリミティブ描画を行うシンプルなサンプルです。
*
* @section PageSampleGfxPrimitiveRenderer_SectionFileStructure ファイル構成
*  本サンプルプログラムは @link ../../../Samples/Sources/Applications/GfxPrimitiveRenderer
*  Samples/Sources/Applications/GfxPrimitiveRenderer @endlink 以下にあります。
*
* @section PageSampleGfxPrimitiveRenderer_SectionNecessaryEnvironment 必要な環境
*  画面表示が利用可能である必要があります。
*
* @section PageSampleGfxPrimitiveRenderer_SectionHowToOperate 操作方法
*  特にありません。
*
* @section PageSampleGfxPrimitiveRenderer_SectionPrecaution 注意事項
*  特にありません。
*
* @section PageSampleGfxPrimitiveRenderer_SectionHowToExecute 実行手順
*  サンプルプログラムをビルドし、実行してください。
*
* @section PageSampleGfxPrimitiveRenderer_SectionDetail 解説
*
* @subsection PageSampleGfxPrimitiveRenderer_SectionSampleProgram サンプルプログラム
*  以下に本サンプルプログラムのソースコードを引用します。
*
*  GfxPrimitiveRenderer/GfxPrimitiveRenderer.cpp
*  @includelineno GfxPrimitiveRenderer/GfxPrimitiveRenderer.cpp
*
* @subsection PageSampleGfxPrimitiveRenderer_SectionSampleDetail サンプルプログラムの解説
*  サンプルプログラムの処理の流れは以下の通りです。
*
* - デバイスを初期化
* - メモリプールを初期化
* - スワップチェーンを初期化
* - キューを初期化
* - コマンドバッファを初期化
* - ビューポートを初期化
* - ラスタライザステートを初期化
* - カラーバッファを初期化
* - 深度ステンシルバッファを初期化
* - UserShader向けバッファを初期化
* - サンプラを初期化
* - サンプラデスクリプタプールを初期化
* - テクスチャデスクリプタプールを初期化
* - テクスチャリソースを初期化
* - nns::gfx::PrimitiveRendererを初期化
* - UserShaderを初期化
* - ループ開始
* - シェーダバリエーションを選択
* - コマンドリストを作成
* - コマンドリストを実行
* - ディスプレイへプレゼンテーション
* - ループ開始に戻る
* - 各種オブジェクトを破棄
*
*/

#include <cstdlib>
#include <cstring>

#include <nn/nn_Assert.h>

#include <nn/fs.h>
#include <nn/init.h>
#include <nn/gfx.h>
#include <nn/vi.h>
#include <nn/gfx/util/gfx_DebugFontTextWriter.h>
#include <nn/nn_Log.h>
#include <nn/util/util_Color.h>

#include <nn/hws/hws_Message.h>

#if NN_GFX_IS_TARGET_NVN
#include <nvn/nvn.h>
#include <nvn/nvn_FuncPtrInline.h>
#if defined( NN_BUILD_TARGET_PLATFORM_OS_NN )
#include <nvnTool/nvnTool_GlslcInterface.h>
#endif
#endif

#if defined( NN_BUILD_TARGET_PLATFORM_OS_NN ) && defined( NN_BUILD_APISET_NX )
#include <nv/nv_MemoryManagement.h>
#endif

#define NN_PERF_PROFILE_ENABLED
#include <nn/perf.h>

#include <nns/gfx/gfx_PrimitiveRenderer.h>
#include <nns/gfx/gfx_PrimitiveRendererMeterDrawer.h>
#include <nns/gfx/gfx_PrimitiveRendererMeshRes.h>

#include <nn/nn_TimeSpan.h>
#include <nn/os.h>

#include <nnt.h>

namespace {

    int g_BufferDescriptorBaseIndex = 0;
    int g_TextureDescriptorBaseIndex = 0;
    int g_SamplerDescriptorBaseIndex = 0;

    int g_RenderWidth = 0;
    int g_RenderHeight = 0;

    nns::gfx::PrimitiveRenderer::MeterDrawer g_MeterDrawer;

    //-----------------------------------------------------------------------------
    // メモリ
    nn::util::BytePtr g_pMemoryHeap(NULL);
    nn::util::BytePtr g_pMemory(NULL);

    static size_t g_VisiblePoolMemorySize = 16 * 1024 * 1024;
    static size_t g_InvisiblePoolMemorySize = 64 * 1024 * 1024;

    void* g_pVisiblePoolMemory = NULL;
    void* g_pInvisiblePoolMemory = NULL;
    void* g_pMemoryPoolStart = NULL;
    ptrdiff_t g_MemoryPoolOffset = 0;
    void* g_pInvisibleMemoryPoolStart = NULL;
    ptrdiff_t g_InvisibleMemoryPoolOffset = 0;

    void* g_MountRomCacheBuffer = NULL;

    //  nn::vi オブジェクト
    nn::vi::Display*                g_pDisplay;
    nn::vi::Layer*                  g_pLayer;

    // デバイスを初期化
    nn::gfx::Device g_Device;
    void InitializeDevice()
    {
        nn::gfx::Device::InfoType info;
        info.SetDefault();
        info.SetApiVersion(nn::gfx::ApiMajorVersion, nn::gfx::ApiMinorVersion);
        g_Device.Initialize(info);
    }

    //  レイヤの初期処理
    void InitializeLayer()
    {
        nn::vi::Initialize();

        nn::Result result = nn::vi::OpenDefaultDisplay(&g_pDisplay);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        NN_UNUSED(result);

        g_RenderWidth = 1280;
        g_RenderHeight = 720;

        result = nn::vi::CreateLayer(&g_pLayer, g_pDisplay);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        result = nn::vi::SetLayerScalingMode(g_pLayer, nn::vi::ScalingMode_FitToLayer);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    }

    //  レイヤ終了処理
    void FinalizeLayer()
    {
        nn::vi::DestroyLayer(g_pLayer);
        nn::vi::CloseDisplay(g_pDisplay);
        nn::vi::Finalize();
    }

    nn::gfx::MemoryPool g_MemoryPool;
    void InitializeMemoryPool()
    {
        nn::gfx::MemoryPool::InfoType info;
        info.SetDefault();
        info.SetMemoryPoolProperty(nn::gfx::MemoryPoolProperty_CpuUncached
            | nn::gfx::MemoryPoolProperty_GpuCached);

        g_VisiblePoolMemorySize = nn::util::align_up(g_VisiblePoolMemorySize,
            nn::gfx::MemoryPool::GetPoolMemorySizeGranularity(&g_Device, info));

        // Determine alignment to allocate space on an alignment boundary
        size_t alignment = nn::gfx::MemoryPool::GetPoolMemoryAlignment(&g_Device, info);
        g_pVisiblePoolMemory = malloc(g_VisiblePoolMemorySize + alignment);

        g_pMemoryPoolStart = nn::util::BytePtr(g_pVisiblePoolMemory).AlignUp(alignment).Get();
        info.SetPoolMemory(g_pMemoryPoolStart, nn::util::align_down(g_VisiblePoolMemorySize,
            nn::gfx::MemoryPool::GetPoolMemorySizeGranularity(&g_Device, info)));
        g_MemoryPool.Initialize(&g_Device, info);

        g_MemoryPoolOffset = 0;
    }

    nn::gfx::MemoryPool g_InvisibleMemoryPool;
    void InitializeInvisibleMemoryPool()
    {
        nn::gfx::MemoryPool::InfoType info;
        info.SetDefault();
        info.SetMemoryPoolProperty(nn::gfx::MemoryPoolProperty_CpuInvisible |
            nn::gfx::MemoryPoolProperty_GpuCached |
            nn::gfx::MemoryPoolProperty_Compressible);

        g_InvisiblePoolMemorySize = nn::util::align_up(g_InvisiblePoolMemorySize,
            nn::gfx::MemoryPool::GetPoolMemorySizeGranularity(&g_Device, info));

        // Determine alignment to allocate space on an alignment boundary
        size_t alignment = nn::gfx::MemoryPool::GetPoolMemoryAlignment(&g_Device, info);
        g_pInvisiblePoolMemory = malloc(g_InvisiblePoolMemorySize + alignment);

        g_pInvisibleMemoryPoolStart = nn::util::BytePtr(g_pInvisiblePoolMemory).AlignUp(alignment).Get();
        info.SetPoolMemory(g_pInvisibleMemoryPoolStart, nn::util::align_down(g_InvisiblePoolMemorySize,
            nn::gfx::MemoryPool::GetPoolMemorySizeGranularity(&g_Device, info)));
        g_InvisibleMemoryPool.Initialize(&g_Device, info);

        g_InvisibleMemoryPoolOffset = 0;
    }


    // スワップチェーンを初期化
    nn::gfx::SwapChain g_SwapChain;
    void InitializeSwapChain()
    {
        nn::gfx::SwapChain::InfoType info;

        info.SetDefault();
        info.SetLayer(g_pLayer);
        info.SetWidth(g_RenderWidth);
        info.SetHeight(g_RenderHeight);
        info.SetFormat(nn::gfx::ImageFormat_R8_G8_B8_A8_UnormSrgb);
        info.SetBufferCount(2);
        if (NN_STATIC_CONDITION(nn::gfx::SwapChain::IsMemoryPoolRequired))
        {
            size_t size = g_SwapChain.CalculateScanBufferSize(&g_Device, info);
            g_InvisibleMemoryPoolOffset = nn::util::align_up(g_InvisibleMemoryPoolOffset,
                nn::gfx::SwapChain::GetScanBufferAlignment(&g_Device, info));
            g_SwapChain.Initialize(&g_Device, info, &g_InvisibleMemoryPool, g_InvisibleMemoryPoolOffset, size);
            g_InvisibleMemoryPoolOffset += size;
        }
        else
        {
            g_SwapChain.Initialize(&g_Device, info, NULL, 0, 0);
        }
    }

    // キューを初期化
    nn::gfx::Queue g_Queue;
    void InitializeQueue()
    {
        nn::gfx::Queue::InfoType info;
        info.SetDefault();
        info.SetCapability(nn::gfx::QueueCapability_Graphics);
        g_Queue.Initialize(&g_Device, info);
    }

    // コマンドバッファを初期化
    nn::gfx::CommandBuffer g_CommandBuffer;
    nn::gfx::CommandBuffer g_CommandBufferBegin;
    nn::gfx::CommandBuffer g_CommandBufferEnd;
    void InitializeCommandBuffer()
    {
        nn::gfx::CommandBuffer::InfoType info;
        info.SetDefault();
        info.SetQueueCapability(nn::gfx::QueueCapability_Graphics);
        info.SetCommandBufferType(nn::gfx::CommandBufferType_Direct);
        g_CommandBuffer.Initialize(&g_Device, info);

        g_CommandBufferBegin.Initialize(&g_Device, info);
        g_CommandBufferEnd.Initialize(&g_Device, info);
    }

    // フレームワーク関連
    nn::gfx::Fence g_DisplayFence[2];
    nn::gfx::Fence g_GpuFence[2];
    nn::gfx::Semaphore g_DisplaySemaphore[2];
    nn::gfx::ColorTargetView* g_pScanBufferViews[2];
    int g_NextScanBufferIndex;
    void InitializeFramework()
    {
        nn::gfx::Fence::InfoType fenceInfo;
        fenceInfo.SetDefault();
        g_DisplayFence[0].Initialize(&g_Device, fenceInfo);
        g_DisplayFence[1].Initialize(&g_Device, fenceInfo);
        g_GpuFence[0].Initialize(&g_Device, fenceInfo);
        g_GpuFence[1].Initialize(&g_Device, fenceInfo);

        nn::gfx::Semaphore::InfoType semaphoreInfo;
        semaphoreInfo.SetDefault();
        g_DisplaySemaphore[0].Initialize(&g_Device, semaphoreInfo);
        g_DisplaySemaphore[1].Initialize(&g_Device, semaphoreInfo);

        g_SwapChain.GetScanBufferViews(g_pScanBufferViews, 2);
    }

    void FinalizeFramework()
    {
        g_DisplayFence[0].Finalize(&g_Device);
        g_DisplayFence[1].Finalize(&g_Device);
        g_GpuFence[0].Finalize(&g_Device);
        g_GpuFence[1].Finalize(&g_Device);

        g_DisplaySemaphore[0].Finalize(&g_Device);
        g_DisplaySemaphore[1].Finalize(&g_Device);
    }

    // ビューポートシザーを初期化
    nn::gfx::ViewportScissorState g_ViewportScissor;
    void InitializeViewport()
    {
        nn::gfx::ViewportScissorState::InfoType info;
        info.SetDefault();
        info.SetScissorEnabled(true);
        nn::gfx::ViewportStateInfo viewportInfo;
        {
            viewportInfo.SetDefault();
            viewportInfo.SetWidth(static_cast< float >(g_RenderWidth));
            viewportInfo.SetHeight(static_cast< float >(g_RenderHeight));
        }
        nn::gfx::ScissorStateInfo scissorInfo;
        {
            scissorInfo.SetDefault();
            scissorInfo.SetWidth(g_RenderWidth);
            scissorInfo.SetHeight(g_RenderHeight);
        }
        info.SetViewportStateInfoArray(&viewportInfo, 1);
        info.SetScissorStateInfoArray(&scissorInfo, 1);
        g_ViewportScissor.Initialize(&g_Device, info);
    }

    // Initialize the rasterizer state
    nn::gfx::RasterizerState g_RasterizerState;
    static void InitializeRasterizerState()
    {
        nn::gfx::RasterizerState::InfoType info;
        info.SetDefault();
        info.SetCullMode(nn::gfx::CullMode_None);
        info.SetPrimitiveTopologyType(nn::gfx::PrimitiveTopologyType_Triangle);
        info.SetScissorEnabled(true);
        info.SetDepthClipEnabled(false);
        g_RasterizerState.Initialize(&g_Device, info);
    }

    // カラーバッファを初期化
    nn::gfx::Texture g_ColorBuffer;
    nn::gfx::ColorTargetView g_ColorTargetView;
    nn::gfx::TextureView g_ColorTextureView;
    nn::gfx::DescriptorSlot g_ColorTextureDescSlot;
    void InitializeColorBuffer()
    {
        nn::gfx::Texture::InfoType info;
        info.SetDefault();
        info.SetWidth(g_RenderWidth);
        info.SetHeight(g_RenderHeight);
        info.SetGpuAccessFlags(nn::gfx::GpuAccess_ColorBuffer);
        info.SetImageStorageDimension(nn::gfx::ImageStorageDimension_2d);
        info.SetImageFormat(nn::gfx::ImageFormat_R16_G16_B16_A16_Float);
        info.SetMipCount(1);

        g_InvisibleMemoryPoolOffset = nn::util::align_up(g_InvisibleMemoryPoolOffset, nn::gfx::Texture::CalculateMipDataAlignment(&g_Device, info));
        size_t size = nn::gfx::Texture::CalculateMipDataSize(&g_Device, info);
        g_ColorBuffer.Initialize(&g_Device, info, &g_InvisibleMemoryPool, g_InvisibleMemoryPoolOffset, size);
        g_InvisibleMemoryPoolOffset += size;

        // カラーターゲットビューを初期化
        nn::gfx::ColorTargetView::InfoType targetInfo;
        targetInfo.SetDefault();
        targetInfo.SetImageDimension(nn::gfx::ImageDimension_2d);
        targetInfo.SetImageFormat(nn::gfx::ImageFormat_R16_G16_B16_A16_Float);
        targetInfo.SetTexturePtr(&g_ColorBuffer);
        g_ColorTargetView.Initialize(&g_Device, targetInfo);

        // カラーバッファのテクスチャビューを初期化
        nn::gfx::TextureView::InfoType viewInfo;
        viewInfo.SetDefault();
        viewInfo.SetImageDimension(nn::gfx::ImageDimension_2d);
        viewInfo.SetImageFormat(nn::gfx::ImageFormat_R16_G16_B16_A16_Float);
        viewInfo.SetTexturePtr(&g_ColorBuffer);
        g_ColorTextureView.Initialize(&g_Device, viewInfo);

        // デスクリプタスロット設定
        //    g_TextureDescPoolAllocator.AllocateSlotForFontTexture( &g_ColorTextureDescSlot, g_ColorTextureView );
    }

    nn::gfx::Texture g_BlurColorBuffer;
    nn::gfx::ColorTargetView g_BlurColorTargetView;
    nn::gfx::TextureView g_BlurColorTextureView;
    nn::gfx::DescriptorSlot g_BlurColorTextureDescSlot;
    void InitializeBlurBuffer()
    {
        nn::gfx::Texture::InfoType info;
        info.SetDefault();
        info.SetWidth(g_RenderWidth);
        info.SetHeight(g_RenderHeight);
        info.SetGpuAccessFlags(nn::gfx::GpuAccess_ColorBuffer);
        info.SetImageStorageDimension(nn::gfx::ImageStorageDimension_2d);
        info.SetImageFormat(nn::gfx::ImageFormat_R16_G16_B16_A16_Float);
        info.SetMipCount(1);

        g_InvisibleMemoryPoolOffset = nn::util::align_up(g_InvisibleMemoryPoolOffset, nn::gfx::Texture::CalculateMipDataAlignment(&g_Device, info));
        size_t size = nn::gfx::Texture::CalculateMipDataSize(&g_Device, info);
        g_BlurColorBuffer.Initialize(&g_Device, info, &g_InvisibleMemoryPool, g_InvisibleMemoryPoolOffset, size);
        g_InvisibleMemoryPoolOffset += size;

        // カラーターゲットビューを初期化
        nn::gfx::ColorTargetView::InfoType targetInfo;
        targetInfo.SetDefault();
        targetInfo.SetImageDimension(nn::gfx::ImageDimension_2d);
        targetInfo.SetImageFormat(nn::gfx::ImageFormat_R16_G16_B16_A16_Float);
        targetInfo.SetTexturePtr(&g_BlurColorBuffer);
        g_BlurColorTargetView.Initialize(&g_Device, targetInfo);

        // カラーバッファのテクスチャビューを初期化
        nn::gfx::TextureView::InfoType viewInfo;
        viewInfo.SetDefault();
        viewInfo.SetImageDimension(nn::gfx::ImageDimension_2d);
        viewInfo.SetImageFormat(nn::gfx::ImageFormat_R16_G16_B16_A16_Float);
        viewInfo.SetTexturePtr(&g_BlurColorBuffer);
        g_BlurColorTextureView.Initialize(&g_Device, viewInfo);
    }

    // 深度ステンシルバッファを初期化
    nn::gfx::Texture g_DepthStencilBuffer;
    nn::gfx::DepthStencilView g_DepthStencilView;
    void InitializeDepthStencilBuffer()
    {
        // 深度ステンシルバッファ用のテクスチャを初期化
        nn::gfx::Texture::InfoType info;
        info.SetDefault();
        info.SetWidth(g_RenderWidth);
        info.SetHeight(g_RenderHeight);
        info.SetGpuAccessFlags(nn::gfx::GpuAccess_DepthStencil);
        info.SetImageStorageDimension(nn::gfx::ImageStorageDimension_2d);
        info.SetImageFormat(nn::gfx::ImageFormat_D32_Float);
        info.SetMipCount(1);
        g_InvisibleMemoryPoolOffset = nn::util::align_up(g_InvisibleMemoryPoolOffset, nn::gfx::Texture::CalculateMipDataAlignment(&g_Device, info));
        size_t size = nn::gfx::Texture::CalculateMipDataSize(&g_Device, info);
        g_DepthStencilBuffer.Initialize(&g_Device, info, &g_InvisibleMemoryPool, g_InvisibleMemoryPoolOffset, size);
        g_InvisibleMemoryPoolOffset += size;

        // 深度ステンシルビューを初期化
        nn::gfx::DepthStencilView::InfoType viewInfo;
        viewInfo.SetDefault();
        viewInfo.SetImageDimension(nn::gfx::ImageDimension_2d);
        viewInfo.SetTexturePtr(&g_DepthStencilBuffer);
        g_DepthStencilView.Initialize(&g_Device, viewInfo);
    }

    // サンプラを初期化
    nn::gfx::Sampler g_Sampler;
    void InitializeSampler()
    {
        nn::gfx::Sampler::InfoType info;
        info.SetDefault();
        info.SetFilterMode(nn::gfx::FilterMode_MinLinear_MagLinear_MipPoint);
        info.SetAddressU(nn::gfx::TextureAddressMode_Mirror);
        info.SetAddressV(nn::gfx::TextureAddressMode_Mirror);
        info.SetAddressW(nn::gfx::TextureAddressMode_Mirror);
        g_Sampler.Initialize(&g_Device, info);
    }

    nn::gfx::DescriptorPool g_BufferDescriptorPool;
    void InitializeBufferDescriptorPool()
    {
        nn::gfx::DescriptorPool::InfoType info;
        info.SetDefault();
        info.SetDescriptorPoolType(nn::gfx::DescriptorPoolType_BufferView);
        info.SetSlotCount(g_BufferDescriptorBaseIndex + 1);
        size_t size = nn::gfx::DescriptorPool::CalculateDescriptorPoolSize(&g_Device, info);
        g_MemoryPoolOffset = nn::util::align_up(g_MemoryPoolOffset,
            nn::gfx::DescriptorPool::GetDescriptorPoolAlignment(&g_Device, info));
        g_BufferDescriptorPool.Initialize(&g_Device, info, &g_MemoryPool, g_MemoryPoolOffset, size);
        g_MemoryPoolOffset += size;
    }

    nn::gfx::DescriptorPool g_SamplerDescriptorPool;
    void InitializeSamplerDescriptorPool()
    {
        nn::gfx::DescriptorPool::InfoType info;
        info.SetDefault();
        info.SetDescriptorPoolType(nn::gfx::DescriptorPoolType_Sampler);
        info.SetSlotCount(g_SamplerDescriptorBaseIndex + 1);
        size_t size = nn::gfx::DescriptorPool::CalculateDescriptorPoolSize(&g_Device, info);
        g_MemoryPoolOffset = nn::util::align_up(g_MemoryPoolOffset,
            nn::gfx::DescriptorPool::GetDescriptorPoolAlignment(&g_Device, info));
        g_SamplerDescriptorPool.Initialize(&g_Device, info,
            &g_MemoryPool, g_MemoryPoolOffset, size);
        g_MemoryPoolOffset += size;
    }

    nn::gfx::DescriptorPool g_TextureDescriptorPool;
    void InitializeTextureDescriptorPool()
    {
        nn::gfx::DescriptorPool::InfoType info;
        info.SetDefault();
        info.SetDescriptorPoolType(nn::gfx::DescriptorPoolType_TextureView);
        info.SetSlotCount(g_TextureDescriptorBaseIndex + 4);
        size_t size = nn::gfx::DescriptorPool::CalculateDescriptorPoolSize(&g_Device, info);
        g_MemoryPoolOffset = nn::util::align_up(g_MemoryPoolOffset,
            nn::gfx::DescriptorPool::GetDescriptorPoolAlignment(&g_Device, info));
        g_TextureDescriptorPool.Initialize(&g_Device, info, &g_MemoryPool, g_MemoryPoolOffset, size);
        g_MemoryPoolOffset += size;
    }

#if defined( NN_BUILD_TARGET_PLATFORM_OS_NN ) && defined( NN_BUILD_APISET_NX )
    void* Allocate(size_t size, size_t alignment, void*)
    {
        return aligned_alloc(alignment, nn::util::align_up(size, alignment));
    }
    void Free(void* addr, void*)
    {
        free(addr);
    }
    void* Reallocate(void* addr, size_t newSize, void*)
    {
        return realloc(addr, newSize);
    }
#endif

    void InitializeResources()
    {
#if defined( NN_BUILD_TARGET_PLATFORM_OS_NN ) && defined( NN_BUILD_APISET_NX )
        // グラフィックスシステムのためのメモリ周りの初期化を行います。
        {
            const size_t GraphicsSystemMemorySize = 8 * 1024 * 1024;
            nv::SetGraphicsAllocator(Allocate, Free, Reallocate, NULL);
            nv::InitializeGraphics(malloc(GraphicsSystemMemorySize), GraphicsSystemMemorySize);
        }
        // グラフィックス開発者向けツールおよびデバッグレイヤのためのメモリアロケータを設定します。
        nv::SetGraphicsDevtoolsAllocator(Allocate, Free, Reallocate, NULL);
        // GLSLC のためのメモリアロケータを設定します。
        glslcSetAllocator(Allocate, Free, Reallocate, NULL);
#endif

        // メモリ
        g_pMemoryHeap.Reset(malloc(32 * 1024 * 1024));
        g_pMemory = g_pMemoryHeap;

        // レイヤの初期化
        InitializeLayer();

        // ライブラリを初期化
        nn::gfx::Initialize();

        InitializeDevice();

#if NN_GFX_IS_TARGET_NVN
        nn::gfx::Device::DataType& deviceData = nn::gfx::AccessorToData(g_Device);
        nvnDeviceGetInteger(deviceData.pNvnDevice,
            NVN_DEVICE_INFO_RESERVED_TEXTURE_DESCRIPTORS, &g_TextureDescriptorBaseIndex);
        nvnDeviceGetInteger(deviceData.pNvnDevice,
            NVN_DEVICE_INFO_RESERVED_SAMPLER_DESCRIPTORS, &g_SamplerDescriptorBaseIndex);
#endif

        InitializeMemoryPool();
        InitializeInvisibleMemoryPool();

        InitializeSwapChain();
        InitializeQueue();
        InitializeFramework();

        InitializeCommandBuffer();
        InitializeViewport();

        InitializeRasterizerState();

        InitializeColorBuffer();
        InitializeDepthStencilBuffer();
        InitializeBlurBuffer();

        InitializeSampler();

        InitializeBufferDescriptorPool();
        InitializeSamplerDescriptorPool();
        InitializeTextureDescriptorPool();

        //InitializeResTextureFile();

        NN_ASSERT(g_pMemoryHeap.Distance(g_pMemory.Get()) < 32 * 1024 * 1024);
        NN_ASSERT(g_MemoryPoolOffset < static_cast<ptrdiff_t>(g_VisiblePoolMemorySize));
        NN_ASSERT(g_InvisibleMemoryPoolOffset < static_cast<ptrdiff_t>(g_InvisiblePoolMemorySize));
    }

    void FinalizeResources()
    {
        // 各オブジェクトを破棄
        //FinalizeResTextureFile();

        FinalizeFramework();

        g_Sampler.Finalize(&g_Device);
        g_BlurColorTextureView.Finalize(&g_Device);
        g_BlurColorTargetView.Finalize(&g_Device);
        g_BlurColorBuffer.Finalize(&g_Device);
        g_DepthStencilView.Finalize(&g_Device);
        g_DepthStencilBuffer.Finalize(&g_Device);
        g_ColorTextureView.Finalize(&g_Device);
        g_ColorTargetView.Finalize(&g_Device);
        g_ColorBuffer.Finalize(&g_Device);
        g_RasterizerState.Finalize(&g_Device);
        g_BufferDescriptorPool.Finalize(&g_Device);
        g_TextureDescriptorPool.Finalize(&g_Device);
        g_SamplerDescriptorPool.Finalize(&g_Device);

        g_ViewportScissor.Finalize(&g_Device);
        g_CommandBuffer.Finalize(&g_Device);
        g_SwapChain.Finalize(&g_Device);
        g_Queue.Finalize(&g_Device);
        g_InvisibleMemoryPool.Finalize(&g_Device);
        g_MemoryPool.Finalize(&g_Device);
        g_Device.Finalize();

        // ライブラリを終了
        nn::gfx::Finalize();

        // レイヤの終了処理
        FinalizeLayer();

        free(g_pMemoryHeap.Get());
        free(g_pVisiblePoolMemory);
        free(g_pInvisiblePoolMemory);
    }

    void InitializeFs()
    {
        nn::Result result;

        size_t cacheSize = 0;
        result = nn::fs::QueryMountRomCacheSize(&cacheSize);
        NN_ABORT_UNLESS(result.IsSuccess());

        g_MountRomCacheBuffer = malloc(cacheSize);
        NN_ABORT_UNLESS_NOT_NULL(g_MountRomCacheBuffer);

        //result = nn::fs::MountRom("Contents", g_MountRomCacheBuffer, cacheSize);
        //NN_ABORT_UNLESS(result.IsSuccess());
    }

    void FinalizeFs()
    {
        //nn::fs::Unmount("Contents");

        free(g_MountRomCacheBuffer);
        g_MountRomCacheBuffer = NULL;
    }

}

//------------------------------------------------------------------------------
// ユーザーピクセルシェーダ
//------------------------------------------------------------------------------
#define SHADER_SOURCE( ... ) #__VA_ARGS__
// モザイクシェーダ
static const char MOSAIC_PS_SOURCE[] =
"#version 430 \n"
SHADER_SOURCE(

    layout(std140, binding = 1) uniform Model
{
    uniform mat4 u_userMatrix;
    uniform vec4 u_color0;
    uniform vec4 u_color1;
    uniform vec2 u_uv_src;
    uniform vec2 u_uv_size;
    uniform vec4 u_layer;
    float rate;
};


// ユーザー定義のコンスタントバッファ
layout(std140, binding = 2) uniform TextureParam
{
    uniform vec2        u_pixel_size;
};

layout(binding = 0) uniform sampler2D texture0;

layout(location = 0) in vec4 v_texCoord;
layout(location = 1) in vec4 v_color;

out vec4 o_Color;

void main()
{
    vec4 out_color = vec4(0);
    vec2 uv = v_texCoord.xy;
    vec2 baseUv = floor(uv / (u_pixel_size * 4.0));
    vec4 color0 = vec4(0);
    for (float x = 0.5; x < 4.0; x += 1.0) {
        for (float y = 0.5; y < 4.0; y += 1.0) {
            color0 += textureLod(texture0, u_pixel_size * (baseUv * 4.0 + vec2(x, y)), 0.0);
        }
    }
    color0 /= 16.0;
    o_Color = color0;
}
);

// カスタムシェーダで使用するコンスタントバッファの構造体
struct PixelSize
{
    nn::util::Float2 u_pixel_size;
};


//---------------------------------------------------------------
// ユーザーシェーダの初期化
//---------------------------------------------------------------
nn::gfx::Shader g_MosaicPixelShader;
int g_SlotPixelSize;
void InitializeUserShader()
{
    nn::gfx::Shader::InfoType info;
    info.SetDefault();
    info.SetSeparationEnabled(true);

    nn::gfx::ShaderCode psCode;
    psCode.codeSize = static_cast<uint32_t>(strlen(MOSAIC_PS_SOURCE));
    psCode.pCode = MOSAIC_PS_SOURCE;

    info.SetShaderCodePtr(nn::gfx::ShaderStage_Pixel, &psCode);
    info.SetSourceFormat(nn::gfx::ShaderSourceFormat_Glsl);
    info.SetCodeType(nn::gfx::ShaderCodeType_Source);
    nn::gfx::ShaderInitializeResult result = g_MosaicPixelShader.Initialize(&g_Device, info);
    NN_SDK_ASSERT(result == nn::gfx::ShaderInitializeResult_Success, "InitializeShader() failed (%d)\n", result);
    NN_UNUSED(result);
    g_SlotPixelSize = g_MosaicPixelShader.GetInterfaceSlot(nn::gfx::ShaderStage_Pixel, nn::gfx::ShaderInterfaceType_ConstantBuffer, "TextureParam");
}


//---------------------------------------------------------------
// プリミティブレンダラの初期化
//---------------------------------------------------------------
nn::mem::StandardAllocator  g_PrimitiveRendererAllocator;
nns::gfx::PrimitiveRenderer::Renderer* g_pPrimitiveRenderer;
void*                       g_pPrimitiveRendererAllocatorMemory = nullptr;

//--------------------------------------------------------------------------------------
//  メモリ確保関数
//--------------------------------------------------------------------------------------
void* AllocateFunction(size_t size, size_t alignment, void* pUserData)
{
    NN_SDK_ASSERT_NOT_NULL(pUserData);
    nn::mem::StandardAllocator* pAllocator = static_cast<nn::mem::StandardAllocator*>(pUserData);
    return pAllocator->Allocate(size, alignment);
}

//--------------------------------------------------------------------------------------
//  メモリ開放関数
//--------------------------------------------------------------------------------------
void DeallocateFunction(void* ptr, void* pUserData)
{
    NN_SDK_ASSERT_NOT_NULL(pUserData);
    nn::mem::StandardAllocator* pAllocator = static_cast<nn::mem::StandardAllocator*>(pUserData);
    pAllocator->Free(ptr);
}

void InitializePrimitiveRenderer()
{
    const size_t workMemorySize = 1024 * 1024 * 64;
    g_pPrimitiveRendererAllocatorMemory = malloc(workMemorySize);
    g_PrimitiveRendererAllocator.Initialize(g_pPrimitiveRendererAllocatorMemory, workMemorySize);

    nns::gfx::PrimitiveRenderer::RendererInfo info;
    info.SetDefault();
    info.SetAllocator(AllocateFunction, &g_PrimitiveRendererAllocator);
    info.SetAdditionalBufferSize(1024 * 4);

    // ダブルバッファで運用する場合は、SetMultiBufferQuantity で 2 を指定し、
    // プリミティブレンダラ内部のバッファをダブルにしたうえで、
    // g_pPrimitiveRenderer->Update(); で利用するバッファを選択( 0 -> 1 -> 0 -> 1 )する
    info.SetMultiBufferQuantity(2);

    // PrimitiveRendererのインスタンス
    g_pPrimitiveRenderer = nns::gfx::PrimitiveRenderer::CreateRenderer(&g_Device, info);
    g_pPrimitiveRenderer->SetScreenWidth(g_RenderWidth);
    g_pPrimitiveRenderer->SetScreenHeight(g_RenderHeight);

    // ユーザーシェーダを初期化する
    InitializeUserShader();
}


//---------------------------------------------------------------
// プリミティブレンダラの破棄
//---------------------------------------------------------------
void FinalizePrimitiveRenderer()
{
    // ユーザーシェーダ破棄
    g_MosaicPixelShader.Finalize(&g_Device);

    nns::gfx::PrimitiveRenderer::DestroyRenderer(g_pPrimitiveRenderer, &g_Device, DeallocateFunction, &g_PrimitiveRendererAllocator);
    g_PrimitiveRendererAllocator.Finalize();
    free(g_pPrimitiveRendererAllocatorMemory);
}

//---------------------------------------------------------------
// デバッグフォントの初期化
//---------------------------------------------------------------
nn::gfx::util::DebugFontTextWriter g_Writer;
nn::util::BytePtr g_DebugFontHeap(NULL);
void InitializeDebugFont()
{
    const int charCountMax = 1024;
    nn::gfx::util::DebugFontTextWriterInfo info;
    info.SetDefault();
    info.SetCharCountMax(charCountMax);
    info.SetUserMemoryPoolEnabled(false);
    info.SetBufferCount(2);

    size_t debugFontHeapSize = nn::gfx::util::DebugFontTextWriter::GetRequiredMemorySize(
        &g_Device,
        info
    );
    g_DebugFontHeap = nn::util::BytePtr(new uint8_t[debugFontHeapSize]);

    g_Writer.Initialize(
        &g_Device,
        info,
        g_DebugFontHeap.Get(),
        debugFontHeapSize,
        NULL,
        0,
        0
    );

    g_Writer.SetDisplayWidth(g_RenderWidth);
    g_Writer.SetDisplayHeight(g_RenderHeight);
    g_Writer.SetTextureDescriptor(&g_TextureDescriptorPool, g_TextureDescriptorBaseIndex + 3);
    g_Writer.SetSamplerDescriptor(&g_SamplerDescriptorPool, g_SamplerDescriptorBaseIndex);
    g_Writer.SetTextColor(nn::util::Color4u8::Black());
}

//---------------------------------------------------------------
// デバッグフォントの破棄
//---------------------------------------------------------------
void FinalizeDebugFont()
{
    g_Writer.Finalize();
    delete[] reinterpret_cast<uint8_t*>(g_DebugFontHeap.Get());
    g_DebugFontHeap.Reset(nullptr);
}

//---------------------------------------------------------------
// 負荷メーターの初期化
//---------------------------------------------------------------
void*  meterMemory;
void InitializeLoadMeter()
{
    nn::perf::LoadMeterCenterInfo info;
    info.SetCoreCount(1);
    info.SetCpuBufferCount(2);
    info.SetGpuBufferCount(3);
    info.SetCpuSectionCountMax(64);
    info.SetGpuSectionCountMax(64);

    size_t memorySize = NN_PERF_GET_BUFFER_SIZE(info);
    size_t memoryAlignment = NN_PERF_GET_BUFFER_ALIGNMENT();
    meterMemory = AllocateFunction(memorySize, memoryAlignment, &g_PrimitiveRendererAllocator);
    size_t memoryPoolSize = NN_PERF_GET_MEMORY_POOL_SIZE(&g_Device, info);
    g_MemoryPoolOffset = nn::util::align_up(g_MemoryPoolOffset, NN_PERF_GET_MEMORY_POOL_ALIGNMENT(&g_Device, info));
    NN_PERF_INITIALIZE_METER(&g_Device, info,
        meterMemory, memorySize,
        &g_MemoryPool, g_MemoryPoolOffset, memoryPoolSize);
    g_MemoryPoolOffset += memoryPoolSize;
}

//---------------------------------------------------------------
// 負荷メーターの破棄
//---------------------------------------------------------------
void FinalizeLoadMeter()
{
    NN_PERF_FINALIZE_METER(&g_Device);
    DeallocateFunction(meterMemory, &g_PrimitiveRendererAllocator);
}

//---------------------------------------------------------------
// 独自頂点バッファを生成
//---------------------------------------------------------------
int CreateUserMesh(nns::gfx::PrimitiveRenderer::PrimitiveMesh* pMeshBuffer, bool bUseVertexColor, bool bUseVertexUv)
{
    const float width = 0.5f;
    const float height = 0.5f;
    const float depth = 0.5f;

    // インデックスバッファを設定します。
    static const uint32_t FaceTable[6][4] =
    {
        { 0, 1, 2, 3 }, // Front
        { 3, 2, 6, 7 }, // Right
        { 7, 6, 5, 4 }, // Back
        { 4, 5, 1, 0 }, // Left
        { 1, 5, 6, 2 }, // Top
        { 4, 0, 3, 7 }, // Bottom
    };

    // 頂点バッファを設定します。
    const nn::util::Float3 VertexPos[8] =
    {
        NN_UTIL_FLOAT_3_INITIALIZER(width / 2.f,  height, -depth / 2.f),
        NN_UTIL_FLOAT_3_INITIALIZER(width, -height,       -depth),
        NN_UTIL_FLOAT_3_INITIALIZER(-width, -height,       -depth),
        NN_UTIL_FLOAT_3_INITIALIZER(-width / 2.f,  height, -depth / 2.f),
        NN_UTIL_FLOAT_3_INITIALIZER(width / 2.f,  height,  depth / 2.f),
        NN_UTIL_FLOAT_3_INITIALIZER(width, -height,        depth),
        NN_UTIL_FLOAT_3_INITIALIZER(-width, -height,        depth),
        NN_UTIL_FLOAT_3_INITIALIZER(-width / 2.f,  height,  depth / 2.f),
    };

    static const nn::util::Float2 VertexUv[4] =
    {
        NN_UTIL_FLOAT_2_INITIALIZER(0.0f,  0.0f),
        NN_UTIL_FLOAT_2_INITIALIZER(0.0f,  1.0f),
        NN_UTIL_FLOAT_2_INITIALIZER(1.0f,  1.0f),
        NN_UTIL_FLOAT_2_INITIALIZER(1.0f,  0.0f),
    };

    static const nn::util::Float4 VertexColor[8] =
    {
        NN_UTIL_FLOAT_4_INITIALIZER(1.0f, 0.2f, 0.2f, 1.0f),
        NN_UTIL_FLOAT_4_INITIALIZER(0.2f, 1.0f, 0.2f, 1.0f),
        NN_UTIL_FLOAT_4_INITIALIZER(0.2f, 0.2f, 1.0f, 1.0f),
        NN_UTIL_FLOAT_4_INITIALIZER(1.0f, 0.2f, 0.2f, 1.0f),
        NN_UTIL_FLOAT_4_INITIALIZER(0.2f, 1.0f, 0.2f, 1.0f),
        NN_UTIL_FLOAT_4_INITIALIZER(0.2f, 0.2f, 1.0f, 1.0f),
        NN_UTIL_FLOAT_4_INITIALIZER(1.0f, 0.2f, 0.2f, 1.0f),
        NN_UTIL_FLOAT_4_INITIALIZER(0.2f, 1.0f, 0.2f, 1.0f),
    };

    const float xScale = 0.8f;
    const float yScale = 0.8f;
    const float zScale = 0.8f;
    uint32_t idxFace = 0;
    const int numIndices = 36;
    const int numVertices = 24;

    // 頂点フォーマット決定
    uint32_t format = nns::gfx::PrimitiveRenderer::VertexFormat_Pos;
    if (bUseVertexColor == true)
    {
        format |= nns::gfx::PrimitiveRenderer::VertexFormat_Color;
    }
    if (bUseVertexUv == true)
    {
        format |= nns::gfx::PrimitiveRenderer::VertexFormat_Uv;
    }

    // メッシュを初期化
    if (pMeshBuffer->Initialize(g_pPrimitiveRenderer->GetGpuBuffer(), numVertices, numIndices, static_cast<nns::gfx::PrimitiveRenderer::VertexFormat>(format)) == false)
    {
        return false;
    }


    // インデックスバッファを設定します。
    {
        size_t idx = 0;
        uint32_t* pIndexData = pMeshBuffer->GetIndexBufferCpuAddress();
        for (idxFace = 0; idxFace < 6; idxFace++)
        {
            pIndexData[idx++] = idxFace * 4 + 0;
            pIndexData[idx++] = idxFace * 4 + 1;
            pIndexData[idx++] = idxFace * 4 + 2;

            pIndexData[idx++] = idxFace * 4 + 0;
            pIndexData[idx++] = idxFace * 4 + 2;
            pIndexData[idx++] = idxFace * 4 + 3;
        }
    }

    size_t idxVertex = 0;
    idxFace = 0;

    {
        nn::util::Float3* pPos = static_cast<nn::util::Float3*>(pMeshBuffer->GetVertexBufferCpuAddress(nns::gfx::PrimitiveRenderer::VertexAttribute_Pos));

        // 頂点座標
        int idxCorner = 0;
        for (idxFace = 0; idxFace < 6; idxFace++)
        {
            for (idxCorner = 0; idxCorner < 4; idxCorner++)
            {
                const size_t index = FaceTable[idxFace][idxCorner];

                // 位置
                pPos[idxVertex].x = VertexPos[index].x * xScale;
                pPos[idxVertex].y = VertexPos[index].y * yScale;
                pPos[idxVertex].z = VertexPos[index].z * zScale;
                idxVertex++;
            }
        }

        // 頂点カラーを利用する場合
        if (bUseVertexColor == true || bUseVertexUv == true)
        {
            nn::util::Float2* pUv = static_cast<nn::util::Float2*>(pMeshBuffer->GetVertexBufferCpuAddress(nns::gfx::PrimitiveRenderer::VertexAttribute_Uv));
            nn::util::Float4* pColor = static_cast<nn::util::Float4*>(pMeshBuffer->GetVertexBufferCpuAddress(nns::gfx::PrimitiveRenderer::VertexAttribute_Color));
            idxCorner = 0;
            idxVertex = 0;
            for (idxFace = 0; idxFace < 6; idxFace++)
            {
                for (idxCorner = 0; idxCorner < 4; idxCorner++)
                {
                    const size_t index = FaceTable[idxFace][idxCorner];

                    // カラー
                    if (pColor)
                    {
                        pColor[idxVertex].x = VertexColor[index].x;
                        pColor[idxVertex].y = VertexColor[index].y;
                        pColor[idxVertex].z = VertexColor[index].z;
                        pColor[idxVertex].w = VertexColor[index].w;
                    }

                    // UV
                    if (pUv)
                    {
                        pUv[idxVertex].x = VertexUv[idxCorner].x;
                        pUv[idxVertex].y = VertexUv[idxCorner].y;
                    }
                    idxVertex++;
                }
            }
        }
    }

    return numIndices;
} //NOLINT(impl/function_size)

//---------------------------------------------------------------
// コマンド生成
//---------------------------------------------------------------
bool MakeCommand(int frame, int bufferIndex)
{
    NN_UNUSED(frame);

    nn::util::BytePtr reset = g_pMemory;

    g_CommandBuffer.Reset();
    g_MemoryPoolOffset = nn::util::align_up(g_MemoryPoolOffset,
        nn::gfx::CommandBuffer::GetCommandMemoryAlignment(&g_Device));
    // ワンタイムのコマンドバッファには nn::gfx::MemoryPoolProperty_CpuUncached | nn::gfx::MemoryPoolProperty_GpuUncached のメモリプールを使うことがより適しています
    g_CommandBuffer.AddCommandMemory(&g_MemoryPool, g_MemoryPoolOffset + bufferIndex * (1024 * 1024 * 6), 1024 * 1024 * 2);
    g_pMemory.AlignUp(256);
    g_CommandBuffer.AddControlMemory(g_pMemory.Get(), 256);
    g_CommandBuffer.Begin();
    NN_PERF_SET_COLOR_GPU(nn::util::Color4u8::Red());
    NN_PERF_BEGIN_MEASURE_GPU(&g_CommandBuffer);
    {
        nn::gfx::ColorTargetView* pTarget = &g_ColorTargetView;

        g_CommandBuffer.InvalidateMemory(nn::gfx::GpuAccess_Descriptor | nn::gfx::GpuAccess_ShaderCode);

        g_CommandBuffer.SetDescriptorPool(&g_BufferDescriptorPool);
        g_CommandBuffer.SetDescriptorPool(&g_TextureDescriptorPool);
        g_CommandBuffer.SetDescriptorPool(&g_SamplerDescriptorPool);

        g_CommandBuffer.ClearColor(pTarget, 0.0f, 0.0f, 0.0f, 1.0f, NULL);
        g_CommandBuffer.ClearDepthStencil(&g_DepthStencilView, 1.0f, 0,
            nn::gfx::DepthStencilClearMode_DepthStencil, nullptr);

        g_CommandBuffer.SetRenderTargets(1, &pTarget, &g_DepthStencilView);
        g_CommandBuffer.SetViewportScissorState(&g_ViewportScissor);
        g_CommandBuffer.SetRasterizerState(&g_RasterizerState);

        g_CommandBuffer.InvalidateMemory(nn::gfx::GpuAccess_Texture | nn::gfx::GpuAccess_IndexBuffer
            | nn::gfx::GpuAccess_ConstantBuffer | nn::gfx::GpuAccess_VertexBuffer);

        nn::gfx::DescriptorSlot constantBufferDescriptor;
        nn::gfx::DescriptorSlot textureDescriptor;
        nn::gfx::DescriptorSlot samplerDescriptor;
        g_BufferDescriptorPool.GetDescriptorSlot(&constantBufferDescriptor, g_BufferDescriptorBaseIndex);
        g_TextureDescriptorPool.GetDescriptorSlot(&textureDescriptor, g_TextureDescriptorBaseIndex);
        g_SamplerDescriptorPool.GetDescriptorSlot(&samplerDescriptor, g_SamplerDescriptorBaseIndex);


        // プリミティブレンダラの更新処理
        g_pPrimitiveRenderer->Update(bufferIndex);

        // model, view, projection行列をデフォルトに
        g_pPrimitiveRenderer->SetDefaultParameters();

        nn::util::Matrix4x3fType viewMatrix;
        nn::util::Matrix4x4fType projectionMatrix;
        nn::util::Matrix4x3f modelMatrix;
        nn::util::MatrixIdentity(&modelMatrix);
        nn::util::Vector3f translate;
        nn::util::VectorSet(&translate, 0.f, 0.f, 1.f);
        nn::util::MatrixSetAxisW(&modelMatrix, translate);

        nn::util::Uint8x4 white = { { 255, 255, 255, 255 } };
        nn::util::Uint8x4 red = { { 255, 0, 0, 255 } };
        nn::util::Uint8x4 green = { { 0, 255, 0, 255 } };
        nn::util::Uint8x4 blue = { { 0, 0, 255, 255 } };


        // Blend
        g_CommandBuffer.SetBlendState(g_pPrimitiveRenderer->GetBlendState(nns::gfx::PrimitiveRenderer::BlendType::BlendType_Normal));

        // Depth Enable
        g_pPrimitiveRenderer->SetDepthStencilState(&g_CommandBuffer, nns::gfx::PrimitiveRenderer::DepthStencilType::DepthStencilType_DepthNoWriteTest);


        // 画面全体にテクスチャを張る
        nn::util::MatrixIdentity(&viewMatrix);
        nn::util::MatrixIdentity(&projectionMatrix);
        g_pPrimitiveRenderer->SetViewMatrix(&viewMatrix);
        g_pPrimitiveRenderer->SetProjectionMatrix(&projectionMatrix);
        g_pPrimitiveRenderer->SetModelMatrix(&modelMatrix);
        g_pPrimitiveRenderer->DrawScreenQuad(&g_CommandBuffer, textureDescriptor, samplerDescriptor);

        // ビューとプロジェクションを設定
        float radius = 20.f;
        float x = radius * sin(frame / 500.f);
        float z = radius * cos(frame / 500.f);

        nn::util::Vector3fType camPos = { x, 10.f, z };
        nn::util::Vector3fType camTarget = { 0.f, 0.f, 0.f };
        nn::util::Vector3fType camUp = { 0.f, 1.f, 0.f };
        nn::util::MatrixLookAtRightHanded(&viewMatrix, camPos, camTarget, camUp);
        g_pPrimitiveRenderer->SetViewMatrix(&viewMatrix);

        // プロジェクションを初期化
        const float fovy = nn::util::FloatPi / 3.0f;
        const float aspect = static_cast< float >(g_RenderWidth) / static_cast< float >(g_RenderHeight);
        nn::util::MatrixPerspectiveFieldOfViewRightHanded(&projectionMatrix, fovy, aspect, 0.1f, 1000.f);
        g_pPrimitiveRenderer->SetProjectionMatrix(&projectionMatrix);

        // Depth Enable
        g_pPrimitiveRenderer->SetDepthStencilState(&g_CommandBuffer, nns::gfx::PrimitiveRenderer::DepthStencilType::DepthStencilType_DepthWriteTest);

        // 軸を描画
        float interval = -10.f;
        nn::util::Vector3fType begin;
        nn::util::Vector3fType end;
        g_pPrimitiveRenderer->SetLineWidth(1.f);
        for (int i = 0; i < 21; i++)
        {

            nn::util::VectorSet(&begin, -10.f, 0.f, interval);
            nn::util::VectorSet(&end, 10.f, 0.f, interval);
            g_pPrimitiveRenderer->SetColor(white);
            g_pPrimitiveRenderer->DrawLine(&g_CommandBuffer, begin, end);
            nn::util::VectorSet(&begin, interval, 0.f, -10.f);
            nn::util::VectorSet(&end, interval, 0.f, 10.f);
            g_pPrimitiveRenderer->DrawLine(&g_CommandBuffer, begin, end);
            interval += 1.0f;
        }

        g_pPrimitiveRenderer->SetLineWidth(5.f);
        nn::util::Vector3fType zeroVector;
        nn::util::VectorSet(&zeroVector, 0.f, 0.f, 0.f);
        nn::util::Vector3fType axisPos;
        nn::util::VectorSet(&axisPos, 5.f, 0.f, 0.f);
        g_pPrimitiveRenderer->SetColor(red);
        g_pPrimitiveRenderer->DrawLine(&g_CommandBuffer, zeroVector, axisPos);
        nn::util::VectorSet(&axisPos, 0.f, 0.f, 5.f);
        g_pPrimitiveRenderer->SetColor(blue);
        g_pPrimitiveRenderer->DrawLine(&g_CommandBuffer, zeroVector, axisPos);
        nn::util::VectorSet(&axisPos, 0.f, 5.f, 0.f);
        g_pPrimitiveRenderer->SetColor(green);
        g_pPrimitiveRenderer->DrawLine(&g_CommandBuffer, zeroVector, axisPos);


        nn::util::Vector3fType center = { 0.f, 0.f, 0.f };
        nn::util::Vector3fType size = { 1.f, 1.f, 1.f };

        nn::util::Vector3f vecZero;
        nn::util::VectorZero(&vecZero);

        float rotY = frame * 0.05f;
        nn::util::Vector3f rotValue;
        nn::util::VectorSet(&rotValue, 0.2f, rotY, 0.2f);
        nn::util::MatrixIdentity(&modelMatrix);
        nn::util::MatrixSetRotateXyz(&modelMatrix, rotValue);
        nn::util::MatrixSetAxisW(&modelMatrix, vecZero);

        // QUADSを描画する
        nn::util::VectorSet(&translate, 4.f, 0.f, -2.f);
        nn::util::MatrixSetAxisW(&modelMatrix, translate);
        g_pPrimitiveRenderer->SetModelMatrix(&modelMatrix);
        g_pPrimitiveRenderer->SetColor(red);
        g_pPrimitiveRenderer->DrawQuad(&g_CommandBuffer, center, size);

        nn::util::VectorSet(&translate, 8.f, 0.f, -2.f);
        nn::util::MatrixSetAxisW(&modelMatrix, translate);
        g_pPrimitiveRenderer->SetModelMatrix(&modelMatrix);
        // 乗算するカラーをwhiteにしておく
        g_pPrimitiveRenderer->SetColor(white);
        g_pPrimitiveRenderer->DrawQuad(&g_CommandBuffer, center, size, textureDescriptor, samplerDescriptor);


        // TRIANGLEを描画
        nn::util::VectorSet(&translate, 2.f, 0.f, -4.f);
        nn::util::MatrixSetAxisW(&modelMatrix, translate);
        g_pPrimitiveRenderer->SetModelMatrix(&modelMatrix);
        g_pPrimitiveRenderer->SetLineWidth(1.f);
        g_pPrimitiveRenderer->SetColor(blue);
        g_pPrimitiveRenderer->DrawTriangle(&g_CommandBuffer,
            nns::gfx::PrimitiveRenderer::Surface::Surface_Wired,
            center, size);

        nn::util::VectorSet(&translate, 4.f, 0.f, -4.f);
        nn::util::MatrixSetAxisW(&modelMatrix, translate);
        g_pPrimitiveRenderer->SetModelMatrix(&modelMatrix);
        g_pPrimitiveRenderer->DrawTriangle(&g_CommandBuffer,
            nns::gfx::PrimitiveRenderer::Surface::Surface_Solid,
            center, size);

        nn::util::VectorSet(&translate, 6.f, 0.f, -4.f);
        nn::util::MatrixSetAxisW(&modelMatrix, translate);
        g_pPrimitiveRenderer->SetModelMatrix(&modelMatrix);
        g_pPrimitiveRenderer->DrawTriangle(&g_CommandBuffer,
            nns::gfx::PrimitiveRenderer::Surface::Surface_Normal,
            center, size);

        g_pPrimitiveRenderer->SetColor(white);
        nn::util::VectorSet(&translate, 8.f, 0.f, -4.f);
        nn::util::MatrixSetAxisW(&modelMatrix, translate);
        g_pPrimitiveRenderer->SetModelMatrix(&modelMatrix);
        g_pPrimitiveRenderer->DrawTriangle(&g_CommandBuffer,
            center, size, textureDescriptor, samplerDescriptor);

        // CIRCLEを描画
        nn::util::VectorSet(&translate, 2.f, 0.f, -6.f);
        nn::util::MatrixSetAxisW(&modelMatrix, translate);
        g_pPrimitiveRenderer->SetModelMatrix(&modelMatrix);
        g_pPrimitiveRenderer->SetColor(green);
        g_pPrimitiveRenderer->DrawCircle(&g_CommandBuffer,
            nns::gfx::PrimitiveRenderer::Surface::Surface_Wired,
            nns::gfx::PrimitiveRenderer::Subdiv::Subdiv_Normal,
            center, 0.5f);

        nn::util::VectorSet(&translate, 4.f, 0.f, -6.f);
        nn::util::MatrixSetAxisW(&modelMatrix, translate);
        g_pPrimitiveRenderer->SetModelMatrix(&modelMatrix);
        g_pPrimitiveRenderer->SetLineWidth(1.f);
        g_pPrimitiveRenderer->DrawCircle(&g_CommandBuffer,
            nns::gfx::PrimitiveRenderer::Surface::Surface_Solid,
            nns::gfx::PrimitiveRenderer::Subdiv::Subdiv_Normal,
            center, 0.5f);

        nn::util::VectorSet(&translate, 6.f, 0.f, -6.f);
        nn::util::MatrixSetAxisW(&modelMatrix, translate);
        g_pPrimitiveRenderer->SetModelMatrix(&modelMatrix);
        g_pPrimitiveRenderer->SetLineWidth(1.f);
        g_pPrimitiveRenderer->DrawCircle(&g_CommandBuffer,
            nns::gfx::PrimitiveRenderer::Surface::Surface_Normal,
            nns::gfx::PrimitiveRenderer::Subdiv::Subdiv_Normal,
            center, 0.5f);

        g_pPrimitiveRenderer->SetColor(white);
        nn::util::VectorSet(&translate, 8.f, 0.f, -6.f);
        nn::util::MatrixSetAxisW(&modelMatrix, translate);
        g_pPrimitiveRenderer->SetModelMatrix(&modelMatrix);
        g_pPrimitiveRenderer->SetLineWidth(1.f);
        g_pPrimitiveRenderer->DrawCircle(&g_CommandBuffer,
            nns::gfx::PrimitiveRenderer::Subdiv::Subdiv_Normal,
            center, 0.5f, textureDescriptor, samplerDescriptor);


        // CUBEを描画する
        nn::util::VectorSet(&translate, 2.f, 0.f, 9.f);
        nn::util::MatrixSetAxisW(&modelMatrix, translate);
        g_pPrimitiveRenderer->SetModelMatrix(&modelMatrix);
        g_pPrimitiveRenderer->SetLineWidth(1.f);
        g_pPrimitiveRenderer->SetColor(red);
        g_pPrimitiveRenderer->DrawCube(&g_CommandBuffer,
            nns::gfx::PrimitiveRenderer::Surface::Surface_Wired,
            center, size);

        nn::util::VectorSet(&translate, 4.f, 0.f, 9.f);
        nn::util::MatrixSetAxisW(&modelMatrix, translate);
        g_pPrimitiveRenderer->SetModelMatrix(&modelMatrix);
        g_pPrimitiveRenderer->DrawCube(&g_CommandBuffer,
            nns::gfx::PrimitiveRenderer::Surface::Surface_Solid,
            center, size);

        nn::util::VectorSet(&translate, 6.f, 0.f, 9.f);
        nn::util::MatrixSetAxisW(&modelMatrix, translate);
        g_pPrimitiveRenderer->SetModelMatrix(&modelMatrix);
        g_pPrimitiveRenderer->DrawCube(&g_CommandBuffer,
            nns::gfx::PrimitiveRenderer::Surface::Surface_Normal,
            center, size);

        g_pPrimitiveRenderer->SetColor(white);
        nn::util::VectorSet(&translate, 8.f, 0.f, 9.f);
        nn::util::MatrixSetAxisW(&modelMatrix, translate);
        g_pPrimitiveRenderer->SetModelMatrix(&modelMatrix);
        g_pPrimitiveRenderer->DrawCube(&g_CommandBuffer,
            center, size, textureDescriptor, samplerDescriptor);

        // SPHEREを描画
        nn::util::VectorSet(&translate, 2.f, 0.f, 3.f);
        nn::util::MatrixSetAxisW(&modelMatrix, translate);
        g_pPrimitiveRenderer->SetModelMatrix(&modelMatrix);
        g_pPrimitiveRenderer->SetLineWidth(1.f);
        g_pPrimitiveRenderer->SetColor(green);
        g_pPrimitiveRenderer->DrawSphere(&g_CommandBuffer,
            nns::gfx::PrimitiveRenderer::Surface::Surface_Wired,
            nns::gfx::PrimitiveRenderer::Subdiv::Subdiv_Normal,
            center, 1.f);

        nn::util::VectorSet(&translate, 4.f, 0.f, 3.f);
        nn::util::MatrixSetAxisW(&modelMatrix, translate);
        g_pPrimitiveRenderer->SetModelMatrix(&modelMatrix);
        g_pPrimitiveRenderer->DrawSphere(&g_CommandBuffer,
            nns::gfx::PrimitiveRenderer::Surface::Surface_Solid,
            nns::gfx::PrimitiveRenderer::Subdiv::Subdiv_Normal,
            center, 1.f);

        nn::util::VectorSet(&translate, 6.f, 0.f, 3.f);
        nn::util::MatrixSetAxisW(&modelMatrix, translate);
        g_pPrimitiveRenderer->SetModelMatrix(&modelMatrix);
        g_pPrimitiveRenderer->DrawSphere(&g_CommandBuffer,
            nns::gfx::PrimitiveRenderer::Surface::Surface_Normal,
            nns::gfx::PrimitiveRenderer::Subdiv::Subdiv_Normal,
            center, 1.f);

        g_pPrimitiveRenderer->SetColor(white);
        nn::util::VectorSet(&translate, 8.f, 0.f, 3.f);
        nn::util::MatrixSetAxisW(&modelMatrix, translate);
        g_pPrimitiveRenderer->SetModelMatrix(&modelMatrix);
        g_pPrimitiveRenderer->DrawSphere(&g_CommandBuffer,
            nns::gfx::PrimitiveRenderer::Subdiv::Subdiv_Normal,
            center, 1.f, textureDescriptor, samplerDescriptor);

        // CONEを描画
        nn::util::VectorSet(&translate, -2.f, 0.f, 3.f);
        nn::util::MatrixSetAxisW(&modelMatrix, translate);
        g_pPrimitiveRenderer->SetModelMatrix(&modelMatrix);
        g_pPrimitiveRenderer->SetColor(blue);
        g_pPrimitiveRenderer->DrawCone(&g_CommandBuffer,
            nns::gfx::PrimitiveRenderer::Surface::Surface_Wired,
            center, 0.5, 1.0);

        nn::util::VectorSet(&translate, -4.f, 0.f, 3.f);
        nn::util::MatrixSetAxisW(&modelMatrix, translate);
        g_pPrimitiveRenderer->SetModelMatrix(&modelMatrix);
        g_pPrimitiveRenderer->DrawCone(&g_CommandBuffer,
            nns::gfx::PrimitiveRenderer::Surface::Surface_Solid,
            center, 0.5, 1.0);

        nn::util::VectorSet(&translate, -6.f, 0.f, 3.f);
        nn::util::MatrixSetAxisW(&modelMatrix, translate);
        g_pPrimitiveRenderer->SetModelMatrix(&modelMatrix);
        g_pPrimitiveRenderer->DrawCone(&g_CommandBuffer,
            nns::gfx::PrimitiveRenderer::Surface::Surface_Normal,
            center, 0.5, 1.0);

        g_pPrimitiveRenderer->SetColor(white);
        nn::util::VectorSet(&translate, -8.f, 0.f, 3.f);
        nn::util::MatrixSetAxisW(&modelMatrix, translate);
        g_pPrimitiveRenderer->SetModelMatrix(&modelMatrix);
        g_pPrimitiveRenderer->DrawCone(&g_CommandBuffer,
            center, 0.5, 1.0, textureDescriptor, samplerDescriptor);

        // CAPSULEを描画
        nn::util::VectorSet(&translate, 2.f, 0.f, 6.f);
        nn::util::MatrixSetAxisW(&modelMatrix, translate);
        g_pPrimitiveRenderer->SetModelMatrix(&modelMatrix);
        g_pPrimitiveRenderer->SetColor(red);
        g_pPrimitiveRenderer->DrawCapsule(&g_CommandBuffer,
            nns::gfx::PrimitiveRenderer::Surface::Surface_Wired,
            center, 0.5, 1.f);

        nn::util::VectorSet(&translate, 4.f, 0.f, 6.f);
        nn::util::MatrixSetAxisW(&modelMatrix, translate);
        g_pPrimitiveRenderer->SetModelMatrix(&modelMatrix);
        g_pPrimitiveRenderer->DrawCapsule(&g_CommandBuffer,
            nns::gfx::PrimitiveRenderer::Surface::Surface_Solid,
            center, 0.5, 1.f);

        nn::util::VectorSet(&translate, 6.f, 0.f, 6.f);
        nn::util::MatrixSetAxisW(&modelMatrix, translate);
        g_pPrimitiveRenderer->SetModelMatrix(&modelMatrix);
        g_pPrimitiveRenderer->DrawCapsule(&g_CommandBuffer,
            nns::gfx::PrimitiveRenderer::Surface::Surface_Normal,
            center, 0.5, 1.f);

        g_pPrimitiveRenderer->SetColor(white);
        nn::util::VectorSet(&translate, 8.f, 0.f, 6.f);
        nn::util::MatrixSetAxisW(&modelMatrix, translate);
        g_pPrimitiveRenderer->SetModelMatrix(&modelMatrix);
        g_pPrimitiveRenderer->DrawCapsule(&g_CommandBuffer,
            center, 0.5, 1.f, textureDescriptor, samplerDescriptor);

        // PIPEを描画
        nn::util::VectorSet(&translate, -2.f, 0.f, 6.f);
        nn::util::MatrixSetAxisW(&modelMatrix, translate);
        g_pPrimitiveRenderer->SetModelMatrix(&modelMatrix);
        g_pPrimitiveRenderer->SetColor(red);
        g_pPrimitiveRenderer->DrawPipe(&g_CommandBuffer,
            nns::gfx::PrimitiveRenderer::Surface::Surface_Wired,
            center, 0.5, 1.f);

        nn::util::VectorSet(&translate, -4.f, 0.f, 6.f);
        nn::util::MatrixSetAxisW(&modelMatrix, translate);
        g_pPrimitiveRenderer->SetModelMatrix(&modelMatrix);
        g_pPrimitiveRenderer->DrawPipe(&g_CommandBuffer,
            nns::gfx::PrimitiveRenderer::Surface::Surface_Solid,
            center, 0.5, 1.f);

        nn::util::VectorSet(&translate, -6.f, 0.f, 6.f);
        nn::util::MatrixSetAxisW(&modelMatrix, translate);
        g_pPrimitiveRenderer->SetModelMatrix(&modelMatrix);
        g_pPrimitiveRenderer->DrawPipe(&g_CommandBuffer,
            nns::gfx::PrimitiveRenderer::Surface::Surface_Normal,
            center, 0.5, 1.f);

        g_pPrimitiveRenderer->SetColor(white);
        nn::util::VectorSet(&translate, -8.f, 0.f, 6.f);
        nn::util::MatrixSetAxisW(&modelMatrix, translate);
        g_pPrimitiveRenderer->SetModelMatrix(&modelMatrix);
        g_pPrimitiveRenderer->DrawPipe(&g_CommandBuffer,
            center, 0.5, 1.f, textureDescriptor, samplerDescriptor);

        // Cylinderを描画
        nn::util::VectorSet(&translate, -2.f, 0.f, 0.f);
        nn::util::MatrixSetAxisW(&modelMatrix, translate);
        g_pPrimitiveRenderer->SetModelMatrix(&modelMatrix);
        g_pPrimitiveRenderer->SetColor(red);
        g_pPrimitiveRenderer->DrawCylinder(&g_CommandBuffer,
            nns::gfx::PrimitiveRenderer::Surface::Surface_Wired,
            center, 0.5, 1.f);

        nn::util::VectorSet(&translate, -4.f, 0.f, 0.f);
        nn::util::MatrixSetAxisW(&modelMatrix, translate);
        g_pPrimitiveRenderer->SetModelMatrix(&modelMatrix);
        g_pPrimitiveRenderer->DrawCylinder(&g_CommandBuffer,
            nns::gfx::PrimitiveRenderer::Surface::Surface_Solid,
            center, 0.5, 1.f);

        nn::util::VectorSet(&translate, -6.f, 0.f, 0.f);
        nn::util::MatrixSetAxisW(&modelMatrix, translate);
        g_pPrimitiveRenderer->SetModelMatrix(&modelMatrix);
        g_pPrimitiveRenderer->DrawCylinder(&g_CommandBuffer,
            nns::gfx::PrimitiveRenderer::Surface::Surface_Normal,
            center, 0.5, 1.f);

        g_pPrimitiveRenderer->SetColor(white);
        nn::util::VectorSet(&translate, -8.f, 0.f, 0.f);
        nn::util::MatrixSetAxisW(&modelMatrix, translate);
        g_pPrimitiveRenderer->SetModelMatrix(&modelMatrix);
        g_pPrimitiveRenderer->DrawCylinder(&g_CommandBuffer,
            center, 0.5, 1.f, textureDescriptor, samplerDescriptor);

        // 独自頂点バッファを使用した描画
        nns::gfx::PrimitiveRenderer::PrimitiveMesh posMesh;
        nns::gfx::PrimitiveRenderer::PrimitiveMesh posUvMesh;
        nns::gfx::PrimitiveRenderer::PrimitiveMesh colorMesh;
        nns::gfx::PrimitiveRenderer::PrimitiveMesh colorUvMesh;

        CreateUserMesh(&posMesh, false, false);
        CreateUserMesh(&posUvMesh, false, true);
        CreateUserMesh(&colorMesh, true, false);
        CreateUserMesh(&colorUvMesh, true, true);

        // 頂点カラーを利用しない場合
        nn::util::VectorSet(&translate, -2.f, 0.f, 9.f);
        nn::util::MatrixSetAxisW(&modelMatrix, translate);
        g_pPrimitiveRenderer->SetColor(green);
        g_pPrimitiveRenderer->SetModelMatrix(&modelMatrix);
        g_pPrimitiveRenderer->DrawUserMesh(&g_CommandBuffer,
            nn::gfx::PrimitiveTopology::PrimitiveTopology_TriangleList, &posMesh);

        nn::util::VectorSet(&translate, -4.f, 0.f, 9.f);
        nn::util::MatrixSetAxisW(&modelMatrix, translate);
        g_pPrimitiveRenderer->SetColor(white);
        g_pPrimitiveRenderer->SetModelMatrix(&modelMatrix);
        g_pPrimitiveRenderer->DrawUserMesh(&g_CommandBuffer,
            nn::gfx::PrimitiveTopology::PrimitiveTopology_TriangleList, &posUvMesh, textureDescriptor, samplerDescriptor);

        // 頂点カラーを利用する場合
        nn::util::VectorSet(&translate, -6.f, 0.f, 9.f);
        nn::util::MatrixSetAxisW(&modelMatrix, translate);
        g_pPrimitiveRenderer->SetColor(white);
        g_pPrimitiveRenderer->SetModelMatrix(&modelMatrix);
        g_pPrimitiveRenderer->DrawUserMesh(&g_CommandBuffer,
            nn::gfx::PrimitiveTopology::PrimitiveTopology_TriangleList, &colorMesh);

        nn::util::VectorSet(&translate, -8.f, 0.f, 9.f);
        nn::util::MatrixSetAxisW(&modelMatrix, translate);
        g_pPrimitiveRenderer->SetModelMatrix(&modelMatrix);
        g_pPrimitiveRenderer->DrawUserMesh(&g_CommandBuffer,
            nn::gfx::PrimitiveTopology::PrimitiveTopology_TriangleList, &colorUvMesh, textureDescriptor, samplerDescriptor);

        //負荷メーターを描画
        // NN_PERF_IS_ENABLED() が false の場合はメーターの描画をしない
        if (NN_STATIC_CONDITION(NN_PERF_IS_ENABLED()))
        {
            nn::perf::CpuMeter* pFrameMeter = NN_PERF_GET_FRAME_METER();
            nn::util::Float2 pos = NN_UTIL_FLOAT_2_INITIALIZER(32.f, g_RenderHeight - g_MeterDrawer.GetHeight(pFrameMeter) - g_MeterDrawer.GetBarHeight());
            g_MeterDrawer.SetDebugFontTextWriter(&g_Writer);
            g_MeterDrawer.SetPosition(pos);
            g_MeterDrawer.SetWidth(g_RenderWidth - 64.f);
            g_MeterDrawer.Draw(&g_CommandBuffer, g_pPrimitiveRenderer, pFrameMeter);
        }

        // テキストを描画
        g_Writer.Draw(&g_CommandBuffer);

        // PrimitiveRendererを利用してスキャンバッファにコピー
        pTarget = g_pScanBufferViews[g_NextScanBufferIndex];
        g_CommandBuffer.SetRenderTargets(1, &pTarget, nullptr);

        // カラーバッファをテクスチャとして利用するのでキャッシュをフラッシュ
        g_CommandBuffer.FlushMemory(nn::gfx::GpuAccess_ColorBuffer);

        // 上書き設定
        g_CommandBuffer.SetBlendState(g_pPrimitiveRenderer->GetBlendState(nns::gfx::PrimitiveRenderer::BlendType::BlendType_Opacity));

        nn::gfx::DescriptorSlot colorDescriptor;
        g_TextureDescriptorPool.GetDescriptorSlot(&colorDescriptor, g_TextureDescriptorBaseIndex + 2);
        g_pPrimitiveRenderer->SetColor(white);
        g_pPrimitiveRenderer->DrawScreenQuadYFlip(&g_CommandBuffer, colorDescriptor, samplerDescriptor);
    }
    NN_PERF_END_MEASURE_GPU(&g_CommandBuffer);
    g_CommandBuffer.FlushMemory(nn::gfx::GpuAccess_QueryBuffer);
    g_CommandBuffer.End();


    // PresentTexture 計測用コマンドバッファ
    ptrdiff_t currentOffset = g_MemoryPoolOffset + bufferIndex * (1024 * 1024 * 6) + (1024 * 1024 * 2);

    g_CommandBufferBegin.Reset();
    currentOffset = nn::util::align_up(currentOffset, nn::gfx::CommandBuffer::GetCommandMemoryAlignment(&g_Device));
    g_CommandBufferBegin.AddCommandMemory(&g_MemoryPool, currentOffset, 1024 * 1024 * 0.5);
    g_pMemory.Advance(256);
    g_CommandBufferBegin.AddControlMemory(g_pMemory.Get(), 256);

    currentOffset += 1024 * 1024 * 0.5;
    g_CommandBufferEnd.Reset();
    currentOffset = nn::util::align_up(currentOffset, nn::gfx::CommandBuffer::GetCommandMemoryAlignment(&g_Device));
    g_CommandBufferEnd.AddCommandMemory(&g_MemoryPool, currentOffset, 1024 * 1024 * 0.5);
    g_pMemory.Advance(256);
    g_CommandBufferEnd.AddControlMemory(g_pMemory.Get(), 256);

    // MakeBeginTimeStampQueryCommand
    g_CommandBufferBegin.Begin();
    NN_PERF_SET_COLOR_GPU(nn::util::Color4u8::Blue());
    NN_PERF_BEGIN_MEASURE_NAME_GPU(&g_CommandBufferBegin, "PresentTextureGpu");
    g_CommandBufferBegin.FlushMemory(nn::gfx::GpuAccess_QueryBuffer);
    g_CommandBufferBegin.End();

    // MakeEndTimeStampQueryCommand
    g_CommandBufferEnd.Begin();
    NN_PERF_END_MEASURE_GPU(&g_CommandBufferEnd);
    g_CommandBufferEnd.FlushMemory(nn::gfx::GpuAccess_QueryBuffer);
    g_CommandBufferEnd.End();

    g_pMemory.Advance(-g_pMemory.Distance(reset.Get()));

    return true;
} //NOLINT(impl/function_size)

int64_t g_SumPresentTexture = 0;
int64_t g_SumAcquireTexture = 0;
int64_t g_SumPresentTextureGpu = 0;
int64_t g_MeasureCount = 0;

//---------------------------------------------------------------
// フレーム処理
//---------------------------------------------------------------
bool DeferredSubmission(int frame)
{
    // FrameworkMode: DeferredSubmission

    static int currentBufferIndex = 0;
    int previousBufferIndex = 1 - currentBufferIndex;

    if (frame > 0)
    {
        // ExecuteCommand(previousBufferIndex)
        NN_PERF_SET_COLOR(nn::util::Color4u8::Blue());
        NN_PERF_BEGIN_MEASURE_NAME("ExecuteCommand");
        {
            g_Queue.ExecuteCommand(&g_CommandBuffer, NULL);
        }
        NN_PERF_END_MEASURE();

        // QueuePresentTexture
        g_Queue.ExecuteCommand(&g_CommandBufferBegin, NULL);
        NN_PERF_SET_COLOR(nn::util::Color4u8::Yellow());
        NN_PERF_BEGIN_MEASURE_NAME("PresentTexture");
        {
            g_Queue.Present(&g_SwapChain, 1);
        }
        NN_PERF_END_MEASURE();
        g_Queue.ExecuteCommand(&g_CommandBufferEnd, &g_GpuFence[previousBufferIndex]);
        g_Queue.Flush();
    }

    // AcquireTexture(currentBufferIndex)
    NN_PERF_SET_COLOR(nn::util::Color4u8::Blue());
    NN_PERF_BEGIN_MEASURE_NAME("AcquireTexture");
    {
        g_NextScanBufferIndex = -1;
        nn::gfx::AcquireScanBufferResult acquireResult = g_SwapChain.AcquireNextScanBufferIndex(
            &g_NextScanBufferIndex, &g_DisplaySemaphore[currentBufferIndex], &g_DisplayFence[currentBufferIndex]);
        NN_SDK_ASSERT(acquireResult == nn::gfx::AcquireScanBufferResult_Success);
        NN_UNUSED(acquireResult);
    }
    NN_PERF_END_MEASURE();

    // 不要
    // QueueWaitSync(currentBufferIndex)
    //g_Queue.SyncSemaphore(&g_DisplaySemaphore[currentBufferIndex]);
    //g_Queue.Flush();

    // MakeCommand(currentBufferIndex)
    NN_PERF_SET_COLOR(nn::util::Color4u8::Green());
    NN_PERF_BEGIN_MEASURE_NAME("MakeCommand");
    if (MakeCommand(frame, currentBufferIndex) == false)
    {
        return false;
    }
    NN_PERF_END_MEASURE();

    if (frame > 60)
    {
        g_SumPresentTexture += NN_PERF_GET_ELAPSED_TIME_CPU(NULL, "PresentTexture", 0).GetMicroSeconds();
        g_SumAcquireTexture += NN_PERF_GET_ELAPSED_TIME_CPU(NULL, "AcquireTexture", 0).GetMicroSeconds();
        g_SumPresentTextureGpu += NN_PERF_GET_ELAPSED_TIME_GPU("PresentTextureGpu", 0).GetMicroSeconds();
        g_MeasureCount++;
    }

    // WaitDisplaySync(currentBufferIndex)
    g_DisplayFence[currentBufferIndex].Sync(nn::TimeSpan::FromSeconds(1));

    if (frame > 0)
    {
        // WaitGpuSync(previousBufferIndex)
        g_GpuFence[previousBufferIndex].Sync(nn::TimeSpan::FromSeconds(1));
    }

    currentBufferIndex = 1 - currentBufferIndex;
    return true;
}

bool DeferredExecution(int frame)
{
    // FrameworkMode: DeferredSubmission

    static int currentBufferIndex = 0;
    int previousBufferIndex = 1 - currentBufferIndex;

    // AcquireTexture(currentBufferIndex)
    NN_PERF_SET_COLOR(nn::util::Color4u8::Blue());
    NN_PERF_BEGIN_MEASURE_NAME("AcquireTexture");
    {
        g_NextScanBufferIndex = -1;
        nn::gfx::AcquireScanBufferResult acquireResult = g_SwapChain.AcquireNextScanBufferIndex(
            &g_NextScanBufferIndex, &g_DisplaySemaphore[currentBufferIndex], &g_DisplayFence[currentBufferIndex]);
        NN_SDK_ASSERT(acquireResult == nn::gfx::AcquireScanBufferResult_Success);
        NN_UNUSED(acquireResult);
    }
    NN_PERF_END_MEASURE();

    g_Queue.SyncSemaphore(&g_DisplaySemaphore[currentBufferIndex]);
    g_Queue.Flush();

    // MakeCommand(currentBufferIndex)
    NN_PERF_SET_COLOR(nn::util::Color4u8::Green());
    NN_PERF_BEGIN_MEASURE_NAME("MakeCommand");
    if (MakeCommand(frame, currentBufferIndex) == false)
    {
        return false;
    }
    NN_PERF_END_MEASURE();

    // ExecuteCommand(previousBufferIndex)
    NN_PERF_SET_COLOR(nn::util::Color4u8::Blue());
    NN_PERF_BEGIN_MEASURE_NAME("ExecuteCommand");
    {
        g_Queue.ExecuteCommand(&g_CommandBuffer, NULL);
    }
    NN_PERF_END_MEASURE();

    // QueuePresentTexture
    g_Queue.ExecuteCommand(&g_CommandBufferBegin, NULL);
    NN_PERF_SET_COLOR(nn::util::Color4u8::Yellow());
    NN_PERF_BEGIN_MEASURE_NAME("PresentTexture");
    {
        g_Queue.Present(&g_SwapChain, 1);
    }
    NN_PERF_END_MEASURE();
    g_Queue.ExecuteCommand(&g_CommandBufferEnd, &g_GpuFence[currentBufferIndex]);
    g_Queue.Flush();

    if (frame > 60)
    {
        g_SumPresentTexture += NN_PERF_GET_ELAPSED_TIME_CPU(NULL, "PresentTexture", 0).GetMicroSeconds();
        g_SumAcquireTexture += NN_PERF_GET_ELAPSED_TIME_CPU(NULL, "AcquireTexture", 0).GetMicroSeconds();
        g_SumPresentTextureGpu += NN_PERF_GET_ELAPSED_TIME_GPU("PresentTextureGpu", 0).GetMicroSeconds();
        g_MeasureCount++;
    }

    // WaitDisplaySync(currentBufferIndex)
    g_DisplayFence[currentBufferIndex].Sync(nn::TimeSpan::FromSeconds(1));

    if (frame > 0)
    {
        // WaitGpuSync(previousBufferIndex)
        g_GpuFence[previousBufferIndex].Sync(nn::TimeSpan::FromSeconds(1));
    }

    currentBufferIndex = 1 - currentBufferIndex;
    return true;
}

//---------------------------------------------------------------
//  Main Function
//  メイン関数です。
//---------------------------------------------------------------
//extern "C" void nnMain()
TEST(CheckAcquirePresentPerformance, Run)
{
    InitializeFs();
    InitializeResources();

    g_TextureDescriptorPool.BeginUpdate();
    {
        g_TextureDescriptorPool.SetTextureView(g_TextureDescriptorBaseIndex + 2,
            &g_ColorTextureView);
    }
    g_TextureDescriptorPool.EndUpdate();

    g_SamplerDescriptorPool.BeginUpdate();
    {
        g_SamplerDescriptorPool.SetSampler(g_SamplerDescriptorBaseIndex, &g_Sampler);
    }
    g_SamplerDescriptorPool.EndUpdate();

    // プリミティブレンダラの初期化
    InitializePrimitiveRenderer();

    // デバッグフォントの初期化
    InitializeDebugFont();

    // 処理メータの初期化
    InitializeLoadMeter();

    // 現在のスレッド(メインスレッド)を 0 番コアに割り当て
    nn::os::SetThreadCoreMask(nn::os::GetCurrentThread(), 0, 1);



    // DeferredSubmission のレンダリング
    g_SumPresentTexture = 0;
    g_SumAcquireTexture = 0;
    g_SumPresentTextureGpu = 0;
    g_MeasureCount = 0;
    for (int frame = 0; frame < 60 * 20; ++frame)
    {
        NN_PERF_BEGIN_FRAME();
        {
            DeferredSubmission(frame);
        }
        NN_PERF_END_FRAME();
    }
    g_Queue.Sync();
    NN_LOG("present %lld\nacquire %lld\npresentGpu %lld\n", g_SumPresentTexture / g_MeasureCount, g_SumAcquireTexture / g_MeasureCount, g_SumPresentTextureGpu / g_MeasureCount);
    // DeferredSubmission は vsync 直後の presentTexture の CPU 負荷が高くなりがち

    //EXPECT_LT(g_SumPresentTexture / g_MeasureCount, 340);
    //EXPECT_LT(g_SumAcquireTexture / g_MeasureCount, 410);
    //EXPECT_LT(g_SumPresentTextureGpu / g_MeasureCount, 380);
    // CI で日々の変化を確認できるようにログに残しておく
    NN_LOG("##teamcity[buildStatisticValue key='Video_DeferredSubmission_PresentTexture' value='%lld']\n", g_SumPresentTexture / g_MeasureCount);
    NN_LOG("##teamcity[buildStatisticValue key='Video_DeferredSubmission_AcquireTexture' value='%lld']\n", g_SumAcquireTexture / g_MeasureCount);
    NN_LOG("##teamcity[buildStatisticValue key='Video_DeferredSubmission_PresentTextureGpu' value='%lld']\n", g_SumPresentTextureGpu / g_MeasureCount);



    g_SumPresentTexture = 0;
    g_SumAcquireTexture = 0;
    g_SumPresentTextureGpu = 0;
    g_MeasureCount = 0;
    // DeferredExecution のレンダリング
    for (int frame = 0; frame < 60 * 20; ++frame)
    {
        NN_PERF_BEGIN_FRAME();
        {
            DeferredExecution(frame);
        }
        NN_PERF_END_FRAME();
    }
    g_Queue.Sync();
    NN_LOG("present %lld\nacquire %lld\npresentGpu %lld\n", g_SumPresentTexture / g_MeasureCount, g_SumAcquireTexture / g_MeasureCount, g_SumPresentTextureGpu / g_MeasureCount);
    // DeferredSubmission は vsync 直後の acquireTexture の CPU 負荷が高くなりがち

    //EXPECT_LT(g_SumPresentTexture / g_MeasureCount, 280);
    //EXPECT_LT(g_SumAcquireTexture / g_MeasureCount, 495);
    //EXPECT_LT(g_SumPresentTextureGpu / g_MeasureCount, 380);
    NN_LOG("##teamcity[buildStatisticValue key='Video_DeferredExecution_PresentTexture' value='%lld']\n", g_SumPresentTexture / g_MeasureCount);
    NN_LOG("##teamcity[buildStatisticValue key='Video_DeferredExecution_AcquireTexture' value='%lld']\n", g_SumAcquireTexture / g_MeasureCount);
    NN_LOG("##teamcity[buildStatisticValue key='Video_DeferredExecution_PresentTextureGpu' value='%lld']\n", g_SumPresentTextureGpu / g_MeasureCount);



    // 処理メータ破棄
    FinalizeLoadMeter();

    // デバッグフォントの破棄
    FinalizeDebugFont();

    // プリミティブレンダラの破棄
    FinalizePrimitiveRenderer();

    FinalizeResources();

    FinalizeFs();
    SUCCEED();
}
