﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

////===========================================================================
///  DEMOGfd.c
///
///     This is gfd file interface code for the DEMO library.
///
////===========================================================================

#include <gfx/demo.h>
#ifdef CAFE
#include <cafe/gx2ut.h>

static nn::gfx::ImageDimension _DEMOConvertImageDimensions( GX2SurfaceDim surfaceDimensions )
{
    nn::gfx::ImageDimension dimensions[] = {
        nn::gfx::ImageDimension_1d, // GX2_SURFACE_DIM_1D
        nn::gfx::ImageDimension_2d, // GX2_SURFACE_DIM_2D
        nn::gfx::ImageDimension_3d, // GX2_SURFACE_DIM_3D
        nn::gfx::ImageDimension_CubeMap, // GX2_SURFACE_DIM_CUBE
        nn::gfx::ImageDimension_1dArray, // GX2_SURFACE_DIM_1D_ARRAY
        nn::gfx::ImageDimension_2dArray, // GX2_SURFACE_DIM_2D_ARRAY
        nn::gfx::ImageDimension_2dMultisample, // GX2_SURFACE_DIM_2D_MSAA
        nn::gfx::ImageDimension_2dMultisampleArray, // GX2_SURFACE_DIM_2D_MSAA_ARRAY
    };

    return dimensions[ surfaceDimensions ];
}

static nn::gfx::ImageFormat _DEMOConvertImageFormat( GX2SurfaceFormat surfaceFormat )
{
    switch( surfaceFormat )
    {
    case GX2_SURFACE_FORMAT_TC_R8_UNORM:              return nn::gfx::ImageFormat_R8_Unorm;
    case GX2_SURFACE_FORMAT_TC_R8_UINT:               return nn::gfx::ImageFormat_R8_Uint;
    case GX2_SURFACE_FORMAT_TC_R8_SNORM:              return nn::gfx::ImageFormat_R8_Snorm;
    case GX2_SURFACE_FORMAT_TC_R8_SINT:               return nn::gfx::ImageFormat_R8_Sint;
    case GX2_SURFACE_FORMAT_TCD_R16_UNORM:            return nn::gfx::ImageFormat_R16_Unorm;
    case GX2_SURFACE_FORMAT_TC_R16_UINT:              return nn::gfx::ImageFormat_R16_Uint;
    case GX2_SURFACE_FORMAT_TC_R16_SNORM:             return nn::gfx::ImageFormat_R16_Snorm;
    case GX2_SURFACE_FORMAT_TC_R16_SINT:              return nn::gfx::ImageFormat_R16_Sint;
    case GX2_SURFACE_FORMAT_TC_R16_FLOAT:             return nn::gfx::ImageFormat_R16_Float;
    case GX2_SURFACE_FORMAT_TC_R8_G8_UNORM:           return nn::gfx::ImageFormat_R8_G8_Unorm;
    case GX2_SURFACE_FORMAT_TC_R8_G8_UINT:            return nn::gfx::ImageFormat_R8_G8_Uint;
    case GX2_SURFACE_FORMAT_TC_R8_G8_SNORM:           return nn::gfx::ImageFormat_R8_G8_Snorm;
    case GX2_SURFACE_FORMAT_TC_R8_G8_SINT:            return nn::gfx::ImageFormat_R8_G8_Sint;
    case GX2_SURFACE_FORMAT_TCS_R5_G6_B5_UNORM:       return nn::gfx::ImageFormat_R5_G6_B5_Unorm;
    case GX2_SURFACE_FORMAT_TC_R5_G5_B5_A1_UNORM:     return nn::gfx::ImageFormat_R5_G5_B5_A1_Unorm;
    case GX2_SURFACE_FORMAT_TC_R4_G4_B4_A4_UNORM:     return nn::gfx::ImageFormat_R4_G4_B4_A4_Unorm;
    case GX2_SURFACE_FORMAT_TC_A1_B5_G5_R5_UNORM:     return nn::gfx::ImageFormat_A1_B5_G5_R5_Unorm;
    case GX2_SURFACE_FORMAT_TC_R32_UINT:              return nn::gfx::ImageFormat_R32_Uint;
    case GX2_SURFACE_FORMAT_TC_R32_SINT:              return nn::gfx::ImageFormat_R32_Sint;
    case GX2_SURFACE_FORMAT_TCD_R32_FLOAT:            return nn::gfx::ImageFormat_R32_Float;
    case GX2_SURFACE_FORMAT_TC_R16_G16_UNORM:         return nn::gfx::ImageFormat_R16_G16_Unorm;
    case GX2_SURFACE_FORMAT_TC_R16_G16_UINT:          return nn::gfx::ImageFormat_R16_G16_Uint;
    case GX2_SURFACE_FORMAT_TC_R16_G16_SNORM:         return nn::gfx::ImageFormat_R16_G16_Snorm;
    case GX2_SURFACE_FORMAT_TC_R16_G16_SINT:          return nn::gfx::ImageFormat_R16_G16_Sint;
    case GX2_SURFACE_FORMAT_TC_R16_G16_FLOAT:         return nn::gfx::ImageFormat_R16_G16_Float;
    case GX2_SURFACE_FORMAT_D_D24_S8_UNORM:           return nn::gfx::ImageFormat_D24_Unorm_S8_Uint;
    case GX2_SURFACE_FORMAT_TC_R11_G11_B10_FLOAT:     return nn::gfx::ImageFormat_R11_G11_B10_Float;
    case GX2_SURFACE_FORMAT_TCS_R10_G10_B10_A2_UNORM: return nn::gfx::ImageFormat_R10_G10_B10_A2_Unorm;
    case GX2_SURFACE_FORMAT_TC_R10_G10_B10_A2_UINT:   return nn::gfx::ImageFormat_R10_G10_B10_A2_Uint;
    case GX2_SURFACE_FORMAT_TCS_R8_G8_B8_A8_UNORM:    return nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm;
    case GX2_SURFACE_FORMAT_TC_R8_G8_B8_A8_UINT:      return nn::gfx::ImageFormat_R8_G8_B8_A8_Uint;
    case GX2_SURFACE_FORMAT_TC_R8_G8_B8_A8_SNORM:     return nn::gfx::ImageFormat_R8_G8_B8_A8_Snorm;
    case GX2_SURFACE_FORMAT_TC_R8_G8_B8_A8_SINT:      return nn::gfx::ImageFormat_R8_G8_B8_A8_Sint;
    case GX2_SURFACE_FORMAT_TCS_R8_G8_B8_A8_SRGB:     return nn::gfx::ImageFormat_R8_G8_B8_A8_UnormSrgb;
    case GX2_SURFACE_FORMAT_D_D32_FLOAT_S8_UINT_X24:  return nn::gfx::ImageFormat_D32_Float_S8_Uint_X24;
    case GX2_SURFACE_FORMAT_TC_R32_G32_UINT:          return nn::gfx::ImageFormat_R32_G32_Uint;
    case GX2_SURFACE_FORMAT_TC_R32_G32_SINT:          return nn::gfx::ImageFormat_R32_G32_Sint;
    case GX2_SURFACE_FORMAT_TC_R32_G32_FLOAT:         return nn::gfx::ImageFormat_R32_G32_Float;
    case GX2_SURFACE_FORMAT_TC_R16_G16_B16_A16_UNORM: return nn::gfx::ImageFormat_R16_G16_B16_A16_Unorm;
    case GX2_SURFACE_FORMAT_TC_R16_G16_B16_A16_UINT:  return nn::gfx::ImageFormat_R16_G16_B16_A16_Uint;
    case GX2_SURFACE_FORMAT_TC_R16_G16_B16_A16_SNORM: return nn::gfx::ImageFormat_R16_G16_B16_A16_Snorm;
    case GX2_SURFACE_FORMAT_TC_R16_G16_B16_A16_SINT:  return nn::gfx::ImageFormat_R16_G16_B16_A16_Sint;
    case GX2_SURFACE_FORMAT_TC_R16_G16_B16_A16_FLOAT: return nn::gfx::ImageFormat_R16_G16_B16_A16_Float;
    case GX2_SURFACE_FORMAT_TC_R32_G32_B32_A32_UINT:  return nn::gfx::ImageFormat_R32_G32_B32_A32_Uint;
    case GX2_SURFACE_FORMAT_TC_R32_G32_B32_A32_SINT:  return nn::gfx::ImageFormat_R32_G32_B32_A32_Sint;
    case GX2_SURFACE_FORMAT_TC_R32_G32_B32_A32_FLOAT: return nn::gfx::ImageFormat_R32_G32_B32_A32_Float;
    case GX2_SURFACE_FORMAT_T_BC1_UNORM:              return nn::gfx::ImageFormat_Bc1_Unorm;
    case GX2_SURFACE_FORMAT_T_BC1_SRGB:               return nn::gfx::ImageFormat_Bc1_UnormSrgb;
    case GX2_SURFACE_FORMAT_T_BC2_UNORM:              return nn::gfx::ImageFormat_Bc2_Unorm;
    case GX2_SURFACE_FORMAT_T_BC2_SRGB:               return nn::gfx::ImageFormat_Bc2_UnormSrgb;
    case GX2_SURFACE_FORMAT_T_BC3_UNORM:              return nn::gfx::ImageFormat_Bc3_Unorm;
    case GX2_SURFACE_FORMAT_T_BC3_SRGB:               return nn::gfx::ImageFormat_Bc3_UnormSrgb;
    case GX2_SURFACE_FORMAT_T_BC4_UNORM:              return nn::gfx::ImageFormat_Bc4_Unorm;
    case GX2_SURFACE_FORMAT_T_BC4_SNORM:              return nn::gfx::ImageFormat_Bc4_Snorm;
    case GX2_SURFACE_FORMAT_T_BC5_UNORM:              return nn::gfx::ImageFormat_Bc5_Unorm;
    case GX2_SURFACE_FORMAT_T_BC5_SNORM:              return nn::gfx::ImageFormat_Bc5_Snorm;
    default: NN_UNEXPECTED_DEFAULT;
    }
}

// Read Vertex Shader from buffer
BOOL DEMOGFDReadVertexShader(void *ppVoid, u32 index, const void *pData)
{
    GX2VertexShader **ppShader = static_cast< GX2VertexShader **> ( ppVoid );
    GX2VertexShader *pHeader;
    void *pProgram;
    u32 ret;
    u32 headerSize;
    u32 programSize;

    // Check inputs
    if(pData == NULL || ppShader == NULL)
    {
        return FALSE;
    }

    // Check index number of shaders
    if(index >= GFDGetVertexShaderCount(pData))
    {
        return FALSE;
    }

    // Get the size
    headerSize = GFDGetVertexShaderHeaderSize(index, pData);
    programSize = GFDGetVertexShaderProgramSize(index, pData);

    if(!headerSize || !programSize)
    {
        return FALSE;
    }

    // Alloc header
    pHeader = (GX2VertexShader *)DEMOAllocEx(headerSize, PPC_IO_BUFFER_ALIGN);

    GX2RBuffer shaderProgram;
    GX2UTCreateShaderProgram(&shaderProgram, programSize);
    pProgram = GX2RLockBuffer(&shaderProgram);

    // Get the shader structure and program
    // from file buffer into user aligned buffer which created above
    ret = GFDGetVertexShader(pHeader, pProgram, index, pData);

    if (ret)
    {
        GX2RUnlockBuffer(&shaderProgram);
        pHeader->shaderPtr = NULL;
        pHeader->shaderProgram = shaderProgram;
        *ppShader = pHeader;
    }
    else
    {
        GX2RUnlockBufferEx(&shaderProgram, GX2R_OPTION_NO_INVALIDATE);
        OSReport("Warning: Invalid Vertex Shader :%d", ret);
        // Free if fail
        if(pHeader)
        {
            DEMOFree(pHeader);
        }

        GX2RDestroyBuffer(&shaderProgram);
    }

    return ret;
}

// Read Pixel Shader from buffer
BOOL DEMOGFDReadPixelShader(void *ppVoid, u32 index, const void *pData)
{
    GX2PixelShader **ppShader = static_cast< GX2PixelShader ** > ( ppVoid );
    GX2PixelShader *pHeader;
    void *pProgram;
    u32 ret;
    u32 headerSize;
    u32 programSize;

    // Check inputs
    if(pData == NULL || ppShader == NULL)
    {
        OSReport("DEMOGFDReadPixelShader: NULL inputs\n");
        return FALSE;
    }

    // Check index number of shaders
    if(index >= GFDGetPixelShaderCount(pData))
    {
        OSReport("DEMOGFDReadPixelShader: bad index\n");
        return FALSE;
    }
    // Get the size
    headerSize = GFDGetPixelShaderHeaderSize(index, pData);
    programSize = GFDGetPixelShaderProgramSize(index, pData);

    if(!headerSize || !programSize)
    {
        OSReport("DEMOGFDReadPixelShader: header or program size is 0\n");
        return FALSE;
    }

    // Alloc header
    pHeader = (GX2PixelShader *)DEMOAllocEx(headerSize, PPC_IO_BUFFER_ALIGN);

    // Alloc shader program
    GX2RBuffer shaderProgram;
    GX2UTCreateShaderProgram(&shaderProgram, programSize);
    pProgram = GX2RLockBuffer(&shaderProgram);

    // Get the shader structure and program
    // from file buffer into user aligned buffer which created above
    ret = GFDGetPixelShader(pHeader, pProgram, index, pData);

    if (ret)
    {
        GX2RUnlockBuffer(&shaderProgram);
        pHeader->shaderPtr = NULL;
        pHeader->shaderProgram = shaderProgram;
        *ppShader = pHeader;
    }
    else
    {
        GX2RUnlockBufferEx(&shaderProgram, GX2R_OPTION_NO_INVALIDATE);
        OSReport("Warning: Invalid Pixel Shader :%d\n", ret);
        // Free if fail
        if(pHeader)
        {
            DEMOFree(pHeader);
        }

        GX2RDestroyBuffer(&shaderProgram);
    }

    return ret;
}

// Read Geometry Shader from buffer
BOOL DEMOGFDReadGeometryShader(void *pVoid, u32 index, const void *pData)
{
    GX2GeometryShader **ppShader = static_cast< GX2GeometryShader ** >( pVoid );
    GX2GeometryShader *pHeader;
    void *pProgram;
    void *pCopyProgram;
    u32 ret;
    u32 headerSize;
    u32 programSize;
    u32 copyProgramSize;

    // Check inputs
    if(pData == NULL || ppShader == NULL)
    {
        return FALSE;
    }

    // Check index number of shaders
    if(index >= GFDGetGeometryShaderCount(pData))
    {
        return FALSE;
    }

    // Get the size
    headerSize = GFDGetGeometryShaderHeaderSize(index, pData);
    programSize = GFDGetGeometryShaderProgramSize(index, pData);
    copyProgramSize = GFDGetGeometryShaderCopyProgramSize(index, pData);

    if(!headerSize || !programSize)
    {
        return FALSE;
    }

    // Alloc header
    pHeader = (GX2GeometryShader *)DEMOAllocEx(headerSize, PPC_IO_BUFFER_ALIGN);

    // Create shader program buffer
    GX2RBuffer shaderProgram;
    GX2UTCreateShaderProgram(&shaderProgram, programSize);

    // Alloc copy shader program (Must be aligned on 256 byte boundary.)
    GX2RBuffer copyShaderProgram;
    GX2UTCreateShaderProgram(&copyShaderProgram, copyProgramSize);

    pProgram = GX2RLockBuffer(&shaderProgram);
    pCopyProgram = GX2RLockBuffer(&copyShaderProgram);

    // Get the shader structure and program
    // from file buffer into user aligned buffer which created above
    ret = GFDGetGeometryShader(pHeader, pProgram, pCopyProgram, index, pData);

    if(ret)
    {
        GX2RUnlockBuffer(&shaderProgram);
        GX2RUnlockBuffer(&copyShaderProgram);

        pHeader->shaderPtr = NULL;
        pHeader->shaderProgram = shaderProgram;
        pHeader->copyShaderPtr = NULL;
        pHeader->copyShaderProgram = copyShaderProgram;
        *ppShader = pHeader;
    }
    else
    {
        OSReport("Warning: Invalid Geometry Shader :%d", ret);
        // Free if fail
        if(pHeader)
        {
            DEMOFree(pHeader);
        }

        GX2RUnlockBufferEx(&shaderProgram, GX2R_OPTION_NO_INVALIDATE);
        GX2RUnlockBufferEx(&copyShaderProgram, GX2R_OPTION_NO_INVALIDATE);

        GX2RDestroyBuffer(&shaderProgram);
        GX2RDestroyBuffer(&copyShaderProgram);
    }

    return ret;
}

// Read Compute Shader from buffer
BOOL DEMOGFDReadComputeShader(void *pVoid, u32 index, const void *pData)
{
    GX2ComputeShader **ppShader = static_cast< GX2ComputeShader** >( pVoid );
    GX2ComputeShader *pHeader;
    void *pProgram;
    u32 ret;
    u32 headerSize;
    u32 programSize;

    // Check inputs
    if(pData == NULL || ppShader == NULL)
    {
        return FALSE;
    }

    // Check index number of shaders
    if(index >= GFDGetComputeShaderCount(pData))
    {
        return FALSE;
    }

    // Get the size
    headerSize = GFDGetComputeShaderHeaderSize(index, pData);
    programSize = GFDGetComputeShaderProgramSize(index, pData);

    if(!headerSize || !programSize)
    {
        return FALSE;
    }

    // Alloc header
    pHeader = (GX2ComputeShader *)DEMOAllocEx(headerSize, PPC_IO_BUFFER_ALIGN);

    GX2RBuffer shaderProgram;
    GX2UTCreateShaderProgram(&shaderProgram, programSize);
    pProgram = GX2RLockBuffer(&shaderProgram);

    // Get the shader structure and program
    // from file buffer into user aligned buffer which created above
    ret = GFDGetComputeShader(pHeader, pProgram, index, pData);

    if (ret)
    {
        GX2RUnlockBuffer(&shaderProgram);
        pHeader->shaderPtr = NULL;
        pHeader->shaderProgram = shaderProgram;
        *ppShader = pHeader;
    }
    else
    {
        GX2RUnlockBufferEx(&shaderProgram, GX2R_OPTION_NO_INVALIDATE);
        OSReport("Warning: Invalid Compute Shader :%d", ret);
        // Free if fail
        if(pHeader)
        {
            DEMOFree(pHeader);
        }

        GX2RDestroyBuffer(&shaderProgram);
    }

    return ret;
}

// Read texture from buffer
BOOL DEMOGFDReadTexture(nn::gfx::Texture* pTexture, nn::gfx::TextureInfo* pInfo, nn::gfx::TextureView* pView, DEMOGfxMemPool** ppPool, u32 index, const void *pData)
{
    GX2Texture *pHeader;
    void* pImage = NULL;
    void* pMipImage = NULL;
    u32 ret;
    u32 headerSize;
    u32 imageSize;
    u32 mipImageSize;
    u32 alignment;

    // Check inputs
    if(pData == NULL || pTexture == NULL || pInfo == NULL)
    {
        return FALSE;
    }

    // Check index number of shaders
    if(index >= GFDGetTextureCount(pData))
    {
        return FALSE;
    }

    // Get the size of texture header, image and align
    headerSize = GFDGetTextureHeaderSize(index, pData);
    imageSize  = GFDGetTextureImageSize(index, pData);
    mipImageSize = GFDGetTextureMipImageSize(index, pData);
    alignment = GFDGetTextureAlignmentSize(index, pData);

    if(!headerSize || !imageSize)
    {
        return FALSE;
    }

    // nn::gfx requires the imageSize/mipSize be aligned so that it can put
    // the aux buffer after it if necessary
    u32 alignedImageSize = ( imageSize + ( alignment - 1 ) ) & ( ~( alignment - 1 ) );
    u32 alignedMipImageSize = ( mipImageSize + ( alignment - 1 ) ) & ( ~( alignment - 1 ) );
    u32 totalImageSize = alignedImageSize + alignedMipImageSize;

    pHeader = (GX2Texture *)DEMOAllocEx(headerSize, PPC_IO_BUFFER_ALIGN);
    pImage = DEMOGfxAllocMEM2( totalImageSize, alignment );
    if ( mipImageSize )
    {
        pMipImage = ( ( u8* ) pImage ) + alignedImageSize;
    }

    ret = GFDGetTexture( pHeader, pImage, pMipImage, index, pData );

    if(!ret)
    {
        OSReport("Warning: Invalid Texture :%d", ret);
        // Free if fail
        if(pHeader)
        {
            DEMOFree(pHeader);
        }

        return ret;
    }
    else
    {
        // Setup the texture info
        pInfo->SetDefault();
        pInfo->SetWidth( pHeader->surface.width );
        pInfo->SetHeight( pHeader->surface.height );

        switch( pHeader->surface.dim )
        {
        case GX2_SURFACE_DIM_1D_ARRAY:
        case GX2_SURFACE_DIM_2D_ARRAY:
        case GX2_SURFACE_DIM_2D_MSAA_ARRAY:
        case GX2_SURFACE_DIM_CUBE:
            {
                pInfo->SetArrayLength( pHeader->surface.depth );
            }
            break;

        default:
            {
                pInfo->SetDepth( pHeader->surface.depth );
            }
            break;
        }

        nn::gfx::ImageDimension pInfoDim = _DEMOConvertImageDimensions(pHeader->surface.dim);
        pInfo->SetGpuAccessFlags( nn::gfx::GpuAccess_Texture | nn::gfx::GpuAccess_Read );
        pInfo->SetImageStorageDimension( DEMOConvertToStorageDimension( pInfoDim ) );
        pInfo->SetImageFormat( _DEMOConvertImageFormat( pHeader->surface.format ) );
        pInfo->SetMultiSampleCount( 1 << pHeader->surface.aa );
        pInfo->SetSwizzle( GX2GetSurfaceSwizzle( &pHeader->surface ) );
        pInfo->SetTileMode( pHeader->surface.tileMode == GX2_TILE_MODE_LINEAR_ALIGNED ? nn::gfx::TileMode_Linear : nn::gfx::TileMode_Optimal );
        pInfo->SetMipCount( pHeader->surface.numMips );

        *ppPool = DEMOGfxMemPool::CreateFromMemory( pImage, totalImageSize, nn::gfx::MemoryPoolProperty_CpuInvisible | nn::gfx::MemoryPoolProperty_GpuCached );
        pTexture->Initialize( &DEMODevice, *pInfo, (*ppPool)->GetPool(), 0, totalImageSize );

        if ( pView )
        {
            nn::gfx::TextureViewInfo info;

            info.SetDefault();
            info.SetImageDimension( pInfoDim );
            info.SetImageFormat( pInfo->GetImageFormat() );
            info.SetTexturePtr( pTexture );
            info.EditSubresourceRange().EditMipRange().SetMinMipLevel( pHeader->viewFirstMip );
            info.EditSubresourceRange().EditMipRange().SetMipCount( pHeader->viewNumMips );
            info.EditSubresourceRange().EditArrayRange().SetBaseArrayIndex( pHeader->viewFirstSlice );
            info.EditSubresourceRange().EditArrayRange().SetArrayLength( pHeader->viewNumSlices );
            pView->Initialize( &DEMODevice, info );
        }
    }

    return ret;
}

// Free vertex shader
void DEMOGFDFreeVertexShader(void *pVoid)
{
    GX2VertexShader* pShader = static_cast< GX2VertexShader* >( pVoid );
    if(pShader)
    {
        if(pShader->shaderPtr)
        {
            DEMOGfxFreeMEM2(pShader->shaderPtr);
        } else {
            GX2RDestroyBuffer(&pShader->shaderProgram);
        }

        DEMOFree(pShader);
    }
}

// Free pixel shader
void DEMOGFDFreePixelShader(void *pVoid)
{
    GX2PixelShader* pShader = static_cast< GX2PixelShader* >( pVoid );
    if(pShader)
    {
        if(pShader->shaderPtr)
        {
            DEMOGfxFreeMEM2(pShader->shaderPtr);
        } else {
            GX2RDestroyBuffer(&pShader->shaderProgram);
        }

        DEMOFree(pShader);
    }
}

// Free geometry shader
void DEMOGFDFreeGeometryShader(void* pVoid)
{
    GX2GeometryShader* pShader = static_cast< GX2GeometryShader* >( pVoid );
    if(pShader)
    {
        if(pShader->shaderPtr)
        {
            DEMOGfxFreeMEM2(pShader->shaderPtr);
        } else {
            GX2RDestroyBuffer(&pShader->shaderProgram);
        }

        if(pShader->copyShaderPtr)
        {
            DEMOGfxFreeMEM2(pShader->copyShaderPtr);
        } else {
            GX2RDestroyBuffer(&pShader->copyShaderProgram);
        }

        DEMOFree(pShader);
    }
}

// Free compute shader
void DEMOGFDFreeComputeShader(void* pVoid)
{
    GX2ComputeShader* pShader = static_cast< GX2ComputeShader* >( pVoid );
    if(pShader)
    {
        if(pShader->shaderPtr)
        {
            DEMOGfxFreeMEM2(pShader->shaderPtr);
        } else {
            GX2RDestroyBuffer(&pShader->shaderProgram);
        }

        DEMOFree(pShader);
    }
}

// Free texture
void DEMOGFDFreeTexture(nn::gfx::Texture *pTexture, nn::gfx::TextureView* pTextureView, DEMOGfxMemPool* pMemPool )
{
    if(pTexture)
    {
        pTexture->Finalize( &DEMODevice );
        pTextureView->Finalize( &DEMODevice );
        pMemPool->Finalize();
    }
}


// Read Aligned Texture
GX2Texture* DEMOGFDReadAlignedTexture(void **pFileBuf, u32 index, u32 align, const char *fileName)
{
    u32 len;
    GX2Texture *pTexture;

    // Read the texture file into a buffer
    *pFileBuf = DEMOGfxLoadAssetFileAlign(fileName, &len, align);

    // Check Align mode of the buffer
    if(GFDGetAlignMode(*pFileBuf) != GFD_ALIGN_MODE_ENABLE)
    {
        DEMOGFDFreeAlignedTexture(*pFileBuf);
        ASSERT(!"Input file isn't aligned. Please use \"-align\" option when it is created by tool.");
        return NULL;
    }

    pTexture = GFDGetGX2RTexturePointer(index, *pFileBuf);

    return pTexture;
}

// Free Aligned Texture
void DEMOGFDFreeAlignedTexture(void *pFileBuf)
{
    if(pFileBuf)
    {
        DEMOFree(pFileBuf);

        // Destroy the GX2RSurface objects
        u32 textureCount = GFDGetTextureCount(pFileBuf);
        for (u32 i=0; i<textureCount; i++)
        {
            GX2Texture* pTexture = GFDGetTexturePointer(i, pFileBuf);
            GX2RDestroySurface(&pTexture->surface);
        }
    }
}

// End of CAFE code

#endif

