﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

////===========================================================================
///  demoFont.c
///
///     This is font code for the demo library.
///
////===========================================================================
#include <cstdarg>
#include <cstdio>

#include <gfx/demo.h>
#include <gfx/demo_FontData.h> // Font Data
#ifdef CAFE
#include <cafe/gfd.h>
#endif

static const char * const FONT_GSH_SHADER_FILE = "shaders/demoFont";
static int s_slotPixelUniform;
static int s_slotVertexUniform;
static int s_slotTexture;
static DEMOGfxShader s_fontShader;

//
// uniform buffers
//
static int s_vsBufferSize;
static int s_psBufferSize;

// --------------------------------------------------------------------------
//  Macro definitions
// --------------------------------------------------------------------------
// Max Buffer Size for vsnprintf_s
static const int FONT_MAX_BUFSIZ = 512;


// Simple structure that contains all the data
// we need to know about characters in the font texture
typedef struct
{
    u32   id;
    s32 minS;
    s32 minT;
    s32 maxS;
    s32 maxT;
    s32 minX;
    s32 minY;
    s32 maxX;
    s32 maxY;
    s32 xAdvance;
} CharContainer_t;

typedef struct
{
    // Contains all the data we need to render each character
    CharContainer_t* pCharDataBuffer;

    // Font texture height
    f32 fontTextureWidth;

    // Font texture width
    f32 fontTextureHeight;

    // The number of characters stored in the raw char data
    u32 numCharData;

    // The offset height of grid
    f32 gridOffsetY;

    // The offset width of grid
    f32 gridOffsetX;

    // Pointer to Font Image Data
    u8* pFontImageData;
    DEMOGfxMemPool* pFontMemPool;

    // --- GPU MEM ---
    // includes pointer to buffer containing the Texture image Data
    nn::gfx::Texture texture;
    nn::gfx::TextureView textureView;
    nn::gfx::DescriptorSlot textureDescriptor;

} FontData_t;

// P: Proportional, F: Fixed mono_space
static FontData_t s_FontP, s_FontF;
static bool s_sampler_init = false;
static nn::gfx::Sampler s_sampler;
static nn::gfx::DescriptorSlot s_samplerDescriptor;

static BOOL s_enabled=TRUE;

// Max number of unique characters defined in each font
static const int MAX_NUM_CHARS = 100;
static CharContainer_t s_CharDataBufferP[MAX_NUM_CHARS];
static CharContainer_t s_CharDataBufferF[MAX_NUM_CHARS];

// Instance
static DEMOFontInstance* gDemoFontCurInstance = NULL;

//--------------------------------------------------------------------------
//   Forward references
//--------------------------------------------------------------------------
static FontData_t* GetCurrentFont(void);
static void UpdateScale(DEMOFontFontData* pFont, f32 scaleX, f32 scaleY);
static void GetFontCharData(FontData_t* pFont,
                            const u32* pFontHeader,
                            const u32* pCharData,
                            const u8* pFontImageData);
static void InitFontTexture(FontData_t* pFont);
static void InitShader(void);
static u32  BSearchIndex(FontData_t* pFont, u32 id);

//---------------------------------------------------------------------------
// structure to record uniform buffers used in a frame
//---------------------------------------------------------------------------
struct FrameUniBuffer
{
    struct FrameUniBuffer* next;

    DEMOGfxBuffer s_psUniformBuffer;
    DEMOGfxBuffer s_vsUniformBuffer;
    DEMOGfxBuffer indexBuffer;
};
static const int MAX_INDEX_COUNT = 80;
static const int MAX_INDICES =  MAX_INDEX_COUNT * 8;

// list of buffers used for rendering strings
static FrameUniBuffer* curFrameData = NULL;     // currently used entry in list
static FrameUniBuffer* firstFrameData = NULL;  // head of list

// pointer to per-frame memory data
static DEMOGfxMemPool* s_FrameMemPool;
static const int DEMO_FONT_FRAME_DATA_SIZE = 2 * 1024 * 1024;

// --------------------------------------------------------------------------
//  Init
// --------------------------------------------------------------------------
DEMOFontInstance* DEMOFontInit()
{

    // Get Font char data
    GetFontCharData(&s_FontF,
                    s_FontHeaderF, (u32*)s_CharDataF, s_FontImageDataF);

    GetFontCharData(&s_FontP,
                    s_FontHeaderP, (u32*)s_CharDataP, s_FontImageDataP);

    // Initialise Shader
    InitShader();

    // Initialize Font Texture
    InitFontTexture(&s_FontP);
    InitFontTexture(&s_FontF);

    DEMOFontAddInstance();

    s_FrameMemPool = DEMOGfxSharedPool->AllocSubPool(DEMO_FONT_FRAME_DATA_SIZE, 4096);

    return gDemoFontCurInstance;
}

// --------------------------------------------------------------------------
//  Free per-frame data
// --------------------------------------------------------------------------
void
DEMOFontFreeData()
{
    FrameUniBuffer *next;

    curFrameData = firstFrameData;
    for (;;)
    {
        next = curFrameData;
        if (!next)
        {
            break;
        }
        curFrameData = next->next;
        next->s_psUniformBuffer.Finalize();
        next->s_vsUniformBuffer.Finalize();
        next->indexBuffer.Finalize();
        delete next;
    }
    s_FrameMemPool->Reset();
    curFrameData = firstFrameData = NULL;
}

// --------------------------------------------------------------------------
//  ShutDown
// --------------------------------------------------------------------------
void DEMOFontShutdown()
{
    DEMOFontFreeData(); // free per-frame data

    s_sampler.Finalize( &DEMODevice );

    // Delete the instance
    DEMOFontDeleteInstance(gDemoFontCurInstance);

    // Free the shader
    DEMOGfxFreeShaders( &s_fontShader );

    // Free texture
    s_FontP.textureView.Finalize( &DEMODevice );
    s_FontP.texture.Finalize( &DEMODevice );
    s_FontF.textureView.Finalize( &DEMODevice );
    s_FontF.texture.Finalize( &DEMODevice );

    s_FontP.pFontMemPool->Finalize();
    delete s_FontP.pFontMemPool;
    s_FontF.pFontMemPool->Finalize();
    delete s_FontF.pFontMemPool;

    s_FrameMemPool->Finalize();
    delete s_FrameMemPool;
}


//
// start a new frame
//
void
DEMOFontNewFrame()
{
    curFrameData = firstFrameData;
}

// --------------------------------------------------------------------------
//  Enable
// --------------------------------------------------------------------------
void DEMOFontDrawEnable(BOOL enable)
{
    s_enabled = enable;

        // MUST verify the instance exists before setting it!
    if (gDemoFontCurInstance)
    {
        gDemoFontCurInstance->enabled = enable;
    }
}

// --------------------------------------------------------------------------
//  Printf
// --------------------------------------------------------------------------
void DEMOFontPrintf(f32 column, f32 line, const char* pFmt, ... )
{
    char str[FONT_MAX_BUFSIZ];
    va_list args;
    s32 stringSize;

    // Don't draw if fonts are disabled
    if (!gDemoFontCurInstance->enabled)
    {
        return;
    }

    // Get string
    va_start(args, pFmt);
    stringSize = vsnprintf( str, FONT_MAX_BUFSIZ, pFmt, args );

    // Assert for over string size
    if ( stringSize < 0 )
    {
        DEMOAssert(!"String is too long\n");
    }

    va_end(args);

    DEMOFontPuts( column, line, str );
}

static void DEMOFontSetupBuffers( FrameUniBuffer* curdata )
{
    // set up uniform buffer interfaces
    curdata->s_psUniformBuffer.AllocFromPool(s_FrameMemPool, s_psBufferSize, NULL, nn::gfx::GpuAccess_ConstantBuffer, 0);
    curdata->s_vsUniformBuffer.AllocFromPool(s_FrameMemPool, s_vsBufferSize, NULL, nn::gfx::GpuAccess_ConstantBuffer, 0);
    curdata->indexBuffer.AllocFromPool(s_FrameMemPool, MAX_INDICES * sizeof(u32), NULL, nn::gfx::GpuAccess_IndexBuffer, 0);
}

static FrameUniBuffer* DEMOFontGetNextFrameData()
{
    if (!curFrameData)
    {
        curFrameData = firstFrameData = new FrameUniBuffer;
        curFrameData->next = NULL;
        DEMOFontSetupBuffers(curFrameData);
        return curFrameData;
    }
    if (!curFrameData->next)
    {
        curFrameData->next = new FrameUniBuffer;
        curFrameData->next->next = NULL;
        DEMOFontSetupBuffers(curFrameData->next);
    }
    curFrameData = curFrameData->next;
    return curFrameData;
}

static void DEMOFontDrawCharacters( FrameUniBuffer* curdata, FontData_t* pFont, u32* indices, u32 indCount )
{
    if ( indCount > MAX_INDICES )
    {
        indCount = MAX_INDICES;
    }

    u32* indPtr = curdata->indexBuffer.Map<u32>();
    memcpy(indPtr, indices, indCount * sizeof(u32));
    curdata->indexBuffer.Unmap();
    // Draw the characters
    {
#if NN_GFX_IS_TARGET_GX
        GX2SetShaderMode( GX2_SHADER_MODE_UNIFORM_BLOCK );
#endif
        DEMOCommandBuffer.SetPipeline( &gDemoFontCurInstance->pipeline );

        DEMOCommandBuffer.InvalidateMemory( nn::gfx::GpuAccess_Texture | nn::gfx::GpuAccess_IndexBuffer | nn::gfx::GpuAccess_ConstantBuffer | nn::gfx::GpuAccess_VertexBuffer );
        DEMOCommandBuffer.SetConstantBuffer(s_slotPixelUniform, nn::gfx::ShaderStage_Pixel, curdata->s_psUniformBuffer.gpuAddress, curdata->s_psUniformBuffer.size);
        DEMOCommandBuffer.SetConstantBuffer( s_slotVertexUniform, nn::gfx::ShaderStage_Vertex, curdata->s_vsUniformBuffer.gpuAddress, curdata->s_psUniformBuffer.size );
        DEMOCommandBuffer.SetTextureAndSampler(s_slotTexture, nn::gfx::ShaderStage_Pixel, pFont->textureDescriptor, s_samplerDescriptor);

        DEMOCommandBuffer.DrawIndexed(nn::gfx::PrimitiveTopology_TriangleList, nn::gfx::IndexFormat_Uint32, curdata->indexBuffer.gpuAddress, indCount, 0);

    }
}

void DEMOFontPuts(f32 column, f32 line, const char* pStr)
{
    u32 stringLength;
    s32 offsetX;
    s32 offsetY;
    FontData_t* pFont = GetCurrentFont();
    DEMOFontFontData* pFontData = &gDemoFontCurInstance->font;
    f32 basePos[4];
    u32 advanceX = 0;
    u32 *indices;
    u32 indCount;
    u32 cIn = 0;
    FrameUniBuffer *curdata;
    u32 s_index_buffer[MAX_INDICES];

    // Don't draw if fonts are disabled
    if (!gDemoFontCurInstance->enabled)
    {
        return;
    }

    // Check the initialize
    DEMOAssert(pStr &&
        "Need to initialize pStr.\n");
    DEMOAssert(pFont->pCharDataBuffer &&
        "Need to call DEMOFontInit(). Before this function.\n");

    // set up the per-string info and add it to the frame data
    curdata = DEMOFontGetNextFrameData();

    //
    stringLength = (u32)(strlen((const char *)pStr));

    // Calc offsets
    offsetX = (s32)(column * pFont->gridOffsetX);
    offsetY = -(s32)((line + 1) * pFont->gridOffsetY);

    float *pFloatBuffer;

    pFloatBuffer = curdata->s_psUniformBuffer.Map<float>();
    // Update color uniform
    memcpy(pFloatBuffer, &pFontData->color, 4 * sizeof(float));
#ifdef CAFE
    GX2EndianSwap( pFloatBuffer, 4 * sizeof(float) );
#endif
    curdata->s_psUniformBuffer.Unmap();

    basePos[2] = pFontData->depth;

    indices = s_index_buffer;
    while (cIn < stringLength)
    {

        basePos[0] = (f32) offsetX;
        basePos[1] = (f32) offsetY;

        indCount = 0;

        for(; cIn < stringLength; cIn++)
        {

            u32 id = (u32) pStr[cIn];
            u32 index = 0;
            u32 px1, px2, py1, py2;
            u32 ps1, ps2, pt1, pt2;

            if(id == 10)
            {
                advanceX = 0;
                offsetY -= (s32)pFont->gridOffsetY;
                cIn++;
                break;
            }
            else if(id >= 32 && id <= 127)
            {
                // Get index of character id
                index = BSearchIndex(pFont,id);
            }

            px1 = (u32)(pFont->pCharDataBuffer[index].minX + advanceX);
            px2 = (u32)(pFont->pCharDataBuffer[index].maxX + advanceX);
            py1 = (u32)(pFont->pCharDataBuffer[index].minY);
            py2 = (u32)(pFont->pCharDataBuffer[index].maxY);

            // If we exceed the bitfield for advanceX, reset base
            if (px2 > 0xfff)
            {
                offsetX += advanceX;
                advanceX = 0;
                break;
            }

            ps1 = (u32)pFont->pCharDataBuffer[index].minS;
            ps2 = (u32)pFont->pCharDataBuffer[index].maxS;
            pt1 = (u32)pFont->pCharDataBuffer[index].minT;
            pt2 = (u32)pFont->pCharDataBuffer[index].maxT;

#define CHECK(px, py, tx, ty) \
    DEMOAssert(!((px)&~0xfff)&&!((py)&~0x1f)&&!((tx)&~0xff)&&!((ty)&~0x7f))
#define PACK(px, py, tx, ty) (((px)<<20)|((py)<<15)|((tx)<<7)|(ty))

            CHECK(px1, py1, ps1, pt1);
            CHECK(px2, py2, ps2, pt2);

            // need to draw a quad, which is 2 triangles
            indices[indCount++] = PACK(px1, py1, ps1, pt1);
            indices[indCount++] = PACK(px1, py2, ps1, pt2);
            indices[indCount++] = PACK(px2, py2, ps2, pt2);

            indices[indCount++] = PACK(px1, py1, ps1, pt1);
            indices[indCount++] = PACK(px2, py2, ps2, pt2);
            indices[indCount++] = PACK(px2, py1, ps2, pt1);
            ASSERT(indCount <= MAX_INDICES);
            advanceX += pFont->pCharDataBuffer[index].xAdvance;
        }

        // Skip blank lines
        if (indCount == 0)
        {
            continue;
        }

        // Set base position
        pFloatBuffer = curdata->s_vsUniformBuffer.Map< float >();
        // Update base position (first vector in uniform buffer)
        memcpy(pFloatBuffer, &basePos, 4 * sizeof(float));
        // Update scale (next 2 floats)
        pFloatBuffer[4] = pFontData->charMagScale[0];
        pFloatBuffer[5] = pFontData->charMagScale[1];
#ifdef CAFE
        GX2EndianSwap( pFloatBuffer, 6 * sizeof(float) );
#endif
        curdata->s_vsUniformBuffer.Unmap();
        //GX2SetVertexUniformReg(s_Shader.uniformsVS.location[0], 1*4, &basePos);

        DEMOFontDrawCharacters( curdata, pFont, indices, indCount );

    }
}

// --------------------------------------------------------------------------
//  Update
// --------------------------------------------------------------------------
void DEMOFontSetSpacing(BOOL proportional)
{
    // Check the initialize
    DEMOAssert(s_FontP.pCharDataBuffer &&
        "Need to call DEMOFontInit(). Before this function.\n");

    // Update proportional boolean
    gDemoFontCurInstance->proportional = proportional;

    // Switch which scale we use
    if (proportional)
    {
        gDemoFontCurInstance->font.charMagScale = gDemoFontCurInstance->font.charMagScaleP;
    } else {
        gDemoFontCurInstance->font.charMagScale = gDemoFontCurInstance->font.charMagScaleF;
    }
}

void DEMOFontSetGridSize(f32 xGrid, f32 yGrid)
{
    // Check the initialize
    DEMOAssert(s_FontP.pCharDataBuffer &&
        "Need to call DEMOFontInit(). Before this function.\n");

    // Update scale
    UpdateScale(&gDemoFontCurInstance->font, xGrid, yGrid);
}

void DEMOFontSetColor(f32 r, f32 g, f32 b, f32 a)
{
    // Check the initialize
    DEMOAssert(s_FontP.pCharDataBuffer &&
        "Need to call DEMOFontInit(). Before this function.\n");

    // Update color
    gDemoFontCurInstance->font.color[0] = r;
    gDemoFontCurInstance->font.color[1] = g;
    gDemoFontCurInstance->font.color[2] = b;
    gDemoFontCurInstance->font.color[3] = a;
}

void DEMOFontSetZValue(f32 zValue)
{
    // Check the initialize
    DEMOAssert(s_FontP.pCharDataBuffer &&
        "Need to call DEMOFontInit(). Before this function.\n");

    // Update depth value
    gDemoFontCurInstance->font.depth = zValue;
}

// --------------------------------------------------------------------------
//  Getter
// --------------------------------------------------------------------------

void DEMOFontGetCharSize(f32 *pCharWidth, f32 *pCharHeight)
{
    FontData_t* pFont = GetCurrentFont();

    // Get size of font
    *pCharWidth   = pFont->gridOffsetX;
    *pCharHeight  = pFont->gridOffsetY;
}

// --------------------------------------------------------------------------
//  Private Functions
// --------------------------------------------------------------------------
/// Check Proportional boolean and return Font handle
static FontData_t* GetCurrentFont()
{
    if(gDemoFontCurInstance->proportional)
    {
        return &s_FontP;
    }
    else
    {
        return &s_FontF;
    }
}

/// Update scale
static void UpdateScale(DEMOFontFontData* pFont, f32 scaleX, f32 scaleY)
{
    // Calculate char scale
    pFont->charMagScaleF[0] = 2.0f / (scaleX * s_FontF.gridOffsetX);
    pFont->charMagScaleF[1] = 2.0f / (scaleY * s_FontF.gridOffsetY);

    pFont->charMagScaleP[0] = 2.0f / (scaleX * s_FontP.gridOffsetX);
    pFont->charMagScaleP[1] = 2.0f / (scaleY * s_FontP.gridOffsetY);
}

/// Get Font char data from demoFontData.h
static void GetFontCharData(FontData_t* pFont,
                            const u32* pFontHeader,
                            const u32* pCharData,
                            const u8* pFontImageData)
{
    u32 i;
    f32 lineHeight;
    f32 maxCharWidth = 0.0f;
    f32 maxCharHeight = 0.0f;

    // Check Font Texture Data
    DEMOAssert(pFontHeader != NULL && pCharData != NULL &&
        pFontImageData != NULL && "No texture data.\n");

    // Check char data buffer
    if (pFont->pCharDataBuffer)
    {
        // Skip Data Initialization
        return;
    }

    // Set Font Texture Data Information
    pFont->fontTextureWidth  = (f32)pFontHeader[0];
    pFont->fontTextureHeight = (f32)pFontHeader[1];
    lineHeight               = (f32)pFontHeader[4];
    pFont->numCharData       = pFontHeader[5];
    pFont->gridOffsetX       = 0;
    pFont->gridOffsetY       = 0;

    if(pFont == &s_FontP)
    {
        pFont->pCharDataBuffer = s_CharDataBufferP;
    }
    else
    {
        pFont->pCharDataBuffer = s_CharDataBufferF;
    }

    // Check the max number
    DEMOAssert(pFont->numCharData <= MAX_NUM_CHARS &&
        "Font has over the max number of characters.\n");

    // Format of data is: id, x, y, width, height, xOffset, yOffset, xAdvance
    for(i = 0; i < pFont->numCharData; ++i)
    {
        u32 id         = pCharData[i * 8 + 0];
        u32 x          = pCharData[i * 8 + 1];
        u32 y          = pCharData[i * 8 + 2];
        u32 w          = pCharData[i * 8 + 3];
        u32 h          = pCharData[i * 8 + 4];
                         // +3 below is a hack to deal with negative offsets
        s32 xOffset    = (s32)pCharData[i * 8 + 5] + 3;
        s32 yOffset    = (s32)pCharData[i * 8 + 6];
        u32 xAdvance   = pCharData[i * 8 + 7];
        f32 charHeight = (f32)(h + yOffset);

        pFont->pCharDataBuffer[i].id       = id;
        pFont->pCharDataBuffer[i].minS     = (s32)x;
        pFont->pCharDataBuffer[i].minT     = (s32)(pFont->fontTextureHeight - h - y - 1);
        pFont->pCharDataBuffer[i].maxS     = (s32)(x + w);
        pFont->pCharDataBuffer[i].maxT     = (s32)(pFont->fontTextureHeight - y - 1);
        pFont->pCharDataBuffer[i].minX     = (s32)xOffset;
        pFont->pCharDataBuffer[i].minY     = (s32)(lineHeight - yOffset - h);
        pFont->pCharDataBuffer[i].maxX     = (s32)(xOffset + w);
        pFont->pCharDataBuffer[i].maxY     = (s32)(lineHeight - yOffset);
        pFont->pCharDataBuffer[i].xAdvance = (s32)xAdvance;

        // Set max height of char in GL cordinate space
        if(charHeight > maxCharHeight)
        {
            maxCharHeight = charHeight;
        }

        if(pFont->pCharDataBuffer[i].xAdvance > maxCharWidth)
        {
            maxCharWidth = (f32)pFont->pCharDataBuffer[i].xAdvance;
        }
    }

    // Set grid offsetX
    pFont->gridOffsetX = maxCharWidth;

    // Set grid offsetY
    pFont->gridOffsetY = maxCharHeight;

    // Set pointer of Font Image Data
    pFont->pFontImageData = (u8*)pFontImageData;
}

/// Init Font Texture
static void InitFontTexture(FontData_t* pFont)
{
    // Texture setup
    u32 width =(u32)pFont->fontTextureWidth;
    u32 height=(u32)pFont->fontTextureHeight;

    DEMOGfxSetupTextureBuffer( &pFont->texture, &pFont->textureView,
        &pFont->textureDescriptor, NULL, NULL, &pFont->pFontMemPool,
        width, height, 1, 1, nn::gfx::ImageDimension_2d, nn::gfx::ImageFormat_R8_Unorm,
        nn::gfx::DepthStencilFetchMode_DepthComponent, 0 );

    // Create Buffer to copy the data in
    DEMOGfxBuffer myBuffer;
    myBuffer.Initialize( width * height, pFont->pFontImageData, nn::gfx::GpuAccess_Read, 0 );

    nn::gfx::TextureCopyRegion dstRegion;
    dstRegion.SetDefault();
    dstRegion.SetWidth( width );
    dstRegion.SetHeight( height );

    DEMOCommandBuffer.Begin();
    DEMOCommandBuffer.CopyBufferToImage( &pFont->texture, dstRegion, &myBuffer.buffer, 0 );
    DEMOCommandBuffer.End();
    DEMOQueue.ExecuteCommand( &DEMOCommandBuffer, NULL );

    // Set up the sampler object (only needs to be done once)
    if (!s_sampler_init)
    {
        DEMOGfxInitSampler( &s_sampler, &s_samplerDescriptor, nn::gfx::TextureAddressMode_Repeat, nn::gfx::FilterMode_MinLinear_MagLinear_MipPoint, nn::gfx::ComparisonFunction_Always );

        s_sampler_init = true;
    }

    myBuffer.Finalize();
}

/// Initialize Font shader
static void InitShader()
{

    DEMOGfxLoadShadersFromFile( &s_fontShader, 0, FONT_GSH_SHADER_FILE );

    // Init attribute to shader
    s_slotVertexUniform = s_fontShader.GetInterfaceSlot( nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_ConstantBuffer, "Vin" );
    s_slotPixelUniform = s_fontShader.GetInterfaceSlot( nn::gfx::ShaderStage_Pixel, nn::gfx::ShaderInterfaceType_ConstantBuffer, "Color" );
    s_slotTexture = s_fontShader.GetInterfaceSlot( nn::gfx::ShaderStage_Pixel, nn::gfx::ShaderInterfaceType_Sampler, "u_CharTex" );

    s_psBufferSize = 8 * sizeof(float);
    s_vsBufferSize = 8 * sizeof(float);
}

// Search index of character id
static u32 BSearchIndex(FontData_t* pFont, u32 id)
{
    u32 first;
    u32 last;
    u32 index = 0;

    // quick check for sanely-ordered fonts
    if (id >= 32 && id < 32 + pFont->numCharData &&
        pFont->pCharDataBuffer[id - 32].id == id )
    {
        return id - 32;
    }

    first = 0;
    last  = pFont->numCharData - 1;

    while(first <= last)
    {
        u32 mid = first + (last - first) / 2;

        if( pFont->pCharDataBuffer[mid].id < id )
        {
            first = mid + 1;
        }
        else if( id < pFont->pCharDataBuffer[mid].id )
        {
            last = mid - 1;
        }
        else
        {
            index = mid;
            break;
        }
    }
    return index;
}


DEMOFontInstance* DEMOFontAddInstance()
{
    nn::gfx::Pipeline::InfoType info;
    nn::gfx::RasterizerStateInfo rasterizerStateInfo;
    nn::gfx::DepthStencilStateInfo depthStencilStateInfo;
    nn::gfx::RenderTargetStateInfo renderTargetStateInfo;
    nn::gfx::BlendStateInfo blendStateInfo;
    nn::gfx::VertexStateInfo vertexStateInfo;

    info.SetDefault();
    rasterizerStateInfo.SetDefault();
    depthStencilStateInfo.SetDefault();
    renderTargetStateInfo.SetDefault();
    blendStateInfo.SetDefault();
    vertexStateInfo.SetDefault();

    // Setup Shaders
    info.SetShaderPtr( s_fontShader.GetShader() );

    // Setup Rasterizer
    rasterizerStateInfo.SetCullMode( nn::gfx::CullMode_None );
    rasterizerStateInfo.SetScissorEnabled( false );
    rasterizerStateInfo.SetMultisampleEnabled( DEMOColorBufferInfo.GetMultisampleCount() > 1 );
    rasterizerStateInfo.EditMultisampleStateInfo().SetSampleCount( DEMOColorBufferInfo.GetMultisampleCount() );

    // Setup Depth/Stencil State
    depthStencilStateInfo.SetDepthTestEnabled( false );
    depthStencilStateInfo.SetDepthWriteEnabled( false );

    // Setup Render Target State
    nn::gfx::ColorTargetStateInfo colorTargets[ 1 ];
    colorTargets[ 0 ].SetDefault();
    colorTargets[ 0 ].SetFormat( DEMOColorBufferInfo.GetImageFormat() );
    renderTargetStateInfo.SetColorTargetStateInfoArray( colorTargets, 1 );
    renderTargetStateInfo.SetDepthStencilFormat( DEMODepthBufferInfo.GetImageFormat() );

    // Setup Blend State
    nn::gfx::BlendTargetStateInfo blendTargetInfo[1];
    {
        blendTargetInfo[0].SetDefault();
        blendTargetInfo[0].SetBlendEnabled( true );
        blendTargetInfo[0].SetColorBlendFunction( nn::gfx::BlendFunction_Add );
        blendTargetInfo[0].SetSourceColorBlendFactor( nn::gfx::BlendFactor_SourceAlpha );
        blendTargetInfo[0].SetDestinationColorBlendFactor( nn::gfx::BlendFactor_OneMinusSourceAlpha );
        blendTargetInfo[0].SetAlphaBlendFunction( nn::gfx::BlendFunction_Add );
        blendTargetInfo[0].SetSourceAlphaBlendFactor( nn::gfx::BlendFactor_SourceAlpha );
        blendTargetInfo[0].SetDestinationAlphaBlendFactor( nn::gfx::BlendFactor_OneMinusSourceAlpha );
    }
    blendStateInfo.SetBlendTargetStateInfoArray( blendTargetInfo, 1 );

    info.SetRasterizerStateInfo(&rasterizerStateInfo);
    info.SetDepthStencilStateInfo(&depthStencilStateInfo);
    info.SetRenderTargetStateInfo(&renderTargetStateInfo);
    info.SetBlendStateInfo(&blendStateInfo);
    info.SetVertexStateInfo(&vertexStateInfo);

    size_t dataSize = nn::gfx::Pipeline::GetRequiredMemorySize( info );
    uint8_t *data = NULL;
    if ( dataSize )
    {
        data = ( uint8_t * )DEMOGfxAllocMEM2( dataSize, nn::gfx::Pipeline::RequiredMemoryInfo_Alignment );
    }
    // NOTE: Vertex Attribute State is not necessary for this shader.

/////////////////////////////////////////////////////////////////////////////
    // Create the new instance
    gDemoFontCurInstance = (DEMOFontInstance*) DEMOAlloc(sizeof(DEMOFontInstance));
    memset(gDemoFontCurInstance, 0, sizeof(DEMOFontInstance));

    gDemoFontCurInstance->pipeline.SetMemory( data, dataSize );
    gDemoFontCurInstance->pipeline.Initialize(&DEMODevice, info);

/////////////////////////////////////////////////////////////////////////////

    // Initialize Proportional Font
    DEMOFontSetSpacing(1);
    DEMOFontSetGridSize(60, 24);
    DEMOFontSetColor(1.0f, 1.0f, 1.0f, 1.0f);
    DEMOFontSetZValue(-1.0f);

    gDemoFontCurInstance->enabled = s_enabled;
    s_enabled = TRUE;

    return gDemoFontCurInstance;
}

void DEMOFontDeleteInstance(DEMOFontInstance *instance)
{
    void* pipelineData = instance->pipeline.GetMemory();
    instance->pipeline.Finalize( &DEMODevice );
    DEMOGfxFreeMEM2( pipelineData );

    DEMOFree(instance);

    // Edge case:  delete current
    if (gDemoFontCurInstance == instance)
    {
        gDemoFontCurInstance = NULL;
    }
}

