﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "testG3d_ViewerTestAllocator.h"

#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>

#include "nnt/g3d/testG3d_TestUtility.h"

#if defined(NN_SDK_BUILD_DEBUG)
#define NN_G3D_VIEWER_TSET_DEBUG_LOG( ... ) NN_LOG("[testG3d_Viewer]: "); NN_LOG(__VA_ARGS__)
#else
#define NN_G3D_VIEWER_TSET_DEBUG_LOG( ... ) static_cast<void>(0)
#endif

class G3dTest::ViewerTestAllocator::Buffer
{
public:
    Buffer(void* pBuffer, uintptr_t addressOffset, size_t bufferSize)
        : m_pBuffer(pBuffer)
        , m_AddressOffset(addressOffset)
        , m_BufferSize(bufferSize)
    {
    }

    void* GetBuffer()
    {
        return m_pBuffer;
    }

    uintptr_t GetAddressOffset() const
    {
        return m_AddressOffset;
    }

    size_t GetBufferSize() const
    {
        return m_BufferSize;
    }

private:
    void* m_pBuffer;
    uintptr_t m_AddressOffset;
    size_t m_BufferSize;
};

G3dTest::ViewerTestAllocator::Buffer* G3dTest::ViewerTestAllocator::EraseBuffer(void* pOriginalBufferData)
{
    for (std::vector<Buffer*>::iterator i = m_Buffers.begin(), end = m_Buffers.end(); i != end; ++i)
    {
        G3dTest::ViewerTestAllocator::Buffer* pTargetBuffer = *i;
        void* pTargetBufferData = pTargetBuffer->GetBuffer();
        if (pOriginalBufferData == pTargetBufferData)
        {
            m_Buffers.erase(i);
            return pTargetBuffer;
        }
    }

    return NULL;
}

G3dTest::ViewerTestAllocator::ViewerTestAllocator()
{
}

void* G3dTest::ViewerTestAllocator::Allocate(size_t size)
{
    return nnt::g3d::Allocate(size);
}

void G3dTest::ViewerTestAllocator::Deallocate(void* ptr, size_t size)
{
    NN_UNUSED(size);
    nnt::g3d::Deallocate(ptr);
}

void* G3dTest::ViewerTestAllocator::AllocateWithUserData(size_t size, size_t alignment, void* pUserData)
{
    G3dTest::ViewerTestAllocator* pAllocator = static_cast<G3dTest::ViewerTestAllocator*>(pUserData);
    return pAllocator->Allocate(size, alignment);
}

void G3dTest::ViewerTestAllocator::FreeWithUserData(void* ptr, void* pUserData)
{
    G3dTest::ViewerTestAllocator* pAllocator = static_cast<G3dTest::ViewerTestAllocator*>(pUserData);
    pAllocator->Free(ptr);
}

void* G3dTest::ViewerTestAllocator::Allocate(size_t size, uintptr_t alignment)
{
    size_t allocSize = (size / alignment + 2) * alignment;

    void* pOriginalBufferData = Allocate(allocSize);
    uintptr_t originalAddress = reinterpret_cast<uintptr_t>(pOriginalBufferData);
    uintptr_t dividedAdress = originalAddress / alignment;
    uintptr_t alignedAddress = (dividedAdress + 1) * alignment;
    void* pAlignedBufferData = reinterpret_cast<void*>(alignedAddress);

    NN_ASSERT(pOriginalBufferData <= pAlignedBufferData);
    NN_ASSERT((alignedAddress + (uintptr_t)size) <= (originalAddress + (uintptr_t)allocSize));

    Buffer* pBuffer = new Buffer(pAlignedBufferData, (ptrdiff_t)alignedAddress - (ptrdiff_t)originalAddress, size);
    m_Buffers.push_back(pBuffer);

    NN_G3D_VIEWER_TSET_DEBUG_LOG(
        "Alloc: size = %zu, alignment = %zu, original address = 0x%08x, aligned address = 0x%08x\n",
        size, alignment, originalAddress, alignedAddress);
    return pAlignedBufferData;
}

void G3dTest::ViewerTestAllocator::Free(void* pBufferData)
{
    Buffer* pTargetBuffer = EraseBuffer(pBufferData);
    NN_ASSERT_NOT_NULL(pTargetBuffer);

    void* pAlignedBufferData = pTargetBuffer->GetBuffer();
    uintptr_t alignedAddress = reinterpret_cast<uintptr_t>(pAlignedBufferData);
    ptrdiff_t originalAddress = (ptrdiff_t)alignedAddress - (ptrdiff_t)pTargetBuffer->GetAddressOffset();
    void* pOriginalBufferData = reinterpret_cast<void*>(originalAddress);

    NN_G3D_VIEWER_TSET_DEBUG_LOG("Free: original address = 0x%08x, aligned address = 0x%08x\n", originalAddress, alignedAddress);

    delete pTargetBuffer;
    Deallocate(pOriginalBufferData, 0 /* 使用されない */ );
}
