﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "testG3d_Main.h"
#include <nn/fs.h>
#include <nn/os.h>
#if defined(NN_BUILD_CONFIG_OS_WIN)
#ifndef WIN32_LEAN_AND_MEAN
#define WIN32_LEAN_AND_MEAN
#endif
#ifndef NOMINMAX
#define NOMINMAX
#endif
#include <nn/nn_Windows.h>
#elif defined( NN_BUILD_CONFIG_OS_HORIZON )
#include <nv/nv_MemoryManagement.h>
#endif

#include <nnt/base/testBase_Exit.h>
#include "nnt/g3d/testG3d_TestUtility.h"
#include <nn/gfx/gfx_ResShaderData-api.nvn.h>
#include <nn/nn_Log.h>

namespace
{
nn::gfx::Device g_Device;

#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON )

void* NvAllocate(size_t size, size_t alignment, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    return aligned_alloc(alignment, size);
}

void NvFree(void* addr, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    free(addr);
}

void* NvReallocate(void* addr, size_t newSize, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    return realloc(addr, newSize);
}
#endif
}

// ResFileのアライメントはBinaryFileHeaderから取得
size_t GetFileAlignment(const char* path)
{
    nn::fs::FileHandle fileHandle;
    nn::Result result = nn::fs::OpenFile(&fileHandle, path, nn::fs::OpenMode_Read);
    NN_ASSERT(result.IsSuccess());
    void* buffer;
    buffer = nnt::g3d::AlignedAllocate<void>(sizeof(nn::util::BinaryFileHeader), 8);
    size_t readSize;
    result = nn::fs::ReadFile(&readSize, fileHandle, 0, buffer, sizeof(nn::util::BinaryFileHeader));
    NN_ASSERT(result.IsSuccess());
    nn::fs::CloseFile(fileHandle);
    size_t alignment = reinterpret_cast<nn::util::BinaryFileHeader*>(buffer)->GetAlignment();
    nnt::g3d::Deallocate(buffer);
    return alignment;
}

size_t GetFileSize(const char* path)
{
    nn::fs::FileHandle fileHandle;
    nn::Result result = nn::fs::OpenFile(&fileHandle, path, nn::fs::OpenMode_Read);
    NN_ASSERT(result.IsSuccess());
    void* buffer;
    buffer = nnt::g3d::AlignedAllocate<void>(sizeof(nn::util::BinaryFileHeader), 8);
    size_t readSize;
    result = nn::fs::ReadFile(&readSize, fileHandle, 0, buffer, sizeof(nn::util::BinaryFileHeader));
    NN_ASSERT(result.IsSuccess());
    nn::fs::CloseFile(fileHandle);
    size_t size = reinterpret_cast<nn::util::BinaryFileHeader*>(buffer)->GetFileSize();
    nnt::g3d::Deallocate(buffer);
    return size;
}

void CompareFsd(const char* fsdPath, const char* referenceFsdPath)
{
    size_t size;
    void* pFile = nnt::g3d::LoadFile(&size, fsdPath, 4);
    void* pReferenceFile = nnt::g3d::LoadFile(&size, referenceFsdPath, 4);

    EXPECT_EQ(memcmp(pFile, pReferenceFile, size), 0);

    nnt::g3d::UnloadFile(pFile);
    nnt::g3d::UnloadFile(pReferenceFile);
}

void ResetCodeHash(const nn::gfx::NvnShaderCode* pNvnShaderCode)
{
    // pNvnShaderCode->pControl の内容を破棄する
    if (pNvnShaderCode)
    {
        nn::util::BytePtr pControl(const_cast<void*>(pNvnShaderCode->pControl.Get()));
        memset(pControl.Get(), 0, pNvnShaderCode->controlSize);
    }
}

void Compare(const char* bfshaPath, const char* referenceBfshaPath, bool matchingFull = false)
{
    size_t size;
    void* pFile = nnt::g3d::LoadFile(&size, bfshaPath, GetFileAlignment(bfshaPath));
    void* pReferenceFile = nnt::g3d::LoadFile(&size, referenceBfshaPath, GetFileAlignment(referenceBfshaPath));

    // 手動デバッグ用に全一致かどうかを判定。glslc 等でバイナリーが変わるので自動テストではOFFにしておく。
    if (matchingFull)
    {
#if NN_GFX_IS_TARGET_NVN
        // merge では codeHash が一致しない場合があるため、codeHash 値をもつ pControl を丸ごと 0 で埋めておく
        nn::g3d::ResShaderFile* pResShaderFile = nn::g3d::ResShaderFile::ResCast(pFile);
        nn::g3d::ResShaderFile* pReferenceResShaderFile = nn::g3d::ResShaderFile::ResCast(pReferenceFile);
        for (int shadingModelIndex = 0; shadingModelIndex < pResShaderFile->GetResShaderArchive()->GetShadingModelCount(); shadingModelIndex++)
        {
            nn::g3d::ResShadingModel* pResShadingModel = pResShaderFile->GetResShaderArchive()->GetShadingModel(shadingModelIndex);
            nn::g3d::ResShadingModel* pReferenceResShadingModel = pReferenceResShaderFile->GetResShaderArchive()->GetShadingModel(shadingModelIndex);
            nn::gfx::ResShaderFile* pGfxResShaderFile = const_cast<nn::gfx::ResShaderFile*>(pResShadingModel->GetGfxResShaderFile());
            nn::gfx::ResShaderFile* pGfxReferenceResShaderFile = const_cast<nn::gfx::ResShaderFile*>(pReferenceResShadingModel->GetGfxResShaderFile());
            for (int shaderVariationIndex = 0; shaderVariationIndex < pGfxResShaderFile->GetShaderContainer()->GetShaderVariationCount(); shaderVariationIndex++)
            {
                nn::gfx::ResShaderVariation* pResShaderVaritaion = pGfxResShaderFile->GetShaderContainer()->GetResShaderVariation(shaderVariationIndex);
                const nn::gfx::ShaderInfoData& infoData = pResShaderVaritaion->ToData().pBinaryProgram.Get()->info;
                ResetCodeHash(static_cast<const nn::gfx::NvnShaderCode*>(infoData.pVertexShaderCode.ptr));
                ResetCodeHash(static_cast<const nn::gfx::NvnShaderCode*>(infoData.pPixelShaderCode.ptr));
                ResetCodeHash(static_cast<const nn::gfx::NvnShaderCode*>(infoData.pGeometryShaderCode.ptr));
                ResetCodeHash(static_cast<const nn::gfx::NvnShaderCode*>(infoData.pHullShaderCode.ptr));
                ResetCodeHash(static_cast<const nn::gfx::NvnShaderCode*>(infoData.pDomainShaderCode.ptr));
                ResetCodeHash(static_cast<const nn::gfx::NvnShaderCode*>(infoData.pComputeShaderCode.ptr));

                const nn::gfx::ResShaderVariation* pReferenceResShaderVaritaion = pGfxReferenceResShaderFile->GetShaderContainer()->GetResShaderVariation(shaderVariationIndex);
                const nn::gfx::ShaderInfoData& referenceInfoData = pReferenceResShaderVaritaion->ToData().pBinaryProgram.Get()->info;
                ResetCodeHash(static_cast<const nn::gfx::NvnShaderCode*>(referenceInfoData.pVertexShaderCode.ptr));
                ResetCodeHash(static_cast<const nn::gfx::NvnShaderCode*>(referenceInfoData.pPixelShaderCode.ptr));
                ResetCodeHash(static_cast<const nn::gfx::NvnShaderCode*>(referenceInfoData.pGeometryShaderCode.ptr));
                ResetCodeHash(static_cast<const nn::gfx::NvnShaderCode*>(referenceInfoData.pHullShaderCode.ptr));
                ResetCodeHash(static_cast<const nn::gfx::NvnShaderCode*>(referenceInfoData.pDomainShaderCode.ptr));
                ResetCodeHash(static_cast<const nn::gfx::NvnShaderCode*>(referenceInfoData.pComputeShaderCode.ptr));
            }
        }
        pResShaderFile->Unrelocate();
        pReferenceResShaderFile->Unrelocate();
#endif
        EXPECT_EQ(memcmp(pFile, pReferenceFile, GetFileSize(bfshaPath)), 0);
    }

    nn::g3d::ResShaderFile* pResShaderFile = nn::g3d::ResShaderFile::ResCast(pFile);
    nn::g3d::ResShaderFile* pReferenceResShaderFile = nn::g3d::ResShaderFile::ResCast(pReferenceFile);
    CheckResShaderFileDataEqual(pResShaderFile->ToData(), pReferenceResShaderFile->ToData());
    pResShaderFile->Unrelocate();
    pReferenceResShaderFile->Unrelocate();

    nnt::g3d::UnloadFile(pFile);
    nnt::g3d::UnloadFile(pReferenceFile);
}

TEST_F(G3dTest, DemoFsd)
{
    CompareFsd("Resource:/demo.fsdb", "Resource:/ReferenceDemo.fsdb");
}

TEST_F(G3dTest, TownFsd)
{
    CompareFsd("Resource:/town.fsdb", "Resource:/ReferenceTown.fsdb");
}

// 各バイナリーに対して以下をチェックする
// ・事前ビルドした正解データと比較
// ・--merge-shader-archive-file の入力として得られたバイナリーと比較
TEST_F(G3dTest, DemoBasic)
{
    Compare("Resource:/Demo.bfsha", "Resource:/ReferenceDemo.bfsha");
    Compare("Resource:/Demo.bfsha", "Resource:/MergeDemo.bfsha", true);
}

TEST_F(G3dTest, DemoVariationRatio)
{
    Compare("Resource:/DemoVariationRatio.bfsha", "Resource:/ReferenceDemoVariationRatio.bfsha");
    Compare("Resource:/DemoVariationRatio.bfsha", "Resource:/MergeDemoVariationRatio.bfsha", true);
}

TEST_F(G3dTest, DemoForceVariation)
{
    Compare("Resource:/DemoForceVariation.bfsha", "Resource:/ReferenceDemoForceVariation.bfsha");
    Compare("Resource:/DemoForceVariation.bfsha", "Resource:/MergeDemoForceVariation.bfsha", true);
}

TEST_F(G3dTest, DemoPreprocessorVariation)
{
    Compare("Resource:/DemoPreprocessorVariation.bfsha", "Resource:/ReferenceDemoPreprocessorVariation.bfsha");
    Compare("Resource:/DemoPreprocessorVariation.bfsha", "Resource:/MergeDemoPreprocessorVariation.bfsha", true);
}

TEST_F(G3dTest, TownBasic)
{
    Compare("Resource:/Town.bfsha", "Resource:/ReferenceTown.bfsha");
    Compare("Resource:/Town.bfsha", "Resource:/MergeTown.bfsha", true);
}

TEST_F(G3dTest, TownVariationRatio)
{
    Compare("Resource:/TownVariationRatio.bfsha", "Resource:/ReferenceTownVariationRatio.bfsha");
    Compare("Resource:/TownVariationRatio.bfsha", "Resource:/MergeTownVariationRatio.bfsha", true);
}

TEST_F(G3dTest, TownFsva)
{
    Compare("Resource:/TownFsva.bfsha", "Resource:/ReferenceTownFsva.bfsha");
    Compare("Resource:/TownFsva.bfsha", "Resource:/MergeTownFsva.bfsha", true);
}

TEST_F(G3dTest, TownPreprocessorVariation)
{
    Compare("Resource:/TownPreprocessorVariation.bfsha", "Resource:/ReferenceTownPreprocessorVariation.bfsha");
    Compare("Resource:/TownPreprocessorVariation.bfsha", "Resource:/MergeTownPreprocessorVariation.bfsha", true);
}

// --merge-shader-archive-file のテスト
// 分割コンパイルした複数バイナリーを一つにマージして、分割なしのバイナリーと一致するかをチェック
TEST_F(G3dTest, demoMergeABC)
{
    Compare("Resource:/Merge/DemoMergeABC.bfsha", "Resource:/Merge/DemoMergeAnswer.bfsha", true);
}

TEST_F(G3dTest, townMergeABC)
{
    Compare("Resource:/Merge/TownMergeABCD.bfsha", "Resource:/Merge/TownMergeAnswer.bfsha", true);
}

// 複数の fsv を使用したバイナリーをマージして、全入りの fsv を使用したバイナリーと一致するかをチェック
// bfsha 間でシェーダープログラムが被っているため(gfx の shaderVariation の数に差分が出るので) g3d の構造チェックだけ行う
TEST_F(G3dTest, townMergeFsvaABC)
{
    Compare("Resource:/Merge/TownMergeFsvaABC.bfsha", "Resource:/Merge/TownMergeFsvaAnswer.bfsha");
}

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON )
    const size_t GraphicsSystemMemorySize = 8 * 1024 * 1024;
    void* pGraphicMemory = malloc(GraphicsSystemMemorySize);
    nv::SetGraphicsAllocator(NvAllocate, NvFree, NvReallocate, NULL);
    nv::SetGraphicsDevtoolsAllocator(NvAllocate, NvFree, NvReallocate, NULL);
    nv::InitializeGraphics(pGraphicMemory, GraphicsSystemMemorySize);
#endif

    nnt::g3d::InitializeG3dTest(g_Device);

    int result = RUN_ALL_TESTS();

    nnt::g3d::FinalizeG3dTest(g_Device);
    nnt::Exit(result);
}
