﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include "testG3d_Main.h"

void CheckResDicDataEqual(const nn::util::ResDicData& data, const nn::util::ResDicData& referenceData)
{
    // 各値
    EXPECT_EQ(data.count, referenceData.count);
    for (int i = 0; i < data.count; ++i)
    {
        EXPECT_EQ(data.entries[1 + i].refBit, referenceData.entries[1 + i].refBit);
        EXPECT_EQ(data.entries[1 + i].children[0], referenceData.entries[1 + i].children[0]);
        EXPECT_EQ(data.entries[1 + i].children[1], referenceData.entries[1 + i].children[1]);
        EXPECT_EQ(std::strcmp(data.entries[1 + i].pKey.Get()->GetData(), referenceData.entries[1 + i].pKey.Get()->GetData()), 0);
    }
}

void CheckResShaderOptionDataEqual(const nn::g3d::ResShaderOptionData& data, const nn::g3d::ResShaderOptionData& referenceData)
{
    // 構造体サイズ
    EXPECT_EQ(sizeof(data), 40);

    // ポインタ
    EXPECT_EQ(std::strcmp(data.pName.Get()->GetData(), referenceData.pName.Get()->GetData()), 0);
    CheckResDicDataEqual(data.pChoiceDic.Get()->ToData(), referenceData.pChoiceDic.Get()->ToData());

    if (data.flag & nn::g3d::ResShaderOption::Flag_Branch && data.flag & nn::g3d::ResShaderOption::Flag_Static)
    {
        for (int i = 0; i < data.choiceCount; ++i)
        {
            EXPECT_EQ(data.pChoiceValues.Get()[i], referenceData.pChoiceValues.Get()[i]);
        }
    }

    // 各値
    EXPECT_EQ(data.choiceCount, referenceData.choiceCount);
    EXPECT_EQ(data.defaultIndex, referenceData.defaultIndex);
    EXPECT_EQ(data.branchOffset, referenceData.branchOffset);
    EXPECT_EQ(data.flag, referenceData.flag);
    EXPECT_EQ(data.keyOffset, referenceData.keyOffset);
    EXPECT_EQ(data.bit32Index, referenceData.bit32Index);
    EXPECT_EQ(data.bit32Shift, referenceData.bit32Shift);
    EXPECT_EQ(data.bit32Mask, referenceData.bit32Mask);

    // 予約領域
    EXPECT_EQ(sizeof(data.reserved), 4);
}

void CheckResAttrVarDataEqual(const nn::g3d::ResAttrVarData& data, const nn::g3d::ResAttrVarData& referenceData)
{
    // 構造体サイズ
    EXPECT_EQ(sizeof(data), 2);

    // 各値
    EXPECT_EQ(data.index, referenceData.index);
    EXPECT_EQ(data.location, referenceData.location);
}

void CheckResSamplerVarDataEqual(const nn::g3d::ResSamplerVarData& data, const nn::g3d::ResSamplerVarData& referenceData)
{
    // 構造体サイズ
    EXPECT_EQ(sizeof(data), 16);

    // ポインタ
    EXPECT_EQ(std::strcmp(data.pAlt.Get()->GetData(), referenceData.pAlt.Get()->GetData()), 0);

    // 各値
    EXPECT_EQ(data.index, referenceData.index);

    // 予約領域
    EXPECT_EQ(sizeof(data.reserved), 7);
}

void CheckResUniformVarDataEqual(const nn::g3d::ResUniformVarData& data, const nn::g3d::ResUniformVarData& referenceData)
{
    // 構造体サイズ
    EXPECT_EQ(sizeof(data), 16);

    // ポインタ
    if (data.pConverterName.Get())
    {
        EXPECT_EQ(std::strcmp(data.pConverterName.Get()->GetData(), referenceData.pConverterName.Get()->GetData()), 0);
    }

    // 各値
    EXPECT_EQ(data.index, referenceData.index);
    EXPECT_EQ(data.offset, referenceData.offset);
    EXPECT_EQ(data.blockIndex, referenceData.blockIndex);

    // 予約領域
    EXPECT_EQ(sizeof(data.reserved), 1);
}

void CheckResUniformBlockVarDataEqual(const nn::g3d::ResUniformBlockVarData& data, const nn::g3d::ResUniformBlockVarData& referenceData)
{
    // 構造体サイズ
    EXPECT_EQ(sizeof(data), 32);

    // ポインタ
    for (int i = 0; i < data.uniformCount; ++i)
    {
        CheckResUniformVarDataEqual(data.pUniformArray.Get()[i].ToData(), referenceData.pUniformArray.Get()[i].ToData());
    }
    if (data.uniformCount > 0)
    {
        CheckResDicDataEqual(data.pUniformDic.Get()->ToData(), referenceData.pUniformDic.Get()->ToData());
    }
    if (data.type == nn::g3d::ResUniformBlockVar::Type_Material)
    {
        //ユニフォームブロックのデフォルト値。Type_Material の場合のみ値が存在する。
        EXPECT_EQ(memcmp(data.pDefault.Get(), referenceData.pDefault.Get(), data.size), 0);
    }
    else
    {
        EXPECT_FALSE(data.pDefault.Get());
    }

    // 各値
    EXPECT_EQ(data.index, referenceData.index);
    EXPECT_EQ(data.type, referenceData.type);
    EXPECT_EQ(data.size, referenceData.size);
    EXPECT_EQ(data.uniformCount, referenceData.uniformCount);

    // 予約領域
    EXPECT_EQ(sizeof(data.reserved), 2);
}

void CheckResShaderStorageBlockVarDataEqual(const nn::g3d::ResShaderStorageBlockVarData& data, const nn::g3d::ResShaderStorageBlockVarData& referenceData)
{
    // 構造体サイズ
    EXPECT_EQ(sizeof(data), 32);

    // ポインタ
    for (int i = 0; i < data.shaderStorageCount; ++i)
    {
        CheckResUniformVarDataEqual(data.pShaderStorageArray.Get()[i].ToData(), referenceData.pShaderStorageArray.Get()[i].ToData());
    }
    if (data.shaderStorageCount > 0)
    {
        CheckResDicDataEqual(data.pShaderStorageDic.Get()->ToData(), referenceData.pShaderStorageDic.Get()->ToData());
    }
    // ssbo は type=material は存在しないので常に NULL
    EXPECT_FALSE(data.pDefault.Get());

    // 各値
    EXPECT_EQ(data.index, referenceData.index);
    EXPECT_EQ(data.type, referenceData.type);
    EXPECT_EQ(data.size, referenceData.size);
    EXPECT_EQ(data.shaderStorageCount, referenceData.shaderStorageCount);

    // 予約領域
    EXPECT_EQ(sizeof(data.reserved), 2);
}

void CheckResShaderProgramDataEqual(const nn::g3d::ResShaderProgramData& data, const nn::g3d::ResShaderProgramData& referenceData)
{
    // 構造体サイズ
    EXPECT_EQ(sizeof(data), 56);

    // ポインタ
    for (int i = 0; i < data.samplerCount; ++i)
    {
        EXPECT_EQ(data.pSamplerTable.Get()[i], referenceData.pSamplerTable.Get()[i]);
    }
    for (int i = 0; i < data.uniformBlockCount; ++i)
    {
        EXPECT_EQ(data.pUniformBlockTable.Get()[i], referenceData.pUniformBlockTable.Get()[i]);
    }
    for (int i = 0; i < data.shaderStorageBlockCount; ++i)
    {
        EXPECT_EQ(data.pShaderStorageBlockTable.Get()[i], referenceData.pShaderStorageBlockTable.Get()[i]);
    }
    // nn::gfx::ResShaderVariation は NULL でないことのみを確認する
    EXPECT_TRUE(data.pShader.Get() != NULL);
    EXPECT_TRUE(data.pShadingModel.Get() != NULL);

    // 各値
    EXPECT_EQ(data.attribActiveFlag, referenceData.attribActiveFlag);
    EXPECT_EQ(data.flag, referenceData.flag);
    EXPECT_EQ(data.samplerCount, referenceData.samplerCount);
    EXPECT_EQ(data.uniformBlockCount, referenceData.uniformBlockCount);
    EXPECT_EQ(data.shaderStorageBlockCount, referenceData.shaderStorageBlockCount);

    // 予約領域
    EXPECT_EQ(sizeof(data.reserved), 4);
}

void CheckResShadingModelDataEqual(const nn::g3d::ResShadingModelData& data, const nn::g3d::ResShadingModelData& referenceData)
{
    // 構造体サイズ
    EXPECT_EQ(sizeof(data), 232);

    // ポインタ
    EXPECT_EQ(std::strcmp(data.pName.Get()->GetData(), referenceData.pName.Get()->GetData()), 0);
    for (int i = 0; i < data.staticOptionCount; ++i)
    {
        CheckResShaderOptionDataEqual(data.pStaticOptionArray.Get()[i].ToData(), referenceData.pStaticOptionArray.Get()[i].ToData());
    }
    if (data.staticOptionCount > 0)
    {
        CheckResDicDataEqual(data.pStaticOptionDic.Get()->ToData(), referenceData.pStaticOptionDic.Get()->ToData());
    }
    for (int i = 0; i < data.dynamicOptionCount; ++i)
    {
        CheckResShaderOptionDataEqual(data.pDynamicOptionArray.Get()[i].ToData(), referenceData.pDynamicOptionArray.Get()[i].ToData());
    }
    if (data.dynamicOptionCount > 0)
    {
        CheckResDicDataEqual(data.pDynamicOptionDic.Get()->ToData(), referenceData.pDynamicOptionDic.Get()->ToData());
    }
    for (int i = 0; i < data.attrCount; ++i)
    {
        CheckResAttrVarDataEqual(data.pAttrArray.Get()[i].ToData(), referenceData.pAttrArray.Get()[i].ToData());
    }
    if (data.attrCount > 0)
    {
        CheckResDicDataEqual(data.pAttrDic.Get()->ToData(), referenceData.pAttrDic.Get()->ToData());
    }
    for (int i = 0; i < data.samplerCount; ++i)
    {
        CheckResSamplerVarDataEqual(data.pSamplerArray.Get()[i].ToData(), referenceData.pSamplerArray.Get()[i].ToData());
    }
    if (data.samplerCount > 0)
    {
        CheckResDicDataEqual(data.pSamplerDic.Get()->ToData(), referenceData.pSamplerDic.Get()->ToData());
    }
    for (int i = 0; i < data.uniformBlockCount; ++i)
    {
        CheckResUniformBlockVarDataEqual(data.pUniformBlockArray.Get()[i].ToData(), referenceData.pUniformBlockArray.Get()[i].ToData());
    }
    if (data.uniformBlockCount > 0)
    {
        CheckResDicDataEqual(data.pUniformBlockDic.Get()->ToData(), referenceData.pUniformBlockDic.Get()->ToData());
    }
    for (int i = 0; i < data.uniformCount; ++i)
    {
        CheckResUniformVarDataEqual(data.pUniformArray.Get()[i].ToData(), referenceData.pUniformArray.Get()[i].ToData());
    }
    for (int i = 0; i < data.shaderStorageBlockCount; ++i)
    {
        CheckResShaderStorageBlockVarDataEqual(data.pShaderStorageBlockArray.Get()[i].ToData(), referenceData.pShaderStorageBlockArray.Get()[i].ToData());
    }
    if (data.shaderStorageBlockCount > 0)
    {
        CheckResDicDataEqual(data.pShaderStorageBlockDic.Get()->ToData(), referenceData.pShaderStorageBlockDic.Get()->ToData());
    }
    for (int i = 0; i < data.shaderStorageCount; ++i)
    {
        CheckResUniformVarDataEqual(data.pShaderStorageArray.Get()[i].ToData(), referenceData.pShaderStorageArray.Get()[i].ToData());
    }
    for (int i = 0; i < data.shaderProgramCount; ++i)
    {
        CheckResShaderProgramDataEqual(data.pShaderProgramArray.Get()[i].ToData(), referenceData.pShaderProgramArray.Get()[i].ToData());
    }

    // keyTable の中身は [statickey,dynamickey] * programCount
    int keyTableSize = sizeof(nn::Bit32) * (data.staticKeyLength + data.dynamicKeyLength) * data.shaderProgramCount;
    EXPECT_EQ(memcmp(data.pKeyTable.Get(), referenceData.pKeyTable.Get(), keyTableSize), 0);

    EXPECT_TRUE(data.pShaderArchive.Get() != NULL);
    {
        const nn::g3d::ResShaderInfo* infoData = data.pShaderInfo.Get();
        const nn::g3d::ResShaderInfo* referenveInfoData = referenceData.pShaderInfo.Get();
        const nn::util::BinPtrToString* pSamplerSymbol = infoData->pSamplerTable.Get();
        const nn::util::BinPtrToString* pRSamplerSymbol = referenveInfoData->pSamplerTable.Get();
        for (int idxSampler = 0, samplerCount = data.samplerCount;idxSampler < samplerCount; ++idxSampler)
        {
            for (int stage = 0; stage < nn::g3d::Stage_End; ++stage)
            {
                EXPECT_EQ(std::strcmp(pSamplerSymbol->Get()->GetData(), pRSamplerSymbol->Get()->GetData()), 0);
                ++pSamplerSymbol;
                ++pRSamplerSymbol;
            }
        }
        const nn::util::BinPtrToString* pUniformBlockSymbol = infoData->pUniformBlockTable.Get();
        const nn::util::BinPtrToString* pRUniformBlockSymbol = referenveInfoData->pUniformBlockTable.Get();
        for (int idxBlock = 0, blockCount = data.uniformBlockCount;idxBlock < blockCount; ++idxBlock)
        {
            for (int stage = 0; stage < nn::g3d::Stage_End; ++stage)
            {
                EXPECT_EQ(std::strcmp(pUniformBlockSymbol->Get()->GetData(), pRUniformBlockSymbol->Get()->GetData()), 0);
                ++pUniformBlockSymbol;
                ++pRUniformBlockSymbol;
            }
        }
        const nn::util::BinPtrToString* pShaderStorageBlockSymbol = infoData->pShaderStorageBlockTable.Get();
        const nn::util::BinPtrToString* pRShaderStorageBlockSymbol = referenveInfoData->pShaderStorageBlockTable.Get();
        for (int idxBlock = 0, blockCount = data.shaderStorageBlockCount;idxBlock < blockCount; ++idxBlock)
        {
            for (int stage = 0; stage < nn::g3d::Stage_End; ++stage)
            {
                EXPECT_EQ(std::strcmp(pShaderStorageBlockSymbol->Get()->GetData(), pRShaderStorageBlockSymbol->Get()->GetData()), 0);
                ++pShaderStorageBlockSymbol;
                ++pRShaderStorageBlockSymbol;
            }
        }
    }
    EXPECT_TRUE(data.pShaderFile.Get() != NULL);
    EXPECT_FALSE(data.pMutexTypePtr.Get());
    EXPECT_FALSE(data.pUserPtr.Get());
    EXPECT_FALSE(data.pCallbackParam.Get());

    // 各値
    EXPECT_EQ(data.uniformCount, referenceData.uniformCount);
    EXPECT_EQ(data.shaderStorageCount, referenceData.shaderStorageCount);
    EXPECT_EQ(data.defaultProgramIndex, referenceData.defaultProgramIndex);
    EXPECT_EQ(data.staticOptionCount, referenceData.staticOptionCount);
    EXPECT_EQ(data.dynamicOptionCount, referenceData.dynamicOptionCount);
    EXPECT_EQ(data.shaderProgramCount, referenceData.shaderProgramCount);
    EXPECT_EQ(data.staticKeyLength, referenceData.staticKeyLength);
    EXPECT_EQ(data.dynamicKeyLength, referenceData.dynamicKeyLength);
    EXPECT_EQ(data.attrCount, referenceData.attrCount);
    EXPECT_EQ(data.samplerCount, referenceData.samplerCount);
    EXPECT_EQ(data.uniformBlockCount, referenceData.uniformBlockCount);

    // systemBlockIndices[4] は emnum 1～4 のシステムユニフォームブロックに対応する。index に注意。
    for (int i = nn::g3d::ResUniformBlockVar::Type_Material; i < nn::g3d::ResUniformBlockVar::Type_Num; ++i)
    {
        EXPECT_EQ(data.systemBlockIndices[i - 1], referenceData.systemBlockIndices[i - 1]);
    }
    EXPECT_EQ(data.shaderStorageBlockCount, referenceData.shaderStorageBlockCount);
    for (int i = nn::g3d::ResUniformBlockVar::Type_Material; i < nn::g3d::ResShaderStorageBlockVar::Type_Num; ++i)
    {
        if (i == nn::g3d::ResShaderStorageBlockVar::Type_None || i == nn::g3d::ResShaderStorageBlockVar::Type_Skeleton)
        {
            EXPECT_EQ(data.systemShaderStorageBlockIndices[i - 1], referenceData.systemShaderStorageBlockIndices[i - 1]);
        }
        else
        {
            // ssbo は none と skeleton 以外は許可していない
            EXPECT_EQ(data.systemShaderStorageBlockIndices[i - 1], -1);
        }
    }

    // 予約領域
    EXPECT_EQ(sizeof(data.reserved), 16);
} // NOLINT(impl/function_size)

void CheckResShaderArchiveDataEqual(const nn::g3d::ResShaderArchiveData& data, const nn::g3d::ResShaderArchiveData& referenceData)
{
    // 構造体サイズ
    EXPECT_EQ(sizeof(data), 72);

    // ポインタ
    EXPECT_EQ(std::strcmp(data.pName.Get()->GetData(), referenceData.pName.Get()->GetData()), 0);
    EXPECT_EQ(std::strcmp(data.pPath.Get()->GetData(), referenceData.pPath.Get()->GetData()), 0);
    for (int i = 0; i < data.shadingModelCount; ++i)
    {
        CheckResShadingModelDataEqual(data.pShadingModelArray.Get()[i].ToData(), referenceData.pShadingModelArray.Get()[i].ToData());
    }
    if (data.shadingModelCount > 0)
    {
        CheckResDicDataEqual(data.pShadingModelDic.Get()->ToData(), referenceData.pShadingModelDic.Get()->ToData());
    }
    EXPECT_FALSE(data.pUserPtr.Get());

    // 各値
    EXPECT_EQ(data.shadingModelCount, referenceData.shadingModelCount);
    EXPECT_EQ(data.flag, referenceData.flag);

    // 予約領域
    EXPECT_EQ(sizeof(data.reserved1), 8);
    EXPECT_EQ(sizeof(data.reserved2), 4);
}

void CheckResShaderFileDataEqual(const nn::g3d::ResShaderFileData& data, const nn::g3d::ResShaderFileData& referenceData)
{
    // 構造体サイズ
    EXPECT_EQ(sizeof(data), 56);

    CheckResShaderArchiveDataEqual(data.pShaderArchive.Get()->ToData(), referenceData.pShaderArchive.Get()->ToData());

    // 各値
    EXPECT_EQ(data.sizeStringPool, referenceData.sizeStringPool);

    // 予約領域
    EXPECT_EQ(sizeof(data.reserved), 4);
}
