﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#define USE_STD

#include "testG3d_Main.h"
#include "nnt/g3d/testG3d_TestUtility.h"

namespace
{
#define RESOURCE_PATH  "Resource:/"
const char* g_ShapeAnimPath = RESOURCE_PATH"shape_anim.bfres";

const nn::g3d::CameraAnimResult g_AnswerAnimCameraResult =
{
    // proj
    {
        0.100000001f, // nearZ
        1000.00000f, // farZ
        1.50000000f, // aspect
        {
            0.523598790f // height
        },
    },
    // view
    {
        {3711.72339f, 3711.72339f, 3711.72339f}, // pos
        // aim, rotate
        {
            {4.65425062f, 4.65425062f, 4.65425062f}
        },
        0.000000000f /* twist */
    }
};

const nn::g3d::LightAnimResult g_AnswerAnimLightResult =
{
    0x00000001, // enable
    {75.0733185f, 0.000000000f, 0.000000000f}, // pos
    // aim, dir
    {
        {0.000000000f, 25.7655315f, 0.000000000f}
    },
    {0.000000000f, 101.129997f}, // distAttn
    {0.000000000f, 1.30576479f}, // angleAttn
    // color
    {
        {0.740740776f, 0.000000000f, 0.259259224f},
        {0.000000000f, 0.740740776f, 0.000000000f}
    }
};

const nn::g3d::FogAnimResult g_AnswerAnimFogResult =
{
    {57.4074059f, 92.5925903f}, // distAttn
    {0.851851821f, 0.851851821f, 0.851851821f} // color
};

const nn::util::Matrix4x3fType g_AnswerAimCameraMtx0 = NN_UTIL_MATRIX_4X3F_INITIALIZER(
    -0.490584344f,  0.560375452f,  0.667344034f,
     0.705208004f,  0.705208004f, -0.0737454072f,
    -0.511941493f,  0.434438020f, -0.741089404f,
    -0.106785774f, -0.629687071f, -7.04216719f
);

const nn::util::Matrix4x3fType g_AnswerAimCameraMtx1 = NN_UTIL_MATRIX_4X3F_INITIALIZER(
     1.00000000f,   0.000000000f,  0.000000000f,
     0.000000000f,  0.000000000f,  1.00000000f,
     0.000000000f, -1.00000000f,   0.000000000f,
    -5.00000000f,   5.00000000f,  -10.0000000
);

const nn::util::Matrix4x3fType g_AnswerAimCameraMtx2 = NN_UTIL_MATRIX_4X3F_INITIALIZER(
     1.00000000f,   0.000000000f,  0.000000000f,
     0.000000000f,  0.000000000f, -1.00000000f,
     0.000000000f,  1.00000000f,   0.000000000f,
    -5.00000000f,  -5.00000000f,  -10.0000000
);

const nn::util::Matrix4x3fType g_AnswerRotateCameraMtx = NN_UTIL_MATRIX_4X3F_INITIALIZER(
     0.869958043f, -0.0991418660f,  0.482934475f,
     0.249985531f,  0.932958007f,  -0.258811653f,
    -0.424926788f,  0.345889509f,   0.836467326f,
    -6.47442436f,   2.22515702f,    1.76766443f
);


const nn::util::Matrix4x3fType g_AnswerOrthoProjTexMtx = NN_UTIL_MATRIX_4X3F_INITIALIZER(
    0.268573970f,  0.000000000f,  0.000000000f,
    0.000000000f, -0.477464825f,  0.000000000f,
    0.000000000f,  0.000000000f,  0.000000000f,
    0.500000000f,  0.500000000f,  1.00000000f
);

const nn::util::Matrix4x3fType g_AnswerPerspProjTexMtx = NN_UTIL_MATRIX_4X3F_INITIALIZER(
     0.487139314f,  0.000000000f,  0.000000000f,
     0.000000000f, -0.866025448f,  0.000000000f,
    -0.500000000f, -0.500000000f, -1.00000000f,
     0.000000000f,  0.000000000f,  0.000000000f
);

nn::g3d::TextureRef TextureBindCallback(const char* name, void* pUserData)
{
    NN_UNUSED(pUserData);
    nn::g3d::TextureRef textureRef;
    if (std::strcmp(name, "human_A") == 0)
    {
        textureRef.SetTextureView(reinterpret_cast<nn::gfx::TextureView*>(0x1));
        nn::gfx::DescriptorSlot descriptorSlot;
        memset(&descriptorSlot, 0xa5, sizeof(descriptorSlot));
        textureRef.SetDescriptorSlot(descriptorSlot);
        return textureRef;
    }
    else if (std::strcmp(name, "human_B") == 0)
    {
        textureRef.SetTextureView(reinterpret_cast<nn::gfx::TextureView*>(0x2));
        nn::gfx::DescriptorSlot descriptorSlot;
        memset(&descriptorSlot, 0x5a, sizeof(descriptorSlot));
        textureRef.SetDescriptorSlot(descriptorSlot);
        return textureRef;
    }
    return textureRef;
}

}

TEST_F(G3dTest, SceneAnimHelper)
{
    nn::g3d::CameraAnimResult cameraResult;
    cameraResult.proj.aspect = 16.0f / 9.0f;
    cameraResult.proj.farZ = 1000.0f;
    cameraResult.proj.nearZ = 0.1f;
    cameraResult.proj.height = 1000.0f;
    cameraResult.proj.fovy = nn::util::DegreeToRadian(60.0f);
    cameraResult.view.aim[0] = 0.0f;
    cameraResult.view.aim[1] = 5.0f;
    cameraResult.view.aim[2] = 10.0f;
    cameraResult.view.pos[0] = 5.0f;
    cameraResult.view.pos[1] = 0.0f;
    cameraResult.view.pos[2] = -5.0f;
    cameraResult.view.rotate[0] = nn::util::DegreeToRadian(15.0f);
    cameraResult.view.rotate[1] = nn::util::DegreeToRadian(30.0f);
    cameraResult.view.rotate[2] = nn::util::DegreeToRadian(15.0f);
    cameraResult.view.twist = nn::util::DegreeToRadian(45.0f);

    nn::util::Matrix4x3fType mtx43;
    nn::g3d::SceneAnimHelper::CalculateAimCameraMtx(&mtx43, &cameraResult);
    EXPECT_TRUE(IsMtx43Equal(mtx43, g_AnswerAimCameraMtx0));

    nn::g3d::SceneAnimHelper::CalculateRotateCameraMtx(&mtx43, &cameraResult);
    EXPECT_TRUE(IsMtx43Equal(mtx43, g_AnswerRotateCameraMtx));

    {
        nn::util::Matrix4x4fType mtx;
        nn::g3d::SceneAnimHelper::CalculateOrthoProjMtx(&mtx, &cameraResult);
        nn::util::Matrix4x4fType answerMtx;
        float halfHeight = cameraResult.proj.height;
        float halfWidth = cameraResult.proj.aspect * halfHeight;
        float nearZ =  cameraResult.proj.nearZ;
        float farZ = cameraResult.proj.farZ;
        nn::util::MatrixOrthographicOffCenterRightHanded(&answerMtx, -halfWidth, halfWidth, -halfHeight, halfHeight, nearZ, farZ);
        EXPECT_TRUE(IsMtx44Equal(mtx, answerMtx));
    }

    {
        nn::util::Matrix4x4fType mtx;
        nn::g3d::SceneAnimHelper::CalculatePerspProjMtx(&mtx, &cameraResult);
        nn::util::Matrix4x4fType answerMtx;
        float fovy = cameraResult.proj.fovy;
        float aspect = cameraResult.proj.aspect;
        float nearZ =  cameraResult.proj.nearZ;
        float farZ = cameraResult.proj.farZ;
        nn::util::MatrixPerspectiveFieldOfViewRightHanded(&answerMtx, fovy, aspect, nearZ, farZ);
        EXPECT_TRUE(IsMtx44Equal(mtx, answerMtx));
    }

    nn::g3d::SceneAnimHelper::CalculateOrthoProjTexMtx(&mtx43, &cameraResult);
    EXPECT_TRUE(IsMtx43Equal(mtx43, g_AnswerOrthoProjTexMtx));

    nn::g3d::SceneAnimHelper::CalculatePerspProjTexMtx(&mtx43, &cameraResult);
    EXPECT_TRUE(IsMtx43Equal(mtx43, g_AnswerPerspProjTexMtx));

    cameraResult.view.aim[0] = 5.0f;
    cameraResult.view.aim[1] = 0.0f;
    cameraResult.view.aim[2] = 5.0f;
    cameraResult.view.pos[0] = 5.0f;
    cameraResult.view.pos[1] = 10.0f;
    cameraResult.view.pos[2] = 5.0f;

    nn::g3d::SceneAnimHelper::CalculateAimCameraMtx(&mtx43, &cameraResult);
    EXPECT_TRUE(IsMtx43Equal(mtx43, g_AnswerAimCameraMtx1));

    cameraResult.view.aim[1] = 0.0f;
    cameraResult.view.pos[1] = -10.0f;

    nn::g3d::SceneAnimHelper::CalculateAimCameraMtx(&mtx43, &cameraResult);
    EXPECT_TRUE(IsMtx43Equal(mtx43, g_AnswerAimCameraMtx2));
}

TEST_F(G3dTest, SkeletalAnim)
{
    nn::gfx::Device* pDevice = GetDevice();
    nn::g3d::ModelObj modelObj;
    nn::g3d::ResModel* pResModel = GetResFile()->FindModel("human");
    pResModel->Reset();

    nn::g3d::ModelObj::Builder builder(pResModel);
    builder.CalculateMemorySize();
    EXPECT_TRUE(builder.IsMemoryCalculated());
    size_t size = builder.GetWorkMemorySize();

    SimplePtr buffer(nnt::g3d::AlignedAllocate<void>(size, nn::g3d::ModelObj::Alignment_Buffer));
    builder.Build(&modelObj, buffer.Get(), size);

    size = modelObj.CalculateBlockBufferSize(pDevice);
    ptrdiff_t offset = AllocateMemoryPool(size, modelObj.GetBlockBufferAlignment(pDevice));
    EXPECT_TRUE(modelObj.SetupBlockBuffer(pDevice, GetWriteCombineMemoryPool(), offset, size));
    EXPECT_TRUE(modelObj.IsBlockBufferValid());

    // スケルタルアニメの構築。
    nn::g3d::ResSkeletalAnim* pResSkeletalAnim = NULL;
    pResSkeletalAnim = GetResFile()->FindSkeletalAnim("human_walk");
    EXPECT_TRUE(pResSkeletalAnim != NULL);

    nn::g3d::BindResult bindCheckResult = pResSkeletalAnim->BindCheck(pResModel->GetSkeleton());
    EXPECT_TRUE(bindCheckResult.IsBound());
    EXPECT_TRUE(bindCheckResult.IsComplete());
    bindCheckResult = pResSkeletalAnim->PreBind(pResModel->GetSkeleton());
    EXPECT_TRUE(bindCheckResult.IsBound());
    EXPECT_TRUE(bindCheckResult.IsComplete());

    nn::g3d::SkeletalAnimObj skeletalAnimObj;
    nn::g3d::SkeletalAnimObj::Builder animBuilder;
    animBuilder.Reserve(pResModel);
    animBuilder.Reserve(pResSkeletalAnim);
    animBuilder.CalculateMemorySize();
    size = animBuilder.GetWorkMemorySize();
    SimplePtr animBuffer(nnt::g3d::AlignedAllocate<void>(size, nn::g3d::SkeletalAnimObj::Alignment_Buffer));
    bool success = animBuilder.Build(&skeletalAnimObj, animBuffer.Get(), size);
    EXPECT_TRUE(success);

    // リソースの設定を行います。再設定可能です。
    skeletalAnimObj.SetResource(pResSkeletalAnim);
    // (テスト用) ClearResult を呼びます。
    skeletalAnimObj.ClearResult();
    // モデルへの関連付けを行います。
    nn::g3d::BindResult result = skeletalAnimObj.Bind(&modelObj);
    EXPECT_TRUE(result.IsComplete());

    // 計算
    for (int loop = 0; loop < 5; ++loop)
    {
        // スケルタルアニメーション。
        if (skeletalAnimObj.GetResource())
        {
            skeletalAnimObj.Calculate();
            skeletalAnimObj.GetFrameCtrl().UpdateFrame();
        }
    }
    skeletalAnimObj.ApplyTo(&modelObj);
    {
        nn::util::Float3 scale = NN_UTIL_FLOAT_3_INITIALIZER(1.0f, 1.0f, 1.0f);
        nn::util::Float3 translate = NN_UTIL_FLOAT_3_INITIALIZER(0.0f, 2.91724753f, 0.140000001f);
        nn::util::Float3 euler = NN_UTIL_FLOAT_3_INITIALIZER(0.0225380976f, 0.139626339f, 1.57160175f);
        const nn::g3d::BoneAnimResult* pBoneAnimResult = skeletalAnimObj.GetResultArray();
        EXPECT_TRUE(memcmp(&scale, &pBoneAnimResult[0].scale, sizeof(scale)) == 0);
        EXPECT_TRUE(memcmp(&translate, &pBoneAnimResult[0].translate, sizeof(translate)) == 0);
        EXPECT_TRUE(memcmp(&euler, &pBoneAnimResult[0].euler, sizeof(euler)) == 0);

        // 期待値計算 モデルを変えると変更が必要箇所
        nn::util::Vector3fType scaleVector;
        nn::util::Vector3fType translateVector;
        nn::util::Vector3fType eulerVector;
        nn::util::VectorLoad(&scaleVector, scale);
        nn::util::VectorLoad(&translateVector, translate);
        nn::util::VectorLoad(&eulerVector, euler);
        nn::util::Matrix4x3fType expectMtx;
        nn::util::MatrixSetTranslate(&expectMtx, translateVector);
        nn::util::MatrixSetRotateXyz(&expectMtx, eulerVector);

        nn::g3d::LocalMtx* pLocalMtxArray = modelObj.GetSkeleton()->GetLocalMtxArray();
        EXPECT_TRUE(IsVector3fEqual(scaleVector, pLocalMtxArray[0].scale));
        EXPECT_TRUE(IsMtx43Equal(expectMtx, pLocalMtxArray[0].mtxRT));
    }

    modelObj.CleanupBlockBuffer(GetDevice());
    FreeMemoryPool();
}

TEST_F(G3dTest, ShapeAnim)
{
    nn::gfx::Device* pDevice = GetDevice();
    size_t alignment = GetFileAlignment(g_ShapeAnimPath);
    size_t size;
    void* pFile = nnt::g3d::LoadFile(&size, g_ShapeAnimPath, alignment);
    // リロケーション
    static_cast<nn::util::BinaryFileHeader*>(pFile)->GetRelocationTable()->Relocate();
    nn::g3d::ResFile* pResFile = nn::g3d::ResFile::ResCast(pFile);
    pResFile->Setup(pDevice);

    nn::g3d::ResModel* pShapeAnimResModel = NULL;
    pShapeAnimResModel = pResFile->FindModel("shape_anim");
    EXPECT_TRUE(pShapeAnimResModel != NULL);

    nn::g3d::ModelObj shapeAnimModelObj;
    nn::g3d::ModelObj::Builder builder(pShapeAnimResModel);
    builder.CalculateMemorySize();
    size = builder.GetWorkMemorySize();
    SimplePtr buffer(nnt::g3d::AlignedAllocate<void>(size, nn::g3d::ModelObj::Alignment_Buffer));
    EXPECT_TRUE(buffer.Get() != NULL);
    bool success = builder.Build(&shapeAnimModelObj, buffer.Get(), size);
    EXPECT_TRUE(success == true);

    // Uniform Block のバッファは所定のアライメントが必要です。
    size = shapeAnimModelObj.CalculateBlockBufferSize(pDevice);
    ptrdiff_t offset = AllocateMemoryPool(size, shapeAnimModelObj.GetBlockBufferAlignment(pDevice));
    EXPECT_TRUE(offset != -1);
    success = shapeAnimModelObj.SetupBlockBuffer(pDevice, GetWriteCombineMemoryPool(), offset, size);
    EXPECT_TRUE(success == true);

    nn::g3d::ResShapeAnim* pResShapeAnim = pResFile->FindShapeAnim("shape_anim");
    nn::g3d::BindResult bindCheckResult = pResShapeAnim->BindCheck(pShapeAnimResModel);
    EXPECT_TRUE(bindCheckResult.IsBound());
    EXPECT_TRUE(bindCheckResult.IsComplete());
    bindCheckResult = pResShapeAnim->PreBind(pShapeAnimResModel);
    EXPECT_TRUE(bindCheckResult.IsBound());
    EXPECT_TRUE(bindCheckResult.IsComplete());

    nn::g3d::ShapeAnimObj shapeAnimObj;
    nn::g3d::ShapeAnimObj::Builder animBuilder;
    animBuilder.Reserve(pShapeAnimResModel);
    animBuilder.Reserve(pResShapeAnim);
    animBuilder.CalculateMemorySize();
    size = animBuilder.GetWorkMemorySize();
    SimplePtr animBuffer(nnt::g3d::AlignedAllocate<void>(size, nn::g3d::ShapeAnimObj::Alignment_Buffer));
    EXPECT_TRUE(animBuffer.Get() != NULL);
    success = animBuilder.Build(&shapeAnimObj, animBuffer.Get(), size);
    EXPECT_TRUE(success == true);

    // リソースの設定を行います。再設定可能です。
    shapeAnimObj.SetResource(pResFile->FindShapeAnim("shape_anim"));
    // モデルへの関連付けを行います。
    nn::g3d::BindResult result = shapeAnimObj.Bind(&shapeAnimModelObj);
    EXPECT_TRUE(result.IsComplete());

    nn::g3d::ResShape* pResShape = pShapeAnimResModel->GetShape(0);
    EXPECT_EQ(pResShape->GetKeyShapeCount(), 3);
    EXPECT_STREQ(pResShape->GetKeyShapeName(0), "pPlane7");
    EXPECT_STREQ(pResShape->GetKeyShapeName(1), "pPlane9");
    EXPECT_STREQ(pResShape->GetKeyShapeName(2), "pPlane8");

    nn::g3d::ResKeyShape* pResKeyShape = pResShape->GetKeyShape(0);
    EXPECT_EQ(pResKeyShape->GetPositionAttribIndex(), 0);
    EXPECT_EQ(pResKeyShape->GetNormalAttribIndex(), -1);
    EXPECT_EQ(pResKeyShape->GetTangentAttribIndex(), -1);
    EXPECT_EQ(pResKeyShape->GetBinormalAttribIndex(), -1);
    EXPECT_EQ(pResKeyShape->GetColorAttribIndex(), -1);
    pResKeyShape = pResShape->GetKeyShape(1);
    EXPECT_EQ(pResKeyShape->GetPositionAttribIndex(), 1);
    EXPECT_EQ(pResKeyShape->GetNormalAttribIndex(), -1);
    EXPECT_EQ(pResKeyShape->GetTangentAttribIndex(), -1);
    EXPECT_EQ(pResKeyShape->GetBinormalAttribIndex(), -1);
    EXPECT_EQ(pResKeyShape->GetColorAttribIndex(), -1);
    pResKeyShape = pResShape->GetKeyShape(2);
    EXPECT_EQ(pResKeyShape->GetPositionAttribIndex(), 2);
    EXPECT_EQ(pResKeyShape->GetNormalAttribIndex(), -1);
    EXPECT_EQ(pResKeyShape->GetTangentAttribIndex(), -1);
    EXPECT_EQ(pResKeyShape->GetBinormalAttribIndex(), -1);
    EXPECT_EQ(pResKeyShape->GetColorAttribIndex(), -1);

    for (int count = 0; count <= 40; ++count)
    {
        shapeAnimObj.Calculate();
        shapeAnimObj.ApplyTo(&shapeAnimModelObj);
        shapeAnimObj.GetFrameCtrl().UpdateFrame();
    }

    float weight = shapeAnimModelObj.GetShape(0)->GetBlendWeight(0);
    EXPECT_EQ(weight, 0.462531328f);
    weight = shapeAnimModelObj.GetShape(0)->GetBlendWeight(1);
    EXPECT_EQ(weight, 0.0f);
    weight = shapeAnimModelObj.GetShape(0)->GetBlendWeight(2);
    EXPECT_EQ(weight, 0.537468672f);

    shapeAnimModelObj.CleanupBlockBuffer(pDevice);
    pShapeAnimResModel->Cleanup(pDevice);

    pResFile->Cleanup(pDevice);
    nnt::g3d::UnloadFile(pFile);
    FreeMemoryPool();
}

namespace
{
void ConvertMayaTexSrtMtx(nn::g3d::TexSrt& srt, nn::util::Float2 mtx[3])
{
    float sinValue = nn::util::SinTable(nn::util::RadianToAngleIndex(srt.r));
    float cosValue = nn::util::CosTable(nn::util::RadianToAngleIndex(srt.r));
    const float sinPart = 0.5f * sinValue - 0.5f;
    const float cosPart = -0.5f * cosValue;

    mtx[0].x =  srt.sx * cosValue;
    mtx[0].y = -srt.sy * sinValue;
    mtx[1].x =  srt.sx * sinValue;
    mtx[1].y =  srt.sy * cosValue;
    mtx[2].x =  srt.sx * (cosPart - sinPart - srt.tx);
    mtx[2].y =  srt.sy * (cosPart + sinPart + srt.ty) + 1.0f;
}
}
#if defined(NN_BUILD_CONFIG_SPEC_NX)
TEST_F(G3dTest, MaterialAnim)
{
    nn::gfx::Device* pDevice = GetDevice();
    GetResFile()->BindTexture(TextureBindCallback, NULL);
    nn::g3d::ModelObj modelObj;
    nn::g3d::ResModel* pResModel = GetResFile()->FindModel("human");
    pResModel->Reset();

    nn::g3d::ResShaderArchive* pShaderArchive = GetResShaderFile()->GetResShaderArchive();
    nn::g3d::ResShadingModel* pShadingModel = pShaderArchive->FindShadingModel("basic");
    nn::g3d::ResMaterial* pMaterial = pResModel->GetMaterial(0);
    nn::g3d::ShaderUtility::BindShaderParam(pMaterial, pShadingModel);

    nn::g3d::ResShaderParam* pResShaderParam = pResModel->GetMaterial(0)->FindShaderParam("diffuse");
    ptrdiff_t colorOffset = pResShaderParam->GetOffset();

    pResShaderParam = pResModel->GetMaterial(0)->FindShaderParam("texsrt0");
    pResShaderParam->SetConvertShaderParamCallback(nn::g3d::ResShaderParam::ConvertTexSrtCallback);
    ptrdiff_t texSrtOffset = pResShaderParam->GetOffset();

    pResShaderParam = pResModel->GetMaterial(0)->FindShaderParam("texsrt_proj");
    pResShaderParam->SetConvertShaderParamCallback(nn::g3d::ResShaderParam::ConvertTexSrtExCallback);

    nn::g3d::ModelObj::Builder builder(pResModel);
    builder.CalculateMemorySize();
    EXPECT_TRUE(builder.IsMemoryCalculated());
    size_t size = builder.GetWorkMemorySize();

    SimplePtr buffer(nnt::g3d::AlignedAllocate<void>(size, nn::g3d::ModelObj::Alignment_Buffer));
    builder.Build(&modelObj, buffer.Get(), size);

    size = modelObj.CalculateBlockBufferSize(pDevice);
    ptrdiff_t offset = AllocateMemoryPool(size, modelObj.GetBlockBufferAlignment(pDevice));
    EXPECT_TRUE(modelObj.SetupBlockBuffer(pDevice, GetWriteCombineMemoryPool(), offset, size));
    EXPECT_TRUE(modelObj.IsBlockBufferValid());

    // dependポインタのチェック。
    {
        nn::g3d::MaterialObj* pMaterialObj = modelObj.GetMaterial(0);
        const nn::g3d::ResMaterial* pResMaterial = pMaterialObj->GetResource();
        const nn::g3d::ResShaderParam* pSpecPowerResShaderParam = pResMaterial->FindShaderParam("spec_power");
        int dependIndex = pSpecPowerResShaderParam->GetDependIndex();
        EXPECT_EQ(dependIndex, pResMaterial->FindShaderParamIndex("spec_int"));
        const void* specIntPtr = pMaterialObj->GetShaderParam(pMaterialObj->FindShaderParamIndex("spec_int"));
        const void* specPowerPtr = pMaterialObj->GetShaderParam(pMaterialObj->FindShaderParamIndex("spec_power"));
        void* dependPtr;
        pSpecPowerResShaderParam->GetDependPointer(&dependPtr, specPowerPtr);
        EXPECT_EQ(specIntPtr, dependPtr);

        // シェーダーで自身を依存先に設定していない場合は depend 先が再設定できない
        EXPECT_FALSE(pSpecPowerResShaderParam->SetDependPointer(const_cast<void*>(specPowerPtr), specPowerPtr));
        EXPECT_TRUE(pSpecPowerResShaderParam->GetDependPointer(&dependPtr, specPowerPtr));
        EXPECT_EQ(specIntPtr, dependPtr);

        // texsrt_proj は自身を依存先に設定しているため、depend 先の変更ができるかをチェック
        const nn::g3d::ResShaderParam* pTexsrtProjResShaderParam = pResMaterial->FindShaderParam("texsrt_proj");
        dependIndex = pTexsrtProjResShaderParam->GetDependIndex();
        EXPECT_EQ(dependIndex, pResMaterial->FindShaderParamIndex("texsrt_proj"));
        const void* texsrtProjPtr = pMaterialObj->GetShaderParam(pMaterialObj->FindShaderParamIndex("texsrt_proj"));
        EXPECT_TRUE(pTexsrtProjResShaderParam->SetDependPointer(const_cast<void*>(texsrtProjPtr), specPowerPtr));
        EXPECT_TRUE(pTexsrtProjResShaderParam->GetDependPointer(&dependPtr, texsrtProjPtr));
        EXPECT_EQ(specPowerPtr, dependPtr);
    }

    // カラーアニメの構築。
    nn::g3d::ResMaterialAnim* pResColorAnim = NULL;
    pResColorAnim = GetResFile()->FindMaterialAnim("human_color");
    nn::g3d::MaterialAnimObj colorAnimObj;

    nn::g3d::BindResult shaderParamBindCheckResult = pResColorAnim->BindCheck(pResModel);
    EXPECT_TRUE(shaderParamBindCheckResult.IsBound());
    EXPECT_TRUE(shaderParamBindCheckResult.IsComplete());
    shaderParamBindCheckResult = pResColorAnim->PreBind(pResModel);
    EXPECT_TRUE(shaderParamBindCheckResult.IsBound());
    EXPECT_TRUE(shaderParamBindCheckResult.IsComplete());

    nn::g3d::MaterialAnimObj::Builder colorAnimBuilder;
    colorAnimBuilder.Reserve(pResModel);
    colorAnimBuilder.Reserve(pResColorAnim);
    colorAnimBuilder.CalculateMemorySize();
    size = colorAnimBuilder.GetWorkMemorySize();
    SimplePtr colorAnimBuffer(nnt::g3d::AlignedAllocate<void>(size, nn::g3d::MaterialAnimObj::Alignment_Buffer));
    {
        bool success = colorAnimBuilder.Build(&colorAnimObj, colorAnimBuffer.Get(), size);
        EXPECT_TRUE(success);
    }
    // リソースの設定を行います。再設定可能です。
    colorAnimObj.SetResource(pResColorAnim);
    // モデルへの関連付けを行います。
    nn::g3d::BindResult result = colorAnimObj.Bind(&modelObj);
    EXPECT_TRUE(result.IsComplete());

    {
        nn::g3d::MaterialAnimObj::Builder testMaterialAnimBuilder;
        testMaterialAnimBuilder.Reserve(pResModel);
        testMaterialAnimBuilder.Reserve(pResColorAnim);
        testMaterialAnimBuilder.SetMaxCurveCount(2);
        testMaterialAnimBuilder.CalculateMemorySize();
        size = testMaterialAnimBuilder.GetWorkMemorySize();
        SimplePtr testMaterialAnimBuffer(nnt::g3d::AlignedAllocate<void>(size, nn::g3d::MaterialAnimObj::Alignment_Buffer));
        nn::g3d::MaterialAnimObj testMaterialAnimObj;
        bool success = testMaterialAnimBuilder.Build(&testMaterialAnimObj, testMaterialAnimBuffer.Get(), size);
        EXPECT_TRUE(success);

        EXPECT_FALSE(testMaterialAnimObj.IsAcceptable(pResColorAnim));
    }

    // テクスチャSRTアニメの構築。
    nn::g3d::ResMaterialAnim* pResTexSrtAnim = NULL;
    pResTexSrtAnim = GetResFile()->FindMaterialAnim("human_texture_srt");
    nn::g3d::MaterialAnimObj texSrtAnimObj;

    nn::g3d::MaterialAnimObj::Builder texSrtAnimBuilder;
    texSrtAnimBuilder.Reserve(pResModel);
    texSrtAnimBuilder.Reserve(pResTexSrtAnim);
    texSrtAnimBuilder.CalculateMemorySize();
    size = texSrtAnimBuilder.GetWorkMemorySize();
    SimplePtr texSrtAnimBuffer(nnt::g3d::AlignedAllocate<void>(size, nn::g3d::MaterialAnimObj::Alignment_Buffer));
    {
        bool success = texSrtAnimBuilder.Build(&texSrtAnimObj, texSrtAnimBuffer.Get(), size);
        EXPECT_TRUE(success);
    }
    // リソースの設定を行います。再設定可能です。
    texSrtAnimObj.SetResource(pResTexSrtAnim);
    // モデルへの関連付けを行います。
    result = texSrtAnimObj.Bind(&modelObj);
    EXPECT_TRUE(result.IsComplete());

    // マテリアルビジビリティアニメの構築
    nn::g3d::ResMaterialAnim* pResVisibilityAnim = NULL;
    pResVisibilityAnim = GetResFile()->FindMaterialAnim("human_material_visibility");
    nn::g3d::BindResult visibilityBindCheckResult = pResVisibilityAnim->BindCheck(pResModel);
    EXPECT_TRUE(visibilityBindCheckResult.IsBound());
    EXPECT_TRUE(visibilityBindCheckResult.IsComplete());
    visibilityBindCheckResult = pResVisibilityAnim->PreBind(pResModel);
    EXPECT_TRUE(visibilityBindCheckResult.IsBound());
    EXPECT_TRUE(visibilityBindCheckResult.IsComplete());

    nn::g3d::MaterialAnimObj visibilityAnimObj;
    nn::g3d::MaterialAnimObj::Builder visibilityAnimBuilder;
    visibilityAnimBuilder.Reserve(pResModel);
    visibilityAnimBuilder.Reserve(pResVisibilityAnim);
    visibilityAnimBuilder.CalculateMemorySize();
    size = visibilityAnimBuilder.GetWorkMemorySize();
    SimplePtr visibilityAnimBuffer(nnt::g3d::AlignedAllocate<void>(size, nn::g3d::MaterialAnimObj::Alignment_Buffer));
    {
        bool success = visibilityAnimBuilder.Build(&visibilityAnimObj, visibilityAnimBuffer.Get(), size);
        EXPECT_TRUE(success);
    }

    // リソースの設定を行います。再設定可能です。
    visibilityAnimObj.SetResource(pResVisibilityAnim);
    // モデルへの関連付けを行います。
    result = visibilityAnimObj.Bind(&modelObj);
    EXPECT_TRUE(result.IsComplete());

    // テクスチャパターンアニメの構築
    nn::g3d::ResMaterialAnim* pResTexPatternAnim = NULL;
    pResTexPatternAnim = GetResFile()->FindMaterialAnim("human_texture_pattern");
    nn::g3d::BindResult texturePatternBindCheckResult = pResTexPatternAnim->BindCheck(pResModel);
    nn::g3d::BindResult test;
    EXPECT_TRUE(texturePatternBindCheckResult.IsBound());
    EXPECT_TRUE(texturePatternBindCheckResult.IsComplete());
    texturePatternBindCheckResult = pResTexPatternAnim->PreBind(pResModel);
    EXPECT_TRUE(texturePatternBindCheckResult.IsBound());
    EXPECT_TRUE(texturePatternBindCheckResult.IsComplete());

    nn::g3d::MaterialAnimObj texPatternAnimObj;
    nn::g3d::MaterialAnimObj::Builder texPatternAnimBuilder;
    texPatternAnimBuilder.Reserve(pResModel);
    texPatternAnimBuilder.Reserve(pResTexPatternAnim);
    texPatternAnimBuilder.CalculateMemorySize();
    size = texPatternAnimBuilder.GetWorkMemorySize();
    SimplePtr texPatternBuffer(nnt::g3d::AlignedAllocate<void>(size, nn::g3d::MaterialAnimObj::Alignment_Buffer));
    {
        bool success = texPatternAnimBuilder.Build(&texPatternAnimObj, texPatternBuffer.Get(), size);
        EXPECT_TRUE(success);
    }
    // リソースの設定を行います。再設定可能です。
    texPatternAnimObj.SetResource(pResTexPatternAnim);
    // モデルへの関連付けを行います。
    result = texPatternAnimObj.Bind(&modelObj);
    EXPECT_TRUE(result.IsComplete());

    // 計算
    for (int loop = 0; loop < 300; ++loop)
    {
        // カメラ行列。
        nn::util::Vector3fType pos = {0.0f, 2.5f, 7.5f};
        nn::util::Vector3fType target = {0.0f, 2.5f, 0.0f};
        nn::util::Vector3fType up = {0.0f, 1.0f, 0.0f};

        nn::util::Matrix4x3fType cameraMtx;
        MatrixLookAtRightHanded(&cameraMtx, pos, target, up);

        // 投影マップ用行列。
        nn::util::Matrix4x3fType texMtx;
        MatrixTextureProjectionPerspectiveFieldOfViewRightHanded(&texMtx, nn::util::DegreeToRadian(45.0f), 1.0f);
        MatrixMultiply(&texMtx, cameraMtx, texMtx);
        nn::util::FloatColumnMajor4x3 floatTexMtx;
        MatrixStore(&floatTexMtx, texMtx);

        nn::g3d::MaterialObj* pMaterialObj = modelObj.GetMaterial(0);
        nn::g3d::TexSrtEx* pTexSrtEx = pMaterialObj->EditShaderParam<nn::g3d::TexSrtEx>(pMaterialObj->FindShaderParamIndex("texsrt_proj"));
        pTexSrtEx->pEffectMtx =&floatTexMtx;

        if (colorAnimObj.GetResource())
        {
            colorAnimObj.Calculate();
            colorAnimObj.ApplyTo(&modelObj);
            colorAnimObj.GetFrameCtrl().UpdateFrame();
        }

        if (texSrtAnimObj.GetResource())
        {
            texSrtAnimObj.Calculate();
            texSrtAnimObj.ApplyTo(&modelObj);
            texSrtAnimObj.GetFrameCtrl().UpdateFrame();
        }

        if (visibilityAnimObj.GetResource())
        {
            visibilityAnimObj.Calculate();
            visibilityAnimObj.ApplyTo(&modelObj);
            visibilityAnimObj.GetFrameCtrl().UpdateFrame();
        }

        if (texPatternAnimObj.GetResource())
        {
            texPatternAnimObj.Calculate();
            texPatternAnimObj.ApplyTo(&modelObj);
            texPatternAnimObj.GetFrameCtrl().UpdateFrame();
        }

        if (loop == 150)
        {
            nn::g3d::TexSrt *pSrt = pMaterialObj->EditShaderParam<nn::g3d::TexSrt>(pMaterialObj->FindShaderParamIndex("texsrt0"));
            EXPECT_PRED3(nn::util::AreEqual, pSrt->sx, 2.0f, 0.01f);
            EXPECT_PRED3(nn::util::AreEqual, pSrt->sy, 2.0f, 0.01f);
            EXPECT_PRED3(nn::util::AreEqual, nn::util::RadianToDegree(pSrt->r), 30.0f, 0.2f);
            EXPECT_PRED3(nn::util::AreEqual, pSrt->tx, 10.0f, 0.01f);
            EXPECT_PRED3(nn::util::AreEqual, pSrt->ty, 10.0f, 0.01f);
        }

        modelObj.CalculateMaterial(0);

        if (loop == 150)
        {
            nn::gfx::Buffer* pBuffer = pMaterialObj->GetMaterialBlock(0);
            uint8_t* ptr = pBuffer->Map<uint8_t>();
            pBuffer->InvalidateMappedRange(0, pMaterialObj->GetMaterialBlockSize());
            nn::util::Float3* pColor = reinterpret_cast<nn::util::Float3*>(ptr + colorOffset);
            EXPECT_PRED3(nn::util::AreEqual, pColor->x, 0.5f, 0.001f);
            EXPECT_PRED3(nn::util::AreEqual, pColor->y, 0.75f, 0.001f);
            EXPECT_PRED3(nn::util::AreEqual, pColor->z, 0.25f, 0.001f);
            nn::util::Float2 texSrtMtx[3];
            nn::g3d::TexSrt *pSrt = pMaterialObj->EditShaderParam<nn::g3d::TexSrt>(pMaterialObj->FindShaderParamIndex("texsrt0"));
            // humanでなくなった場合変更が必要
            ConvertMayaTexSrtMtx(*pSrt, texSrtMtx);
            EXPECT_TRUE(std::memcmp(ptr + texSrtOffset, texSrtMtx, sizeof(texSrtMtx)) == 0);
            pBuffer->Unmap();
        }

        if (0 <= loop && loop < 100)
        {
            EXPECT_TRUE(modelObj.IsMaterialVisible(0) == false);
        }
        else if (100 <= loop && loop < 200)
        {
            EXPECT_TRUE(modelObj.IsMaterialVisible(0) == true);
        }
        else
        {
            EXPECT_TRUE(modelObj.IsMaterialVisible(0) == false);
        }

        if (0 <= loop && loop < 100)
        {
            EXPECT_TRUE(modelObj.GetMaterial(0)->GetTexture(0).GetTextureView() == reinterpret_cast<nn::gfx::TextureView*>(0x1));
        }
        else if (100 <= loop && loop < 200)
        {
            EXPECT_TRUE(modelObj.GetMaterial(0)->GetTexture(0).GetTextureView() == reinterpret_cast<nn::gfx::TextureView*>(0x2));
        }
        else
        {
            EXPECT_TRUE(modelObj.GetMaterial(0)->GetTexture(0).GetTextureView() == reinterpret_cast<nn::gfx::TextureView*>(0x1));
        }

    }

    modelObj.CleanupBlockBuffer(pDevice);

    GetResFile()->ReleaseTexture();
    FreeMemoryPool();
} //NOLINT
#endif

TEST_F(G3dTest, BoneVisibilityAnim)
{
    nn::gfx::Device* pDevice = GetDevice();
    nn::g3d::ModelObj modelObj;
    nn::g3d::ResModel* pResModel = GetResFile()->FindModel("human");
    pResModel->Reset();

    nn::g3d::ModelObj::Builder builder(pResModel);
    builder.CalculateMemorySize();
    EXPECT_TRUE(builder.IsMemoryCalculated());
    size_t size = builder.GetWorkMemorySize();

    SimplePtr buffer(nnt::g3d::AlignedAllocate<void>(size, nn::g3d::ModelObj::Alignment_Buffer));
    builder.Build(&modelObj, buffer.Get(), size);

    size = modelObj.CalculateBlockBufferSize(pDevice);
    ptrdiff_t offset = AllocateMemoryPool(size, modelObj.GetBlockBufferAlignment(pDevice));
    EXPECT_TRUE(modelObj.SetupBlockBuffer(GetDevice(), GetWriteCombineMemoryPool(), offset, size));
    EXPECT_TRUE(modelObj.IsBlockBufferValid());

    // ボーンビジビリティアニメの構築。
    nn::g3d::ResBoneVisibilityAnim* pResBoneVisibilityAnim = NULL;
    pResBoneVisibilityAnim = GetResFile()->FindBoneVisibilityAnim("human_bone_visibility");
    EXPECT_TRUE(pResBoneVisibilityAnim != NULL);
    nn::g3d::BindResult bindCheckResult = pResBoneVisibilityAnim->BindCheck(pResModel);
    EXPECT_TRUE(bindCheckResult.IsBound());
    EXPECT_TRUE(bindCheckResult.IsComplete());
    bindCheckResult = pResBoneVisibilityAnim->PreBind(pResModel);
    EXPECT_TRUE(bindCheckResult.IsBound());
    EXPECT_TRUE(bindCheckResult.IsComplete());

    nn::g3d::BoneVisibilityAnimObj boneVisibilityAnimObj;
    nn::g3d::BoneVisibilityAnimObj::Builder animBuilder;
    animBuilder.Reserve(pResModel);
    animBuilder.Reserve(pResBoneVisibilityAnim);
    animBuilder.CalculateMemorySize();
    size = animBuilder.GetWorkMemorySize();
    SimplePtr animBuffer(nnt::g3d::AlignedAllocate<void>(size, nn::g3d::BoneVisibilityAnimObj::Alignment_Buffer));
    bool success = animBuilder.Build(&boneVisibilityAnimObj, animBuffer.Get(), size);
    EXPECT_TRUE(success);

    // リソースの設定を行います。再設定可能です。
    boneVisibilityAnimObj.SetResource(pResBoneVisibilityAnim);
    // モデルへの関連付けを行います。
    nn::g3d::BindResult result = boneVisibilityAnimObj.Bind(&modelObj);
    EXPECT_TRUE(result.IsComplete());

    int hipIndex = modelObj.GetSkeleton()->FindBoneIndex("hip_1");
    int spinIndex = modelObj.GetSkeleton()->FindBoneIndex("spin_1");
    // 計算
    for (int loop = 0; loop < 300; ++loop)
    {
        if (boneVisibilityAnimObj.GetResource())
        {
            boneVisibilityAnimObj.Calculate();
            boneVisibilityAnimObj.ApplyTo(&modelObj);
            boneVisibilityAnimObj.GetFrameCtrl().UpdateFrame();
        }

        if (0 <= loop && loop < 100)
        {
            EXPECT_TRUE(modelObj.IsBoneVisible(hipIndex) == true);
            EXPECT_TRUE(modelObj.IsBoneVisible(spinIndex) == false);
        }
        else if (100 <= loop && loop < 200)
        {
            EXPECT_TRUE(modelObj.IsBoneVisible(hipIndex) == false);
            EXPECT_TRUE(modelObj.IsBoneVisible(spinIndex) == true);
        }
        else
        {
            EXPECT_TRUE(modelObj.IsBoneVisible(hipIndex) == true);
            EXPECT_TRUE(modelObj.IsBoneVisible(spinIndex) == false);
        }
    }

    modelObj.CleanupBlockBuffer(pDevice);
    FreeMemoryPool();
}

TEST_F(G3dTest, SceneAnim)
{
    nn::g3d::ResSceneAnim* pResSceneAnim = GetResFile()->FindSceneAnim("human_scene");

    // カメラアニメ
    nn::g3d::ResCameraAnim* pResCameraAnim = pResSceneAnim->FindCameraAnim("human_camera");
    if (pResCameraAnim)
    {
        nn::g3d::CameraAnimObj cameraAnimObj;
        nn::g3d::CameraAnimObj::Builder builder;
        builder.Reserve(pResCameraAnim);
        builder.CalculateMemorySize();
        size_t size = builder.GetWorkMemorySize();
        SimplePtr buffer(nnt::g3d::AlignedAllocate<void>(size, nn::g3d::CameraAnimObj::Alignment_Buffer));
        bool success = builder.Build(&cameraAnimObj, buffer.Get(), size);
        EXPECT_TRUE(success);

        // リソースの設定を行います。再設定可能です。
        cameraAnimObj.SetResource(pResCameraAnim);

        for (int i = 0; i < 150; i++)
        {
            cameraAnimObj.Calculate();
            cameraAnimObj.GetFrameCtrl().UpdateFrame();

            nn::g3d::CameraAnimResult* pResult = cameraAnimObj.GetResult();
            if (i == 100)
            {
                EXPECT_TRUE(IsFloatEqual(pResult->proj.nearZ, g_AnswerAnimCameraResult.proj.nearZ));
                EXPECT_TRUE(IsFloatEqual(pResult->proj.farZ, g_AnswerAnimCameraResult.proj.farZ));
                EXPECT_TRUE(IsFloatEqual(pResult->proj.aspect, g_AnswerAnimCameraResult.proj.aspect));
                EXPECT_TRUE(IsFloatEqual(pResult->proj.height, g_AnswerAnimCameraResult.proj.height));
                EXPECT_TRUE(IsFloatArrayEqual(pResult->view.pos, g_AnswerAnimCameraResult.view.pos, 3));
                EXPECT_TRUE(IsFloatArrayEqual(pResult->view.aim, g_AnswerAnimCameraResult.view.aim, 3));
                EXPECT_TRUE(IsFloatEqual(pResult->view.twist, g_AnswerAnimCameraResult.view.twist));
            }
        }
    }

    // ライトアニメ
    nn::g3d::ResLightAnim* pResLightAnim = pResSceneAnim->FindLightAnim("human_light");
    if (pResLightAnim)
    {
        nn::g3d::LightAnimObj lightAnimObj;
        nn::g3d::LightAnimObj::Builder builder;
        builder.Reserve(pResLightAnim);
        builder.CalculateMemorySize();
        size_t size = builder.GetWorkMemorySize();
        SimplePtr buffer(nnt::g3d::AlignedAllocate<void>(size, nn::g3d::LightAnimObj::Alignment_Buffer));
        bool success = builder.Build(&lightAnimObj, buffer.Get(), size);
        EXPECT_TRUE(success);

        // リソースの設定を行います。再設定可能です。
        lightAnimObj.SetResource(pResLightAnim);

        for (int i = 0; i < 150; i++)
        {
            lightAnimObj.Calculate();
            lightAnimObj.GetFrameCtrl().UpdateFrame();

            nn::g3d::LightAnimResult* pResult = lightAnimObj.GetResult();
            if (i == 100)
            {
                EXPECT_TRUE(IsFloatArrayEqual(pResult->pos, g_AnswerAnimLightResult.pos, 3));
                EXPECT_TRUE(IsFloatArrayEqual(pResult->aim, g_AnswerAnimLightResult.aim, 3));
                EXPECT_TRUE(IsFloatArrayEqual(pResult->distAttn, g_AnswerAnimLightResult.distAttn, 2));
                EXPECT_TRUE(IsFloatArrayEqual(pResult->angleAttn, g_AnswerAnimLightResult.angleAttn, 2));
                EXPECT_TRUE(IsFloatArrayEqual(pResult->color[0], g_AnswerAnimLightResult.color[0], 3));
                EXPECT_TRUE(IsFloatArrayEqual(pResult->color[1], g_AnswerAnimLightResult.color[1], 3));
            }
        }
    }

    // フォグアニメ
    nn::g3d::ResFogAnim* pResFogAnim = pResSceneAnim->FindFogAnim("human_fog");
    if (pResFogAnim)
    {
        nn::g3d::FogAnimObj fogAnimObj;
        nn::g3d::FogAnimObj::Builder builder;
        builder.Reserve(pResFogAnim);
        builder.CalculateMemorySize();
        size_t size = builder.GetWorkMemorySize();
        SimplePtr buffer(nnt::g3d::AlignedAllocate<void>(size, nn::g3d::FogAnimObj::Alignment_Buffer));
        bool success = builder.Build(&fogAnimObj, buffer.Get(), size);
        EXPECT_TRUE(success);

        // リソースの設定を行います。再設定可能です。
        fogAnimObj.SetResource(pResFogAnim);

        for (int i = 0; i < 150; i++)
        {
            fogAnimObj.Calculate();
            fogAnimObj.GetFrameCtrl().UpdateFrame();

            nn::g3d::FogAnimResult* pResult = fogAnimObj.GetResult();
            if (i == 100)
            {
                EXPECT_TRUE(IsFloatArrayEqual(pResult->distAttn, g_AnswerAnimFogResult.distAttn, 2));
                EXPECT_TRUE(IsFloatArrayEqual(pResult->color, g_AnswerAnimFogResult.color, 3));
            }
        }
    }
}

TEST_F(G3dTest, SceneAnimUserDataAccess)
{
    nn::g3d::ResSceneAnim* pResSceneAnim = GetResFile()->FindSceneAnim("human_scene");

    // カメラアニメ
    nn::g3d::ResCameraAnim* pResCameraAnim = pResSceneAnim->FindCameraAnim("human_camera");
    if (pResCameraAnim)
    {
        EXPECT_EQ( 2, pResCameraAnim->GetUserDataCount() );
        EXPECT_STREQ( "camera_anim_user_data_0", pResCameraAnim->GetUserDataName( 0 ) );
        EXPECT_STREQ( "camera_anim_user_data_1", pResCameraAnim->GetUserDataName( 1 ) );
        EXPECT_EQ( 0, pResCameraAnim->FindUserDataIndex( "camera_anim_user_data_0" ) );
        EXPECT_EQ( 1, pResCameraAnim->FindUserDataIndex( "camera_anim_user_data_1" ) );

        auto pResUserData = pResCameraAnim->FindUserData( "camera_anim_user_data_0" );
        EXPECT_STREQ( "camera_anim_user_data_0", pResUserData->GetName() );
        EXPECT_EQ( 1, pResUserData->GetCount() );
        const float* pFloatVal = pResUserData->GetFloat();
        EXPECT_FLOAT_EQ( 0.f, pFloatVal[0] );
        EXPECT_FLOAT_EQ( pFloatVal[0], pResUserData->GetFloat(0));
        pResUserData = pResCameraAnim->FindUserData( "camera_anim_user_data_1" );
        EXPECT_STREQ( "camera_anim_user_data_1", pResUserData->GetName() );
        EXPECT_EQ( 1, pResUserData->GetCount() );
        pFloatVal = pResUserData->GetFloat();
        EXPECT_FLOAT_EQ( 1.f, pFloatVal[0] );
        EXPECT_FLOAT_EQ( pFloatVal[0], pResUserData->GetFloat(0) );
    }

    // ライトアニメ
    nn::g3d::ResLightAnim* pResLightAnim = pResSceneAnim->FindLightAnim("human_light");
    if (pResLightAnim)
    {
        EXPECT_EQ( 2, pResLightAnim->GetUserDataCount() );
        EXPECT_STREQ( "light_anim_user_data_0", pResLightAnim->GetUserDataName( 0 ) );
        EXPECT_STREQ( "light_anim_user_data_1", pResLightAnim->GetUserDataName( 1 ) );
        EXPECT_EQ( 0, pResLightAnim->FindUserDataIndex( "light_anim_user_data_0" ) );
        EXPECT_EQ( 1, pResLightAnim->FindUserDataIndex( "light_anim_user_data_1" ) );

        auto pResUserData = pResLightAnim->FindUserData( "light_anim_user_data_0" );
        EXPECT_STREQ( "light_anim_user_data_0", pResUserData->GetName() );
        EXPECT_EQ( 1, pResUserData->GetCount() );
        const float* pFloatVal = pResUserData->GetFloat();
        EXPECT_FLOAT_EQ( 0.f, pFloatVal[0] );
        EXPECT_FLOAT_EQ( pFloatVal[0], pResUserData->GetFloat(0) );
        pResUserData = pResLightAnim->FindUserData( "light_anim_user_data_1" );
        EXPECT_STREQ( "light_anim_user_data_1", pResUserData->GetName() );
        EXPECT_EQ( 1, pResUserData->GetCount() );
        pFloatVal = pResUserData->GetFloat();
        EXPECT_FLOAT_EQ( 1.f, pFloatVal[0] );
        EXPECT_FLOAT_EQ( pFloatVal[0], pResUserData->GetFloat(0));
    }

    // フォグアニメ
    nn::g3d::ResFogAnim* pResFogAnim = pResSceneAnim->FindFogAnim("human_fog");
    if (pResFogAnim)
    {
        EXPECT_EQ( 2, pResFogAnim->GetUserDataCount() );
        EXPECT_STREQ( "fog_anim_user_data_0", pResFogAnim->GetUserDataName( 0 ) );
        EXPECT_STREQ( "fog_anim_user_data_1", pResFogAnim->GetUserDataName( 1 ) );
        EXPECT_EQ( 0, pResFogAnim->FindUserDataIndex( "fog_anim_user_data_0" ) );
        EXPECT_EQ( 1, pResFogAnim->FindUserDataIndex( "fog_anim_user_data_1" ) );

        auto pResUserData = pResFogAnim->FindUserData( "fog_anim_user_data_0" );
        EXPECT_STREQ( "fog_anim_user_data_0", pResUserData->GetName() );
        EXPECT_EQ( 1, pResUserData->GetCount() );
        const float* pFloatVal = pResUserData->GetFloat();
        EXPECT_FLOAT_EQ( 0.f, pFloatVal[0] );
        EXPECT_FLOAT_EQ( pFloatVal[0], pResUserData->GetFloat(0) );
        pResUserData = pResFogAnim->FindUserData( "fog_anim_user_data_1" );
        EXPECT_STREQ( "fog_anim_user_data_1", pResUserData->GetName() );
        EXPECT_EQ( 1, pResUserData->GetCount() );
        pFloatVal = pResUserData->GetFloat();
        EXPECT_FLOAT_EQ( 1.f, pFloatVal[0] );
        EXPECT_FLOAT_EQ( pFloatVal[0], pResUserData->GetFloat(0) );
    }

    // ユーザデータ 1
    const nn::gfx::ResUserData* pResUserData = pResSceneAnim->FindUserData("human_user_data_1");
    if (pResUserData)
    {
        EXPECT_STREQ( "human_user_data_1", pResUserData->GetName() );
        EXPECT_EQ( 1, pResUserData->GetCount() );
        const float* pFloatVal = pResUserData->GetFloat();
        EXPECT_FLOAT_EQ( 4.f, pFloatVal[0] );
        EXPECT_FLOAT_EQ( pFloatVal[0], pResUserData->GetFloat(0));
    }

    // ユーザデータ 2
    pResUserData = pResSceneAnim->FindUserData("human_user_data_2");
    if (pResUserData)
    {
        EXPECT_STREQ( "human_user_data_2", pResUserData->GetName() );
        EXPECT_EQ( 1, pResUserData->GetCount() );
        const char* pStringVal = pResUserData->GetString(0);
        EXPECT_STREQ( "user_data_2", pStringVal );
    }
}
