﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cmath>
#include <nnt/g3d/testG3d_TestUtility.h>
#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/fs.h>
#include <nn/init.h>
#include <nn/mii.h>
#include <nn/util/util_MathTypes.h>
#include <nn/g3d/g3d_ResFile.h>
#include <nn/g3d/g3d_ResShader.h>
#include <nn/g3d/g3d_ModelObj.h>
#include <nn/g3d/g3d_MaterialAnimObj.h>
#include <nn/g3d/mii/g3d_MiiConverter.h>
#include "testG3d_MiiConvertTest.h"
#include "testG3d_MiiCharModel.h"
#include "testG3d_MiiCharModelCreateArg.h"
#include "testG3d_MiiG3dMiiUtil.h"
#include "testG3d_MiiG3dTestModelSimple.h"

const char* CharInfoPath = "Resource:/Mii/charinfo/TestCharInfo009.dat";

nn::gfx::Device g_Device;
nn::gfx::Device* GetDevice()
{
    return &g_Device;
}

#if defined( NN_BUILD_CONFIG_SPEC_NX )
#if NN_BUILD_CONFIG_OS_WIN
/// WinNvn用テクスチャリソースのパス
const char* TextureResourcePath = "Resource:/Mii/resource/WinNvnTextureMidSRGB.dat";
#else
/// NX用テクスチャリソースのパス
const char* TextureResourcePath = "Resource:/Mii/resource/NXTextureMidSRGB.dat";
#endif    // #if NN_BUILD_CONFIG_OS_HORIZON
#else
const char* TextureResourcePath = "Resource:/Mii/resource/WinGenericTextureMidSRGB.dat";
#endif    // #if NN_BUILD_CONFIG_SPEC_NX

const char* ShapeResourcePath = "Resource:/Mii/resource/ShapeMid.dat";

const char* BfresPath = "Resource:/Mii.bfres";
nn::g3d::ResFile*    g_pResFile = nullptr;
void LoadBfres()
{
    size_t size = 0;
    size_t align = 4 * 1024;
    g_pResFile = reinterpret_cast<nn::g3d::ResFile*>( nnt::g3d::LoadFile( &size, BfresPath, align ) );
    NN_ASSERT_NOT_NULL( g_pResFile );
}

class MiiG3dTest : public ::testing::Test
{
protected:
    virtual void SetUp() NN_NOEXCEPT
    {
    }
    virtual void TearDown() NN_NOEXCEPT
    {
    }
};

const nn::gfx::PrimitiveTopology    g_cMiiPrimitiveTopology = nn::mii::DrawParam::PrimitiveTopology;    //!< TODO: TEST_F 内で直接参照すると undefined のエラーに対するワークアラウンド
const nn::gfx::IndexFormat            g_cMiiIndexFormat        = nn::mii::DrawParam::IndexFormat;            //!< TODO: TEST_F 内で直接参照すると undefined のエラーに対するワークアラウンド

TEST_F( MiiG3dTest, AssignCharModel )
{
    // リソースロード
    LoadBfres();

    // オブジェクト初期化
    CharModelCreateArg charModelCreateArg;
    charModelCreateArg.SetTexPath( TextureResourcePath );
    charModelCreateArg.SetShapePath( ShapeResourcePath );
    charModelCreateArg.SetTexFormat( nn::gfx::ImageFormat_R8_G8_B8_A8_Uint );
    charModelCreateArg.SetTexMipCount( 1 );
    charModelCreateArg.SetTexResolution( 512 );
    charModelCreateArg.SetCharInfoPath( CharInfoPath );
    charModelCreateArg.SetExpressionFlags( nn::mii::ExpressionFlag_All );
    charModelCreateArg.SetCreateFlags( nn::mii::CreateFlag_Normal | nn::mii::CreateFlag_Hat | nn::mii::CreateFlag_NoseNormal );
    MiiCharModel miiCharModel;
    miiCharModel.Initialize( GetDevice(), &charModelCreateArg );

    std::vector<MiiG3dTestModelSimple*> g3dMiiCharModelPtrArray;
    g3dMiiCharModelPtrArray.reserve( g_pResFile->GetModelCount() );
    g3dMiiCharModelPtrArray.push_back( new MiiG3dTestModelSimple() );

    // G3dMiCharModel 初期化
    for( auto iterG3dMiiCharModel = g3dMiiCharModelPtrArray.begin(); iterG3dMiiCharModel != g3dMiiCharModelPtrArray.end(); ++iterG3dMiiCharModel )
    {
        ( *iterG3dMiiCharModel )->Initialize();
    }

    // G3dMiCharModel CharModel のアサイン
    g_pResFile = nn::g3d::ResFile::ResCast( g_pResFile );
    for( int idxModel = 0; idxModel < g_pResFile->GetModelCount(); ++idxModel )
    {
        auto pModel = g_pResFile->GetModel( idxModel );
        auto pG3dMiiCharModel = g3dMiiCharModelPtrArray[idxModel];
        nn::g3d::mii::AssignCharModel( pModel, miiCharModel.GetCharModel(),
                                        pG3dMiiCharModel->GetMaterialCallback(),
                                        pG3dMiiCharModel->GetMaterialCallbackParam(),
                                        pG3dMiiCharModel->GetDrawParamCallback(),
                                        pG3dMiiCharModel->GetDrawParamCallbackParam() );
    }

    // g3dMii 機能テスト
    // DrawParamCallbackOutput::pDrawParam == nullptr の時
    {
        auto pModel = g_pResFile->GetModel( 0 );
        auto shapeCount = pModel->GetShapeCount();
        auto& pG3dMiiCharModel = g3dMiiCharModelPtrArray[0];

        for( auto idxVisibleBoneNameIdxArray = 0; idxVisibleBoneNameIdxArray < pG3dMiiCharModel->GetVisibleBoneCount(); ++idxVisibleBoneNameIdxArray )
        {
            for( auto idxShape = 0; idxShape < shapeCount; ++idxShape )
            {
                auto pShape = pModel->GetShape( idxShape );

                // bone
                auto boneIdx = pShape->GetBoneIndex();
                const char* pVisibleBoneName = pG3dMiiCharModel->GetVisibleBoneName( idxVisibleBoneNameIdxArray );
                if( std::string( pModel->GetSkeleton()->GetBoneName( boneIdx ) ) != pVisibleBoneName )
                {
                    continue;    // visible なボーンについてのみテストします。
                }

                // shape bounding
                auto shape = pShape->ToData();
                auto pSubMeshBoundingArray = shape.pSubMeshBoundingArray.Get();
                const int subMeshCount = pShape->GetSubMeshCount();
                auto& shapeBounding = pSubMeshBoundingArray[subMeshCount];    // shape の bounding は submesh bounding の次に格納されています。
                auto& pMiiBounding = pG3dMiiCharModel->GetGoldenDrawParamOutput( idxShape ).pBoundingBox;
                nn::util::Vector3fType bbMax = NN_UTIL_VECTOR_3F_INITIALIZER( pMiiBounding->max.x, pMiiBounding->max.y, pMiiBounding->max.z );
                nn::util::Vector3fType bbMin = NN_UTIL_VECTOR_3F_INITIALIZER( pMiiBounding->min.x, pMiiBounding->min.y, pMiiBounding->min.z );
                nn::util::Vector3fType center;
                nn::util::Vector3fType extent;
                nn::util::VectorAdd( &center, bbMax, bbMin );
                center._v[0] /= 2.f;
                center._v[1] /= 2.f;
                center._v[2] /= 2.f;
                nn::util::VectorSubtract( &extent, bbMax, bbMin );
                extent._v[0] /= 2.f;
                extent._v[1] /= 2.f;
                extent._v[2] /= 2.f;
                EXPECT_FLOAT_EQ( shapeBounding.center.x, nn::util::VectorGetX( center ) );
                EXPECT_FLOAT_EQ( shapeBounding.center.y, nn::util::VectorGetY( center ) );
                EXPECT_FLOAT_EQ( shapeBounding.center.z, nn::util::VectorGetZ( center ) );
                EXPECT_FLOAT_EQ( shapeBounding.extent.x, nn::util::VectorGetX( extent ) );
                EXPECT_FLOAT_EQ( shapeBounding.extent.y, nn::util::VectorGetY( extent ) );
                EXPECT_FLOAT_EQ( shapeBounding.extent.z, nn::util::VectorGetZ( extent ) );

                // 本来 mii のサブメッシュバウンディングはユーザは mii を構築するまで index 情報を分かりえないですが
                // サブメッシュバウンディングが fmd 上に存在する場合は同値で上書きする仕様です。
                for( int idxSubMesh = 0; idxSubMesh < subMeshCount; ++idxSubMesh )
                {
                    auto& subMeshBounding = pSubMeshBoundingArray[idxSubMesh];
                    EXPECT_FLOAT_EQ( subMeshBounding.center.x, nn::util::VectorGetX( center ) );
                    EXPECT_FLOAT_EQ( subMeshBounding.center.y, nn::util::VectorGetY( center ) );
                    EXPECT_FLOAT_EQ( subMeshBounding.center.z, nn::util::VectorGetZ( center ) );
                    EXPECT_FLOAT_EQ( subMeshBounding.extent.x, nn::util::VectorGetX( extent ) );
                    EXPECT_FLOAT_EQ( subMeshBounding.extent.y, nn::util::VectorGetY( extent ) );
                    EXPECT_FLOAT_EQ( subMeshBounding.extent.z, nn::util::VectorGetZ( extent ) );
                }

                // バウンディング球
                float radius = 0.f;
                radius = std::sqrt( extent._v[0] * extent._v[0] +
                                    extent._v[1] * extent._v[1] +
                                    extent._v[2] * extent._v[2] );

                for( int idxMesh = 0; idxMesh < pShape->GetMeshCount(); ++idxMesh )
                {
                    EXPECT_FLOAT_EQ( shape.pRadiusArray.Get()[idxMesh], radius );
                }

                auto pMesh = pShape->GetMesh( 0 );    // 似顔絵ライブラリに LOD モデルは存在しないので index 0 のメッシュを使用してテストします。
                auto& meshData = pMesh->ToData();
                auto pDrawParam = pG3dMiiCharModel->GetGoldenDrawParamOutput( idxShape ).pDrawParam;

                // mesh
                EXPECT_EQ( meshData.pIndexBuffer.Get(), pDrawParam->GetIndexBuffer() );
                EXPECT_EQ( meshData.count, pDrawParam->GetIndexCount() );
                EXPECT_EQ( meshData.primType, ::g_cMiiPrimitiveTopology );
                EXPECT_EQ( meshData.format, ::g_cMiiIndexFormat );
                auto& subMeshData = meshData.pSubMeshArray.Get()->ToData();
                EXPECT_EQ( subMeshData.count, pDrawParam->GetIndexCount() );
                EXPECT_EQ( subMeshData.offset, 0 );

                // vertex
                auto pVtx = pShape->GetVertex();
                auto& vtxData = pShape->GetVertex()->ToData();
                const uint32_t vtxNum = static_cast<uint32_t>( pDrawParam->GetBufferSize( nn::mii::DrawParam::AttributeType_Position ) / nn::mii::DrawParam::PositionStride );
                EXPECT_EQ( vtxData.count, vtxNum );
                const char* miiAttribNameArray[] =    // DCC Exporter の仕様上必ずこの名前で出力されます。
                {
                    "_p0",    // idxMiiAttrib: 0 for mii position
                    "_n0",    // idxMiiAttrib: 1 for mii normal
                    "_u0",    // idxMiiAttrib: 2 for mii uv
                    "_t0",    // idxMiiAttrib: 3 for mii tangent
                    "_c0",    // idxMiiAttrib: 4 for mii param
                };
                const int miiAttribCount = sizeof( miiAttribNameArray ) / sizeof( char* );
                for( int idxMiiAttrib = 0; idxMiiAttrib < miiAttribCount; ++idxMiiAttrib )
                {
                    const char* miiAttribName = miiAttribNameArray[idxMiiAttrib];
                    int idxVtxAttrib = pVtx->FindVertexAttrIndex( miiAttribName );
                    if( idxVtxAttrib == nn::util::ResDic::Npos )
                    {
                        continue;
                    }

                    auto miiAttribType = static_cast<nn::mii::DrawParam::AttributeType>( idxMiiAttrib );
                    if( !pDrawParam->IsValidAttribute( miiAttribType ) )    // ライブラリは無効な attribute については何もせず fmd の値が保持されます。
                    {
                        continue;
                    }

                    int idxBuff = pVtx->GetVertexAttr( idxVtxAttrib )->GetBufferIndex();
                    auto pVtxBufferStateInfo = &vtxData.pVertexBufferStateInfoArray.Get()[idxBuff];  // アクセサが private なので直接取得する
                    auto ppVtxBuffer = vtxData.pVertexBufferPtrArray.Get();  // 未初期化の ResFile なのでアクセサ使用不可
                    auto pVtxBufferInfo = pVtx->GetVertexBufferInfo( idxBuff );

                    EXPECT_EQ( ppVtxBuffer[idxBuff], const_cast<nn::gfx::Buffer*>( pDrawParam->GetAttribute( miiAttribType ) ) );
                    EXPECT_EQ( pVtxBufferStateInfo->ToData()->stride, static_cast<uint32_t>( pDrawParam->GetVertexAttributeStride( miiAttribType ) ) );
                    EXPECT_EQ( pVtxBufferInfo->ToData()->size, static_cast<uint32_t>( vtxData.count * pVtxBufferStateInfo->GetStride() ) );

                    auto& vtxAttrib = pVtx->GetVertexAttr( idxMiiAttrib )->ToData();
                    EXPECT_EQ( vtxAttrib.format, pDrawParam->GetVertexAttributeFormat( miiAttribType ) );
                    EXPECT_EQ( vtxAttrib.offset, static_cast<uint16_t>( pDrawParam->GetVertexAttributeOffset( miiAttribType ) ) );
                }
            }
        }
    }

    // G3dMiCharModel 終了処理
    miiCharModel.Finalize( GetDevice() );
    for( auto iterG3dMiiCharModel = g3dMiiCharModelPtrArray.begin(); iterG3dMiiCharModel != g3dMiiCharModelPtrArray.end(); ++iterG3dMiiCharModel )
    {
        ( *iterG3dMiiCharModel )->Finalize();
    }
} //NOLINT
