﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>
#include <nn/nn_Log.h>
#include <nn/init.h>
#include <nn/os.h>
#include <nn/g3d.h>

#include "g3ddemo_DemoUtility.h"
#include "g3ddemo_GfxUtility.h"
#include "testG3d_FrameCapture.h"
#include "testG3d_CaptureSetting.h"

#if defined(NN_BUILD_CONFIG_OS_WIN)
#include <filesystem>
#include <nn/nn_Windows.h>
#endif

#include <nnt/graphics/testGraphics_Path.h>
#include <nnt/graphics/testGraphics_GetHostExecutableFilepath.h>
#include <nnt/graphics/testGraphics_CreateDirectories.h>
#include <nnt/graphics/testGraphics_InitializeAllocatorFunctionForStandardAllocator.h>
#include <nnt/graphics/testGraphics_Png.h>

namespace
{
nnt::g3d::CaptureSetting g_CaptureSetting;
nnt::g3d::FrameCapture g_FrameCapture;
nn::gfx::MemoryPool g_FrameCaptureMemoryPool;
const uint64_t FrameCaptureMemoryPoolSize = 8 * 1024 * 1024;
void* g_pFrameCaptureMemoryPoolMemory = NULL;
}

extern int TownMain(
    const nnt::g3d::CaptureSetting* pCaptureSetting);


nnt::g3d::FrameCapture* GetFrameCapture()
{
    return &g_FrameCapture;
}

void InitializeScreenCapture()
{
    nn::gfx::Device* pDevice = nn::g3d::demo::GetGfxFramework()->GetDevice();

    // キャプチャ画像の出力先。
    nnt::graphics::Path outputsPath(nnt::graphics::GetHostExecutableFilepath());
    outputsPath.MakeParent();
    NN_LOG("Mount Outputs: %s\n", outputsPath.GetString());
    nn::Result result = nn::fs::MountHost("Outputs", outputsPath.GetString());
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    result = nnt::graphics::CreateDirectories("Outputs:/Outputs");
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    NN_UNUSED(result);

    // キャプチャシステム用バッファ。
    g_pFrameCaptureMemoryPoolMemory = nn::g3d::demo::AllocateMemory(FrameCaptureMemoryPoolSize);

    // キャプチャシステム用メモリプール。
    nn::gfx::MemoryPool::InfoType memoryPoolInfo;
    memoryPoolInfo.SetDefault();
    memoryPoolInfo.SetMemoryPoolProperty(nn::gfx::MemoryPoolProperty_CpuCached | nn::gfx::MemoryPoolProperty_GpuCached);
    memoryPoolInfo.SetPoolMemory(g_pFrameCaptureMemoryPoolMemory, FrameCaptureMemoryPoolSize);
    g_FrameCaptureMemoryPool.Initialize(pDevice, memoryPoolInfo);

    // キャプチャシステムの初期化。
    nn::util::BytePtr frameCaptureMemoryPoolBase(g_pFrameCaptureMemoryPoolMemory);
    nn::util::BytePtr frameCaptureMemoryPoolCurrent(g_pFrameCaptureMemoryPoolMemory);
    const int viewportWidth = nn::g3d::demo::GetGfxFramework()->GetDisplayWidth();
    const int viewportHeight = nn::g3d::demo::GetGfxFramework()->GetDisplayHeight();
    g_FrameCapture.Initialize(pDevice, &g_FrameCaptureMemoryPool,
        &frameCaptureMemoryPoolBase, &frameCaptureMemoryPoolCurrent, viewportWidth, viewportHeight);
}

void ScreenCapture(const int idxFrame)
{
    // 画像をpngに保存。
    g_FrameCapture.FetchCaptureResult();
    nnt::graphics::Path fileName("Outputs:/%s%02d.png", g_CaptureSetting.GetOutputImageName(), idxFrame);
    g_FrameCapture.SaveToPng(fileName.GetString());
}

uint64_t GetFrameCaptureCountMax()
{
    return g_CaptureSetting.GetCameraCount();
}

void FinalizeScreenCapture()
{
    nn::gfx::Device* pDevice = nn::g3d::demo::GetGfxFramework()->GetDevice();

    // キャプチャシステムの破棄。
    g_FrameCapture.Finalize();
    g_FrameCaptureMemoryPool.Finalize(pDevice);
    nn::g3d::demo::FreeMemory(g_pFrameCaptureMemoryPoolMemory);
}

void GetScreenCaptureCamera(const int idxCamera, nn::util::Vector3fType* pCameraPosition, nn::util::Vector3fType* pCameraTarget)
{
    if (idxCamera < g_CaptureSetting.GetCameraCount())
    {
        pCameraPosition->_v[0] = g_CaptureSetting.GetCamera(idxCamera)->posX;
        pCameraPosition->_v[1] = g_CaptureSetting.GetCamera(idxCamera)->posY;
        pCameraPosition->_v[2] = g_CaptureSetting.GetCamera(idxCamera)->posZ;
        pCameraTarget->_v[0] = g_CaptureSetting.GetCamera(idxCamera)->targetX;
        pCameraTarget->_v[1] = g_CaptureSetting.GetCamera(idxCamera)->targetY;
        pCameraTarget->_v[2] = g_CaptureSetting.GetCamera(idxCamera)->targetZ;
    }
}

TEST(ScreenCaptureTest, Basic)
{
    // デモの初期化処理。
    nn::g3d::demo::Initialize();
    nn::g3d::demo::InitializeDemo();

    int argumentCount = nnt::GetHostArgc();
    EXPECT_GE(argumentCount, 2);

    // 引数で渡された設定ファイルを読み込む
    char** pArguments = nnt::GetHostArgv();
    for (int argIndex = 1; argIndex < argumentCount; ++argIndex)
    {
        if ((pArguments[argIndex] == nullptr) || (strlen(pArguments[argIndex]) == 0))
        {
            // テストランナーから実行すると最後の引数が (null) になってしまう
            continue;
        }

        NN_LOG("argv[%d] = %s\n", argIndex, pArguments[argIndex]);
        g_CaptureSetting.LoadSetting(pArguments[argIndex]);
    }

    // デモを実行する。
    TownMain(&g_CaptureSetting);

    // デモの終了処理。
    NN_LOG("nn::g3d::demo::FinalizeDemo\n");
    nn::g3d::demo::FinalizeDemo();
    NN_LOG("nn::g3d::demo::Finalize\n");
    nn::g3d::demo::Finalize();
}
