﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <iostream>
#include <nn/fs.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/mem/mem_StandardAllocator.h>
#include <nnt/nnt_Argument.h>
#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/graphics/testGraphics_Path.h>

#if defined( NN_BUILD_CONFIG_OS_WIN )
#include <GL/glew.h>
#endif

#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
#include <nv/nv_MemoryManagement.h>
#endif

#include "framework/testEft_TestHeap.h"
#include "framework/testEft_RenderSystem.h"
#include "framework/testEft_EffectSystem.h"
#include "testEft_HandlingTestSet.h"
#include "testEft_FrameCapture.h"
#include "testEft_ProgramOptions.h"

//---------------------------------------------------------------------------
//  変数
//---------------------------------------------------------------------------
static nnt::eft::TestHeap                                   g_TestHeap;                     // テスト用ヒープ
static nnt::eft::EftRenderSystem                            g_RenderSystem;                 // 描画システム
static nnt::eft::EffectSystem                               g_EffectSystem;                 // エフェクトシステム
static nnt::eft::TextureDescriptorIndexAllocator            g_TextureDescPoolAllocator;     // テクスチャデスクリプタインデックスアロケータ
static nnt::eft::SamplerDescriptorIndexAllocator            g_SamplerDescPoolAllocator;     // サンプラーデスクリプタインデックスアロケータ
static nnt::eft::FrameCapture                               g_FrameCapture;
static nn::mem::StandardAllocator                           g_TestGraphicsStandardAllocator;

//------------------------------------------------------------------------------
//  nn::fs用メモリ取得用関数
//------------------------------------------------------------------------------
void* AllocateFs( size_t size )
{
    return malloc( size );
}

//------------------------------------------------------------------------------
//  nn::fs用メモリ解放用関数
//------------------------------------------------------------------------------
void DeallocateFs( void* p, size_t size )
{
    NN_UNUSED( size );
    free(p);
}

//------------------------------------------------------------------------------
//  テクスチャディスクリプタスロットをアロケート
//------------------------------------------------------------------------------
bool AllocateSlotForFontTexture_( nn::gfx::DescriptorSlot* dstSlot, const nn::gfx::TextureView& textureView, void* pUserData )
{
    return g_TextureDescPoolAllocator.AllocateSlotForFontTexture( dstSlot, textureView, pUserData );
}

//------------------------------------------------------------------------------
//  Font サンプラディスクリプタスロットをアロケート
//------------------------------------------------------------------------------
bool AllocateSlotForFontSampler_(nn::gfx::DescriptorSlot* dstSlot, const nn::gfx::Sampler& sampler, void* pUserData)
{
    return g_SamplerDescPoolAllocator.AllocateSlotForFontSampler( dstSlot, sampler, pUserData );
}

//---------------------------------------------------------------------------
//  エフェクトリソースの登録。
//---------------------------------------------------------------------------
uint32_t EntryResource()
{
    return g_EffectSystem.EntryResource( nnt::eft::ProgramOptions::GetInstancePointer()->ptclPath.GetString() );
}

//------------------------------------------------------------------------------
// メモリ割り当て・破棄関数
//------------------------------------------------------------------------------
void* Allocate( size_t size )
{
    return malloc( size );
}

void Deallocate( void* ptr, size_t size )
{
    NN_UNUSED( size );
    free( ptr );
}

#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
//------------------------------------------------------------------------------
// グラフィックスシステム用メモリ割り当て・破棄関数
//------------------------------------------------------------------------------
static void* NvAllocateFunction(size_t size, size_t alignment, void* userPtr)
{
    NN_UNUSED(userPtr);
    return aligned_alloc(alignment, size);
}
static void NvFreeFunction(void* addr, void* userPtr)
{
    NN_UNUSED(userPtr);
    free(addr);
}
static void* NvReallocateFunction(void* addr, size_t newSize, void* userPtr)
{
    NN_UNUSED(userPtr);
    return realloc(addr, newSize);
}
#endif

//---------------------------------------------------------------------------
//  Main 関数
//---------------------------------------------------------------------------
extern "C" void nnMain()
{
    nn::fs::SetAllocator(Allocate, Deallocate);

#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
    // グラフィックスシステムのためのメモリ周りの初期化を行います。
    {
        const size_t GraphicsSystemMemorySize = 8 * 1024 * 1024;
        nv::SetGraphicsAllocator(NvAllocateFunction, NvFreeFunction, NvReallocateFunction, NULL);
        nv::SetGraphicsDevtoolsAllocator(NvAllocateFunction, NvFreeFunction, NvReallocateFunction, NULL);
        nv::InitializeGraphics( malloc( GraphicsSystemMemorySize ), GraphicsSystemMemorySize );
    }
#endif

    int argc = ::nnt::GetHostArgc();
    char** argv = ::nnt::GetHostArgv();

    // GoogleTEST 初期化
    ::testing::InitGoogleTest(&argc, argv);

    // 引数のチェック
    nnt::eft::ProgramOptions::Parse(argc, argv);
    NN_LOG("ptcl   : \"%s\"\n", nnt::eft::ProgramOptions::GetInstancePointer()->ptclPath.GetString());
    NN_LOG("outDir : \"%s\"\n", nnt::eft::ProgramOptions::GetInstancePointer()->capturedImageDirectoryPath.GetString());
    NN_LOG("width  : %d\n", nnt::eft::ProgramOptions::GetInstancePointer()->screenWidth);
    NN_LOG("height : %d\n", nnt::eft::ProgramOptions::GetInstancePointer()->screenHeight);
    NN_ASSERT(!nnt::eft::ProgramOptions::GetInstancePointer()->capturedImageDirectoryPath.IsEmpty());
    NN_ASSERT(!nnt::eft::ProgramOptions::GetInstancePointer()->ptclPath.IsEmpty());
    NN_ASSERT(nnt::eft::ProgramOptions::GetInstancePointer()->screenWidth != 0);
    NN_ASSERT(nnt::eft::ProgramOptions::GetInstancePointer()->screenHeight != 0);

    // メモリ関連の初期化
    nn::mem::StandardAllocator allocatorForEft;
    nn::mem::StandardAllocator allocatorForGfx;
    const uint32_t memSizeForEffect         = 256 * 1024 * 1024;    // システム用メモリサイズ
    const uint32_t memSizeForGfx            = 256 * 1024 * 1024;    // GFX用メモリサイズ
    nn::util::BytePtr pEftMemoryHeap(NULL);
    nn::util::BytePtr pGfxMemoryHeap(NULL);
    {
        // エフェクトシステム用メモリブロックの取得
        pEftMemoryHeap.Reset(new uint8_t[memSizeForEffect]);
        NN_ASSERT(pEftMemoryHeap.Get());
        allocatorForEft.Initialize(pEftMemoryHeap.Get(), memSizeForEffect);

        pGfxMemoryHeap.Reset(new uint8_t[memSizeForGfx]);
        NN_ASSERT(pGfxMemoryHeap.Get());
        allocatorForGfx.Initialize(pGfxMemoryHeap.Get(), memSizeForGfx);
    }
    // ヒープを設定
    g_TestHeap.SetStandardAllocator(&allocatorForEft);

    // ファイルシステム初期化
    nn::fs::MountHostRoot();

    // 描画システムの初期化
    g_RenderSystem.Initialize(&allocatorForGfx,
        nnt::eft::ProgramOptions::GetInstancePointer()->screenWidth,
        nnt::eft::ProgramOptions::GetInstancePointer()->screenHeight);

    g_TextureDescPoolAllocator.SetDescriptorPool(g_RenderSystem.GetTextureDescriptorPool());
    g_SamplerDescPoolAllocator.SetDescriptorPool(g_RenderSystem.GetSamplerDescriptorPool());

    g_RenderSystem.InitializeDescriptors( &g_TextureDescPoolAllocator );

    // エフェクトシステムを初期化
    g_EffectSystem.Initialize(&g_TestHeap, &g_RenderSystem);
    g_EffectSystem.GetSystem()->RegisterTextureViewToDescriptorPool( AllocateSlotForFontTexture_, NULL );

    g_EffectSystem.GetSystem()->RegisterSamplerToDescriptorPool( AllocateSlotForFontSampler_, NULL );

    // シードの設定
    nn::vfx::detail::Random::SetGlobalSeed(true, 32);

    // エフェクトリソースの読み込み
    EntryResource();
    g_EffectSystem.GetSystem()->RegisterTextureViewToDescriptorPool( AllocateSlotForFontTexture_, NULL );

    // フレームキャプチャの生成
    g_FrameCapture.Initialize(
        g_RenderSystem.GetDevice(),
        g_RenderSystem.GetMemoryPoolAllocator(),
        g_RenderSystem.GetInvisibleMemoryPoolAllocator(),
        nnt::eft::ProgramOptions::GetInstancePointer()->screenWidth,
        nnt::eft::ProgramOptions::GetInstancePointer()->screenHeight);

    // GoogleTEST 実行
    int result = RUN_ALL_TESTS();

    // キャプチャ破棄
    g_FrameCapture.Finalize();

    // リソース破棄
    g_EffectSystem.ClearResource();

    // エフェクトシステムの終了処理
    g_EffectSystem.Finalize();

    // 描画システムの終了処理
    g_RenderSystem.Finalize();

    // ファイルシステムのアンマウント
    nn::fs::UnmountHostRoot();

    // アロケータ終了処理
    allocatorForEft.Finalize();
    allocatorForGfx.Finalize();

    // ヒープメモリーの終了処理
    delete[] reinterpret_cast<uint8_t*>(pEftMemoryHeap.Get());
    delete[] reinterpret_cast<uint8_t*>(pGfxMemoryHeap.Get());

    ::nnt::Exit(result);
}

//------------------------------------------------------------------------------
//  エフェクトビューアのコールバック関数の定義
//------------------------------------------------------------------------------
#if NN_GFX_IS_TARGET_GL
void _eftDrawPostCallback( const void* pParam )
{
    NN_UNUSED(pParam);

    glBindVertexArray(0);
}
#endif

//---------------------------------------------------------------------------
//  エフェクトのキャプチャを行います。
//---------------------------------------------------------------------------
void RunLoop(HandlingTestSet* pTestSet, const char* pTestName)
{
    nn::util::Vector3fType              gCameraPosition;            // カメラ位置
    nn::util::Vector3fType              gCameraLookAt;              // カメラ視点
    nn::util::Matrix4x3fType            gView;                      // モデルビュー
    nn::util::Matrix4x4fType            gProjctionMatrix;           // プロジェクション

    // カメラを初期化
    {
        nn::util::VectorSet( &gCameraPosition, 0.0f, 30.0f, 50.0f );
        nn::util::VectorSet( &gCameraLookAt, 0.0f, 0.0f, 0.0f );

        nn::util::Vector3fType camUp;
        nn::util::Vector3fType target;
        nn::util::VectorSet( &camUp,  0.0f, 1.0f, 0.0f );
        nn::util::VectorSet( &target,  0.0f, 0.0f, 0.0f );

        nn::util::MatrixIdentity( &gView );
        nn::util::MatrixLookAtRightHanded( &gView, gCameraPosition, target, camUp );
    }

    // プロジェクションの初期化
    float fovy = nn::util::FloatPi / 3.0f;
    float aspect = g_RenderSystem.GetScreenWidth() / static_cast<float>(g_RenderSystem.GetScreenHeight());
    float zNear = 0.1f;
    float zFar = 1000.0f;
    nn::util::MatrixIdentity(&gProjctionMatrix);
    nn::util::MatrixPerspectiveFieldOfViewRightHanded(&gProjctionMatrix, fovy, aspect, zNear, zFar);

    bool bLoop = true;
    int counter = 0;

    bool first = true;
    while (bLoop)
    {
        // フレーム数を設定
        pTestSet->SetFrameNum(counter);

        // コマンドを実行
        pTestSet->ExecuteCommands();

        // コマンドを生成
        g_RenderSystem.ResetCommandBuffer();

        //------------------------------------------
        // コマンドを生成
        //------------------------------------------
        g_RenderSystem.GetCommandBuffer()->Begin();
        {
            g_RenderSystem.GetCommandBuffer()->InvalidateMemory( nn::gfx::GpuAccess_Descriptor | nn::gfx::GpuAccess_ShaderCode );

            // ディスクリプタプールをセットする。
            g_RenderSystem.GetCommandBuffer()->SetDescriptorPool( g_RenderSystem.GetTextureDescriptorPool() );
            g_RenderSystem.GetCommandBuffer()->SetDescriptorPool( g_RenderSystem.GetSamplerDescriptorPool() );

            nn::gfx::ColorTargetView* pTarget = g_RenderSystem.GetColorTargetView();

            g_RenderSystem.GetCommandBuffer()->ClearColor(g_RenderSystem.GetColorTargetView(), 0.0f, 0.0f, 0.0f, 1.0f, NULL);
            g_RenderSystem.GetCommandBuffer()->ClearDepthStencil(g_RenderSystem.GetDepthStencilView(), 1.0f, 0, nn::gfx::DepthStencilClearMode_DepthStencil, NULL);
            g_RenderSystem.GetCommandBuffer()->SetViewportScissorState(g_RenderSystem.GetViewportScissorState());
            g_RenderSystem.GetCommandBuffer()->SetRenderTargets(1, &pTarget, g_RenderSystem.GetDepthStencilView());

            // エフェクトの計算
            g_EffectSystem.Calc();
            // エフェクト描画処理
            g_EffectSystem.DrawEffectSystem(gView, gProjctionMatrix, gCameraPosition, 0.1f, 10000.f);

#if NN_GFX_IS_TARGET_GL
            // ※設定していないと切り替え時にエラー
            g_RenderSystem.GetCommandBuffer()->Gl4SetUserCommand( _eftDrawPostCallback, NULL );
#endif

            g_FrameCapture.PushCaptureCommand(g_RenderSystem.GetCommandBuffer(), g_RenderSystem.GetColorBuffer(), nnt::eft::EftRenderSystem::ColorBufferImageFormat);

        }

        // カラーバッファからスキャンバッファへコピー
        nn::gfx::TextureCopyRegion region;
        region.SetDefault();
        region.SetWidth(g_RenderSystem.GetScreenWidth());
        region.SetHeight(g_RenderSystem.GetScreenHeight());
        g_RenderSystem.GetCommandBuffer()->BlitImage(g_RenderSystem.GetCurrentSwapChain()->AcquireNextScanBuffer(),
                                                     region,
                                                     g_RenderSystem.GetColorBuffer(),
                                                     region, 0);

        g_RenderSystem.GetCommandBuffer()->End();

        if (!first)
        {
            g_RenderSystem.GetQueue()->ExecuteCommand(g_RenderSystem.GetCommandBuffer(), g_RenderSystem.GetFence());
        }

        if (first)
        {
            first = false;
        }
        else
        {
            // 結果の表示
#ifndef NN_BUILD_CONFIG_OS_HORIZON
            // NOTE: 実機で垂直同期数を '0' に設定すると返ってこなくなる為、Present() を行わないようにします。
            //       ローカル環境で描画を直接確かめたい場合は '1' を設定してください。
            //       又、nvn Win 環境で Present() を行わないようにした場合は、Queue の Finalize() から返ってこなくなります。
            g_RenderSystem.GetQueue()->Present( g_RenderSystem.GetCurrentSwapChain(), 0 );
#endif
            g_RenderSystem.GetQueue()->Flush();
            g_RenderSystem.GetFence()->Sync( nn::TimeSpan::FromNanoSeconds( 1000 * 1000 * 1000 / 60 ) );

            // キャプチャ
            if ( pTestSet->DoCapture() )
            {
#if !defined(NN_PLATFORM_CAFE)
                // キャプチャ画像のファイル名を設定
                nn::vfx::Resource* resource = g_EffectSystem.GetSystem()->GetResource( g_EffectSystem.GetResourceId() );

                nnt::graphics::Path filePath("%s/%s_%s_%03d.png", nnt::eft::ProgramOptions::GetInstancePointer()->capturedImageDirectoryPath.GetString(),
                    pTestName, resource->GetEmitterSetName( 0 ), counter);

                // 書き出し
                g_FrameCapture.FetchCaptureResult();
                g_FrameCapture.SaveToPng(filePath.GetString());
#endif
            }

            if ( pTestSet->IsEnd() )
            {
                if ( g_EffectSystem.GetHandle()->IsValid() )
                {
                    g_EffectSystem.DestroyEmitterSet();
                }
                counter = 0;
                bLoop = false;
            }
        }
        counter++;
    }
}

//---------------------------------------------------------------------------
//  テスト本体 (別ファイルに記述)
//---------------------------------------------------------------------------
#include "testEft_EmitterSetHandlingTestBody.h"
