﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstddef>
#include <cstdint>
#include <nn/vfx/vfx_Heap.h>

#include <nn/mem/mem_StandardAllocator.h>

namespace nnt{
namespace eft{

//---------------------------------------------------------------------------
//! @brief        エフェクトヒープ
//---------------------------------------------------------------------------
class TestHeap : public nn::vfx::Heap
{
public:
    //---------------------------------------------------------------------------
    //! @brief        コンストラクタです。
    //---------------------------------------------------------------------------
    TestHeap()
        : m_AllocateFunction(nullptr)
        , m_DeallocateFunction(nullptr)
        , m_AllocateUserParam(nullptr)
        , m_Counter(0)
        , m_AllocedSize(0)
    {
    }

    //---------------------------------------------------------------------------
    //! @brief        コンストラクタです。
    //---------------------------------------------------------------------------
    explicit TestHeap(
        void* (*allocateFunction)(size_t size, int alignment, void* userParam),
        void (*deallocateFunction)(void* ptr, void* userParam),
        void* userParam)
    {
        m_AllocateFunction = allocateFunction;
        m_DeallocateFunction = deallocateFunction;
        m_AllocateUserParam = userParam;
    }

    //---------------------------------------------------------------------------
    //! @brief        デストラクタです。
    //---------------------------------------------------------------------------
    ~TestHeap()
    {
        Finalize();
    }

    //---------------------------------------------------------------------------
    //! @brief        nw::ut::IAllocatorインスタンスを設定します。
    //---------------------------------------------------------------------------
    void SetAllocator(
        void* (*allocateFunction)(size_t size, int alignment, void* userParam),
        void (*deallocateFunction)(void* ptr, void* userParam),
        void* userParam)
    {
        m_AllocateFunction = allocateFunction;
        m_DeallocateFunction = deallocateFunction;
        m_AllocateUserParam = userParam;
    }

    void SetStandardAllocator(nn::mem::StandardAllocator* pAllocator)
    {
        m_AllocateFunction = AllocateStandardAllocator;
        m_DeallocateFunction = DeallocateStandardAllocator;
        m_AllocateUserParam = static_cast<void*>(pAllocator);
    }

    //---------------------------------------------------------------------------
    //! @brief        メモリを確保します。
    //---------------------------------------------------------------------------
    virtual void* Alloc( size_t size, size_t alignment = nn::vfx::Heap::Alignment_Default )
    {
        void* ptr = m_AllocateFunction(size, static_cast<int>(alignment), m_AllocateUserParam);
        NN_SDK_ASSERT( ptr );
        m_Counter++;

        return ptr;
    }

    //---------------------------------------------------------------------------
    //! @brief        メモリを解放します。
    //---------------------------------------------------------------------------
    virtual void Free( void* ptr )
    {
        NN_SDK_ASSERT( ptr );

        if ( m_Counter > 0 )
        {
            m_Counter--;
        }
        else
        {
            NN_SDK_ASSERT( "[EFTUT] Heap Free Error! " );
        }

        m_DeallocateFunction(ptr, m_AllocateUserParam);
    }

    //---------------------------------------------------------------------------
    //! @brief        メモリ確保カウンターを取得します。
    //---------------------------------------------------------------------------
    int32_t GetCounter() const
    {
        return m_Counter;
    }

    //---------------------------------------------------------------------------
    //! @brief        確保したメモリサイズを取得します。
    //---------------------------------------------------------------------------
    int32_t GetAllocedSize() const
    {
        return m_AllocedSize;
    }

    //---------------------------------------------------------------------------
    //! @brief        終了処理
    //---------------------------------------------------------------------------
    void Finalize()
    {
        m_AllocateFunction = nullptr;
        m_DeallocateFunction = nullptr;
        m_AllocateUserParam = nullptr;
        m_Counter = 0;
        m_AllocedSize = 0;
    }

    // StandardAllocator 用
    static void* AllocateStandardAllocator(size_t size, int alignment, void* userParam)
    {
        nn::mem::StandardAllocator* allocator = reinterpret_cast<nn::mem::StandardAllocator*>(userParam);
        return allocator->Allocate(size, static_cast<size_t>(alignment));
    }
    static void DeallocateStandardAllocator(void* ptr, void* userParam)
    {
        nn::mem::StandardAllocator* allocator = reinterpret_cast<nn::mem::StandardAllocator*>(userParam);
        allocator->Free(ptr);
    }

private:
    void* (*m_AllocateFunction)(size_t size, int alignment, void* userParam);
    void (*m_DeallocateFunction)(void* ptr, void* userParam);
    void* m_AllocateUserParam;
    int32_t m_Counter;
    int32_t m_AllocedSize;
};

}}
