﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/eft/eft2_System.h>
#include <nw/eft/eft2_Misc.h>
#include <nw/eft/eftvw2_FileSystem.h>
#include <nnt.h>
#include "nwDemo/System.h"
#include "nwDemo/FrameBuffer.h"
#include "nwUtil/TestHeap.h"
#include "testEft_FileManager.h"
#include "testEft_PngFile.h"
#include "ScreenCapture.h"
#include "testEft_HandlingTestSet.h"
#include "util/PathUtility.h"

#if EFT_IS_WIN
#include <nn/nn_Windows.h>
#include <shlwapi.h>
#endif

#if (defined(EFT_IS_CAFE) && EFT_IS_CAFE)
enum{
    MaxPathLength = 256,
};

    #define sprintf_s sprintf
#else
enum{
    MaxPathLength = 512,
};
#endif


//---------------------------
//  定数
//---------------------------

static const int ScreenWidth  = 640;
static const int ScreenHeight = 480;

#if EFT_IS_WIN
    static const char* PLATFORM_NAME = "Win32";   // プラットフォーム名
#endif
#if EFT_IS_CAFE
    static const char* PLATFORM_NAME = "Cafe";  // プラットフォーム名
#endif


//---------------------------
//  変数
//---------------------------
static nw::ut::MemoryAllocator      g_NwAllocator;                  // Allocator
static nw::eftdemo::System*         g_DemoSystem = NULL;            // DemoSystem
nw::eft2::System*                   g_EffectSystem = NULL;          // Effect System
static nw::dev::PrimitiveRenderer*  g_PrimitiveRenderer = NULL;     // PrimitiveRenderer
static TestHeap                     g_TestHeap;                     // テスト用ヒープ
static nw::eftdemo::FrameBuffer     g_FrameBuffer64;                // デフォルトフレームバッファ
static ScreenCapture                g_ScreenCapture;                // スクリーンキャプチャ
static nw::math::VEC3               g_CameraPosition;               // カメラ位置
static nw::math::VEC3               g_CameraLookAt;                 // カメラ視点
static nw::math::Matrix34           g_View;                         // モデルビュー
static nw::math::MTX44              g_ProjctionMatrix;              // プロジェクション
static nw::eft2::Handle             g_EsetHandle;                   // エフェクト操作ハンドル
static void*                        g_PtclResource = NULL;          // PTCLリソース
static char                         g_OutDir[MaxPathLength]      = "";              // 出力先ディレクトリ
static char                         g_PtclPath[MaxPathLength]    = "";              // エフェクトリソースパス

//---------------------------------------------------
//  カスタムアクション
//---------------------------------------------------
class CustomActionTest
{
public:
    //------------------------------------------------------------------------------
    //  初期化
    //------------------------------------------------------------------------------
    static void Initialize( nw::eft2::System* system )
    {
        nw::eft2::CallbackSet set;
        set.particleEmit      = _ParticleEmitCallback;
        system->SetCallback( nw::eft2::EFT_CUSTOM_ACTION_CALLBACK_ID_2, set );
    }

    //------------------------------------------------------------------------------
    //  パーティクル生成後コールバック
    //------------------------------------------------------------------------------
    static bool _ParticleEmitCallback( nw::eft2::ParticleEmitArg& arg )
    {
        nw::eft2::ResCustomActionData* actionParam = reinterpret_cast<nw::eft2::ResCustomActionData*>(arg.emitter->GetCustomActionParameter());
        arg.SetWorldPos( actionParam->fValue[0], actionParam->fValue[1], actionParam->fValue[2] );
        return true;
    }
};



//---------------------------------------------------
//  カスタムシェーダ
//---------------------------------------------------
class CustomShaderTest
{
public:
    //------------------------------------------------------------------------------
    //  初期化
    //------------------------------------------------------------------------------
    static void Initialize( nw::eft2::System* system )
    {
        nw::eft2::CallbackSet set;
        set.renderStateSet      = _RenderStateSetCallback;
        system->SetCallback( nw::eft2::EFT_CUSTOM_SHADER_CALLBACK_ID_1, set );
    }

    //------------------------------------------------------------------------------
    //  描画設定後コールバック
    //------------------------------------------------------------------------------
    static bool _RenderStateSetCallback( nw::eft2::RenderStateSetArg& arg )
    {
        if ( !arg.emitter->emitterRes->customShaderParam ) return false;

        nw::eft2::System* system = arg.emitter->emitterSet->GetSystem();
        nw::eft2::Shader* shader = arg.GetShader();

        // ユニフォームブロック
        struct CustomShaderSampleUniformBlock0
        {
            nw::math::VEC4      fogParam;
            nw::math::VEC4      fogColor;
            nw::math::VEC4      lightVec;
            nw::math::Matrix44  lightViewProj;
        };
        CustomShaderSampleUniformBlock0* ubo0 = static_cast<CustomShaderSampleUniformBlock0 *>( system->AllocFromTempBuffer( sizeof(CustomShaderSampleUniformBlock0) ) );
        if ( !ubo0 ) return false;

        ubo0->fogParam.x   = 100.f;
        ubo0->fogParam.y   = 200.f;
        ubo0->fogParam.z   = 0.f;
        ubo0->fogParam.w   = 0.f;
        ubo0->fogColor.x   = 0.2f;
        ubo0->fogColor.y   = 0.2f;
        ubo0->fogColor.z   = 0.2f;
        ubo0->fogColor.w   = 1.0f;
        ubo0->lightVec.x   = 0.0f;
        ubo0->lightVec.y   = 200.0f;
        ubo0->lightVec.z   = 200.0f;
        ubo0->lightVec.z   = 0.0f;
        shader->BindCustomShaderUniformBlock( nw::eft2::EFT_CUSTOM_SHADER_UBO_0, ubo0, sizeof(CustomShaderSampleUniformBlock0) );

        // EffectMakerで設定したパラメータをユニフォームブロックとして設定
        shader->BindReservedCustomShaderUniformBlock( arg.emitter );

        return true;
    }
};


//---------------------------
//  エフェクトの初期化。
//---------------------------
void InitializeEffectSystem( nw::eft2::Heap* heap )
{
    nw::eft2::Config  config;
    config.SetEffectHeap( heap );
    config.SetEffectDynamicHeap( heap );
    g_EffectSystem = new nw::eft2::System( config );
    CustomShaderTest::Initialize( g_EffectSystem );
    CustomActionTest::Initialize( g_EffectSystem );
}

//---------------------------
//  エフェクトの終了処理。
//---------------------------
void FinalizeEffectSystem( nw::eft2::Heap* heap )
{
    NW_UNUSED_VARIABLE( heap );

    if ( g_EffectSystem ) delete g_EffectSystem;
    g_EffectSystem = NULL;
}

//---------------------------------------------------------------------------
//  エフェクトリソースの登録。
//---------------------------------------------------------------------------
u32 EntryResource( nw::eft2::Heap* heap )
{
    u32 resId = 0;
    u32 fileSize = 0;

    bool resLoad = nw::eftvw2::FileSystem::Load( heap, g_PtclPath, &g_PtclResource, &fileSize );
    EFT_ASSERT( resLoad == true );

    g_EffectSystem->EntryResource( heap, g_PtclResource, resId );

    return g_EffectSystem->GetResource( resId )->GetEmitterSetNum();
}

//---------------------------------------------------------------------------
//  エフェクトリソースの破棄。
//---------------------------------------------------------------------------
void ClearResource( nw::eft2::Heap* heap )
{
    g_EffectSystem->ClearResource( heap, 0 );
    heap->Free( g_PtclResource );
}

//---------------------------------------------------------------------------
//  エフェクトリソースの登録。
//---------------------------------------------------------------------------
void CreateEmitterSet( nw::eft2::Handle* handle, u32 res, u32 idx )
{
    handle->Invalidate();
    g_EffectSystem->CreateEmitterSetID( handle, idx, res, 0 );
}

//---------------------------------------------------------------------------
//  エフェクトリソースの破棄。
//---------------------------------------------------------------------------
void DestroyEmitterSet( nw::eft2::Handle* handle )
{
    g_EffectSystem->KillEmitterSet( handle->GetEmitterSet() );
}

//---------------------------------------------------------------------------
//  エフェクトの定期処理。
//---------------------------------------------------------------------------
void CalcEffectSystem()
{
    // エフェクトランタイムの計算処理
    g_EffectSystem->BeginFrame();
    g_EffectSystem->Calc( 0, 1.0f, true );
}

//---------------------------------------------------------------------------
//  エフェクトの描画処理。
//---------------------------------------------------------------------------
void DrawEffectSystem( nw::math::Matrix34* viewMatrix,
                      nw::math::Matrix44* projMatrix,
                      nw::math::VEC3*     eyePos,
                      f32                 screenNear,
                      f32                 screenFar )
{
    #define DRAW_PATH_OPAQU_BUFFER              ( nw::eft2::EFT_DRAW_PATH_FLAG_3 )
    #define DRAW_PATH_PRE_COPY_COLOR_BUFFER     ( nw::eft2::EFT_DRAW_PATH_FLAG_2 )
    #define DRAW_PATH_DEFAULT                   ( nw::eft2::EFT_DRAW_PATH_FLAG_0 )
    #define DRAW_PATH_REDUCTION_BUFFER          ( nw::eft2::EFT_DRAW_PATH_FLAG_1 )

    nw::eft2::ViewParam view;
    view.Set( viewMatrix, projMatrix, eyePos, screenNear, screenFar );
    g_EffectSystem->SwapBuffer();
    g_EffectSystem->SetViewParam( &view );
    g_EffectSystem->Draw( 0, DRAW_PATH_OPAQU_BUFFER );
    g_EffectSystem->Draw( 0, DRAW_PATH_PRE_COPY_COLOR_BUFFER );
    g_EffectSystem->Draw( 0, DRAW_PATH_DEFAULT );
    g_EffectSystem->Draw( 0, DRAW_PATH_REDUCTION_BUFFER );
}


//---------------------------------------------------------------------------
//  デバッグ文字列を表示する。
//---------------------------------------------------------------------------
void DebugDraw( nw::dev::DevTextWriter* writer )
{
#if EFT_IS_WIN
    glEnable( GL_BLEND );
    glBlendFunc( GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA );
    glBlendEquationEXT( GL_FUNC_ADD_EXT );
#endif

    writer->SetCursor( 10, 10 );
    writer->Printf( "CpuPtcl   (%8d)\n",            g_EffectSystem->GetProcessingCpuParticleCount() );
    writer->Printf( " GpuPtcl   (%8d)\n",           g_EffectSystem->GetProcessingGpuParticleCount() );
    writer->Printf( " GpuSoPtcl (%8d)\n",           g_EffectSystem->GetProcessingGpuSoParticleCount() );
    writer->Printf( " WARNING:  (0x%08x)\n",        nw::eft2::GetWarnings() );

    writer->Flush();
}

//---------------------------------------------------------------------------
//  文字列の置き換え
//---------------------------------------------------------------------------
std::string ReplaceStrings( std::string String1, std::string String2, std::string String3 )
{
    std::string::size_type  Pos( String1.find( String2 ) );

    while( Pos != std::string::npos )
    {
        String1.replace( Pos, String2.length(), String3 );
        Pos = String1.find( String2, Pos + String3.length() );
    }

    return String1;
}

//---------------------------------------------------------------------------
//  スクリーンキャプチャ。
//---------------------------------------------------------------------------
void CaptureScreen( const char* pTestName, u32 emitterSetID, u32 frame )
{
    // キャプチャ画像のファイル名を設定
    nw::eft2::Resource* resource = g_EffectSystem->GetResource( 0 );

    char filePath[MaxPathLength];
    sprintf_s( filePath, "%s/%s_%s_%03d.png", g_OutDir, pTestName, resource->GetEmitterSetName( emitterSetID ), frame );

    // "/" では PathCanonicalizeA() が正しく動作しないので "\\" に変換
    std::string outString = ReplaceStrings(filePath, "/", "\\");
    strcpy( filePath, outString.c_str() );

    char filePathCanonicalize[MaxPathLength];
    PathCanonicalizeA( filePathCanonicalize, filePath );

    // キャプチャ実行
    g_ScreenCapture.Capture();

    // キャプチャ画像を取得
    int width         = g_ScreenCapture.GetWidth();
    int height        = g_ScreenCapture.GetHeight();
    const u8* pPixels = g_ScreenCapture.GetPixels();

    // キャプチャ画像を保存
    WritePng( width, height, pPixels, filePathCanonicalize );
}

//------------------------------------------------------------------------------
//  初期化処理
//------------------------------------------------------------------------------
void Initialize()
{
    // ファイルシステムの初期化
    nw::dev::FileDeviceManager* fileSystem = nw::dev::FileDeviceManager::GetInstance();
    nw::dev::FileDeviceManager::CreateArg fileDeviceArg;
    fileDeviceArg.allocator = &g_NwAllocator;
    fileSystem->Initialize( fileDeviceArg );

    // デモシステムの作成
    nw::eftdemo::System::CreateArg    arg;

    arg.allocator   = &g_NwAllocator;
    arg.width       = ScreenWidth;
    arg.height      = ScreenHeight;
    arg.drawMeter   = true;
    arg.fontPath    = "common/fonts/nintendo_NTLG-DB_002.bffnt";
    arg.clearColor  = 0x333333FF;

#if defined(NW_PLATFORM_WIN32)
    arg.waitVBlank  = 0;
#endif

#if defined(NW_PLATFORM_CAFE)
    arg.waitVBlank  = 1;
    arg.fontShaderPath = "common/shaders/font_BuildinShader.gsh";
#endif

    nw::eftdemo::System::LoadConfigFile(&g_NwAllocator, &arg);

    g_DemoSystem = new nw::eftdemo::System( arg );

    // デモシステムの初期化
    g_DemoSystem->Initialize();

    // 入力インターフェースの初期化
    g_DemoSystem->InitializeInputInterface();

    // グラフィックシステムの初期化
    g_DemoSystem->InitializeGraphicsSystem();

    nw::gfnd::Graphics::GetInstance()->LockDrawContext();
    {
        // PrimitiveRenderer の初期化
        g_PrimitiveRenderer = nw::dev::PrimitiveRenderer::GetInstance();

#if defined(NW_PLATFORM_WIN32)
        g_PrimitiveRenderer->Initialize( &g_NwAllocator );
#endif
#if defined(NW_PLATFORM_CAFE)
        // PrimitiveRenderer で用いるシェーダーバイナリへのパスを指定する。
        nw::dev::FileDeviceManager* fileSystem = nw::dev::FileDeviceManager::GetInstance();
        nw::dev::FileDevice::LoadArg loadArg;
        loadArg.path = "common/shaders/dev_PrimitiveRenderer.gsh";
        loadArg.allocator = &gNwAllocator;
        u8* binary = fileSystem->Load( loadArg );
        gPrimitiveRenderer->InitializeFromBinary( &gNwAllocator, binary, loadArg.readSize );
        fileSystem->Unload( loadArg, binary );
#endif
    }
    nw::gfnd::Graphics::GetInstance()->UnlockDrawContext();
}

//------------------------------------------------------------------------------
//  終了処理
//------------------------------------------------------------------------------
void Finalize()
{
    // PrimitiveRenderer終了処理
    g_PrimitiveRenderer->Finalize( &g_NwAllocator );

    // デモシステム終了処理
    g_DemoSystem->FinalizeGraphicsSystem();
    g_DemoSystem->FinalizeInputInterface();
    g_DemoSystem->Finalize();

    // ファイルシステム終了処理
    nw::dev::FileDeviceManager* fileSystem = nw::dev::FileDeviceManager::GetInstance();
    fileSystem->Finalize();
}

//---------------------------------------------------------------------------
//
//---------------------------------------------------------------------------
static void AnalyzeCommandLineOption(int argc, char **argv)
{
    // 引数解析
    for (int i = 1; i < argc; i++)
    {
        if (strcmp(argv[i], "-outDir") == 0)
        {
            if (i + 1 < argc)
            {
                sprintf(g_OutDir, "%s", PathUtility::GetAbsolutePathFromCurrentDirectory(argv[i + 1]).c_str());
                i++;
            }
            continue;
        }
        // エフェクトリソース
        if (strcmp(argv[i], "-ptcl") == 0)
        {
            if (i + 1 < argc)
            {
                sprintf(g_PtclPath, "%s", PathUtility::GetFilePathFromCurrentDirectory(argv[i + 1]).c_str());
                i++;
            }
            continue;
        }
    }
}

//------------------------------------------------------------------------------
//  Main 関数
//------------------------------------------------------------------------------
#if defined(NW_PLATFORM_WIN32)
int PCSDKMain(int argc, char **argv)
#elif defined(NW_PLATFORM_CAFE)
int main(int argc, char **argv)
#endif
{
    if (argc == 1)
    {
        return 0;
    }

    // GoogleTEST 初期化
    ::testing::InitGoogleTest(&argc, argv);

    AnalyzeCommandLineOption(argc, argv);

    std::tr2::sys::create_directories(std::tr2::sys::path(g_OutDir));

#if ( EFT_IS_WIN && defined(_DEBUG) && _DEBUG )
    _CrtSetDbgFlag(_CRTDBG_LEAK_CHECK_DF | _CRTDBG_ALLOC_MEM_DF);
#endif

    // アロケーターの初期化
    const size_t memSize = 512 * 1024 * 1024;
#if defined(NW_PLATFORM_WIN32)
    void* addr = malloc( memSize );
#else
    void* addr = MEMAllocFromDefaultHeap( HEAP_SIZE );
#endif
    g_NwAllocator.Initialize( addr, memSize );

    //----------------------------------------
    // フレームワーク初期化処理
    //----------------------------------------
    Initialize();

    g_TestHeap.SetNwAllocator( &g_NwAllocator );

    FileManager::GetInstance().SetHeap( &g_TestHeap );

    //----------------------------------------
    // フレームバッファの初期化
    //----------------------------------------
    f32 width  = ScreenWidth;
    f32 height = ScreenHeight;

    nw::gfnd::Graphics::GetInstance()->LockDrawContext();
    {
        nw::ut::MemoryAllocator* allocator = &g_NwAllocator;

        // 標準フレームバッファ Float
        g_FrameBuffer64.Initialize( allocator, (s32)width, (s32)height, nw::eftdemo::FrameBuffer::FRAMEBUFFER_TYPE_FLOAT16 );

        // スクリーンキャプチャを初期化
        g_ScreenCapture.Initialize( &g_TestHeap, &g_FrameBuffer64 );
    }
    nw::gfnd::Graphics::GetInstance()->UnlockDrawContext();

    //----------------------------------------
    // カメラを初期化
    //----------------------------------------
    g_CameraPosition.Set( 0.0f, 30.0f, 50.0f );
    g_CameraLookAt.Set  ( 0.0f, 00.0f, 0.0f );

    nw::math::VEC3 comUp ( 0.0f, 1.0f, 0.0f );
    nw::math::VEC3 target( 0.0f, 0.0f, 0.0f );

    g_View.Identity();
    g_View.SetLookAt( g_CameraPosition, comUp, target );

    //----------------------------------------
    // プロジェクションの初期化
    //----------------------------------------
    nw::math::MTX44Perspective( &g_ProjctionMatrix, nw::math::F_PI / 3, width / (f32)height, 0.1f, 1000.f );

    //----------------------------------------
    // リソースの初期化
    //----------------------------------------
    nw::gfnd::Graphics::GetInstance()->LockDrawContext();
    {
        // エフェクトシステムの初期化とリソース読込
        InitializeEffectSystem( &g_TestHeap );

        // 乱数を固定する
        nw::eft2::Random::SetGlobalSeed( TRUE, 32 );

        // リソース読み込み
        EntryResource( &g_TestHeap );
    }
    nw::gfnd::Graphics::GetInstance()->UnlockDrawContext();

    //----------------------------------------
    // メイン処理
    //----------------------------------------

    // GoogleTEST 実行
    int result = RUN_ALL_TESTS();

    // スクリーンキャプチャを破棄
    g_ScreenCapture.Release();

    // リソース破棄
    ClearResource( &g_TestHeap );

    // フレームワークの終了処理
    Finalize();

    // アロケータ終了処理
    g_NwAllocator.Finalize();
    free( addr );

    ::nnt::Exit(result);
    return result;
}

//---------------------------------------------------------------------------
//! @brief  エフェクトのキャプチャを行います。
//---------------------------------------------------------------------------
void RunLoop(HandlingTestSet* pTestSet, const char* pTestName)
{
    bool bLoop = true;
    int counter = 0;

    while ( !g_DemoSystem->IsExiting() && bLoop )
    {
        nw::gfnd::Graphics::GetInstance()->LockDrawContext();
        nw::eftdemo::FrameBuffer* currentFrameBuffer = &g_FrameBuffer64;
        {
            pTestSet->SetFrameNum( counter );

            // 書き込み先フレームバッファの選定
            currentFrameBuffer->Bind();
            currentFrameBuffer->SetClearColor( nw::math::VEC4( 0.0f, 0.0f, 0.0f, 0.0f) );
            currentFrameBuffer->Clear( true, true );

            // コマンドを実行
            pTestSet->ExecuteCommands();

            // エフェクトの計算/描画処理
            CalcEffectSystem();

        #if EFT_IS_CAFE
            GX2SetShaderMode( GX2_SHADER_MODE_UNIFORM_BLOCK );
            GX2Invalidate( static_cast<GX2InvalidateType>( GX2_INVALIDATE_SHADER ), NULL, 0xFFFFFFFF );
        #endif

            DrawEffectSystem( &g_View, &g_ProjctionMatrix, &g_CameraPosition, 0.1f, 10000.f );

        #if EFT_IS_CAFE
            GX2SetShaderMode( GX2_SHADER_MODE_UNIFORM_REGISTER );
            GX2Invalidate( static_cast<GX2InvalidateType>( GX2_INVALIDATE_SHADER ), NULL, 0xFFFFFFFF );
        #endif

            //if ( gDemoSystem->GetTextWriterInitialized() )
            //{ DebugDraw( gDemoSystem->GetTextWriter() ); }

            // キャプチャ
            if ( pTestSet->DoCapture() )
            {
                CaptureScreen( pTestName, 0, counter );
            }

            if ( pTestSet->IsEnd() )
            {
                DestroyEmitterSet( &g_EsetHandle );
                g_EsetHandle.Invalidate();
                bLoop = false;
                counter = 0;
            }

            // ForceCalc の動作確認を兼ねたキャプチャもよいかも。
        }
        nw::gfnd::Graphics::GetInstance()->UnlockDrawContext();

        // バッファのスワップ
        g_DemoSystem->SwapBuffer( currentFrameBuffer );

        //  VBlank 待ち
        g_DemoSystem->WaitForFlip();

        counter++;
    }
}

//------------------------------------------------------------------------------
//  テスト本体 (別ファイルに記述)
//------------------------------------------------------------------------------
#include "testEft_EmitterSetHandlingTestBody.h"
