﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "testGraphics_PngIO.h"

#include <new>
#include <sstream>
#include <zlib.h>
#include <nn/nn_Assert.h>
#include <nn/fs.h>
#include <nn/nn_Windows.h>

namespace nnt{ namespace graphics{

//------------------------------------------------------------------------------
//  PNGファイル読み込み
//------------------------------------------------------------------------------
bool PngIO::ReadPng( uint8_t** pImage, const char* pPath, PNG_IHDR& ihdr )
{
    // ファイルからデータを入力
    uint8_t* pData;
    uint32_t size;
    {
        nn::Result result;
        nn::fs::FileHandle file = {};
        int64_t fileSize = 0;
        result = nn::fs::OpenFile(&file, pPath, nn::fs::OpenMode_Read);
        NN_ASSERT(result.IsSuccess());
        result= nn::fs::GetFileSize(&fileSize, file);
        NN_ASSERT(result.IsSuccess());
        size = static_cast<uint32_t>(fileSize);
        pData = reinterpret_cast<uint8_t*>(NNT_GRAPHICS_ALLOCATE(size, NN_ALIGNOF(int)));
        NN_ASSERT_NOT_NULL(pData);
        result = nn::fs::ReadFile(file, 0, pData, size);
        NN_ASSERT(result.IsSuccess());
        nn::fs::CloseFile(file);
    }

    uint8_t* dataHead = pData;

    uint8_t pixelDepth;  // ピクセル深度
    uint32_t rowBytes;

    // PNGヘッダを入力
    {
        uint8_t header[8];
        InputBytes( &pData, &header, sizeof( header ) );
    }

    // IHDRチャンクを出力
    {
        // チャンク長を入力
        Input32( &pData );

        // チャンクデータを入力
        uint8_t ihdrHeader[4];
        InputBytes( &pData, ihdrHeader, sizeof( ihdrHeader ) );
        ihdr.width         = Input32( &pData );
        ihdr.height        = Input32( &pData );
        ihdr.bitDepth      = Input8( &pData );
        ihdr.colorType     = Input8( &pData );
        ihdr.compressionType = Input8( &pData );
        ihdr.filterType    = Input8( &pData );
        ihdr.interlaceType = Input8( &pData );

        // CRCを入力
        Input32( &pData );

        switch (ihdr.colorType )
        {
        case PNG_COLOR_TYPE_GRAY:
        case PNG_COLOR_TYPE_PALETTE:
            ihdr.channels = 1;
            break;
        case PNG_COLOR_TYPE_RGB:
            ihdr.channels = 3;
            break;
        case PNG_COLOR_TYPE_GRAY_ALPHA:
            ihdr.channels = 2;
            break;
        case PNG_COLOR_TYPE_RGB_ALPHA:
            ihdr.channels = 4;
            break;
        default:
            ihdr.channels = 3;
            break;
        }

        pixelDepth = ihdr.bitDepth * ihdr.channels;
        rowBytes = ((ihdr.width * static_cast<uint32_t>(pixelDepth) + 7) >> 3);

    }

    // 圧縮データのバッファを作成
    uint32_t decompressBufferSize = (rowBytes) * ihdr.height;
    *pImage = static_cast<uint8_t*>(AllocateMemory(decompressBufferSize, NN_ALIGNOF(int)) );

    // IDATチャンクを入力
    {
        // チャンク長を入力
        uint32_t length = 0;

        // チャンクデータを入力
        typedef struct Chunk
        {
            uint8_t* pBuf;
            size_t size;
        } Chunk;
        const int maxChunkCount = 255;
        Chunk chunkArray[maxChunkCount] = {};
        uint8_t* pCompressBuffer = nullptr;
        int idxIdatChunk = 0;
        bool isIdatEnd = false;
        while( !isIdatEnd )
        {
            uint32_t chunkLength = Input32( &pData );
            length += chunkLength;
            uint8_t  idatHeader[4];
            InputBytes( &pData, idatHeader, sizeof( idatHeader ) );

            chunkArray[idxIdatChunk].pBuf = NNT_GRAPHICS_NEW(chunkLength, NN_ALIGNOF(int)) uint8_t;
            chunkArray[idxIdatChunk].size = chunkLength;
            InputBytes( &pData, chunkArray[idxIdatChunk].pBuf, chunkLength );

            // CRCを入力
            Input32( &pData );

            char header[5] = {};
            memcpy( header, pData + 4 , 4 );    // 次のチャンクヘッダを取得
            if( std::string( header ) != "IDAT" )
            {
                if( idxIdatChunk == 0 )
                {
                    pCompressBuffer = chunkArray[idxIdatChunk].pBuf;
                }
                else
                {
                    pCompressBuffer = NNT_GRAPHICS_NEW(length, NN_ALIGNOF(int)) uint8_t;
                    uint8_t* ptr = pCompressBuffer;
                    for( int idxBuf = 0; idxBuf <= idxIdatChunk; ++idxBuf )
                    {
                        memcpy( ptr, chunkArray[idxBuf].pBuf, chunkArray[idxBuf].size );
                        ptr += chunkArray[idxBuf].size;
                        NNT_GRAPHICS_DELETE( chunkArray[idxBuf].pBuf );
                        chunkArray[idxBuf] = Chunk();
                    }
                }
                isIdatEnd = true;
            }

            ++idxIdatChunk;
            NN_ASSERT( idxIdatChunk < maxChunkCount, "Exceeded max IDAT chunk count\n" );
        }

        // zstreamを作成
        z_stream zstream;

        zstream.zalloc = Z_NULL;
        zstream.zfree = Z_NULL;
        zstream.opaque = Z_NULL;

        uint32_t filtedImageSize = (1 + rowBytes) * ihdr.height;
        uint8_t* pFilteredImage = NNT_GRAPHICS_NEW(filtedImageSize, NN_ALIGNOF(int)) uint8_t;

        zstream.next_in   = pCompressBuffer;
        zstream.avail_in  = length;
        zstream.next_out  = pFilteredImage;
        zstream.avail_out = filtedImageSize;

        int res = inflateInit2(&zstream, MAX_WBITS);
        NN_ASSERT(res == Z_OK);

        // データを解凍
        res = inflate(&zstream, Z_FINISH);
        NN_ASSERT(res == Z_STREAM_END);

        // zstreamを破棄
        res = inflateEnd(&zstream);
        NN_ASSERT(res == Z_OK);

        // フィルターを展開
        DecompressFilter( pFilteredImage, ihdr.width, ihdr.height, pixelDepth / 8, *pImage );

        NNT_GRAPHICS_DELETE( pCompressBuffer );
        NNT_GRAPHICS_DELETE( pFilteredImage );
    }

    // IENDチャンクを入力
    {
        // チャンク長を入力
        Input32( &pData );

        // チャンクデータを入力
        uint8_t  iendHeader[4];
        InputBytes( &pData, iendHeader, sizeof( iendHeader ) );

        // CRCを入力
        Input32( &pData );
    }

    FreeMemory(dataHead);

    return true;
}// NOLINT(impl/function_size);

//------------------------------------------------------------------------------
//  PNGファイル読み込み
//------------------------------------------------------------------------------
bool PngIO::ReadPng( uint8_t** pImage, const char* pPath)
{
    PNG_IHDR ihdr;
    return ReadPng( pImage, pPath, ihdr );
}

//------------------------------------------------------------------------------
//  PNGファイル書き込み
//------------------------------------------------------------------------------
bool PngIO::WritePng( const char* pPath, const uint8_t* pImage, int width, int height, uint8_t bitDepth, uint8_t colorType, uint8_t compressType, uint8_t filterType, uint8_t interlaceType )
{
    // メモリーストリーム
    std::ostringstream stream;

    uint8_t channels = 4; // チャンネル数

    // PNGヘッダを出力
    {
        uint8_t header[] = { 0x89, 0x50, 0x4e, 0x47, 0x0d, 0x0a, 0x1a, 0x0a };
        OutputBytes( &stream, header, sizeof( header ) );
    }

    switch (colorType )
    {
    case PNG_COLOR_TYPE_GRAY:
    case PNG_COLOR_TYPE_PALETTE:
        channels = 1;
        break;
    case PNG_COLOR_TYPE_RGB:
        channels = 3;
        break;
    case PNG_COLOR_TYPE_GRAY_ALPHA:
        channels = 2;
        break;
    case PNG_COLOR_TYPE_RGB_ALPHA:
        channels = 4;
        break;
    default:
        channels = 3;
        break;
    }

    uint8_t pixelDepth = bitDepth * channels;
    uint32_t rowBytes = ((width * static_cast<uint32_t>(pixelDepth) + 7) >> 3);

    // IHDRチャンクを出力
    {
        uint32_t ihdrLength        = 13;
        uint8_t  ihdrHeader[]      = { 'I', 'H', 'D', 'R' };

        // CRCを初期化
        uint32_t crc = crc32( 0L, Z_NULL, 0 );

        // チャンク長を出力
        Output32( &stream, &ihdrLength );

        // チャンクデータを出力
        OutputBytes( &stream, ihdrHeader, sizeof( ihdrHeader ), &crc );
        Output32( &stream, &width, &crc );
        Output32( &stream, &height, &crc );
        Output8( &stream, &bitDepth, &crc );
        Output8( &stream, &colorType, &crc );
        Output8( &stream, &compressType, &crc );
        Output8( &stream, &filterType, &crc );
        Output8( &stream, &interlaceType, &crc );

        // CRCを出力
        Output32( &stream, &crc );
    }

    // 圧縮データのバッファを作成
    uint32_t compressBufferSize = (1 + rowBytes) * height + 128 * 1024;  // 念のため少し大きめのバッファサイズを確保
    uint8_t* pCompressBuffer = NNT_GRAPHICS_NEW(compressBufferSize, NN_ALIGNOF(int)) uint8_t;

    uint32_t compressedSize;

    // ピクセルデータを圧縮
    {
        // zstreamを作成
        z_stream zstream;

        zstream.zalloc = Z_NULL;
        zstream.zfree = Z_NULL;
        zstream.opaque = Z_NULL;

        int res = deflateInit(&zstream, Z_DEFAULT_COMPRESSION);
        NN_ASSERT(res == Z_OK);

        // 圧縮率を上げるためピクセルデータにフィルターをかける
        uint32_t filteredSize = (1 + rowBytes) * height;
        uint8_t* pFilteredImage = NNT_GRAPHICS_NEW(filteredSize, NN_ALIGNOF(int)) uint8_t;

        CompressFilter( pImage, width, height, pixelDepth / 8, pFilteredImage );

        zstream.next_in   = pFilteredImage;
        zstream.avail_in  = filteredSize;
        zstream.next_out  = pCompressBuffer;
        zstream.avail_out = compressBufferSize;

        // データを圧縮
        res = deflate(&zstream, Z_NO_FLUSH);
        NN_ASSERT(res == Z_OK);

        res = deflate(&zstream, Z_FINISH);
        NN_ASSERT(res == Z_STREAM_END);

        compressedSize = zstream.total_out;

        // zstreamを破棄
        res = deflateEnd(&zstream);
        NN_ASSERT(res == Z_OK);

        NNT_GRAPHICS_DELETE( pFilteredImage );
    }

    // IDATチャンクを出力
    {
        uint32_t idatLength = compressedSize;
        uint8_t  idatHeader[] = { 'I', 'D', 'A', 'T' };

        // CRCを初期化
        uint32_t crc = crc32( 0L, Z_NULL, 0 );

        // チャンク長を出力
        Output32( &stream, &idatLength );

        // チャンクデータを出力
        OutputBytes( &stream, idatHeader, sizeof( idatHeader ), &crc );
        OutputBytes( &stream, pCompressBuffer, compressedSize, &crc );

        // CRCを出力
        Output32( &stream, &crc );
    }

    // 圧縮データを破棄
    NNT_GRAPHICS_DELETE( pCompressBuffer );

    // IENDチャンクを出力
    {
        uint32_t iendLength = 0;
        uint8_t  iendHeader[] = { 'I', 'E', 'N', 'D' };

        // CRCを初期化
        uint32_t crc = crc32( 0L, Z_NULL, 0 );

        // チャンク長を出力
        Output32( &stream, &iendLength );

        // チャンクデータを出力
        OutputBytes( &stream, iendHeader, sizeof( iendHeader ), &crc );

        // CRCを出力
        Output32( &stream, &crc );
    }

    // ファイルにデータを保存
    {
        nn::Result result;
        size_t size = stream.str().size();
        nn::fs::FileHandle file = {};
        //result = nn::fs::DeleteFile(pPath);
        //NN_ASSERT(result.IsSuccess() || nn::fs::ResultPathNotFound::Includes(result));
        result = nn::fs::CreateFile(pPath, size);
        NN_ASSERT(result.IsSuccess());
        result = nn::fs::OpenFile(&file, pPath, nn::fs::OpenMode_Write);
        NN_ASSERT(result.IsSuccess());
        result = nn::fs::WriteFile(file, 0, stream.str().c_str(), size, nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush));
        NN_ASSERT(result.IsSuccess());
        nn::fs::CloseFile(file);
    }

    return true;
}// NOLINT(impl/function_size)

//------------------------------------------------------------------------------
//  PNGファイル書き込み
//------------------------------------------------------------------------------
bool PngIO::WritePng( const char* pPath, const uint8_t* pImage, PNG_IHDR ihdr )
{
    return WritePng( pPath, pImage, ihdr.width, ihdr.height, ihdr.bitDepth, ihdr.colorType, ihdr.compressionType, ihdr.filterType, ihdr.interlaceType );
}
//------------------------------------------------------------------------------
//  PNGファイル解放
//------------------------------------------------------------------------------
void PngIO::FreePng(uint8_t** pImage)
{
    NNT_GRAPHICS_FREE(*pImage);
    *pImage = NULL;
}

//------------------------------------------------------------------------------
//  圧縮率を上げるため画像にフィルターをかける
//------------------------------------------------------------------------------
void PngIO::CompressFilter( const uint8_t* pInput, int width, int height, uint8_t pixelBytes, uint8_t* pOutput )
{
    for ( int y = 0; y < height; ++y )
    {
        int rowBytes = width * pixelBytes;

        const uint8_t* pInputLine = &pInput[y * rowBytes];
        uint8_t* pOutputLine = &pOutput[y * ( 1 + rowBytes )];

        pOutputLine[0] = 0;

        for ( int x = 0; x < rowBytes; ++x )
        {
            pOutputLine[1 + x] = pInputLine[x];
        }
    }
}

//------------------------------------------------------------------------------
// フィルタを適用したピクセルデータを復元
//------------------------------------------------------------------------------
void PngIO::DecompressFilter( const uint8_t* pInput, int width, int height, uint8_t pixelBytes, uint8_t* pOutput )
{
    for ( int y = 0; y < height; ++y )
    {
        // Win版はOpenGLの仕様に合わせて上下反転させる
        const uint8_t* pInputLine = &pInput[y * ( 1 + width * pixelBytes )];
        uint8_t* pOutputLine = &pOutput[y * ( width * pixelBytes )];

        for ( int x = 0; x < width * pixelBytes; ++x )
        {
            pOutputLine[x] = pInputLine[1 + x];
        }
    }
}


}}
