﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <new>
#include <sstream>
#include <zlib.h>
#include <nn/nn_Assert.h>
#include <nn/fs.h>
#include <nn/nn_Windows.h>
#include "testGraphics_ImageIo.h"

namespace nnt{ namespace graphics{

void* ImageIo::AllocateMemory(size_t size, int alignment)
{
    return _aligned_malloc(size, static_cast<size_t>(alignment));
}
void ImageIo::FreeMemory(void* ptr)
{
    _aligned_free(ptr);
}

//------------------------------------------------------------------------------
//  8bitデータをストリームに出力
//------------------------------------------------------------------------------
void ImageIo::Output8( std::ostream* pStream, const void* pData, uint32_t* pCrc )
{
    pStream->write( static_cast<const char*>( pData ), 1 );

    if (pCrc != NULL)
    {
        *pCrc = crc32( *pCrc, static_cast<const Bytef*>( pData ), 1 );
    }
}

//------------------------------------------------------------------------------
//  バイト列データをストリームに出力
//------------------------------------------------------------------------------
void ImageIo::OutputBytes( std::ostream* pStream, const void* pData, size_t size, uint32_t* pCrc )
{
    pStream->write( static_cast<const char*>( pData ), size );

    if (pCrc != NULL)
    {
        *pCrc = crc32( *pCrc, static_cast<const Bytef*>( pData ), static_cast<nnutilZlib_uInt>(size) );
    }
}

//!------------------------------------------------------------------------------
//! 8bitデータをストリームから入力
//!------------------------------------------------------------------------------
uint8_t ImageIo::Input8( uint8_t** pStream )
{
    uint8_t data = *(*pStream);
    *pStream = *pStream + 1;
    return data;
}

//------------------------------------------------------------------------------
// 16bitデータをストリームから入力
// isFlip: エンディアンスワップ
//------------------------------------------------------------------------------
uint16_t ImageIo::Input16( uint8_t** pStream, bool isFlip )
{
    uint16_t data;
    memcpy( &data, *pStream, 2 );

    if( isFlip )
    {
        uint16_t temp = data;
        data = 0;
        const uint8_t* pBytes = reinterpret_cast<const uint8_t*>( &temp );
        for ( int i = 0; i < 2; ++i )
        {
            data = data << 8 | pBytes[i];
        }
    }

    *pStream = *pStream + 2;

    return data;
}

uint16_t ImageIo::Input16( uint8_t** pStream )
{
    return Input16( pStream, true );
}

//!------------------------------------------------------------------------------
//! バイト列データをストリームから入力
//!------------------------------------------------------------------------------
void ImageIo::InputBytes( uint8_t** pStream, void* pData, size_t size )
{
    memcpy( pData, *pStream, size );
    *pStream = *pStream + size;
}


//------------------------------------------------------------------------------
//  32bitデータをストリームに出力
//------------------------------------------------------------------------------
void ImageIo::Output32( std::ostream* pStream, const void* pData, uint32_t* pCrc )
{
    // ビッグエンディアンになるようにWin版のみフリップ
    const uint8_t* pBytes = reinterpret_cast<const uint8_t*>( pData );
    uint32_t data;
    for ( int i = 0; i < 4; ++i )
    {
        data = data << 8 | pBytes[i];
    }

    pStream->write( reinterpret_cast<const char*>( &data ), 4 );

    if (pCrc != NULL)
    {
        *pCrc = crc32( *pCrc, reinterpret_cast<const Bytef*>( &data ), 4 );
    }
}

//------------------------------------------------------------------------------
// 32bitデータをストリームから入力
// isFlip: エンディアンスワップ
//------------------------------------------------------------------------------
uint32_t ImageIo::Input32( uint8_t** pStream, bool isFlip )
{
    uint32_t data;
    memcpy( &data, *pStream, 4 );

    if( isFlip )
    {
        uint32_t temp = data;
        data = 0;
        const uint8_t* pBytes = reinterpret_cast<const uint8_t*>( &temp );
        for ( int i = 0; i < 4; ++i )
        {
            data = data << 8 | pBytes[i];
        }
    }

    *pStream = *pStream + 4;

    return data;
}

uint32_t ImageIo::Input32( uint8_t** pStream )
{
    return Input32( pStream, true );
}

}}
