﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/TargetConfigs/build_Base.h>
#include <nn/nn_Log.h>

#include <nn/os.h>

#include <nn/gpio/gpio.h>
#include <nn/gpio/gpio_PadAccessorDev.h>

#include <nn/gpio/driver/gpio.h>
#include <nn/gpio/driver/gpio_PadAccessorDev.h>

#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>

#include "../common/testGpio_Util.h"

namespace
{

// GPIO パッドの H/L を切り替える時間
const nn::TimeSpan PadValueSwitchTime = nn::TimeSpan::FromMilliSeconds(100);

// 割り込みテストの時の GPIO パッドの H/L を切り替える時間
const nn::TimeSpan PadValueSwitchTimeForInterrupt = nn::TimeSpan::FromMilliSeconds(100);

// 割り込み待ちのタイムアウト時間
const nn::TimeSpan TimeOut = nn::TimeSpan::FromMilliSeconds(5000);

// システムが起動しきるまで待つ時間 (TORIAEZU)
const nn::TimeSpan WaitTime = nn::TimeSpan::FromSeconds(10);

// 割り込みスレッドのスタック領域
const size_t InterruptThreadStackSize = nn::os::StackRegionAlignment;
NN_ALIGNAS(nn::os::StackRegionAlignment) char s_Stack[InterruptThreadStackSize];

// テスト用 pinmux ライブラリ
nnt::gpio::Pinmux g_Pinmux;

}

namespace nnt { namespace gpio {

// 割り込みテストのための H/L を切り替えるスレッドの実態
void RepeatHAndLThreadBodyForOneGpioPad(void* arg)
{
    nnt::gpio::PadName pad = *reinterpret_cast<nnt::gpio::PadName*>(arg);

    int loopCount = 0;

    NN_LOG("%s: Sub Thread Start\n", __FUNCTION__);

    while(1)
    {
        nn::os::SleepThread(PadValueSwitchTimeForInterrupt);

        // 入力を内部的に pulldown する。(TODO:引数からとるようにする)
        g_Pinmux.PullDown(pad);
        NN_LOG("%s: Pulldown(L)\n", __FUNCTION__);

        nn::os::SleepThread(PadValueSwitchTimeForInterrupt);

        // 入力を内部的に pullup する。(TODO:引数からとるようにする)
        g_Pinmux.PullUp(pad);
        NN_LOG("%s: Pullup(H)\n", __FUNCTION__);

        loopCount++;
        if(loopCount == 3)
        {
            break;
        }
    }
}


// 1パッドごとの基本的な出力のテスト
void TestOneGpioPadOutput(PadName pad)
{
    NN_LOG("Start Basic Output Test For GPIO Pad: %d\n", pad);

    // セッション用の構造体
    nn::gpio::GpioPadSession session;

#if (defined(NN_GPIO_GEN) && (NN_GPIO_GEN == 2))
    NNT_ASSERT_RESULT_SUCCESS(nn::gpio::OpenSession(&session, pad));
#else // gen1
    nn::gpio::OpenSessionForDev(&session, static_cast<int>(pad));
#endif

    nn::gpio::SetDirection(&session, nn::gpio::Direction_Output);

    // テストのためにループバックに設定する
    g_Pinmux.SetDirectionForLoopBack(pad);

    // H/L を PadValueSwitchTime おきに 3 回繰り返してチェックする
    for(int i = 0; i < 3; i++)
    {
        nn::gpio::SetValue(&session, nn::gpio::GpioValue_High);
        nn::os::SleepThread(PadValueSwitchTime);
        EXPECT_EQ(nn::gpio::GpioValue_High, nn::gpio::GetValue(&session));
        NN_LOG("%s: Output : H\n", __FUNCTION__);

        nn::gpio::SetValue(&session, nn::gpio::GpioValue_Low);
        nn::os::SleepThread(PadValueSwitchTime);
        EXPECT_EQ(nn::gpio::GpioValue_Low, nn::gpio::GetValue(&session));
        NN_LOG("%s: Output : L\n", __FUNCTION__);
    }

    nn::gpio::CloseSession(&session);
}

// 1パッドごとの基本的な出力のテスト
void TestOneGpioPadOutputForSleep(PadName pad)
{
    NN_LOG("Start Basic Output For Sleep Test For GPIO Pad: %d\n", pad);

    // セッション用の構造体
    nn::gpio::GpioPadSession session;


#if (defined(NN_GPIO_GEN) && (NN_GPIO_GEN == 2))
    NNT_ASSERT_RESULT_SUCCESS(nn::gpio::OpenSession(&session, pad));
#else // gen1
    nn::gpio::OpenSessionForDev(&session, static_cast<int>(pad));
#endif

    nn::gpio::SetDirection(&session, nn::gpio::Direction_Output);

    nn::gpio::SetValueForSleepState(&session, nn::gpio::GpioValue_High);

    nn::gpio::CloseSession(&session);
}


// 1パッドごとの基本的な入力のテスト
void TestOneGpioPadInput(PadName pad)
{
    NN_LOG("Start Basic Input Test For GPIO Pad: %d\n", pad);

    // セッション用の構造体
    nn::gpio::GpioPadSession session;

#if (defined(NN_GPIO_GEN) && (NN_GPIO_GEN == 2))
    NNT_ASSERT_RESULT_SUCCESS(nn::gpio::OpenSession(&session, pad));
#else // gen1
    nn::gpio::OpenSessionForDev(&session, static_cast<int>(pad));
#endif

    nn::gpio::SetDirection(&session, nn::gpio::Direction_Input);

    // pullup/pulldown を PadValueSwitchTime おきに 3 回繰り返してチェックする
    for(int i = 0; i < 3; i++)
    {
        g_Pinmux.PullUp(pad);
        nn::os::SleepThread(PadValueSwitchTime);
        EXPECT_EQ(nn::gpio::GpioValue_High, nn::gpio::GetValue(&session));
        NN_LOG("%s: Input : H\n", __FUNCTION__);

        g_Pinmux.PullDown(pad);
        nn::os::SleepThread(PadValueSwitchTime);
        EXPECT_EQ(nn::gpio::GpioValue_Low, nn::gpio::GetValue(&session));
        NN_LOG("%s: Input : L\n", __FUNCTION__);
    }

    nn::gpio::CloseSession(&session);
}


// 1パッドごとの基本的な割り込みのテスト
void TestOneGpioPadInterrupt(PadName pad, nn::gpio::InterruptMode mode)
{
    NN_LOG("Start Basic Interrupt Test For GPIO Pad: %d\n", pad);

    // セッション用の構造体
    nn::gpio::GpioPadSession session;

    //GPIO ピンの値が変化した際に通知を受けるイベントオブジェクト
    nn::os::SystemEventType event;

    // H/L を繰り返すスレッド
    nn::os::ThreadType thread;


#if (defined(NN_GPIO_GEN) && (NN_GPIO_GEN == 2))
    NNT_ASSERT_RESULT_SUCCESS(nn::gpio::OpenSession(&session, pad));
#else // gen1
    nn::gpio::OpenSessionForDev(&session, static_cast<int>(pad));
#endif

    nn::gpio::SetDirection(&session, nn::gpio::Direction_Input);

    // 状態が不定の可能性があるので、入力を内部的に pulldown しておく。
    g_Pinmux.PullDown(pad);
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(50));

    nn::gpio::SetInterruptMode(&session, mode);

    // 割り込みを enable にする。
    nn::gpio::SetInterruptEnable(&session, true);


    // 割り込みをクリアする
    nn::gpio::ClearInterruptStatus(&session);

    // 設定を表示
    NN_LOG("-----------------------\n");
    NN_LOG(" GPIO Pad        : %d\n", pad);
    NN_LOG(" Direction       : %d\n", nn::gpio::GetDirection(&session));
    NN_LOG(" InterruptMode   : %d\n", nn::gpio::GetInterruptMode(&session));
    NN_LOG(" InterruptEnable : %d\n", nn::gpio::GetInterruptEnable(&session));
    NN_LOG(" isInterrupted   : %d\n", nn::gpio::GetInterruptStatus(&session));
    NN_LOG("-----------------------\n");

    // Interrupt を bind する。
    nn::gpio::BindInterrupt(&event, &session);

    // GPIO の H/L を変更するスレッド(本来はハードウェアからの信号なので、最優先のスレッドで作成する。)
    nn::os::CreateThread(&thread, RepeatHAndLThreadBodyForOneGpioPad, &pad, s_Stack, InterruptThreadStackSize, nn::os::HighestThreadPriority);
    nn::os::StartThread(&thread);

    NN_LOG(" Start Interrupt Test \n");

    // 終了条件に使用する
    int highCount =0;
    int lowCount = 0;

    // 割り込みが発生したら、値を取得する。
    while(1)
    {
        if(!nn::os::TimedWaitSystemEvent(&event, TimeOut))
        {
            ADD_FAILURE();
        }

        if(nn::gpio::GetValue(&session) == nn::gpio::GpioValue_High)
        {
            NN_LOG(" Get Interrupt : H\n");
            highCount++;
        }
        else
        {
            NN_LOG(" Get Interrupt : L\n");
            lowCount++;
        }

        // 終了条件の確認
        // 想定以上の値になっていたら、エラーとして、終了する。
        if(highCount > 3 || lowCount > 2)
        {
            // 割り込みステータスをクリアする
            nn::gpio::ClearInterruptStatus(&session);

            // イベントのクリア
            nn::os::ClearSystemEvent(&event);

            EXPECT_LE(highCount, 3);
            EXPECT_LE(lowCount, 2);
            nn::gpio::UnbindInterrupt(&session);
            nn::gpio::CloseSession(&session);
            nn::os::DestroyThread(&thread);
            return;
        }
        // 想定通りの値になっていたら、終了する。
        if(highCount == 3 && lowCount == 2)
        {
            // 割り込みステータスをクリアする
            nn::gpio::ClearInterruptStatus(&session);

            // イベントのクリア
            nn::os::ClearSystemEvent(&event);

            nn::gpio::UnbindInterrupt(&session);
            nn::gpio::CloseSession(&session);
            nn::os::DestroyThread(&thread);
            return;
        }

        // 割り込みステータスをクリアする
        nn::gpio::ClearInterruptStatus(&session);

        // イベントのクリア
        nn::os::ClearSystemEvent(&event);

        // 割り込み許可状態に戻す
        nn::gpio::SetInterruptEnable(&session,true);

    }
}

} } // nnt::gpio


// GPIO のテストはシステム起動中に実行すると間隔がズレて失敗するため、システムが起動するまで待つ
TEST(testGpio_Basic, WaitSystemBoot)
{
    nn::os::SleepThread(WaitTime);
}

//-----------------------------------------------------------------------------
// GPIO ライブラリの Input の基本的なテスト(内部で GPIO パッドを Pullup/PullDown するため、端子が解放されていること)
TEST(testGpio_Basic, Input)
{

    nn::gpio::Initialize();

    // 現状使用できる Jetson TK1 向けの GPIO パッドの入力を試す
    for(int i = 0; i<nnt::gpio::NumberOfTestPad; i++)
    {
        nnt::gpio::TestOneGpioPadInput(nnt::gpio::PadNameList[i]);
    }

    nn::gpio::Finalize();

}

//-----------------------------------------------------------------------------
// GPIO ライブラリの Output の基本的なテスト(内部で GPIO パッドをループバックするため、端子が解放されていること)
TEST(testGpio_Basic, Output)
{
    nn::gpio::Initialize();

    // 現状使用できる Jetson TK1 向けの GPIO パッドの入力を試す
    for(int i = 0; i<nnt::gpio::NumberOfTestPad; i++)
    {
        nnt::gpio::TestOneGpioPadOutputForSleep(nnt::gpio::PadNameList[i]);
    }

    nn::gpio::Finalize();
}

//-----------------------------------------------------------------------------
// GPIO ライブラリの Output の基本的なテスト(内部で GPIO パッドをループバックするため、端子が解放されていること)
TEST(testGpio_Basic, OutputForSleep)
{
    nn::gpio::Initialize();

    // 現状使用できる Jetson TK1 向けの GPIO パッドの入力を試す
    for(int i = 0; i<nnt::gpio::NumberOfTestPad; i++)
    {
        nnt::gpio::TestOneGpioPadOutput(nnt::gpio::PadNameList[i]);
    }

    nn::gpio::Finalize();
}

//-----------------------------------------------------------------------------
// GPIO ライブラリの Interrupt の基本的なテスト(内部で GPIO パッドを Pullup/PullDown するため、端子が解放されていること)
// AnyEdge 以外のテストは Interrupt のテストにあります。
TEST(testGpio_Basic, InterruptAnyEdge)
{
    nn::gpio::Initialize();

    // 現状使用できる Jetson TK1 向けの GPIO パッドの入力を試す
    for(int i = 0; i<nnt::gpio::NumberOfTestPad; i++)
    {
        nnt::gpio::TestOneGpioPadInterrupt(nnt::gpio::PadNameList[i], nn::gpio::InterruptMode_AnyEdge);
    }

    nn::gpio::Finalize();
}

//---------------------------------------
// wake 要因を取得するテスト
// ひとまず取得するところまでで、自動テストでは値自体の精査はしない
TEST(testGpio_Basic, GetWakeReason)
{
    nn::gpio::Initialize();

#if defined(NN_BUILD_CONFIG_HARDWARE_NX)
#if (defined(NN_GPIO_GEN) && (NN_GPIO_GEN == 2))
    const std::pair<nn::DeviceCode, const char*> WakeEvents[] =
    {
        std::make_pair(NN_DEVICECODE_GPIO_NX_GAME_CARD_CD,      "GAME_CARD_CD"),
        std::make_pair(NN_DEVICECODE_GPIO_NX_BATT_MGIC_IRQ,     "BATT_MGIC_IRQ"),
        std::make_pair(NN_DEVICECODE_GPIO_NX_BQ_24190_IRQ,      "BQ_24190_IRQ"),
        std::make_pair(NN_DEVICECODE_GPIO_NX_CRADLE_IRQ,        "CRADLE_IRQ"),
        std::make_pair(NN_DEVICECODE_GPIO_NX_BT_WAKE_AP,        "BT_WAKE_AP"),
        std::make_pair(NN_DEVICECODE_GPIO_NX_EXTCON_DET_S,      "EXTCON_DET_S"),
        std::make_pair(NN_DEVICECODE_GPIO_NX_EXTCON_DET_U,      "EXTCON_DET_U"),
        std::make_pair(NN_DEVICECODE_GPIO_NX_WIFI_WAKE_HOST,    "WIFI_WAKE_HOST"),
        std::make_pair(NN_DEVICECODE_GPIO_NX_SD_CD,             "SD_CD"),
        std::make_pair(NN_DEVICECODE_GPIO_NX_PMU_IRQ,           "PMU_IRQ"),
        std::make_pair(NN_DEVICECODE_GPIO_NX_EXT_UART_2_CTS,    "EXT_UART_2_CTS"),
        std::make_pair(NN_DEVICECODE_GPIO_NX_EXT_UART_3_CTS,    "EXT_UART_3_CTS"),
    };

    for ( const auto& we : WakeEvents )
    {
        bool isActive;
        NNT_EXPECT_RESULT_SUCCESS(nn::gpio::IsWakeEventActive(&isActive, we.first));
        NN_LOG("[gpio test] %s = %s\n", we.second, isActive ? "true" : "false");
    }

#else // Gen1
    NN_LOG("[gpio test] GpioPadName_GameCardCd   = %d\n", nn::gpio::IsWakeEventActive(nn::gpio::GpioPadName_GameCardCd));
    NN_LOG("[gpio test] GpioPadName_BattMgicIrq  = %d\n", nn::gpio::IsWakeEventActive(nn::gpio::GpioPadName_BattMgicIrq));
    NN_LOG("[gpio test] GpioPadName_Bq24190Irq   = %d\n", nn::gpio::IsWakeEventActive(nn::gpio::GpioPadName_Bq24190Irq));
    NN_LOG("[gpio test] GpioPadName_CradleIrq    = %d\n", nn::gpio::IsWakeEventActive(nn::gpio::GpioPadName_CradleIrq));
    NN_LOG("[gpio test] GpioPadName_BtWakeAp     = %d\n", nn::gpio::IsWakeEventActive(nn::gpio::GpioPadName_BtWakeAp));
    NN_LOG("[gpio test] GpioPadName_ExtconDetS   = %d\n", nn::gpio::IsWakeEventActive(nn::gpio::GpioPadName_ExtconDetS));
    NN_LOG("[gpio test] GpioPadName_ExtconDetU   = %d\n", nn::gpio::IsWakeEventActive(nn::gpio::GpioPadName_ExtconDetU));
    NN_LOG("[gpio test] GpioPadName_WifiWakeHost = %d\n", nn::gpio::IsWakeEventActive(nn::gpio::GpioPadName_WifiWakeHost));
    NN_LOG("[gpio test] GpioPadName_SdCd         = %d\n", nn::gpio::IsWakeEventActive(nn::gpio::GpioPadName_SdCd));
    NN_LOG("[gpio test] GpioPadName_PmuIrq       = %d\n", nn::gpio::IsWakeEventActive(nn::gpio::GpioPadName_PmuIrq));
    NN_LOG("[gpio test] GpioPadName_ExtUart2Cts  = %d\n", nn::gpio::IsWakeEventActive(nn::gpio::GpioPadName_ExtUart2Cts));
    NN_LOG("[gpio test] GpioPadName_ExtUart3Cts  = %d\n", nn::gpio::IsWakeEventActive(nn::gpio::GpioPadName_ExtUart3Cts));
#endif // NN_GPIO_GEN
#endif // NN_BUILD_CONFIG_HARDWARE_NX

    nn::gpio::Finalize();
}
