﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/TargetConfigs/build_Base.h>
#include <nn/nn_Log.h>
#include <nn/os.h>

#include <nn/gpio/gpio.h>
#include <nn/gpio/gpio_PadAccessorDev.h>

#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>

#include "../common/testGpio_Util.h"

namespace
{

// API に渡すチャタリング判定時間(msec)
const uint8_t DebonceTime = 100;

// 1回のパルスを発生させてから次のパルスまでの待ち時間
const int LoopWaitTimeMilliSeconds= 200;
const nn::TimeSpan LoopWaitTime = nn::TimeSpan::FromMilliSeconds(LoopWaitTimeMilliSeconds);

// サブスレッドで H/L を切り替える回数
const int LoopNumber = 3;

// 割り込みスレッドのスタック領域
const size_t InterruptThreadStackSize = nn::os::StackRegionAlignment;
NN_ALIGNAS(nn::os::StackRegionAlignment) char s_Stack[InterruptThreadStackSize];

// テスト用 pinmux ライブラリ
nnt::gpio::Pinmux g_Pinmux;

}

namespace nnt { namespace gpio {

// チャタリングのテストのため、Pulldown -> Pullup -> Pulldown をするスレッド
void RepeatHAndLThreadBodyForDebouncePad(void* arg)
{
    nnt::gpio::PadName pad = *reinterpret_cast<nnt::gpio::PadName*>(arg);

    int loopCount = 0;

    NN_LOG("%s: Sub Thread Start\n", __FUNCTION__);

    while(1)
    {
        nn::os::SleepThread(LoopWaitTime);

        g_Pinmux.PullDown(pad);
        //NN_LOG("PullDown\n");

        // パルス発生時間をだいたい DebounceTime の半分としてテスト
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(DebonceTime / 4));

        g_Pinmux.PullUp(pad);
        //NN_LOG("PullUp\n");

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(DebonceTime / 4));

        g_Pinmux.PullDown(pad);
        //NN_LOG("PullDown\n");

        loopCount++;
        if(loopCount == LoopNumber)
        {
            break;
        }
    }
}

// チャタリング防止機能のテスト(Edge トリガー版)
void TestDebouncePad(nnt::gpio::PadName pad)
{
    NN_LOG("Start Debounce Test For GPIO Pad: %d\n", pad);

    // セッション用の構造体
    nn::gpio::GpioPadSession session;

    //GPIO ピンの値が変化した際に通知を受けるイベントオブジェクト
    nn::os::SystemEventType event;

    // H/L を繰り返すスレッド
    nn::os::ThreadType thread;

#if (defined(NN_GPIO_GEN) && (NN_GPIO_GEN == 2))
    NNT_ASSERT_RESULT_SUCCESS(nn::gpio::OpenSession(&session, pad));
#else // gen1
    nn::gpio::OpenSessionForDev(&session, static_cast<int>(pad));
#endif

    nn::gpio::SetDirection(&session, nn::gpio::Direction_Input);

    // Debounce 除去機能の msec を設定
    nn::gpio::SetDebounceTime(&session, DebonceTime);

    // Debounce 除去機能を enable に
    nn::gpio::SetDebounceEnabled(&session, true);

    // 状態が不定の可能性があるので、入力を内部的に pulldown しておく。
    g_Pinmux.PullDown(pad);
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(5));

    // 本テストでは立ち上がりエッジ決め打ち
    nn::gpio::SetInterruptMode(&session, nn::gpio::InterruptMode_RisingEdge);

    // 割り込みをクリアする
    nn::gpio::ClearInterruptStatus(&session);

    // 割り込みを enable にする。
    nn::gpio::SetInterruptEnable(&session, true);

    // 設定を表示
    NN_LOG("-----------------------\n");
    NN_LOG(" GPIO Pad        : %d\n", pad);
    NN_LOG(" Direction       : %d\n", nn::gpio::GetDirection(&session));
    NN_LOG(" InterruptMode   : %d\n", nn::gpio::GetInterruptMode(&session));
    NN_LOG(" InterruptEnable : %d\n", nn::gpio::GetInterruptEnable(&session));
    NN_LOG(" isInterrupted   : %d\n", nn::gpio::GetInterruptStatus(&session));
    NN_LOG("-----------------------\n");

    // Interrupt を bind する。
    nn::gpio::BindInterrupt(&event, &session);

    // GPIO の H/L を変更するスレッド(本来はハードウェアからの信号なので、最優先のスレッドで作成する。)
    nn::os::CreateThread(&thread, RepeatHAndLThreadBodyForDebouncePad, &pad, s_Stack, InterruptThreadStackSize, nn::os::HighestThreadPriority);
    nn::os::StartThread(&thread);

    // TimedWati で待つ時間(少し余裕をみて、DebonceTime * 4 にしている)
    const nn::TimeSpan Timeout = nn::TimeSpan::FromMilliSeconds((DebonceTime * 4 + LoopWaitTimeMilliSeconds) * LoopNumber);

    // 割込みが入らないことを確認
    EXPECT_FALSE(nn::os::TimedWaitSystemEvent(&event, Timeout));

    // 割り込みステータスをクリアする
    nn::gpio::ClearInterruptStatus(&session);

    // イベントのクリア
    nn::os::ClearSystemEvent(&event);

    // スレッドを削除
    nn::os::DestroyThread(&thread);

    // -------------------------------------------------------
    // ここまでで、Debounce を設定したときに設定時間内でのパルスでは割り込みが入らないことをテストした
    // ここからは、Deobunce を設定したときでも設定時間以上であれば割り込みが入ることをテストする
    // -------------------------------------------------------

    // Debounce 機能の msec を 1/10 に設定
    nn::gpio::SetDebounceTime(&session, DebonceTime / 10);

    int loopCount = 0;

    // 再度スレッドを作成
    nn::os::CreateThread(&thread, RepeatHAndLThreadBodyForDebouncePad, &pad, s_Stack, InterruptThreadStackSize, nn::os::HighestThreadPriority);
    nn::os::StartThread(&thread);

    while(1)
    {
        // 今度は割り込みが 3回入るはず
        EXPECT_TRUE(nn::os::TimedWaitSystemEvent(&event, Timeout));

        // 割り込みステータスをクリアする
        nn::gpio::ClearInterruptStatus(&session);

        // イベントのクリア
        nn::os::ClearSystemEvent(&event);

        loopCount++;
        if(loopCount == LoopNumber)
        {
            break;
        }

        // 割り込み許可状態に戻す
        nn::gpio::SetInterruptEnable(&session,true);
    }

    // スレッドを削除
    nn::os::DestroyThread(&thread);

    nn::gpio::UnbindInterrupt(&session);
    nn::gpio::CloseSession(&session);

    return;
}

} } // nnt::gpio

// チャタリング機能のテスト
TEST(testGpio_Debounce, Edge)
{
    nn::gpio::Initialize();

    for(int i = 0; i<nnt::gpio::NumberOfTestPad; i++)
    {
        nnt::gpio::TestDebouncePad(nnt::gpio::PadNameList[i]);
    }

    nn::gpio::Finalize();
}
