﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

//-----------------------------------------------------------------------------
// HIPC によって Gfx0Driver サービスを利用するためのテストプログラム。
// このテストでは単一のテストプロセス内で、サーバスレッドとクライアントスレッド
// を動作させている。実環境では、サーバとクライアントは別プロセスとなる。

#include <nnt/nntest.h>

#include <nn/nn_Common.h>
#include <nn/os.h>
#include <nn/gfx0.h>
#include <nn/sf/sf_HipcServer.h>

namespace {

//-----------------------------------------------------------------------------
// nn::sf::HipcSimpleAllInOneServerManager は HIPC のサーバのポートと
// セッションを一元管理する。引数は最大セッション数と最大ポート数。
//
class Gfx0ServerManager
    : public nn::sf::HipcSimpleAllInOneServerManager<30, 1>
{
private:
    // OnNeedsToAccept を override し、ポートへのアクセスの動作を決定する。
    // 詳細は OnNeedsToAccept() のリファレンスを参照。
    virtual nn::Result OnNeedsToAccept(int portIndex, PortObjectImpl* pPort) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(portIndex);
        // ポートから開かれたセッションに、
        // サーバ内における DFC 実装（ CreateGfx0DriverByDfc() ）を紐付ける。
        return AcceptImpl(pPort, nn::gfx0::CreateGfx0DriverByDfc());
    }
};

// 繰返しのサーバ起動／終了が可能となるように placement new で初期化を行う。
// 繰返しの起動と終了が必要ない場合には Gfx0ServerManager は直接配置してもよい。
std::aligned_storage<sizeof(Gfx0ServerManager), NN_ALIGNOF(Gfx0ServerManager)>::type g_Gfx0ServerManagerStorage;

//-----------------------------------------------------------------------------
// サーバ実装用のサービスオブジェクトへの共有ポインタ
Gfx0ServerManager*  g_pGfx0ServerManager;


//-----------------------------------------------------------------------------
// サーバスレッド用グローバル変数とサーバ関数
NN_ALIGNAS(4096) char g_ServerStack[4096];
nn::os::ThreadType    g_ServerThread;

void HipcServerFunction(void* arg) NN_NOEXCEPT
{
    NN_UNUSED(arg);
    NN_SDK_ASSERT(g_pGfx0ServerManager);

    // サーバループを実行
    // nn::gfx0::RequestStopGfx0DriverServer() が別スレッドから呼ばれるまで、
    // ループし続ける。
    g_pGfx0ServerManager->LoopAuto();
}

//-----------------------------------------------------------------------------
// サーバの初期化
void InitializeHipcServer() NN_NOEXCEPT
{
    NN_SDK_ASSERT(!g_pGfx0ServerManager);

    // Gfx0ServerManager オブジェクトのコンストラクト（placement new）
    g_pGfx0ServerManager = new (&g_Gfx0ServerManagerStorage) Gfx0ServerManager;

    // サービス名の登録とポートの初期化
    // portIndex は OnNeedsToAccept の第一引数となるが、
    // 今回は使用しないため、適当に 0 を指定している。
    // sessionCountMax は、現時点では十分に大きな値を指定しておけばよい。
    auto portIndex       = 0;
    auto sessionCountMax = 30;
    g_pGfx0ServerManager->InitializePort(portIndex, sessionCountMax, nn::gfx0::Gfx0DriverServiceName);

    // サーバマネージャの開始
    // ただし、実際のサーバ動作は LoopAuto() 関数の呼び出しによって行なわれる。
    g_pGfx0ServerManager->Start();

    // サーバ処理スレッドの作成
    auto result = nn::os::CreateThread(&g_ServerThread, &HipcServerFunction, nullptr, g_ServerStack, sizeof(g_ServerStack), 10);
    ASSERT_TRUE(result.IsSuccess());
    nn::os::StartThread(&g_ServerThread);
}

void FinalizeHipcServer() NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_pGfx0ServerManager);

    // サーバ終了をリクエスト
    // サーバスレッド中の nn::gfx0::LoopGfx0DriverServer() の処理が終了する。
    g_pGfx0ServerManager->RequestStop();

    // サーバスレッド終了待機と破棄
    nn::os::WaitThread(&g_ServerThread);
    nn::os::DestroyThread(&g_ServerThread);

    // placement new でコンストラクトされているので明示的にデストラクタを呼ぶ
    g_pGfx0ServerManager->~Gfx0ServerManager();
    g_pGfx0ServerManager = nullptr;
}

void ExecuteClient() NN_NOEXCEPT
{
    // 基本的に DFC の場合と呼出しは同じ。
    // nn::gfx0::InitializeGfx0Driver() の内部で DFC か HIPC かは決まっている

    nn::gfx0::InitializeGfx0Driver();

    for (auto i = 0; i < 100; ++i)
    {
        auto ret = nn::gfx0::DummyFuncGfx0(i , i * 2);
        EXPECT_EQ( i * 3, ret );
    }

    nn::gfx0::FinalizeGfx0Driver();
}

}   // namespace

//-----------------------------------------------------------------------------

TEST(Gfx0Skeleton, Basic_DummyFuncGfx0ByHipc)
{
    // サーバの起動
    // サーバの処理は別スレッドで行われる。
    InitializeHipcServer();

    // クライアントコード(テストコード)の実行
    ExecuteClient();

    // サーバの終了
    FinalizeHipcServer();
}

