﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/gc/detail/gc_Util.h>

#include "testGc_Unit_GcLibrary_Util.h"

using namespace nn::gc;
using namespace nn::gc::detail;

#ifdef TSET_RB_LOG_ENABLE
#define NN_TEST_RB_LOG(...)   NN_DETAIL_INTERNAL_LOG(__VA_ARGS__)
#else
#define NN_TEST_RB_LOG(...)   static_cast<void>(0)
#endif


bool compareAsicWorkInfo(AsicWorkInfo a, AsicWorkInfo b)
{
    if( a.lockerIndex != b.lockerIndex ||
        a.pReadInfo != b.pReadInfo ||
        a.hasLocker != b.hasLocker ||
        a.workIndex != b.workIndex)
    {
        return false;
    }
    return true;
}

void TESTCMP(AsicWorkInfo a, AsicWorkInfo b)
{
    if(compareAsicWorkInfo(a, b) == false)
    {
        NN_TEST_RB_LOG("TESTCMP: %d, %d\n", a.lockerIndex, b.lockerIndex);
        ASSERT_TRUE(compareAsicWorkInfo(a, b));
    }
}

void TESTCMP_NE(AsicWorkInfo a, AsicWorkInfo b)
{
    ASSERT_FALSE(compareAsicWorkInfo(a, b));
}

TEST(GcUnitTest, RingBufferTest)
{
    const int numReserved = 4;
    AsicWorkInfo buffer[32];
    AsicWorkInfo* bufferStart = &(buffer[numReserved]);
    const size_t bufferLength = sizeof(buffer) / sizeof(AsicWorkInfo) - numReserved * 2;

    // メモリ破壊を調べるため、最初と最後にデータを埋める
    const AsicWorkInfo sentinelInfo = {AsicWork_SetCardToSecureMode, 99999, true, nullptr};
    for(size_t i=0; i<numReserved; i++)
    {
        buffer[i] = sentinelInfo;
        buffer[numReserved + bufferLength + i] = sentinelInfo;
    }
    const AsicWorkInfo defaultInfo = {AsicWork_None, 0, true, nullptr};
    for(size_t i=0; i<bufferLength; i++)
    {
        buffer[numReserved + i] = defaultInfo;
    }

    // 下準備
    AsicWorkInfo ibuf[bufferLength * 2];
    AsicWorkInfo obuf[bufferLength * 2];
    AsicWorkInfo temp;
    const AsicWorkInfo testInfo = {AsicWork_Activate, 1, true, nullptr};
    for(size_t i = 0; i < (sizeof(ibuf) / sizeof(AsicWorkInfo)); i++)
    {
        // インプットに使用するデータは適当な値で初期化
        ibuf[i] = testInfo;
        ibuf[i].lockerIndex = i;
    }
    RingBuffer<AsicWorkInfo> rb(bufferStart, bufferLength);

    // *** テスト開始：ちょっとずつずらしながら試す
    for(size_t ofs = 0; ofs < bufferLength; ofs++)
    {
        size_t testAtIndex = 0;
        for(size_t i = 0; i < (sizeof(obuf) / sizeof(AsicWorkInfo)); i++)
        {
            // アウトプット先に使用するデータはデフォルトの値で初期化
            obuf[i] = defaultInfo;
        }

        // 状態チェック
        {
            NN_TEST_RB_LOG("1 %d, %d, %d\n", rb.m_AtIndex, ofs + testAtIndex, (ofs + testAtIndex) % bufferLength);
            ASSERT_EQ(rb.m_AtIndex, (ofs + testAtIndex) % bufferLength);
        }

        // 初期状態
        ASSERT_TRUE(rb.IsEmpty());
        ASSERT_EQ(0, rb.GetLength());
        ASSERT_EQ(0, rb.DeleteHead(3));
        ASSERT_EQ(0, rb.DeleteTail(3));
        ASSERT_EQ(0, rb.Dequeue(obuf));
        ASSERT_EQ(0, rb.Dequeue(obuf, 3));
        ASSERT_TRUE(rb.IsEmpty());

        // 状態チェック
        {
            NN_TEST_RB_LOG("2 %d, %d, %d\n", rb.m_AtIndex, ofs + testAtIndex, (ofs + testAtIndex) % bufferLength);
            ASSERT_EQ(rb.m_AtIndex, (ofs + testAtIndex) % bufferLength);
        }

        // データ追加
        rb.Enqueue(testInfo);
        testAtIndex++;
        ASSERT_FALSE(rb.IsEmpty());
        ASSERT_EQ(1, rb.GetLength());
        TESTCMP_NE(testInfo, obuf[0]);
        ASSERT_EQ(1, rb.Dequeue(obuf));
        ASSERT_TRUE(rb.IsEmpty());
        TESTCMP(testInfo, obuf[0]);  // 差し替わったはず

        // 状態チェック
        {
            NN_TEST_RB_LOG("3 %d, %d, %d\n", rb.m_AtIndex, ofs + testAtIndex, (ofs + testAtIndex) % bufferLength);
            ASSERT_EQ(rb.m_AtIndex, (ofs + testAtIndex) % bufferLength);

            for(size_t j=0; j<bufferLength; j++)
            {
                AsicWorkInfo ref = (j == 0) ? testInfo : defaultInfo;
                TESTCMP(rb.m_Buffer[(ofs + j) % bufferLength], ref);
            }
        }

        // さらに追加
        int anum = 12;
        size_t numContentIbuf = 12;
        rb.Enqueue(ibuf, anum);
        testAtIndex += anum;
        ASSERT_EQ(anum, rb.GetLength());
        int dnum = 2;
        ASSERT_EQ(dnum, rb.Dequeue(obuf, dnum));
        for(int i=0; i<dnum; i++)
        {
            ASSERT_EQ(i, obuf[i].lockerIndex);
        }
        ASSERT_EQ(anum - dnum, rb.GetLength());

        // 状態チェック
        {
            NN_TEST_RB_LOG("4 %d, %d, %d\n", rb.m_AtIndex, ofs + testAtIndex, (ofs + testAtIndex) % bufferLength);
            ASSERT_EQ(rb.m_AtIndex, (ofs + testAtIndex) % bufferLength);

            TESTCMP(rb.m_Buffer[(ofs) % bufferLength], testInfo);
            // 追加された分
            for(size_t j=0; j<numContentIbuf; j++)
            {
                TESTCMP(rb.m_Buffer[(ofs + 1 + j) % bufferLength], ibuf[j]);
            }
            // のこり
            for(size_t j=0; j<bufferLength - 1 - numContentIbuf; j++)
            {
                TESTCMP(rb.m_Buffer[(ofs + 1 + j + numContentIbuf) % bufferLength], defaultInfo);
            }
        }

        // 削除して取り出し（1 + 1 + 2 + 1 = 5 取り出し）
        for(int i=1; i<=2; i++)
        {
            ASSERT_EQ(i, rb.DeleteHead(i));
            dnum += i;
            // NN_TEST_RB_LOG("a:%d/d:%d\n", anum, dnum);
            ASSERT_EQ(anum - dnum, rb.GetLength());
            ASSERT_EQ(1, rb.Dequeue(&temp));
            dnum++;
            // NN_TEST_RB_LOG("a:%d/d:%d\n", anum, dnum);
            ASSERT_EQ(dnum - 1, temp.lockerIndex);
            ASSERT_EQ(anum - dnum, rb.GetLength());
        }

        // 状態チェック
        {
            NN_TEST_RB_LOG("5 %d, %d, %d\n", rb.m_AtIndex, ofs + testAtIndex, (ofs + testAtIndex) % bufferLength);
            ASSERT_EQ(rb.m_AtIndex, (ofs + testAtIndex) % bufferLength);

            // 中身はさっきと変わらないはず
            TESTCMP(rb.m_Buffer[(ofs) % bufferLength], testInfo);
            for(size_t j=0; j<numContentIbuf; j++)
            {
                TESTCMP(rb.m_Buffer[(ofs + 1 + j) % bufferLength], ibuf[j]);
            }
            for(size_t j=0; j<bufferLength - 1 - numContentIbuf; j++)
            {
                TESTCMP(rb.m_Buffer[(ofs + 1 + j + numContentIbuf) % bufferLength], defaultInfo);
            }
        }

        // 削除して取り出し（1 + 1 + 2 + 1 = 5 取り出し）
        const int lastDnum = dnum;
        for(int i=1; i<=2; i++)
        {
            rb.DeleteTail(i);
            testAtIndex -= i;
            dnum += i;
            // NN_TEST_RB_LOG("a:%d/d:%d\n", anum, dnum);
            ASSERT_EQ(anum - dnum, rb.GetLength());
            rb.Dequeue(&temp);
            dnum++;
            // NN_TEST_RB_LOG("a:%d/d:%d\n", anum, dnum);
            ASSERT_EQ(i == 1 ? lastDnum : lastDnum + 1, temp.lockerIndex);  // 全部削除されたら先頭でなく最後尾の値になる
            ASSERT_EQ(anum - dnum, rb.GetLength());
        }

        // 状態チェック
        {
            NN_TEST_RB_LOG("6 %d, %d, %d\n", rb.m_AtIndex, ofs + testAtIndex, (ofs + testAtIndex) % bufferLength);
            ASSERT_EQ(rb.m_AtIndex, (ofs + testAtIndex) % bufferLength);

            // DeleteTail では m_AtIndex は手前に下がるが、中身はさっきと同じはず
            TESTCMP(rb.m_Buffer[(ofs) % bufferLength], testInfo);
            for(size_t j=0; j<numContentIbuf; j++)
            {
                TESTCMP(rb.m_Buffer[(ofs + 1 + j) % bufferLength], ibuf[j]);
            }
            for(size_t j=0; j<bufferLength - 1 - numContentIbuf; j++)
            {
                TESTCMP(rb.m_Buffer[(ofs + 1 + j + numContentIbuf) % bufferLength], defaultInfo);
            }
        }

        // 入っているよりも削除する
        const int lent1 = 10;
        rb.Enqueue(ibuf, lent1);
        testAtIndex += lent1;
        anum += lent1;
        ASSERT_EQ(anum - dnum, rb.GetLength());
        rb.Dequeue(obuf, lent1 * 2);
        dnum += lent1;
        ASSERT_EQ(anum - dnum, rb.GetLength());

        // 状態チェック
        {
            NN_TEST_RB_LOG("7 %d, %d, %d\n", rb.m_AtIndex, ofs + testAtIndex, (ofs + testAtIndex) % bufferLength);
            ASSERT_EQ(rb.m_AtIndex, (ofs + testAtIndex) % bufferLength);

            numContentIbuf -= 3;
            TESTCMP(rb.m_Buffer[(ofs) % bufferLength], testInfo);
            for(size_t j=0; j<numContentIbuf; j++)
            {
                TESTCMP(rb.m_Buffer[(ofs + 1 + j) % bufferLength], ibuf[j]);
            }
            // 追加分
            for(size_t j=0; j<lent1; j++)
            {
                TESTCMP(rb.m_Buffer[(ofs + 1 + j + numContentIbuf) % bufferLength], ibuf[j]);
            }
            // のこり
            for(size_t j=0; j<bufferLength - 1 - numContentIbuf - lent1; j++)
            {
                TESTCMP(rb.m_Buffer[(ofs + 1 + j + numContentIbuf + lent1) % bufferLength], defaultInfo);
            }
        }

        // 入りきらないものを入れる
        rb.Enqueue(ibuf, bufferLength * 2);
        testAtIndex += bufferLength * 2;
        anum += bufferLength;
        ASSERT_EQ(anum - dnum, rb.GetLength());
        rb.Enqueue(ibuf, bufferLength * 2);
        testAtIndex += bufferLength * 2;
        anum += 0;
        ASSERT_EQ(anum - dnum, rb.GetLength());

        // 状態チェック
        {
            NN_TEST_RB_LOG("8 %d, %d, %d\n", rb.m_AtIndex, ofs + testAtIndex, (ofs + testAtIndex) % bufferLength);
            ASSERT_EQ(rb.m_AtIndex, (ofs + testAtIndex) % bufferLength);

            // 中身がすべて入れ替えられる
            for(size_t j=0; j<bufferLength; j++)
            {
                TESTCMP(rb.m_Buffer[(ofs + testAtIndex + j) % bufferLength], ibuf[j + bufferLength]);
            }
        }

        // （ループを回す関係上、きりのいいところまで足しておく）
        const size_t lent2 = bufferLength - (testAtIndex % bufferLength);
        rb.Enqueue(ibuf, lent2);
        testAtIndex += lent2;
        anum += 0;
        ASSERT_EQ(anum - dnum, rb.GetLength());

        // 状態チェック
        {
            NN_TEST_RB_LOG("9 %d, %d, %d\n", rb.m_AtIndex, ofs + testAtIndex, (ofs + testAtIndex) % bufferLength);
            ASSERT_EQ(rb.m_AtIndex, (ofs + testAtIndex) % bufferLength);

            // 後ろ向きにチェック
            for(size_t j=0; j<lent2; j++)
            {
                TESTCMP(rb.m_Buffer[(ofs + testAtIndex - 1 - j) % bufferLength], ibuf[lent2 - 1 - j]);
            }
            for(size_t j=0; j<(bufferLength - lent2); j++)
            {
                TESTCMP(rb.m_Buffer[(ofs + testAtIndex - lent2 - 1 - j) % bufferLength], ibuf[bufferLength * 2 - 1 - j]);
            }
        }

        // 最大容量よりも多く削除する
        rb.Dequeue(obuf, bufferLength * 2);
        dnum += bufferLength;
        ASSERT_EQ(anum - dnum, rb.GetLength());

        // 状態チェック
        {
            NN_TEST_RB_LOG("10 %d, %d, %d\n", rb.m_AtIndex, ofs + testAtIndex, (ofs + testAtIndex) % bufferLength);
            ASSERT_EQ(rb.m_AtIndex, (ofs + testAtIndex) % bufferLength);

            // 内容に変化はないはず
            for(size_t j=0; j<lent2; j++)
            {
                TESTCMP(rb.m_Buffer[(ofs + testAtIndex - 1 - j) % bufferLength], ibuf[lent2 - 1 - j]);
            }
            for(size_t j=0; j<(bufferLength - lent2); j++)
            {
                TESTCMP(rb.m_Buffer[(ofs + testAtIndex - lent2 - 1 - j) % bufferLength], ibuf[bufferLength * 2 - 1 - j]);
            }
        }

        // 次のループの為、オフセットをひとつずらす
        rb.Enqueue(testInfo);
        testAtIndex++;
        anum++;
        rb.DeleteHead(1);
        dnum++;
        ASSERT_EQ(anum - dnum, rb.GetLength());

        // 状態チェック
        {
            NN_TEST_RB_LOG("11 %d, %d, %d\n", rb.m_AtIndex, ofs + testAtIndex, (ofs + testAtIndex) % bufferLength);
            ASSERT_EQ(rb.m_AtIndex, (ofs + testAtIndex) % bufferLength);

            TESTCMP(rb.m_Buffer[(ofs + testAtIndex - 1) % bufferLength], testInfo);
            for(size_t j=0; j<lent2; j++)
            {
                TESTCMP(rb.m_Buffer[(ofs + testAtIndex - 1 - 1 - j) % bufferLength], ibuf[lent2 - 1 - j]);
            }
            for(size_t j=0; j<(bufferLength - lent2 - 1); j++)
            {
                TESTCMP(rb.m_Buffer[(ofs + testAtIndex - 1 - lent2 - 1 - j) % bufferLength], ibuf[bufferLength * 2 - 1 - j]);
            }
        }

        // メモリ破壊していないか、番兵をチェック
        for(int i=0; i<numReserved; i++)
        {
            TESTCMP(buffer[i], sentinelInfo);
            TESTCMP(buffer[numReserved + bufferLength + i], sentinelInfo);
        }

        NN_TEST_RB_LOG("ring buffer test %d/%d end\n", ofs + 1, bufferLength);

        // 次のループに備え、強制的に内部のデータを初期化
        for(size_t i=0; i<bufferLength; i++)
        {
            rb.m_Buffer[i] = defaultInfo;
        }

        // 次のループは、中身が空でオフセットが 1 ずれたものが渡っていく
    }
}    // NOLINT(impl/function_size)
