﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/gc/detail/gc_AsicRegister.h>
#include <nn/gc/detail/gc_Util.h>

#include "testGc_Unit_GcLibrary_Util.h"

using namespace nn::gc;
using namespace nn::gc::detail;

namespace nn { namespace gc {
namespace detail {

// *** ↓ は gc_AsicRegister.cpp のコピー（頻繁な変更はないと思われる） ***

// レジスタの読み出しは AsicRegister クラスのみが行うため、ここでビット定義
// レジスタのビットフィールド定義（C++ の bit field はプラットフォーム依存の為、使用不可）
struct AsicRegisterMap_AsicStatus
{
    typedef BitPack32::Field<0, 8, u8> cdst;
    typedef BitPack32::Field<8, 2, u8> crcrslt;
    typedef BitPack32::Field<12, 1, bool> toerr;
    typedef BitPack32::Field<15, 1, bool> ltc;
    typedef BitPack32::Field<16, 8, u8> asicst;
    typedef BitPack32::Field<24, 2, u8> asicmode;
};
struct AsicRegisterMap_AccessControl1
{
    typedef BitPack32::Field<0, 3, u8> rdclk;
    typedef BitPack32::Field<4, 3, u8> wrclk;
    typedef BitPack32::Field<16, 1, u8> toset;
    typedef BitPack32::Field<20, 4, u8> to;
    typedef BitPack32::Field<31, 1, u8> clksel;
};
struct AsicRegisterMap_ReadWait1Time
{
    typedef BitPack32::Field<0, 26, u32> wait1;
};
struct AsicRegisterMap_ReadWait2Time
{
    typedef BitPack32::Field<0, 26, u32> wait2;
};
struct AsicRegisterMap_WriteWait1Time
{
    typedef BitPack32::Field<0, 26, u32> wait1;
};
struct AsicRegisterMap_WriteWait2Time
{
    typedef BitPack32::Field<0, 26, u32> wait2;
};
struct AsicRegisterMap_PageRemainder
{
    typedef BitPack32::Field<0, 32, u32> pr;
};
struct AsicRegisterMap_LatencyTime
{
    typedef BitPack32::Field<0, 9, u16> ltct;
};
struct AsicRegisterMap_LimitedArea
{
    typedef BitPack32::Field<0, 32, u32> lra;
};
struct AsicRegisterMap_CupVersion
{
    typedef BitPack32::Field<0, 32, u32> cver;
};
struct AsicRegisterMap_AsicVersion
{
    typedef BitPack32::Field<0, 32, u32> ver;
};
struct AsicRegisterMap_Standby2Control
{
    typedef BitPack32::Field<0, 1, bool> stb;
};

} } }

AsicRegister& ar = AsicRegister::GetInstance();

TEST(GcUnitTest, AsicRegisterTest)
{
    // 最初のアドレスのテスト
    BitPack32* const firstRegAddr = &(ar.m_RegisterMap.asicStatus);
    TESTOUT(ar.ConvertRegisterValueAddressToRegisterAddress(firstRegAddr) == 0, "register first address");
    // 最後のアドレスのテスト
    u32 regAddressDiff = &(ar.m_RegisterMap.standby2Control) - firstRegAddr;
    TESTOUT(ar.ConvertRegisterValueAddressToRegisterAddress(&(ar.m_RegisterMap.standby2Control)) == (regAddressDiff * sizeof(u32)), "register last address");
    NN_LOG(" address: %x, %x, diff: %x, %x\n", firstRegAddr, &(ar.m_RegisterMap.standby2Control),
        ar.ConvertRegisterValueAddressToRegisterAddress(&(ar.m_RegisterMap.standby2Control)), regAddressDiff * sizeof(u32));
    // サイズのテスト
    TESTOUT(((regAddressDiff + 1) * sizeof(u32)) == sizeof(ar.m_RegisterArray), "register size");
    // 中間のテスト
    regAddressDiff = &(ar.m_RegisterMap.latencyTime) - firstRegAddr;
    TESTOUT(ar.ConvertRegisterValueAddressToRegisterAddress(&(ar.m_RegisterMap.latencyTime)) == (regAddressDiff * sizeof(u32)), "register middle address");

    // 初期値のテスト
    char buf[AsicRegister::GcAsicRegisterSize];
    memset(buf, 0, sizeof(buf));
    // 初期値の代入：リトルエンディアンなので、4byte中の先頭に書き込めばよい
    buf[(&(ar.m_RegisterMap.accessControl1) - firstRegAddr) * sizeof(u32)] = 0x11;
    buf[(&(ar.m_RegisterMap.latencyTime) - firstRegAddr) * sizeof(u32)] = 0x64;
    buf[(&(ar.m_RegisterMap.limitedArea) - firstRegAddr) * sizeof(u32)] = 0x01;
    TESTCMP(ar.m_RegisterArray, buf, sizeof(buf), "initial value");
    PrintArray(buf, sizeof(buf));
    PrintArray(ar.m_RegisterArray, sizeof(ar.m_RegisterArray));
    char initial_buf[AsicRegister::GcAsicRegisterSize];
    memcpy(initial_buf, buf, sizeof(buf));

    // バイト代入テスト
    {
        char temp = 123;
        ar.SetLatencyCheckValue(temp);
        buf[ar.ConvertRegisterValueAddressToRegisterAddress(&(ar.m_RegisterMap.latencyTime))] = temp;
        TESTCMP(ar.m_RegisterArray, buf, sizeof(buf), "byte test (latencyTime)");
        PrintArray(ar.m_RegisterArray, sizeof(buf));
        PrintArray(buf, sizeof(buf));
    }

    // マルチバイト代入テスト
    {
        u32 temp = (1<<25) + 1234567;
        NN_LOG(" mask, value: %x, %x; masked: %x\n", AsicRegisterMap_ReadWait1Time::wait1::Mask, (temp << AsicRegisterMap_ReadWait1Time::wait1::Pos),
            (~AsicRegisterMap_ReadWait1Time::wait1::Mask & (temp << AsicRegisterMap_ReadWait1Time::wait1::Pos)));

        ar.m_RegisterMap.readWait1Time.Set<AsicRegisterMap_ReadWait1Time::wait1>(temp);
        MemcpyFromU32(buf + ar.ConvertRegisterValueAddressToRegisterAddress(&(ar.m_RegisterMap.readWait1Time)), temp);
        TESTCMP(ar.m_RegisterArray, buf, sizeof(buf), "word test (readWait1Time)");
        PrintArray(ar.m_RegisterArray, sizeof(buf));
        PrintArray(buf, sizeof(buf));
    }

    // ビット代入テスト
    {
        char temp = 15;
        ar.m_RegisterMap.accessControl1.Set<AsicRegisterMap_AccessControl1::to>(temp);
        buf[ar.ConvertRegisterValueAddressToRegisterAddress(&(ar.m_RegisterMap.accessControl1)) + AsicRegisterMap_AccessControl1::to::Pos / 8] = temp << (AsicRegisterMap_AccessControl1::to::Pos % 8);
        TESTCMP(ar.m_RegisterArray, buf, sizeof(buf), "bit test (accessControl1)");
        PrintArray(ar.m_RegisterArray, sizeof(buf));
        PrintArray(buf, sizeof(buf));
    }

    // リセット
    ar.ResetRegister();
    TESTCMP(ar.m_RegisterArray, initial_buf, sizeof(initial_buf), "reset test");
}

