﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <nn/nn_Log.h>

#include <nn/fs.h>
#include <nn/fs/fs_Bis.h>
#include <nn/fs/fs_Code.h>
#include <nn/fs/fs_SdCardForDebug.h>
#include <nn/fs/fs_Result.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/fsUtil/testFs_util_allocator.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/nnt_Argument.h>
#include <nn/sdmmc/sdmmc_Common.h>
#include <nn/gc/gc.h>
#include <nn/gc/detail/gc_Define.h>
#include <nn/gc/writer/gc_Writer.h>

#include <nn/fs/fs_GameCard.h>
#include <nn/fs/fs_GameCardForInspection.h>

#include <nn/spl/spl_Api.h>

#include <nn/init.h>
#include <nn/os.h>
#include <nn/dd.h>

#include "GcCommon.h"

const int64_t WriteUnitSize           = 8 * 1024 * 1024;
const size_t  MemoryHeapSize          = 24 * 1024 * 1024;
const size_t  WorkBufferSize          = nn::util::align_up(nn::gc::GcWorkBufferSize, nn::os::MemoryBlockUnitSize);
const size_t  DataBufferSize          = WriteUnitSize;
const size_t  MemoryHeapSizeForMalloc = MemoryHeapSize - DataBufferSize - WorkBufferSize;
const uint32_t SecureAreaStartPageAddr = 0x80000;

char*     g_pWorkBuffer;
uintptr_t g_WorkBufferAddress;
char*     g_pDataBuffer;
uintptr_t g_DataBufferAddress;
uintptr_t g_MallocBufferAddress;
char*     g_pReadBufferFromFs;

nn::dd::DeviceAddressSpaceType g_Das;
nn::dd::DeviceVirtualAddress g_WorkBufferDeviceVirtualAddress;
nn::dd::DeviceVirtualAddress g_DataBufferDeviceVirtualAddress;

nn::os::EventType g_CardDetectionEvent;

void PrintArray(const char* buffer, const size_t bufferLength) NN_NOEXCEPT
{
    for(size_t i=0; i<bufferLength; i++)
    {
        if(i!=0 && i % 16 == 0)
        {
            NN_LOG("\n");
        }

        NN_LOG("%02X ", buffer[i]);
    }
    NN_LOG("\n");
}

extern "C" void nninitStartup()
{
    nn::Result result = nn::os::SetMemoryHeapSize(MemoryHeapSize);
    NN_ABORT_UNLESS(result.IsSuccess(), "Cannot set MemoryHeapSize.");

    // メモリヒープから WorkBuffer で使用するメモリ領域を確保
    result = nn::os::AllocateMemoryBlock( &g_WorkBufferAddress, WorkBufferSize );
    NN_ABORT_UNLESS(result.IsSuccess(), "Cannot AllocateMemoryBlock0.");

    // メモリヒープから DataBuffer で使用するメモリ領域を確保
    result = nn::os::AllocateMemoryBlock( &g_DataBufferAddress, DataBufferSize );
    NN_ABORT_UNLESS(result.IsSuccess(), "Cannot AllocateMemoryBlock1.");

    // メモリヒープから malloc で使用するメモリ領域を確保
    result = nn::os::AllocateMemoryBlock( &g_MallocBufferAddress, MemoryHeapSizeForMalloc );
    NN_ABORT_UNLESS(result.IsSuccess(), "Cannot AllocateMemoryBlock2.");

    // malloc 用のメモリ領域を設定する
    nn::init::InitializeAllocator( reinterpret_cast<void*>(g_MallocBufferAddress), MemoryHeapSizeForMalloc );
}

void* Allocate(std::size_t size)
{
    return malloc(size);
}

void Deallocate(void* p, std::size_t size)
{
    NN_UNUSED(size);
    free(p);
}

bool CanMountAndOpenFile(nn::fs::FileHandle* pFileHandle, char* filePath)
{
    nn::fs::SetAllocator(Allocate, Deallocate);
    NN_LOG("Mount\n");
    nn::Result result = nn::fs::MountHostRoot();
    if(result.IsFailure())
    {
        NN_LOG("Mount Root Failure\n");
        return false;
    }
    NN_LOG("Open File\n");
    result = nn::fs::OpenFile(pFileHandle, filePath, nn::fs::OpenMode::OpenMode_Read);
    if(result.IsFailure())
    {
        NN_LOG("Open File Failure\n");
        return false;
    }
    return true;
}

bool CanGetFilePath(char* filePath, size_t filePathSize)
{
    int argC = nnt::GetHostArgc();
    if(argC <= 1)
    {
        NN_LOG("No Argments!\n");
        return false;
    }
    char** argV = nnt::GetHostArgv();
    NN_LOG("%s\n", argV[1]);
    memcpy(filePath ,argV[1], filePathSize);
    return true;
}

void ReadWriteImpl(nn::fs::FileHandle file, int64_t unitIndex, size_t readWriteSize)
{
    int64_t byteOffset = unitIndex * WriteUnitSize;
    int pageOffset     = byteOffset / nn::gc::GcPageSize;
    int pageUnitNum    = readWriteSize / nn::gc::GcPageSize;
    memset(g_pReadBufferFromFs, 0, readWriteSize);
    if(nn::fs::ReadFile(file, byteOffset, g_pReadBufferFromFs, readWriteSize).IsFailure())
    {
        NN_LOG("\n Read HostIo File Failure!!!\n");
        return;
    }
    memcpy(g_pDataBuffer, g_pReadBufferFromFs, readWriteSize);
    if(nn::gc::writer::Write(g_pDataBuffer, readWriteSize, pageOffset, pageUnitNum).IsFailure())
    {
        NN_LOG("\n Write With Verify Failure!!!\n");
        return;
    }
    return;
}

void WriteInputFileToDevCard(nn::fs::FileHandle file)
{
    int64_t fileSize = 0;
    nn::fs::GetFileSize(&fileSize, file);
    NN_LOG("Write File Size is %lld\n",fileSize);
    int64_t maxUnitNum = fileSize / WriteUnitSize;
    for(int64_t i = 0; i < maxUnitNum; i++)
    {
        ReadWriteImpl(file, i, WriteUnitSize);
    }
    int64_t restSize = fileSize % WriteUnitSize;
    if(restSize != 0)
    {
        ReadWriteImpl(file, maxUnitNum, restSize);
    }
    return;
}

// 挿抜があった際に呼ばれるコールバック関数：GC のスレッドをブロックしてしまうため、イベントのシグナルのみにとどめる必要がある
void SignalDetectionEvents(void *pParameter)
{
    nn::os::SignalEvent( &g_CardDetectionEvent );
}

void InitializeDriver()
{
    nn::spl::InitializeForFs();

    NN_LOG("Read From Partition\n");

    ReadGcCalibrationPartitionToSetInternalKeys();

    NN_LOG("Finalize gc driver from fs\n");

    nn::fs::FinalizeGameCardDriver();

    NN_LOG("Initialize gc driver.\n");

    // クロック制御は pcv 経由で行う
    nn::sdmmc::SwitchToPcvClockResetControl();

    // 監視するイベントを用意する
    nn::os::InitializeEvent( &g_CardDetectionEvent, false, nn::os::EventClearMode_ManualClear );

    // バッファの設定
    g_pWorkBuffer = reinterpret_cast<char*>(g_WorkBufferAddress);
    g_pDataBuffer = reinterpret_cast<char*>(g_DataBufferAddress);

    SetupDeviceAddressSpace(&g_Das, nn::dd::DeviceName::DeviceName_Sdmmc2a);

    g_WorkBufferDeviceVirtualAddress = MapDeviceAddressSpaceAligned(&g_Das, g_WorkBufferAddress, WorkBufferSize, 0);
    nn::gc::Initialize(g_pWorkBuffer, WorkBufferSize, g_WorkBufferDeviceVirtualAddress);

    g_DataBufferDeviceVirtualAddress = MapDeviceAddressSpaceAligned(&g_Das, g_DataBufferAddress, DataBufferSize, g_WorkBufferDeviceVirtualAddress + WorkBufferSize);
    nn::gc::RegisterDeviceVirtualAddress(g_DataBufferAddress, DataBufferSize, g_DataBufferDeviceVirtualAddress);

    g_pReadBufferFromFs = reinterpret_cast<char*>(malloc(WriteUnitSize));

    // コールバック関数を登録する
    nn::gc::RegisterDetectionEventCallback( SignalDetectionEvents, nullptr );
}

void FinalizeDriver()
{
    nn::gc::UnregisterDeviceVirtualAddress(g_DataBufferAddress, DataBufferSize, g_DataBufferDeviceVirtualAddress);
    UnmapDeviceAddressSpaceAligned(&g_Das, g_DataBufferAddress, DataBufferSize, g_DataBufferDeviceVirtualAddress);

    nn::gc::Finalize();
    UnmapDeviceAddressSpaceAligned(&g_Das, g_WorkBufferAddress, WorkBufferSize, g_WorkBufferDeviceVirtualAddress);

    CleanDeviceAddressSpace(&g_Das, nn::dd::DeviceName::DeviceName_Sdmmc2a);

    free(g_pReadBufferFromFs);
    nn::os::FreeMemoryBlock(g_MallocBufferAddress, MemoryHeapSizeForMalloc);
    nn::os::FreeMemoryBlock(g_DataBufferAddress, DataBufferSize);
    nn::os::FreeMemoryBlock(g_WorkBufferAddress, WorkBufferSize);

    nn::spl::Finalize();
}

extern "C" void nnMain()
{
    NN_LOG("Hello, I'm gc sample program.\n");

    // ファイルパス取得
    char filePath[256];
    if(!CanGetFilePath(filePath,sizeof(filePath)))
    {
        return;
    }
    NN_LOG("file : %s\n",filePath);
    nn::fs::FileHandle file;
    if(!CanMountAndOpenFile(&file, filePath))
    {
        return;
    }

    // GC 用の処理
    NN_LOG("\n\nFinished. Now I'm going to initialize gc library!\n");

    InitializeDriver();

    NN_LOG("\n\nFinished. Change to Write Mode\n");

    // ------------------- ここから書き込み処理 -------------------

    nn::gc::writer::ChangeMode(nn::gc::writer::AsicMode_WriteMode);


    while (!nn::gc::IsCardInserted())
    {
        NN_LOG("@ card is not inserted.\n");
        // カード Inserted イベントを待つ：実際にはメニューの人が GPIO イベントを待つ
        NN_LOG("@ Waiting for card accessible event after card insertion ... \n");
        nn::os::WaitEvent( &g_CardDetectionEvent );
        // イベントをクリア
        nn::os::ClearEvent( &g_CardDetectionEvent );
    }

    NN_LOG("\n\nFinished. Activate For Writer\n");

    if(nn::gc::writer::ActivateForWriter().IsFailure())
    {
        NN_LOG("\n Activate For Writer failed. ");
        FinalizeDriver();
        return;
    }

    NN_LOG("\n\nFinished.\n");

    int64_t msize;
    nn::gc::writer::GetCardAvailableRawSize(&msize);
    NN_LOG("Card memory size: %llu byte\n", msize);

    NN_LOG("Erase All and Write Parameter\n");

    if(nn::gc::writer::EraseAndWriteParameter(nn::gc::writer::MemorySize_1GB, SecureAreaStartPageAddr).IsFailure())
    {
        NN_LOG("\n Activate For EraseAndWriteParameter failed. \n");
        FinalizeDriver();
        return;
    }

    NN_LOG("\n\nFinished. Write\n");

    WriteInputFileToDevCard(file);

    NN_LOG("\n\nWrite Success\n");

    // ------------------- ここから読み込み・アクセス確認 -------------------

    NN_LOG("\n\nFinished. Change to Read Mode\n");

    nn::gc::writer::ChangeMode(nn::gc::writer::AsicMode_ReadMode);

    if (!nn::gc::IsCardInserted())
    {
        NN_LOG("@ card is not inserted.\n");
        // カード Inserted イベントを待つ：実際にはメニューの人が GPIO イベントを待つ
        NN_LOG("@ Waiting for card accessible event after card insertion ... \n");
        nn::os::WaitEvent( &g_CardDetectionEvent );
        // イベントをクリア
        nn::os::ClearEvent( &g_CardDetectionEvent );
    }

    NN_LOG("\n\nFinished. Activate\n");

    if(nn::gc::Activate().IsFailure())
    {
        NN_LOG("\n Activate Failure, Please Confirm Write Binary Data assigned.\n");
        FinalizeDriver();
        return;
    }

    NN_LOG("\n\nFinished. Change To Card Secure\n");

    if(nn::gc::SetCardToSecureMode().IsFailure())
    {
        NN_LOG("\n Change To Card Secure Mode Failure.Please Confirm Write Binary Data assigned.\n");
        FinalizeDriver();
        return;
    }

    NN_LOG("\n\nFinished. Clear\n");

    FinalizeDriver();

    NN_LOG("\n\nWrite Read Success\n");
    return;
}

