﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>
#include "../../Common/testGamePad_Common.h"
#include "../../Common/testGamePad_IrSensor.h"

namespace nnt { namespace irsensor {
    nn::irsensor::IrCameraHandle IrSensorTest::s_Handles[nnt::gamepad::NpadIdCountMax];
    int IrSensorTest::s_HandleCount;
}} // namaspace nnt::irsensor

namespace
{

const int VerifyStatesTestCountMax = 5;                                         // ステータスのテストを実施する回数
const nn::TimeSpan VerifyStatesTestLimitTime = nn::TimeSpan::FromSeconds(5);    // ステータスが更新されない状態の許容時間
const int NotReadyWaitLoopCountMax = 1000;                                      // ステータス取得の準備が完了するまでに待つ最大のループ回数
const nn::TimeSpan PollingInterval = nn::TimeSpan::FromMilliSeconds(15);        // ステータス取得の準備が完了していない時のポーリング間隔

class PointingProcessorTest : public ::nnt::irsensor::IrSensorTest
{
protected:
    void DoTestRunningProcessor(const nn::irsensor::IrCameraHandle& handle) NN_NOEXCEPT;
    void DoTestReadingPointingStates(const nn::irsensor::IrCameraHandle& handle) NN_NOEXCEPT;
    void DoTestVerifyPointingState(const nn::irsensor::PointingProcessorState& state) NN_NOEXCEPT;
    void DoTestStopImageProcessor(const nn::irsensor::IrCameraHandle& handle) NN_NOEXCEPT;
    void ReadingStateTest(const nn::irsensor::IrCameraHandle& handle) NN_NOEXCEPT;
};

//==================================================
// プロセッサの起動までの時間を計測
//==================================================
void PointingProcessorTest::DoTestRunningProcessor(const nn::irsensor::IrCameraHandle& handle) NN_NOEXCEPT
{
    int counter = 0;
    nn::TimeSpanType startTime = nn::os::GetSystemTick().ToTimeSpan();      // 計測を開始した時間を記憶

    while (NN_STATIC_CONDITION(true))
    {
        auto status = nn::irsensor::GetImageProcessorStatus(handle);

        if (status == nn::irsensor::ImageProcessorStatus_Stopped)
        {
            counter++;
            nn::os::SleepThread(PollingInterval);
            ASSERT_LT(counter, NotReadyWaitLoopCountMax);
        }
        else if (status == nn::irsensor::ImageProcessorStatus_Running)
        {
            NN_LOG("#     Image processor started running. (%lld ms)\n", nn::os::GetSystemTick().ToTimeSpan().GetMilliSeconds() - startTime.GetMilliSeconds());
            break;
        }
    }
    NNT_IRSENSOR_EXIT_SUCCESS;
}

//==================================================
// ステータスの取得と更新時間の計測
//==================================================
void PointingProcessorTest::DoTestReadingPointingStates(const nn::irsensor::IrCameraHandle& handle) NN_NOEXCEPT
{
    NN_LOG("#     <GetPointingProcessorStates>\n");
    int64_t prevSamplingNumber = -1;

    //-------------------------------------------------------------------------------
    // プロセッサが起動するまでの時間の計測します
    //-------------------------------------------------------------------------------
    nn::irsensor::RunPointingProcessor(handle);
    DoTestRunningProcessor(handle);

    //-------------------------------------------------------------------------------
    // ステートが更新されるまでにかかる時間を計測します
    //-------------------------------------------------------------------------------
    for (int i = 0; i < VerifyStatesTestCountMax; ++i)
    {
        nn::irsensor::PointingProcessorState state[nn::irsensor::PointingProcessorStateCountMax];
        auto samplingNumber = prevSamplingNumber;
        auto stateCount = 0;
        auto startTime = nn::os::GetSystemTick().ToTimeSpan();                       // 計測を開始した時間を記憶

        NN_LOG("#     Test : [%2d/%2d]\n", i + 1, VerifyStatesTestCountMax);
        while (NN_STATIC_CONDITION(true))
        {
            nn::Result result = nn::irsensor::GetPointingProcessorStates(state, &stateCount, nn::irsensor::PointingProcessorStateCountMax, handle);
            if (nn::irsensor::ResultIrsensorNotReady::Includes(result) == false)
            {
                if (nn::irsensor::ResultIrsensorDeviceError::Includes(result))
                {
                    NN_LOG("############# ResultIrsensorDeviceError #############\n");
                    return;
                }
                samplingNumber = state[0].samplingNumber;
            }
            if (prevSamplingNumber != samplingNumber)
            {
                break;
            }
            ASSERT_LT(nn::os::GetSystemTick().ToTimeSpan().GetMilliSeconds() - startTime.GetMilliSeconds(), VerifyStatesTestLimitTime.GetMilliSeconds());
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1));
        }
        NN_LOG("#             UpdatePointingProcessorStates (%lld ms)\n", nn::os::GetSystemTick().ToTimeSpan().GetMilliSeconds() - startTime.GetMilliSeconds());
        NN_LOG("#             SamplingNumber %lld -> %lld\n", prevSamplingNumber, samplingNumber);
        prevSamplingNumber = samplingNumber;
        //-------------------------------------------------------------------------------
        // ステータスの検証を行います
        //-------------------------------------------------------------------------------
        NN_LOG("#         VerifyPointingState (StateCount : %d)\n", stateCount);
        for (auto j = 0; j < stateCount; ++j)
        {
            DoTestVerifyPointingState(state[j]);
        }
    }
    NN_LOG("#     SUCCESS\n");
    NNT_IRSENSOR_EXIT_SUCCESS;
}

//==================================================
// ステータスの検証
//==================================================
void PointingProcessorTest::DoTestVerifyPointingState(const nn::irsensor::PointingProcessorState& state) NN_NOEXCEPT
{
    ASSERT_GE(state.samplingNumber, 0);
    ASSERT_GE(state.timeStamp.GetNanoSeconds(), 0);

    ASSERT_GE(state.position.x, -1.f);
    ASSERT_LE(state.position.x, 1.f);
    ASSERT_GE(state.position.y, -1.f);
    ASSERT_LE(state.position.y, 1.f);

    ASSERT_GE(state.pointingStatus, nn::irsensor::PointingStatus_DataValid);
    ASSERT_LE(state.pointingStatus, nn::irsensor::PointingStatus_LackOfObjects);

    NNT_IRSENSOR_EXIT_SUCCESS;
}

//==================================================
// プロセッサの停止までの時間を計測
//==================================================
void PointingProcessorTest::DoTestStopImageProcessor(const nn::irsensor::IrCameraHandle& handle) NN_NOEXCEPT
{
    nn::irsensor::StopImageProcessorAsync(handle);

    nn::TimeSpanType startTime = nn::os::GetSystemTick().ToTimeSpan();
    auto counter = 0;
    while (NN_STATIC_CONDITION(true))
    {
        auto status = nn::irsensor::GetImageProcessorStatus(handle);

        if (status == nn::irsensor::ImageProcessorStatus_Running)
        {
            counter++;
            nn::os::SleepThread(PollingInterval);
            ASSERT_LT(counter, NotReadyWaitLoopCountMax);
        }
        else
        {
            NN_LOG("#     StopImageProcessor (%lld ms)\n", nn::os::GetSystemTick().ToTimeSpan().GetMilliSeconds() - startTime.GetMilliSeconds());
            break;
        }
    }
    NNT_IRSENSOR_EXIT_SUCCESS;
}

//==================================================
// ステータス取得のテスト
//==================================================
void PointingProcessorTest::ReadingStateTest(const nn::irsensor::IrCameraHandle& handle) NN_NOEXCEPT
{
    EXPECT_LT(0, s_HandleCount);

    DoTestReadingPointingStates(handle);
    DoTestStopImageProcessor(handle);

    NNT_IRSENSOR_EXIT_SUCCESS;
}

//==================================================
// 状態遷移のテスト
//==================================================

TEST_F(PointingProcessorTest, StateReading)
{
    nnt::gamepad::Initialize();

    // コントローラの再接続
    nnt::gamepad::DisconnectAll();
    nnt::gamepad::ConnectAll();

    NN_LOG("#\n# Start Pointing Processor Test.\n#\n");
    for (const auto& handle : nnt::irsensor::IrSensorTest::s_Handles)
    {
        // IRカメラが利用できない場合はスキップ
        if (GetIrCameraStatus(handle) != nn::irsensor::IrCameraStatus_Available)
        {
            continue;
        }

        NN_LOG("#-----------------------------------------------\n");
        NN_LOG("# IrCameraHandle : 0x%x\n", handle);
        NN_LOG("#-----------------------------------------------\n");
        ReadingStateTest(handle);
    }
    nnt::gamepad::DisconnectAll();
}

// USB有線接続が有効時にステータスが正常に取得できるか
TEST_F(PointingProcessorTest, StateReadingForUsb)
{
    nnt::gamepad::Initialize();

    // USB 有線接続を有効にする
    ::nnt::gamepad::EnableUsbConnect();

    // コントローラの再接続
    nnt::gamepad::DisconnectAll();
    nnt::gamepad::ConnectAll();

    NN_LOG("#\n# Start Pointing Processor Test.\n#\n");
    for (const auto& handle : nnt::irsensor::IrSensorTest::s_Handles)
    {
        // IRカメラが利用できない場合はスキップ
        if (GetIrCameraStatus(handle) != nn::irsensor::IrCameraStatus_Available)
        {
            continue;
        }

        NN_LOG("#-----------------------------------------------\n");
        NN_LOG("# IrCameraHandle : 0x%x\n", handle);
        NN_LOG("#-----------------------------------------------\n");
        ReadingStateTest(handle);
    }

    nnt::gamepad::DisconnectAll();
}

} // namespace
