﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/hid/hid_NpadCommonTypes.h>
#include <nn/hid/hid_NpadFullKey.h>
#include <nn/hid/hid_NpadHandheld.h>
#include <nn/hid/hid_NpadJoyDual.h>
#include <nn/hid/hid_NpadJoyLeft.h>
#include <nn/hid/hid_NpadJoyRight.h>
#include <nn/hid/hid_NpadGc.h>
#include <nn/hid/system/hid_NpadSystem.h>
#include <nn/hid/system/hid_NpadSystemExt.h>

namespace nnt { namespace gamepad {

/**
* @brief       GamePad テストでペアリング登録されるコントローラ台数(Bluetoothコントローラ台数)です。
*
* @details     参考: HID-APIチーム／検討情報／ジョイコンQA／自動テスト環境
*                    http://spdlybra.nintendo.co.jp/confluence/pages/viewpage.action?pageId=162539026
*/
const int DevicePairingCountMax = 6;

/**
* @brief       GamePad テストで登録される USB コントローラの台数です。
*
* @details     参考: HID-APIチーム／検討情報／ジョイコンQA／自動テスト環境
*                    http://spdlybra.nintendo.co.jp/confluence/pages/viewpage.action?pageId=162539026
*/
const int UsbPadsCountMax = 1;

/**
 * @brief       GamePad テストで接続されるコントローラ台数です。
 *
 * @details     参考: HID-APIチーム／検討情報／ジョイコンQA／自動テスト環境
 *                    http://spdlybra.nintendo.co.jp/confluence/pages/viewpage.action?pageId=162539026
 */
const int DeviceCountMax = DevicePairingCountMax + UsbPadsCountMax;

/**
 * @brief       GamePad テストでサポートされている Npad の操作形態です。
 */
const nn::hid::NpadStyleSet NpadStyleSets = (
    nn::hid::NpadStyleFullKey::Mask           |
    nn::hid::NpadStyleHandheld::Mask          |
    nn::hid::NpadStyleJoyDual::Mask           |
    nn::hid::NpadStyleJoyLeft::Mask           |
    nn::hid::NpadStyleJoyRight::Mask          |
    nn::hid::NpadStyleGc::Mask                |
    nn::hid::system::NpadStyleSystem::Mask    |
    nn::hid::system::NpadStyleSystemExt::Mask
);

/**
* @brief       NpadIDから取得できる、コントローラの種類を現す列挙体です。
*/
enum ControllerType {
    Undefined,
    Handheld,
    Kuina,
    JoyLeft,
    JoyRight,
    ProCon,
    UsbCon
};

/**
 * @brief       GamePad テストでサポートされている NpadId です。
 */
const ::nn::hid::NpadIdType NpadIds[] = {
    ::nn::hid::NpadId::No1,
    ::nn::hid::NpadId::No2,
    ::nn::hid::NpadId::No3,
    ::nn::hid::NpadId::No4,
    ::nn::hid::NpadId::No5,
    ::nn::hid::NpadId::No6,
    ::nn::hid::NpadId::No7,
    ::nn::hid::NpadId::No8,
    ::nn::hid::NpadId::Handheld,
};

const int NpadIdCountMax = NN_ARRAY_SIZE(NpadIds); //<< サポートされている NpadId の数

/**
 * @brief       Npad の初期化を行います。
 *
 * @details     任意の操作形態と NpadId の組み合わせを有効化します。
 *
 */
void Initialize() NN_NOEXCEPT;

/**
 * @brief       ペアリング済の全てのデバイスを接続します。
 *
 * @details     ペアリング済の Bluetooth アドレスとして列挙される全てのデバイスに対して、
 *              SendConnectionTrigger() を呼び出すことで無線接続します。
 *
 */
void ConnectAll() NN_NOEXCEPT;

/**
 * @brief       接続状態にある全てのコントローラを切断します。
 *
 * @details     接続状態にある全てのコントローラを切断します。
 *
 */
void DisconnectAll() NN_NOEXCEPT;

/**
 * @brief       ProコントローラのUSB有線接続機能を有効化します
 *
 * @details     ProコントローラのUSB有線接続機能を、
 *              EnableUsbFullKeyController() を呼び出すことで有効化します。
 *
 */
void EnableUsbConnect() NN_NOEXCEPT;

/**
 * @brief       ProコントローラのUSB有線接続機能を無効化します
 *
 * @details     ProコントローラのUSB有線接続機能を、
 *              EnableUsbFullKeyController() を呼び出すことで無効化します。
 *
 */
void DisableUsbConnect() NN_NOEXCEPT;

/**
 * @brief       任意の USB コントローラの接続処理を行います。
 *
 * @details     呼び出し時点で USB 接続されている任意の USB コントローラを
 *              Npad として登録します。
 */
void ConnectUsbPads() NN_NOEXCEPT;

/**
 * @brief       任意の USB コントローラの切断処理を行います。
 *
 * @details     呼び出し時点で Npad として登録されている任意の USB コントローラの
 *              登録を削除します。
 */
void DisconnectUsbPads() NN_NOEXCEPT;

/**
 * @brief       NpadIDから、現在検証機材に接続されているコントローラの種類を判別します
 *
 * @details     判断できる種別はHandHeld、KUINA、ProCon、USB専用コントローラの4種類
 *
 * @param[in]   npadId                   UniquePad を取得する Npad の NpadIdType
 *
 * @return      Npadに対応したコントローラの種類
 */
nnt::gamepad::ControllerType GetControllerTypeFromNpad(const nn::hid::NpadIdType& npadId) NN_NOEXCEPT;

/**
 * @brief       ペアリング済デバイスの状態をログ出力します。
 */
void PrintRegisteredDevices() NN_NOEXCEPT;

/**
 * @brief       UniquePad の状態をログ出力します。
 */
void PrintUniquePads() NN_NOEXCEPT;

/**
* @brief       接続済 USB コントローラの台数を取得します。
*/
int GetConnectedUsbPadsCount() NN_NOEXCEPT;

}} // namespace nnt::gamepad
