﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>
#include <nnt/gtest/gtest.h>
#include <nnd/ftm/ftm.h>

namespace nnt { namespace ftm {

// 割り込み使用時の正常フローのテスト
TEST(WithInterruptTest, Sequence)
{
    using namespace nnd::ftm;

    Initialize();

    nn::Result result = ResetDevice();
    EXPECT_TRUE(result.IsSuccess());

    result = SleepOutDevice();
    EXPECT_TRUE(result.IsSuccess());

    nn::os::SystemEventType event;
    result = BindInterrupt(&event);
    EXPECT_TRUE(result.IsSuccess());

    result = ActivateSensing();
    EXPECT_TRUE(result.IsSuccess());

    char readData[FtmMaxEventReportCount * FtmMaxEventReportByteSize];
    int leftCount;
    int readCount;
    bool isOverflow;
    result = ReadEventReports(readData, &leftCount, &readCount, &isOverflow, GetMaxEventReportCount());
    EXPECT_TRUE(result.IsSuccess());

    EventReport report[FtmMaxEventReportCount];
    ParseEventReports(report, readData, readCount);

    result = UnbindInterrupt();
    EXPECT_TRUE(result.IsSuccess());

    result = DeactivateSensing();
    EXPECT_TRUE(result.IsSuccess());

    result = ResetDevice();
    EXPECT_TRUE(result.IsSuccess());

    Finalize();
}

// 割り込み不使用時の正常フローのテスト
TEST(WithoutInterruptTest, Sequence)
{
    using namespace nnd::ftm;

    Initialize();

    nn::Result result = ResetDevice();
    EXPECT_TRUE(result.IsSuccess());

    result = SleepOutDevice();
    EXPECT_TRUE(result.IsSuccess());

    result = ActivateSensing();
    EXPECT_TRUE(result.IsSuccess());

    char readData[FtmMaxEventReportCount * FtmMaxEventReportByteSize];
    int leftCount;
    int readCount;
    bool isOverflow;
    result = ReadEventReports(readData, &leftCount, &readCount, &isOverflow, GetMaxEventReportCount());
    EXPECT_TRUE(result.IsSuccess());

    EventReport report[FtmMaxEventReportCount];
    ParseEventReports(report, readData, readCount);

    result = DeactivateSensing();
    EXPECT_TRUE(result.IsSuccess());

    result = ResetDevice();
    EXPECT_TRUE(result.IsSuccess());

    Finalize();
}

// BindInterrupt() が正しい場所でのみ呼び出せることのテスト
TEST(BindInterruptTest, Sequence)
{
    using namespace nnd::ftm;

    nn::os::SystemEventType event;

    // 必要な処理を一切呼ばない状態での呼び出し（期待値 : 失敗）
    EXPECT_DEATH_IF_SUPPORTED(BindInterrupt(&event), "");

    // Initialize() を呼んだ時点での呼び出し（期待値 : 成功）
    Initialize();
    nn::Result result = BindInterrupt(&event);
    EXPECT_TRUE(result.IsSuccess());
    result = UnbindInterrupt();
    EXPECT_TRUE(result.IsSuccess());

    // ResetDevice() を呼んだ時点での呼び出し（期待値 : 成功）
    result = ResetDevice();
    EXPECT_TRUE(result.IsSuccess());
    result = BindInterrupt(&event);
    EXPECT_TRUE(result.IsSuccess());
    result = UnbindInterrupt();
    EXPECT_TRUE(result.IsSuccess());

    // SleepInDevice() を呼んだ時点での呼び出し（期待値 : 成功）
    result = SleepInDevice();
    EXPECT_TRUE(result.IsSuccess());
    result = BindInterrupt(&event);
    EXPECT_TRUE(result.IsSuccess());
    result = UnbindInterrupt();
    EXPECT_TRUE(result.IsSuccess());

    // SleepOutDevice() を呼んだ時点での呼び出し（期待値 : 成功）
    result = SleepOutDevice();
    EXPECT_TRUE(result.IsSuccess());
    result = BindInterrupt(&event);
    EXPECT_TRUE(result.IsSuccess());
    result = UnbindInterrupt();
    EXPECT_TRUE(result.IsSuccess());

    // ActivateSensing() を呼んだ時点での呼び出し（期待値 : 成功）
    result = ActivateSensing();
    EXPECT_TRUE(result.IsSuccess());
    result = BindInterrupt(&event);
    EXPECT_TRUE(result.IsSuccess());
    result = UnbindInterrupt();
    EXPECT_TRUE(result.IsSuccess());

    // ReadEventReports() を呼んだ時点での呼び出し（期待値 : 成功）
    char readData[FtmMaxEventReportCount * FtmMaxEventReportByteSize];
    int leftCount;
    int readCount;
    bool isOverflow;
    result = ReadEventReports(readData, &leftCount, &readCount, &isOverflow, GetMaxEventReportCount());
    EXPECT_TRUE(result.IsSuccess());
    result = BindInterrupt(&event);
    EXPECT_TRUE(result.IsSuccess());
    result = UnbindInterrupt();
    EXPECT_TRUE(result.IsSuccess());

    // ParseEventReports() を呼んだ時点での呼び出し（期待値 : 成功）
    EventReport report[FtmMaxEventReportCount];
    ParseEventReports(report, readData, readCount);
    result = BindInterrupt(&event);
    EXPECT_TRUE(result.IsSuccess());
    result = UnbindInterrupt();
    EXPECT_TRUE(result.IsSuccess());

    // DeactivateSensing() を呼んだ時点での呼び出し（期待値 : 成功）
    DeactivateSensing();
    result = BindInterrupt(&event);
    EXPECT_TRUE(result.IsSuccess());
    result = UnbindInterrupt();
    EXPECT_TRUE(result.IsSuccess());

    // ResetDevice() を呼んだ時点での呼び出し（期待値 : 成功）
    result = ResetDevice();
    EXPECT_TRUE(result.IsSuccess());
    result = BindInterrupt(&event);
    EXPECT_TRUE(result.IsSuccess());
    result = UnbindInterrupt();
    EXPECT_TRUE(result.IsSuccess());

    // Finalize() を呼んだ時点での呼び出し（期待値 : 失敗）
    Finalize();
    EXPECT_DEATH_IF_SUPPORTED(BindInterrupt(&event), "");
}

// ActivateSensing() が正しい場所でのみ呼び出せることのテスト
TEST(ActivateSensingTest, Sequence)
{
    using namespace nnd::ftm;

    // 必要な処理を一切呼ばない状態での呼び出し（期待値 : 失敗）
    EXPECT_DEATH_IF_SUPPORTED(ActivateSensing(), "");

    // Initialize() を呼んだ時点での呼び出し（期待値 : 失敗）
    Initialize();
    EXPECT_DEATH_IF_SUPPORTED(ActivateSensing(), "");

    // ResetDevice() を呼んだ時点での呼び出し（期待値 : 失敗）
    nn::Result result = ResetDevice();
    EXPECT_TRUE(result.IsSuccess());
    EXPECT_DEATH_IF_SUPPORTED(ActivateSensing(), "");

    // SleepInDevice() を呼んだ時点での呼び出し（期待値 : 失敗）
    result = SleepInDevice();
    EXPECT_TRUE(result.IsSuccess());
    EXPECT_DEATH_IF_SUPPORTED(ActivateSensing(), "");

    // SleepOutDevice() を呼んだ時点での呼び出し（期待値 : 成功）
    result = SleepOutDevice();
    EXPECT_TRUE(result.IsSuccess());
    result = ActivateSensing();
    EXPECT_TRUE(result.IsSuccess());
    result = DeactivateSensing();
    EXPECT_TRUE(result.IsSuccess());

    // BindInterrupt() を呼んだ時点での呼び出し（期待値 : 成功）
    nn::os::SystemEventType event;
    result = BindInterrupt(&event);
    EXPECT_TRUE(result.IsSuccess());
    result = ActivateSensing();
    EXPECT_TRUE(result.IsSuccess());

    // DeactivateSensing() を呼んだ時点での呼び出し（期待値 : 成功）
    result = DeactivateSensing();
    EXPECT_TRUE(result.IsSuccess());
    result = ActivateSensing();
    EXPECT_TRUE(result.IsSuccess());
    result = DeactivateSensing();
    EXPECT_TRUE(result.IsSuccess());

    // UnbindInterrupt() を呼んだ時点での呼び出し（期待値 : 成功）
    result = UnbindInterrupt();
    EXPECT_TRUE(result.IsSuccess());
    result = ActivateSensing();
    EXPECT_TRUE(result.IsSuccess());
    result = DeactivateSensing();
    EXPECT_TRUE(result.IsSuccess());

    // ResetDevice() を呼んだ時点での呼び出し（期待値 : 失敗）
    result = ResetDevice();
    EXPECT_TRUE(result.IsSuccess());
    EXPECT_DEATH_IF_SUPPORTED(ActivateSensing(), "");

    // Finalize() を呼んだ時点での呼び出し（期待値 : 失敗）
    Finalize();
    EXPECT_DEATH_IF_SUPPORTED(ActivateSensing(), "");
}

// ReadEventReports() が正しい場所でのみ呼び出せることのテスト
TEST(ReadEventReportsTest, Sequence)
{
    using namespace nnd::ftm;

    // Initialize() を呼んだ時点での呼び出し（期待値 : 失敗）
    Initialize();
    char readData[FtmMaxEventReportCount * FtmMaxEventReportByteSize];
    int leftCount;
    int readCount;
    bool isOverflow;
    EXPECT_DEATH_IF_SUPPORTED(ReadEventReports(readData, &leftCount, &readCount, &isOverflow, GetMaxEventReportCount()), "");

    // ResetDevice() を呼んだ時点での呼び出し（期待値 : 失敗）
    nn::Result result = ResetDevice();
    EXPECT_TRUE(result.IsSuccess());
    EXPECT_DEATH_IF_SUPPORTED(ReadEventReports(readData, &leftCount, &readCount, &isOverflow, GetMaxEventReportCount()), "");

    // SleepInDevice() を呼んだ時点での呼び出し（期待値 : 失敗）
    result = SleepInDevice();
    EXPECT_TRUE(result.IsSuccess());
    EXPECT_DEATH_IF_SUPPORTED(ReadEventReports(readData, &leftCount, &readCount, &isOverflow, GetMaxEventReportCount()), "");

    // SleepOutDevice() を呼んだ時点での呼び出し（期待値 : 失敗）
    result = SleepOutDevice();
    EXPECT_TRUE(result.IsSuccess());
    EXPECT_DEATH_IF_SUPPORTED(ReadEventReports(readData, &leftCount, &readCount, &isOverflow, GetMaxEventReportCount()), "");

    // BindInterrupt() を呼んだ時点での呼び出し（期待値 : 失敗）
    nn::os::SystemEventType event;
    result = BindInterrupt(&event);
    EXPECT_TRUE(result.IsSuccess());
    EXPECT_DEATH_IF_SUPPORTED(ReadEventReports(readData, &leftCount, &readCount, &isOverflow, GetMaxEventReportCount()), "");

    // ActivateSensing() を呼んだ時点での呼び出し（期待値 : 成功）
    result = ActivateSensing();
    EXPECT_TRUE(result.IsSuccess());
    result = ReadEventReports(readData, &leftCount, &readCount, &isOverflow, GetMaxEventReportCount());
    EXPECT_TRUE(result.IsSuccess());

    // DeactivateSensing() を呼んだ時点での呼び出し（期待値 : 失敗）
    result = DeactivateSensing();
    EXPECT_TRUE(result.IsSuccess());
    EXPECT_DEATH_IF_SUPPORTED(ReadEventReports(readData, &leftCount, &readCount, &isOverflow, GetMaxEventReportCount()), "");

    // UnbindInterrupt() を呼んだ時点での呼び出し（期待値 : 失敗）
    result = UnbindInterrupt();
    EXPECT_TRUE(result.IsSuccess());
    EXPECT_DEATH_IF_SUPPORTED(ReadEventReports(readData, &leftCount, &readCount, &isOverflow, GetMaxEventReportCount()), "");

    // ResetDevice() を呼んだ時点での呼び出し（期待値 : 失敗）
    result = ResetDevice();
    EXPECT_TRUE(result.IsSuccess());
    EXPECT_DEATH_IF_SUPPORTED(ReadEventReports(readData, &leftCount, &readCount, &isOverflow, GetMaxEventReportCount()), "");

    // Finalize() を呼んだ時点での呼び出し（期待値 : 失敗）
    Finalize();
    EXPECT_DEATH_IF_SUPPORTED(ReadEventReports(readData, &leftCount, &readCount, &isOverflow, GetMaxEventReportCount()), "");
}

}}
