﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// ビルドスイッチのため、例外的に先頭でインクルードする
#include <nn/fs/detail/fs_ResultHandlingUtilitySuppressRecordingEventOnUnsupportedPlatforms.h>

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/fs.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>

#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h.>
#include <nnt/base/testBase_Exit.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/result/testResult_Assert.h>
#include <nn/fs/detail/fs_ResultHandlingUtility.h>

#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_ResultPrivate.h>
#include <nn/sdmmc/sdmmc_Result.h>

using namespace nn;
using namespace nn::fs;
using namespace nn::fs::detail;
using namespace nnt::fs::util;

namespace {

template <typename ResultType, typename Func>
void TestExecute(ResultType result, Func func)
{
    NN_UNUSED(result);
    NNT_ASSERT_RESULT_FAILURE(ResultType, func());
}



TEST(FsResultDo, Success)
{
    TestExecute(ResultHandledByAllProcess(), []() -> Result {
        NN_FS_RESULT_DO(ResultSuccess());
        return ResultHandledByAllProcess();
    });
}

TEST(FsResultDo, Public)
{
    TestExecute(ResultPathNotFound(), []() -> Result {
        NN_FS_RESULT_DO(ResultPathNotFound());
        NN_RESULT_SUCCESS;
    });
}

TEST(FsResultDoDeath, Private)
{
    auto testFunction = []() NN_NOEXCEPT -> Result
    {
        NN_FS_RESULT_DO(ResultPreconditionViolation());
        NN_RESULT_SUCCESS;
    };
    EXPECT_DEATH_IF_SUPPORTED(
        TestExecute(ResultPreconditionViolation(), testFunction),
        ""
    );
}

TEST(FsResultDoDeath, OtherComponent)
{
    auto testFunction = []() NN_NOEXCEPT -> Result
    {
        NN_FS_RESULT_DO(nn::sdmmc::ResultNoDevice());
        NN_RESULT_SUCCESS;
    };
    EXPECT_DEATH_IF_SUPPORTED(
        TestExecute(nn::sdmmc::ResultNoDevice(), testFunction),
        ""
    );
}

TEST(FsResultDoWithHandler, Success)
{
    TestExecute(ResultHandledByAllProcess(), []() -> Result {
        NN_FS_RESULT_DO_WITH_HANDLER(ResultSuccess(),
            [](Result r) -> Result {
                return r;
            }
        );
        return ResultHandledByAllProcess();
    });
}

TEST(FsResultDoWithHandler, Convert)
{
    TestExecute(ResultPathNotFound(), []() -> Result {
        NN_FS_RESULT_DO_WITH_HANDLER(ResultPathAlreadyExists(),
            [](Result r) -> Result {
                if( ResultPathAlreadyExists::Includes(r) )
                {
                    return ResultPathNotFound();
                }
                else
                {
                    return ResultUnexpected();
                }
            }
        );
        return ResultHandledByAllProcess();
    });
}


TEST(FsResultDoWithHandler, ThroughPublic)
{
    TestExecute(ResultPathNotFound(), []() -> Result {
        NN_FS_RESULT_DO_WITH_HANDLER(ResultPathNotFound(),
            [](Result r) -> Result {
                return r;
            }
        );
        return ResultHandledByAllProcess();
    });
}

TEST(FsResultDoWithHandlerDeath, ThroughPrivate)
{
    auto testFunction = []() NN_NOEXCEPT -> Result
    {
        NN_FS_RESULT_DO_WITH_HANDLER(
            ResultPreconditionViolation(),
            [](Result result) NN_NOEXCEPT -> Result
            {
                return result;
            }
        );
        return ResultHandledByAllProcess();
    };

    EXPECT_DEATH_IF_SUPPORTED(
        TestExecute(ResultPreconditionViolation(), testFunction),
        ""
    );
}



// テストユーティリティ

TEST(TestUtil, IsFilledWith32BitCount)
{
    const size_t BufferSize = 48;
    auto buffer = AllocateAlignedBuffer(BufferSize, 32);

    auto ValueOffset = 0x12345678 * 4;
    FillBufferWith32BitCount(buffer.get(), BufferSize, ValueOffset);
    int64_t borders[] = { 0, 1, 2, 3, 4, 7, 8, 11, 12, 13, 14, 15, 16 };

    for (auto start : borders)
    {
        for (auto last : borders)
        {
            if (start > last)
            {
                continue;
            }

            size_t size = static_cast<size_t>((last + 1) - start);

            NNT_FS_SCOPED_TRACE("%lld %lld %lld\n", start, last, 0ULL);

            EXPECT_TRUE(IsFilledWith32BitCount(buffer.get() + start, size, ValueOffset + start));
        }
    }
}


TEST(TestUtil, FillBufferWith32BitCount)
{
    const size_t BufferSize = 48;
    auto buffer = AllocateAlignedBuffer(BufferSize, 32);

    int64_t borders[] = {0, 1, 2, 3, 4,  7, 8,  11, 12, 13, 14, 15, 16};
    int64_t valueOffsets[] = {
        0x12345678, 0x12345679, 0x1234567a, 0x123456b,
        0x3FFFFFFF, 0x40000000, 0x40000001,
        0x003FFFFF, 0x00400000, 0x00400001,
        0x00003FFF, 0x00004000, 0x00004001,
        0x0000003F, 0x00000040, 0x00000041,
    };

    for (auto valueOffset : valueOffsets)
    {
        for (auto start : borders)
        {
            for (auto last : borders)
            {
                //NN_LOG("%lld %lld %lld\n", start, last, valueOffset);

                if (start > last)
                {
                    continue;
                }

                size_t size = static_cast<size_t>((last + 1) - start);

                NNT_FS_SCOPED_TRACE("%lld %lld %lld\n", start, last, valueOffset);

                FillBufferWith8BitCount(buffer.get(), BufferSize, 0xBA);
                EXPECT_TRUE(IsFilledWith8BitCount(buffer.get(), BufferSize, 0xBA));


                FillBufferWith32BitCount(buffer.get() + start, size, valueOffset);
                EXPECT_TRUE(IsFilledWith32BitCount(buffer.get() + start, size, valueOffset));


                FillBufferWith8BitCount(buffer.get() + start, size, 0xBA + start);
                EXPECT_TRUE(IsFilledWith8BitCount(buffer.get(), BufferSize, 0xBA));
            }

        }
    }
}


}

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    auto ret = RUN_ALL_TESTS();

    nnt::Exit(ret);
}
