﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <chrono>
#include <thread>
#include <cstring>

#include <nn/os.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/fs.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/nn_Windows.h>

#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/nnt_Argument.h>

#include <nn/fssystem/fs_SubdirectoryFileSystem.h>

using namespace nnt::fs::util;
using namespace nn::fs::fsa;
using namespace nn::fssystem;

namespace nn { namespace fs { namespace detail {


class SubdirectoryFileSystemTest : public SubdirectoryFileSystem
{

public:
    explicit SubdirectoryFileSystemTest(const char* pRootPath)
        : SubdirectoryFileSystem(nullptr, pRootPath)
    {
    }
    Result ResolveFullPathPublic(char* outPath, size_t outPathSize, const char* relativePath) const NN_NOEXCEPT
    {
        return ResolveFullPath(outPath, outPathSize, relativePath);
    }
};



TEST(SubdirectoryFileSystem, ResolvePath)
{
    char fullPath[EntryNameLengthMax + 1];

    // TODO: 末尾スラッシュの有無の仕様明確化・統一

    const char* TestCases[][3] = {
        // root, relative, normalized result

        {"/",            "/",                  "/"},
        {"/",            "/foo",               "/foo"},
        {"/",            "/foo/../",           "/"},
        {"/",            "/foo/bar/../",       "/foo"},
        {"/",            "/foo/bar/../../",    "/"},

        {"/foo",         "/",                  "/foo/"},
        {"/foo",         "/bar",               "/foo/bar"},
        {"/foo",         "/bar/",              "/foo/bar"},
        {"/foo/bar",     "/buz/../",           "/foo/bar/"},

        {"/foo/",        "/",                  "/foo/"},
        {"/foo/",        "/bar",               "/foo/bar"},
        {"/foo/bar/",    "/buz/../",           "/foo/bar/"},
    };

    for(auto testCase : TestCases)
    {
        NN_LOG("<%s> <%s> <%s>\n", testCase[0], testCase[1], testCase[2]);
        SubdirectoryFileSystemTest subdirFs(testCase[0]);
        NNT_EXPECT_RESULT_SUCCESS(subdirFs.ResolveFullPathPublic(fullPath, sizeof(fullPath), testCase[1]));
        EXPECT_STREQ(testCase[2], fullPath);
    }
}

TEST(SubdirectoryFileSystem, ResolvePathExcessRoot)
{
    char fullPath[EntryNameLengthMax + 1];

    const char* TestCases[][3] = {
        // root, relative
        {"/",            "/../"},
        {"/foo",         "/../"},
        {"/foo/bar",     "/../"},
        {"/foo/",        "/../"},
        {"/foo/bar/",    "/../"},
        {"/",            "/foo/../../"},
        {"/",            "/foo/bar/../../../"},
        {"/",            "/foo/bar/../buz/../../../"},
    };

    for(auto testCase : TestCases)
    {
        NN_LOG("<%s> <%s>\n", testCase[0], testCase[1]);
        SubdirectoryFileSystemTest subdirFs(testCase[0]);
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultDirectoryUnobtainable, subdirFs.ResolveFullPathPublic(fullPath, sizeof(fullPath), testCase[1])); // TODO: error code
    }
}



}}}


extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    nn::fs::SetAllocator(nnt::fs::util::Allocate, nnt::fs::util::Deallocate);

    ::testing::InitGoogleTest(&argc, argv);
    int result = RUN_ALL_TESTS();

    nnt::Exit(result);
}
