﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>
#include <cstring>
#include <random>

#include <nn/os.h>

#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/fs.h>
#include <nn/result/result_HandlingUtility.h>

#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/fsUtil/testFs_util.h>

#include <nn/fs/fs_MemoryStorage.h>
#include <nn/fssystem/fs_AlignmentMatchingStorage.h>
#include "fs_ReadOnlyBlockCacheStorage.h"
#include "fs_LruListCache.h"

#include "testFs_Unit_StorageLayerTestCase.h"


namespace nn { namespace fssrv { namespace detail {


using namespace nnt::fs::util;
using namespace nn::fssystem;


class ReadOnlyBlockCacheStorageTestBase : public ::testing::Test
{
protected:
    static const int CacheSize = 1 * 1024 * 1024;

    static const int BlockSize = 512 * 1024;
    static const int CacheCount = CacheSize / BlockSize;

protected:
    void SetUpStorage(nn::fs::IStorage* pStorage) NN_NOEXCEPT
    {
        m_CacheBuffer.reset(new char[CacheSize]);

        m_pCacheStorage.reset(new ReadOnlyBlockCacheStorage(pStorage, BlockSize, m_CacheBuffer.get(), CacheSize, CacheCount));
        m_pStorage.reset(new AlignmentMatchingStoragePooledBuffer<1>(m_pCacheStorage.get(), BlockSize));
    }

    nn::fs::IStorage* GetStorage() NN_NOEXCEPT
    {
        return m_pStorage.get();
    }

private:
    std::unique_ptr<char> m_CacheBuffer;
    std::unique_ptr<ReadOnlyBlockCacheStorage> m_pCacheStorage;
    std::unique_ptr<nn::fs::IStorage> m_pStorage;

};

class ReadOnlyBlockCacheStorageTest : public ReadOnlyBlockCacheStorageTestBase
{
protected:
    static const int StorageSize = 16 * 1024 * 1024;

protected:
    virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
    {
        m_pMemoryStorage.reset(new nnt::fs::util::AccessCountedMemoryStorage);
        m_pMemoryStorage->Initialize(StorageSize);
        FillBufferWith32BitCount(m_pMemoryStorage->GetBuffer(), StorageSize, 0);

        SetUpStorage(m_pMemoryStorage.get());
    }

    nnt::fs::util::AccessCountedMemoryStorage* GetMemoryStorage() NN_NOEXCEPT
    {
        return m_pMemoryStorage.get();
    }

private:
    std::unique_ptr<nnt::fs::util::AccessCountedMemoryStorage> m_pMemoryStorage;
};

class ReadOnlyBlockCacheStorageLargeSizeTest : public ReadOnlyBlockCacheStorageTestBase
{
};

TEST_F(ReadOnlyBlockCacheStorageTest, GetSize)
{
    TestGetSize(GetStorage(), StorageSize);
}

TEST_F(ReadOnlyBlockCacheStorageTest, StorageContent)
{
    TestStorageContent(GetStorage());
    TestStorageContent(GetStorage(), BlockSize);
    TestStorageContent(GetStorage(), BlockSize * 2);
    TestStorageContent(GetStorage(), BlockSize / 2);
}

TEST_F(ReadOnlyBlockCacheStorageTest, ReadTwice)
{
    std::unique_ptr<char[]> buffer(new char[BlockSize]);
    std::unique_ptr<char[]> buffer2(new char[BlockSize]);

    NNT_EXPECT_RESULT_SUCCESS(GetStorage()->Read(0, buffer.get(),  BlockSize));
    NNT_EXPECT_RESULT_SUCCESS(GetStorage()->Read(0, buffer2.get(), BlockSize));

    NNT_FS_UTIL_EXPECT_MEMCMPEQ(buffer.get(), buffer2.get(), BlockSize);
}


TEST_F(ReadOnlyBlockCacheStorageTest, ReadRandomRangeStress)
{
    TestRandomRangeRead(GetStorage(), 1000);
    //TestRandomRangeRead(GetStorage(), -1);
}

TEST_F(ReadOnlyBlockCacheStorageTest, Invalidate)
{
    char buffer[1] = {};

    // 下位ストレージからデータを読み込む
    NNT_ASSERT_RESULT_SUCCESS(GetStorage()->Read(0, buffer, 1));

    // 同じ個所を読み込んでもキャッシュされているので回ストレージへのアクセスは発生しないはず
    GetMemoryStorage()->ResetAccessCounter();
    NNT_ASSERT_RESULT_SUCCESS(GetStorage()->Read(0, buffer, 1));
    EXPECT_EQ(0, GetMemoryStorage()->GetReadTimes());

    // キャッシュを破棄してから読むと再び下位ストレージへのアクセスが発生するはず
    GetMemoryStorage()->ResetAccessCounter();
    NNT_EXPECT_RESULT_SUCCESS(GetStorage()->OperateRange(
        nn::fs::OperationId::Invalidate,
        0,
        1));
    NNT_ASSERT_RESULT_SUCCESS(GetStorage()->Read(0, buffer, 1));
    EXPECT_GT(GetMemoryStorage()->GetInvalidateTimes(), 0);
    EXPECT_GT(GetMemoryStorage()->GetReadTimes(), 0);
}

TEST_F(ReadOnlyBlockCacheStorageLargeSizeTest, Read)
{
    const size_t bufferSize = BlockSize * 2;

    nnt::fs::util::VirtualMemoryStorage baseStorage;
    baseStorage.Initialize(static_cast<int64_t>(64) * 1024 * 1024 * 1024 + bufferSize);
    NN_UTIL_SCOPE_EXIT
    {
        baseStorage.Finalize();
    };

    const int64_t offsetList[] =
    {
        0,
        static_cast<int64_t>(4) * 1024 * 1024 * 1024,
        static_cast<int64_t>(8) * 1024 * 1024 * 1024,
        static_cast<int64_t>(16) * 1024 * 1024 * 1024,
        static_cast<int64_t>(32) * 1024 * 1024 * 1024,
        static_cast<int64_t>(64) * 1024 * 1024 * 1024,
    };
    const auto offsetListLength = sizeof(offsetList) / sizeof(int64_t);

    std::unique_ptr<char> readBuffer(new char[bufferSize]);
    std::unique_ptr<char> writeBufferArray[offsetListLength];

    for( int i = 0; i < offsetListLength; ++i )
    {
        const int64_t offset = offsetList[i];
        writeBufferArray[i].reset(new char[bufferSize]);
        FillBufferWithRandomValue(writeBufferArray[i].get(), bufferSize);
        NNT_ASSERT_RESULT_SUCCESS(baseStorage.Write(offset, writeBufferArray[i].get(), bufferSize));
    }

    SetUpStorage(&baseStorage);

    for( int i = 0; i < offsetListLength; ++i )
    {
        const int64_t offset = offsetList[i];
        NNT_ASSERT_RESULT_SUCCESS(GetStorage()->Read(offset, readBuffer.get(), bufferSize));
        NNT_FS_UTIL_EXPECT_MEMCMPEQ(writeBufferArray[i].get(), readBuffer.get(), bufferSize);
    }
}

TEST(LruListCache, Mru)
{
    typedef int64_t Key;
    typedef char*   Value;
    typedef LruListCache<Key, Value> BlockCache;
    typedef BlockCache::Node Node;

    Value const base = nullptr;

    {
        // 入れた順に LRU である
        BlockCache cache;

        cache.PushMruNode(std::move(std::unique_ptr<Node>(new Node(base + 1))), 10);
        cache.PushMruNode(std::move(std::unique_ptr<Node>(new Node(base + 2))), 20);
        cache.PushMruNode(std::move(std::unique_ptr<Node>(new Node(base + 3))), 30);

        EXPECT_EQ(base + 1, cache.PopLruNode()->value);
        EXPECT_EQ(base + 2, cache.PopLruNode()->value);
        EXPECT_EQ(base + 3, cache.PopLruNode()->value);
    }

    {
        // FindValueAndUpdateMru で参照できる
        BlockCache cache;

        cache.PushMruNode(std::move(std::unique_ptr<Node>(new Node(base + 1))), 10);
        cache.PushMruNode(std::move(std::unique_ptr<Node>(new Node(base + 2))), 20);
        cache.PushMruNode(std::move(std::unique_ptr<Node>(new Node(base + 3))), 30);

        Value value;
        EXPECT_TRUE(cache.FindValueAndUpdateMru(&value, 20));
        EXPECT_EQ(value, base + 2);
        EXPECT_TRUE(cache.FindValueAndUpdateMru(&value, 30));
        EXPECT_EQ(value, base + 3);
        EXPECT_TRUE(cache.FindValueAndUpdateMru(&value, 10));
        EXPECT_EQ(value, base + 1);
    }

    {
        // FindValueAndUpdateMru で参照すると MRU になる
        BlockCache cache;

        cache.PushMruNode(std::move(std::unique_ptr<Node>(new Node(base + 1))), 10);
        cache.PushMruNode(std::move(std::unique_ptr<Node>(new Node(base + 2))), 20);
        cache.PushMruNode(std::move(std::unique_ptr<Node>(new Node(base + 3))), 30);

        Value value;
        EXPECT_TRUE(cache.FindValueAndUpdateMru(&value, 20));

        EXPECT_EQ(base + 1, cache.PopLruNode()->value);
        EXPECT_EQ(base + 3, cache.PopLruNode()->value);
        EXPECT_EQ(base + 2, cache.PopLruNode()->value);
    }

}

}}}

