﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>
#include <cstring>

#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/os.h>
#include <nn/fs.h>
#include <nn/fs/fs_SubStorage.h>
#include <nn/fs/fs_MemoryStorage.h>
#include <nn/fs/fs_FileStorage.h>
#include <nn/fs/fsa/fs_IFile.h>
#include <nn/fssystem/fs_AesXtsStorage.h>
#include <nn/fssystem/fs_AlignmentMatchingStorage.h>
#include <nn/fssystem/fs_Utility.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/crypto/crypto_Aes128XtsEncryptor.h>
#include <nn/crypto/crypto_Aes128XtsDecryptor.h>

#include "fs_HierarchicalSha256Storage.h"

#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/fsUtil/testFs_util.h>

#include "testFs_Unit_StorageLayerTestCase.h"

using namespace nn::fssystem;
using namespace nn::crypto;
using namespace nnt::fs::util;

namespace
{

class AesXtsStorageTestBase : public ::testing::Test
{
public:
    static const int XtsBlockSize = 512;

protected:
    static const int AesWorkBufferSize = 512 * 1024;

protected:
    void SetUpStorage(nn::fs::IStorage* pBaseStorage, bool isDefaultEncryptionEnabled) NN_NOEXCEPT
    {
        memset(m_Key[0], 0x00, AesXtsStorage::KeySize);
        memset(m_Key[1], 0x01, AesXtsStorage::KeySize);
        memset(m_Iv, 0x11, sizeof(m_Iv));

        // ストレージ元のバッファをあらかじめ暗号化
        if( isDefaultEncryptionEnabled )
        {
            char counter[AesXtsStorage::IvSize];
            memcpy(counter, m_Iv, AesXtsStorage::IvSize);

            int64_t baseStorageSize = 0;
            NNT_ASSERT_RESULT_SUCCESS(pBaseStorage->GetSize(&baseStorageSize));
            NN_ABORT_UNLESS(nn::util::is_aligned(baseStorageSize, XtsBlockSize));

            auto buffer = nnt::fs::util::AllocateBuffer(XtsBlockSize);
            for( int64_t offset = 0; offset < baseStorageSize; offset += XtsBlockSize )
            {
                NNT_ASSERT_RESULT_SUCCESS(pBaseStorage->Read(offset, buffer.get(), XtsBlockSize));
                const auto decryptedSize = EncryptAes128Xts(
                    buffer.get(),
                    XtsBlockSize,
                    m_Key[0],
                    m_Key[1],
                    AesXtsStorage::KeySize,
                    counter,
                    AesXtsStorage::IvSize,
                    buffer.get(),
                    XtsBlockSize);
                ASSERT_EQ(decryptedSize, XtsBlockSize);
                NNT_ASSERT_RESULT_SUCCESS(pBaseStorage->Write(offset, buffer.get(), XtsBlockSize));
                AddCounter(counter, AesXtsStorage::IvSize, 1);
            }
        }

        m_pCheckStorage.reset(
            new WriteSizeCheckStorage(
                pBaseStorage,
                XtsBlockSize,
                nn::fssystem::PooledBuffer::GetAllocatableSizeMax(),
                [this](nn::fs::IStorage* pBaseStorage) NN_NOEXCEPT
                {
                    std::unique_ptr<nn::fs::IStorage> pAesStorage(new AesXtsStorage(pBaseStorage, m_Key[0], m_Key[1], AesXtsStorage::KeySize, m_Iv, AesXtsStorage::IvSize, XtsBlockSize));
                    return pAesStorage;
                }
            )
        );
        m_pStorage.reset(new AlignmentMatchingStorage<AesXtsStorage::KeySize, 4>(m_pCheckStorage.get()));
    }

    nn::fs::IStorage* GetStorage() NN_NOEXCEPT
    {
        return m_pStorage.get();
    }

private:
    std::unique_ptr<nnt::fs::util::WriteSizeCheckStorage> m_pCheckStorage;
    std::unique_ptr<nn::fs::IStorage> m_pStorage;

    char m_Key[2][AesXtsStorage::KeySize];
    char m_Iv[AesXtsStorage::IvSize];
};

class AesXtsStorageTest : public AesXtsStorageTestBase
{
protected:
    static const int StorageSize = 16 * 1024 * 1024;

protected:
    virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
    {
        m_pMemoryStorage.reset(new nnt::fs::util::AccessCountedMemoryStorage);
        m_pMemoryStorage->Initialize(StorageSize);

        FillBufferWith32BitCount(m_pMemoryStorage->GetBuffer(), StorageSize, 0);

        SetUpStorage(m_pMemoryStorage.get(), true);
    }

    nnt::fs::util::AccessCountedMemoryStorage* GetMemoryStorage() NN_NOEXCEPT
    {
        return m_pMemoryStorage.get();
    }

private:
    std::unique_ptr<nnt::fs::util::AccessCountedMemoryStorage> m_pMemoryStorage;

};

class AesXtsStorageLargeSizeTest : public AesXtsStorageTestBase
{
protected:
    static const auto BufferSize = XtsBlockSize;
    static const int64_t StorageSize = static_cast<int64_t>(64) * 1024 * 1024 * 1024 + BufferSize;

protected:
    virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
    {
        m_BaseStorage.Initialize(StorageSize);

        SetUpStorage(&m_BaseStorage, false);
    }

    virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
    {
        m_BaseStorage.Finalize();
    }

private:
    nnt::fs::util::VirtualMemoryStorage m_BaseStorage;
};

TEST_F(AesXtsStorageTest, GetSize)
{
    TestGetSize(GetStorage(), StorageSize);
}
TEST_F(AesXtsStorageTest, StorageContent)
{
    TestStorageContent(GetStorage());
    TestStorageContent(GetStorage(), XtsBlockSize);
    TestStorageContent(GetStorage(), XtsBlockSize * 2);
    TestStorageContent(GetStorage(), XtsBlockSize + AesXtsStorage::KeySize);
    TestStorageContent(GetStorage(), AesXtsStorage::KeySize);
    TestStorageContent(GetStorage(), AesXtsStorage::KeySize * 2);
}

TEST_F(AesXtsStorageTest, BoundaryOffsetSize)
{
    TestBoundaryOffsetSize(GetStorage(), 1);
}
TEST_F(AesXtsStorageTest, WriteRead)
{
    TestWriteReadStorage(GetStorage(), XtsBlockSize);
    TestWriteReadStorage(GetStorage(), XtsBlockSize * 2);
    TestWriteReadStorage(GetStorage(), XtsBlockSize + AesXtsStorage::KeySize);
    TestWriteReadStorage(GetStorage(), AesXtsStorage::KeySize);
    TestWriteReadStorage(GetStorage(), AesXtsStorage::KeySize * 2);
    TestWriteReadStorage(GetStorage(), 1024 * 1024);
}

TEST_F(AesXtsStorageTest, ConcurrentWriteRead)
{
    TestConcurrentWriteRead(GetStorage());
}

TEST_F(AesXtsStorageTest, Invalidate)
{
    GetMemoryStorage()->ResetAccessCounter();
    NNT_EXPECT_RESULT_SUCCESS(GetStorage()->OperateRange(
        nn::fs::OperationId::Invalidate,
        0,
        StorageSize));
    EXPECT_GT(GetMemoryStorage()->GetInvalidateTimes(), 0);
}



namespace
{
void SetValue(char* pCounter, size_t counterSize, uint64_t high, uint64_t low)
{
    NN_ASSERT(counterSize == AesXtsStorage::IvSize);

    for( int i = 0; i < sizeof(high); ++i )
    {
        pCounter[i] = (high >> (56 - (i * 8))) & 0xFF;
    }
    for( int i = 0; i < sizeof(low); ++i )
    {
        pCounter[8 + i] = (low >> (56 - (i * 8))) & 0xFF;
    }
}
}

TEST(AesXtsStorage, AddCounter)
{
    struct TestCase
    {
        uint64_t baseHigh;
        uint64_t baseLow;

        uint64_t additionalValue;

        uint64_t expectedHigh;
        uint64_t expectedLow;
    };

    const TestCase testCases[] = {
        {0x0000000000000000ULL, 0x0000000000000001ULL,                  1ULL, 0x0000000000000000ULL, 0x0000000000000002ULL},
        {0x0000000000000000ULL, 0x0000000000000001ULL,               0xFFULL, 0x0000000000000000ULL, 0x0000000000000100ULL},
        {0x0000000000000000ULL, 0x0000000000000100ULL,             0xFF00ULL, 0x0000000000000000ULL, 0x0000000000010000ULL},

        {0x0000000000000000ULL, 0x8000000000000008ULL, 0x8000000000000008ULL, 0x0000000000000001ULL, 0x0000000000000010ULL},

        {0x0000000000000000ULL, 0x8000000000000000ULL, 0x8000000000000000ULL, 0x0000000000000001ULL, 0x0000000000000000ULL},
        {0x0000000000000000ULL, 0xFFFFFFFFFFFFFFFFULL,                  1ULL, 0x0000000000000001ULL, 0x0000000000000000ULL},
        {0x0000000000000000ULL, 0x0000000000000001ULL, 0xFFFFFFFFFFFFFFFFULL, 0x0000000000000001ULL, 0x0000000000000000ULL},
    };

    for( auto t : testCases )
    {
        char counter[AesXtsStorage::IvSize];
        SetValue(counter, sizeof(counter), t.baseHigh, t.baseLow);

        AddCounter(counter, AesXtsStorage::IvSize, t.additionalValue);

        char expectedCounter[AesXtsStorage::IvSize];
        SetValue(expectedCounter, sizeof(expectedCounter), t.expectedHigh, t.expectedLow);

        NNT_FS_UTIL_EXPECT_MEMCMPEQ(expectedCounter, counter, sizeof(counter));
    }

}

TEST_F(AesXtsStorageLargeSizeTest, ReadWrite)
{
    TestWriteReadStorageWithLargeOffset(GetStorage(), BufferSize);
}

}
