﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/fssystem/fs_SpeedEmulationConfiguration.h>
#include <nn/fssystem/fs_StorageLayoutTypeSetter.h>
#include <nn/fssrv/fssrv_SpeedEmulationStorage.h>

#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/fsUtil/testFs_util.h>

class FsSpeedEmulationStorageTest : public ::testing::Test
{
protected:
    virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
    {
        m_DefaultMode = nn::fssystem::SpeedEmulationConfiguration::GetSpeedEmulationMode();
    }

    virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
    {
        nn::fssystem::SpeedEmulationConfiguration::SetSpeedEmulationMode(m_DefaultMode);
    }

private:
    nn::fs::SpeedEmulationMode m_DefaultMode;
};

// QueryRange のテスト
TEST_F(FsSpeedEmulationStorageTest, TestQueryRange)
{
    std::shared_ptr<nnt::fs::util::SafeMemoryStorage> baseStorage(
        new nnt::fs::util::SafeMemoryStorage());
    ASSERT_NE(nullptr, baseStorage);
    static const size_t StorageSize = 8 * 1024 * 1024;
    baseStorage->Initialize(StorageSize);

    nn::fssrv::SpeedEmulationStorage storage(baseStorage.get());

    const nn::fs::SpeedEmulationMode speedMulationModes[] = {
        nn::fs::SpeedEmulationMode::None,
        nn::fs::SpeedEmulationMode::Slower,
        nn::fs::SpeedEmulationMode::Faster,
        nn::fs::SpeedEmulationMode::Random,
    };

    for( auto mode : speedMulationModes )
    {
        nn::fssystem::SpeedEmulationConfiguration::SetSpeedEmulationMode(mode);

        nn::fs::QueryRangeInfo info;

        NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultNullptrArgument,
            storage.OperateRange(nullptr, sizeof(info), nn::fs::OperationId::QueryRange, 0, StorageSize, nullptr, 0));
        NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultInvalidSize,
            storage.OperateRange(&info, 0, nn::fs::OperationId::QueryRange, 0, StorageSize, nullptr, 0));

        NNT_ASSERT_RESULT_SUCCESS(storage.OperateRange(
            &info, sizeof(info), nn::fs::OperationId::QueryRange, 0, StorageSize, nullptr, 0));

        int32_t expectSpeedEmulationTypeFlag = 0;
        if( mode != nn::fs::SpeedEmulationMode::None )
        {
            expectSpeedEmulationTypeFlag = static_cast<int32_t>(nn::fs::SpeedEmulationTypeFlag::StorageSpeedEmulationEnabled);
        }
        EXPECT_EQ(expectSpeedEmulationTypeFlag, info.speedEmulationTypeFlag);
        EXPECT_EQ(0, info.aesCtrKeyTypeFlag);
    }
}

// 4 GB を超えるオフセットのテスト
TEST_F(FsSpeedEmulationStorageTest, TestReadWriteLargeOffset)
{
    nn::fssystem::ServiceContext context;
    nn::fssystem::RegisterServiceContext(&context);
    NN_UTIL_SCOPE_EXIT
    {
        nn::fssystem::UnregisterServiceContext();
    };

    nn::fssystem::ScopedStorageLayoutTypeSetter scopedContext(nn::fssystem::StorageFlag_Mmc);

    static const size_t AccessSize = 1024;
    nnt::fs::util::VirtualMemoryStorage baseStorage(
        nnt::fs::util::LargeOffsetMax + static_cast<int64_t>(AccessSize));

    nn::fssrv::SpeedEmulationStorage storage(&baseStorage);

    const nn::fs::SpeedEmulationMode speedMulationModes[] = {
        nn::fs::SpeedEmulationMode::None,
        nn::fs::SpeedEmulationMode::Slower,
        nn::fs::SpeedEmulationMode::Faster,
        nn::fs::SpeedEmulationMode::Random,
    };

    for( auto mode : speedMulationModes )
    {
        nn::fssystem::SpeedEmulationConfiguration::SetSpeedEmulationMode(mode);
        nnt::fs::util::TestStorageAccessWithLargeOffset(&storage, AccessSize);
    }
}

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    auto ret = RUN_ALL_TESTS();

    nnt::Exit(ret);
}
