﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <random>
#include <nn/os/os_Thread.h>
#include <nn/fssystem/save/fs_SaveDataFileSystemCore.h>
#include <nn/fssystem/save/fs_BlockCacheBufferedStorage.h>
#include "testFs_util_CommonStorageTests.h"
#include "testFs_Unit_SaveDataFileSystemDriverLayerTestCase.h"

class TestFileSystemSetup
{
public:
    TestFileSystemSetup() NN_NOEXCEPT
        : m_pFileSystem(nullptr)
    {
    }

    virtual ~TestFileSystemSetup() NN_NOEXCEPT
    {
    }

    //! アーカイブオブジェクトを取得します。
    nn::fssystem::save::IFileSystem* GetFileSystem() NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(m_pFileSystem);
        return m_pFileSystem;
    }

    //! ファイルオブジェクトを設定します。
    void SetFileSystem(nn::fssystem::save::IFileSystem* pFileSystem) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pFileSystem);
        m_pFileSystem = pFileSystem;
    }

    //! アーカイブを初期化します。
    virtual void Initialize(int64_t offset, size_t sizeBlock) = 0;

    //! アーカイブを破棄します。
    virtual void Finalize() = 0;

    //! アーカイブへのアクセスはスレッドセーフかどうか
    virtual bool IsEnableThreadSafeAccess() = 0;

    //! アーカイブへの排他制御は有効かどうか
    virtual bool IsEnableExclusiveAccess() = 0;

    //! ファイルのサイズ変更が可能かどうか
    virtual bool IsEnableResizeFile() = 0;

    //!< ファイルへのデータ書き込み時に自動伸長するかどうか
    virtual bool IsAutoExtendSizeOnWrite() const NN_NOEXCEPT
    {
        return true;
    }

protected:
    Random m_Random;

private:
    nn::fssystem::save::IFileSystem* m_pFileSystem;
};

/*!
    @brief IFileSystem に対する共通テストを実行します。

    tests::FileSystemTest::Test(new MyArchiveTestSetup(pFileSystem, 0, 0, 0, 0));
    のように使用します。
*/
class FileSystemTest
{
public:
    //! ファイル、ディレクトリが作成可能なアーカイブに対して
    //! ディレクトリの作成と削除テストを行います。
    static void TestDirectory(nn::fssystem::save::IFileSystem* pFileSystem) NN_NOEXCEPT;

    //! ファイル、ディレクトリが作成可能なアーカイブに対して
    //! ファイルの作成、削除テストを行います。
    static void TestFile(
                    nn::fssystem::save::IFileSystem* pFileSystem,
                    bool isEnableResizeFile,
                    bool isAutoExtendSizeOnWrite
                ) NN_NOEXCEPT;

    //! ファイル、ディレクトリが作成可能なアーカイブに対して
    //! ファイル、ディレクトリの作成と削除テストを行います。
    static void TestCreation(nn::fssystem::save::IFileSystem* pFileSystem) NN_NOEXCEPT;

    //! ファイル、ディレクトリが作成可能なアーカイブに対して
    //! ファイル、ディレクトリのアクセス制御テストを行います。
    static void TestOpenClose(
        nn::fssystem::save::IFileSystem* pFileSystem, bool isEnableExclusiveAccess) NN_NOEXCEPT;

#if 0
    //! ファイル、ディレクトリが作成可能なアーカイブに対して
    //! 複数のスレッドからアーカイブに同時アクセステストを行います。
    static void TestThread(
        nn::fssystem::save::IFileSystem* pFileSystem, bool isEnableResizeFile) NN_NOEXCEPT;
#endif

    //! リネーム機能のテストを行ないます。
    static void TestRename(nn::fssystem::save::IFileSystem* pFileSystem) NN_NOEXCEPT;

    //! Ensure系機能のテストを行ないます。
    static void TestEnsure(nn::fssystem::save::IFileSystem* pFileSystem) NN_NOEXCEPT;

public:
    //! アーカイブテストを実行します。
    template <class TestSetup>
    static void RunTest() NN_NOEXCEPT
    {
        std::mt19937 mt(nnt::fs::util::GetRandomSeed());
        TestSetup tests;

        for( uint32_t loop = 0; loop < TestLoop; ++loop )
        {
            if( loop > 10 )
            {
                if( (loop + 1) % (TestLoop / 2) == 0 )
                {
                    NN_SDK_LOG(".");
                }
            }
            else
            {
                NN_SDK_LOG(".");
            }

            int64_t offset = std::uniform_int_distribution<>(0, 63)(mt);
            size_t sizeBlock = GenerateRandomBlockSize(512, MaxBlockSize);

            tests.Initialize(offset, sizeBlock);
            TestDirectory(tests.GetFileSystem());
            tests.Finalize();

            tests.Initialize(offset, sizeBlock);
            TestFile(tests.GetFileSystem(), tests.IsEnableResizeFile(), tests.IsAutoExtendSizeOnWrite());
            tests.Finalize();

            tests.Initialize(offset, sizeBlock);
            TestCreation(tests.GetFileSystem());
            tests.Finalize();

            tests.Initialize(offset, sizeBlock);
            TestOpenClose(tests.GetFileSystem(), tests.IsEnableExclusiveAccess());
            tests.Finalize();

#if 0
            if (tests.IsEnableExclusiveAccess() && tests.IsEnableThreadSafeAccess())
            {
                tests.Initialize(offset, sizeBlock);
                TestThread(tests.GetFileSystem(), tests.IsEnableResizeFile());
                tests.Finalize();
            }
#endif

            tests.Initialize(offset, sizeBlock);
            TestRename(tests.GetFileSystem());
            tests.Finalize();

            tests.Initialize(offset, sizeBlock);
            TestEnsure(tests.GetFileSystem());
            tests.Finalize();
        }

        NN_SDK_LOG("\n");
    }

private:
    static const uint32_t TestLoop = 3;
};

template<bool IsLogEnabled>
class MeasurableBufferManager : public nn::fssystem::IBufferManager
{
public:
    explicit MeasurableBufferManager(IBufferManager* pBaseBufferManager) NN_NOEXCEPT
        : m_pBaseBufferManager(pBaseBufferManager),
            m_TotalAllocatableSizeMin(std::numeric_limits<size_t>::max())
    {
        NN_SDK_REQUIRES_NOT_NULL(m_pBaseBufferManager);
        m_IsBufferStarved = false;
    }

    virtual const std::pair<uintptr_t, size_t> DoAllocateBuffer(
                                                    size_t size,
                                                    const BufferAttribute& bufAttr
                                                ) NN_NOEXCEPT NN_OVERRIDE
    {
        auto allocated = m_pBaseBufferManager->AllocateBuffer(size, bufAttr);
        auto currentSize = GetTotalAllocatableSize();
        if( m_TotalAllocatableSizeMin > currentSize )
        {
            m_TotalAllocatableSizeMin = currentSize;
        }
        if( NN_STATIC_CONDITION(IsLogEnabled) )
        {
            NN_LOG("+ %016p %u free %u allocatable %u\n",
                    allocated.first,
                    static_cast<uint32_t>(allocated.second),
                    static_cast<uint32_t>(GetFreeSize()),
                    static_cast<uint32_t>(GetTotalAllocatableSize()));
        }
        else
        {
            nn::os::YieldThread();
        }
        if( allocated.first == 0 )
        {
            m_IsBufferStarved = true;
        }
        return allocated;
    }

    virtual void DoDeallocateBuffer(uintptr_t address, size_t size) NN_NOEXCEPT NN_OVERRIDE
    {
        if( NN_STATIC_CONDITION(IsLogEnabled) )
        {
            NN_LOG("- %016p %u free %u allocatable %u\n",
                    address,
                    static_cast<uint32_t>(size),
                    static_cast<uint32_t>(GetFreeSize()),
                    static_cast<uint32_t>(GetTotalAllocatableSize()));
        }
        m_pBaseBufferManager->DeallocateBuffer(address, size);
    }

    virtual CacheHandle DoRegisterCache(
                            uintptr_t address,
                            size_t size,
                            const BufferAttribute& bufAttr
                        ) NN_NOEXCEPT NN_OVERRIDE
    {
        return m_pBaseBufferManager->RegisterCache(address, size, bufAttr);
    }

    virtual const std::pair<uintptr_t, size_t> DoAcquireCache(
                                                    CacheHandle handle
                                                ) NN_NOEXCEPT NN_OVERRIDE
    {
        return m_pBaseBufferManager->AcquireCache(handle);
    }

    virtual size_t DoGetTotalSize() const NN_NOEXCEPT NN_OVERRIDE
    {
        return m_pBaseBufferManager->GetTotalSize();
    }

    virtual size_t DoGetFreeSize() const NN_NOEXCEPT NN_OVERRIDE
    {
        return m_pBaseBufferManager->GetFreeSize();
    }

    virtual size_t DoGetTotalAllocatableSize() const NN_NOEXCEPT NN_OVERRIDE
    {
        return m_pBaseBufferManager->GetTotalAllocatableSize();
    }

    virtual size_t DoGetFreeSizePeak() const NN_NOEXCEPT NN_OVERRIDE
    {
        return 0;
    }

    virtual size_t DoGetTotalAllocatableSizePeak() const NN_NOEXCEPT NN_OVERRIDE
    {
        return 0;
    }

    virtual size_t DoGetRetriedCount() const NN_NOEXCEPT NN_OVERRIDE
    {
        return 0;
    }

    virtual void DoClearPeak() NN_NOEXCEPT NN_OVERRIDE
    {
    }

    size_t GetTotalAllocatableSizeMin() const NN_NOEXCEPT
    {
        return m_TotalAllocatableSizeMin;
    }

    bool IsBufferStarved() const NN_NOEXCEPT
    {
        return m_IsBufferStarved;
    }

private:
    IBufferManager* m_pBaseBufferManager;
    std::atomic_size_t m_TotalAllocatableSizeMin;
    std::atomic_bool m_IsBufferStarved;
};

// 4 GB を超えるオフセットでファイルにアクセスする共通テスト
void TestLargeOffsetAccess(nn::fssystem::save::IFileSystem* pFileSystem, const char* filePath) NN_NOEXCEPT;
